<?php
defined( 'ABSPATH' ) || exit;

/**
 * Quote Request Actions.
 * 
 * @class QTS_Meta_Box_Quote_Request_Actions
 * @package Class
 */
class QTS_Meta_Box_Quote_Request_Actions extends WC_Meta_Box_Order_Actions {

	/**
	 * Output the metabox.
	 *
	 * @param WP_Post $post Post object.
	 */
	public static function output( $post ) {
		global $thequote_request;

		if ( ! is_object( $thequote_request ) ) {
			$thequote_request = _qts_get_quote_request( $post->ID );
		}

		$quote_request = $thequote_request;

		/**
		 * Get quote request actions.
		 * 
		 * @since 1.0
		 */
		$quote_request_actions = apply_filters( 'qts_quote_request_actions', array(
			'resend_invoice_order' => __( 'Resend invoice order notification', 'quote-request-for-woocommerce' ),
				) );
		?>
		<ul class="order_actions submitbox">

			<?php
			/**
			 * Trigger after quote request actions start.
			 * 
			 * @param int $post_id
			 * @since 1.0
			 */
			do_action( 'qts_quote_request_actions_start', $post->ID );
			?>

			<?php if ( $quote_request->needs_payment() ) { ?>
				<li class="wide" id="actions">
					<select name="wc_order_action">
						<option value=""><?php esc_html_e( 'Choose an action...', 'quote-request-for-woocommerce' ); ?></option>
						<?php foreach ( $quote_request_actions as $action => $title ) { ?>
							<option value="<?php echo esc_attr( $action ); ?>"><?php echo esc_html( $title ); ?></option>
						<?php } ?>
					</select>
					<button class="button wc-reload"><span><?php esc_html_e( 'Apply', 'quote-request-for-woocommerce' ); ?></span></button>
				</li>
			<?php } ?>

			<li class="wide">
				<div id="delete-action">
					<?php
					if ( current_user_can( 'delete_post', $post->ID ) ) {
						if ( ! EMPTY_TRASH_DAYS ) {
							$delete_text = __( 'Delete permanently', 'quote-request-for-woocommerce' );
						} else {
							$delete_text = __( 'Move to Trash', 'quote-request-for-woocommerce' );
						}
						?>
						<a class="submitdelete deletion" href="<?php echo esc_url( get_delete_post_link( $post->ID ) ); ?>"><?php echo esc_html( $delete_text ); ?></a>
						<?php
					}
					?>
				</div>
				<button type="submit" class="button save_order button-primary" name="save" value="<?php echo 'auto-draft' === $post->post_status ? esc_attr__( 'Create', 'quote-request-for-woocommerce' ) : esc_attr__( 'Update', 'quote-request-for-woocommerce' ); ?>"><?php echo 'auto-draft' === $post->post_status ? esc_html__( 'Create', 'quote-request-for-woocommerce' ) : esc_html__( 'Update', 'quote-request-for-woocommerce' ); ?></button>
			</li>

			<?php
			/**
			 * Trigger after quote request actions end.
			 * 
			 * @param int $post_id
			 * @since 1.0
			 */
			do_action( 'qts_quote_request_actions_end', $post->ID );
			?>

		</ul>
		<?php
	}

	/**
	 * Save meta box data.
	 *
	 * @param int     $post_id Post ID.
	 * @param WP_Post $post Post Object.
	 */
	public static function save( $post_id, $post ) {
		if ( 'qts_quote_request' !== $post->post_type || empty( $_POST[ 'woocommerce_meta_nonce' ] ) || ! wp_verify_nonce( sanitize_key( $_POST[ 'woocommerce_meta_nonce' ] ), 'woocommerce_save_data' ) ) {
			return;
		}

		// Get quote request object.
		$quote_request = _qts_get_quote_request( $post_id );

		// Handle button actions.
		if ( ! empty( $_POST[ 'wc_order_action' ] ) ) {
			$action = wc_clean( wp_unslash( $_POST[ 'wc_order_action' ] ) );

			if ( 'resend_invoice_order' === $action ) {
				// Send the customer invoice email.
				WC()->payment_gateways();
				WC()->shipping();
				$email = _qts()->mailer->get_email( 'customer_accepted_quote_to_pay' );

				if ( $email ) {
					/**
					 * Before resend quote request invoice order emails.
					 * 
					 * @param QTS_Quote_Request $quote_request
					 * @since 1.0
					 */
					do_action( 'qts_before_resend_quote_request_invoice_order_emails', $quote_request );

					$email->trigger( $quote_request );

					/**
					 * Resend quote request invoice order emails.
					 * 
					 * @param QTS_Quote_Request $quote_request
					 * @since 1.0
					 */
					do_action( 'qts_after_resend_quote_request_invoice_order_emails', $quote_request );

					// Note the event.
					if ( did_action( 'qts_email_sent', $email ) ) {
						$quote_request->add_order_note( __( 'Quote Request invoice order manually sent to customer.', 'quote-request-for-woocommerce' ), false, true );

						// Change the post saved message.
						add_filter( 'redirect_post_location', array( __CLASS__, 'set_email_sent_message' ) );
					} else {
						// Change the post saved message.
						add_filter( 'redirect_post_location', array( __CLASS__, 'set_email_sent_failed_message' ) );
					}
				}
			} else {
				if ( ! did_action( 'qts_quote_request_action_' . sanitize_title( $action ) ) ) {
					/**
					 * Triggers quote request action.
					 * 
					 * @param QTS_Quote_Request $quote_request
					 * @since 1.0
					 */
					do_action( 'qts_quote_request_action_' . sanitize_title( $action ), $quote_request );
				}
			}
		}
	}

	/**
	 * Set the correct message ID.
	 *
	 * @param string $location Location.
	 * @return string
	 */
	public static function set_email_sent_message( $location ) {
		return add_query_arg( 'message', 11, $location );
	}

	/**
	 * Set the correct message ID.
	 *
	 * @param string $location Location.
	 * @return string
	 */
	public static function set_email_sent_failed_message( $location ) {
		return add_query_arg( 'message', 12, $location );
	}

}
