<?php

defined( 'ABSPATH' ) || exit;

/**
 * Handle our Form Fields by simply inheriting WC Checkout Form Fields.
 * 
 * @class QTS_Form_Fields
 * @package Class
 */
class QTS_Form_Fields {

	/**
	 * Get the fields.
	 * 
	 * @var array 
	 */
	protected static $fields;

	/**
	 * Get the available fields for usage.
	 * 
	 * @var array 
	 */
	protected static $available_fields;

	/**
	 * Post instance of the application form field.
	 *
	 * @var WP_Post[]
	 */
	protected static $post = array();

	/**
	 * Core metadata for the form field.
	 *
	 * @var array
	 */
	protected static $metadata = array();

	/**
	 * Get the default field keys. 
	 * 
	 * @var array 
	 */
	protected static $default_field_keys = array(
		'billing_first_name',
		'billing_last_name',
		'billing_company',
		'billing_country',
		'billing_address_1',
		'billing_address_2',
		'billing_city',
		'billing_state',
		'billing_postcode',
		'billing_phone',
		'billing_email',
		'qts_customer_note'
	);

	/**
	 * Allowed field attribute data for the form field.
	 *
	 * @var array
	 */
	protected static $allowed_attribute_data = array(
		'type'              => 'text',
		'label'             => '',
		'description'       => '',
		'placeholder'       => '',
		'name'              => '',
		'maxlength'         => false,
		'required'          => false,
		'autocomplete'      => false,
		'class'             => array(),
		'label_class'       => array(),
		'input_class'       => array(),
		'options'           => array(),
		'custom_attributes' => array(),
		'validate'          => array(),
		'default'           => '',
		'autofocus'         => '',
		'priority'          => 0,
	);

	const POST_TYPE = 'qts_form_field';

	/**
	 * Init hooks.
	 */
	public static function init() {
		add_filter( 'woocommerce_form_field_qts_multiselect', __CLASS__ . '::prepare_multiselect_fieldset', 99, 4 );
		add_filter( 'woocommerce_form_field_qts_customer_note', __CLASS__ . '::prepare_customer_note_fieldset', 10, 4 );
		add_filter( 'woocommerce_form_field_email', array( __CLASS__, 'prepare_billing_email_checkout_fieldset' ), 99, 4 );
	}

	/**
	 * Return the array of default props.
	 *
	 * @return array
	 */
	public static function get_default_props() {
		$default_metadata             = self::$allowed_attribute_data;
		$default_metadata[ 'status' ] = 'disabled';
		return $default_metadata;
	}

	/**
	 * Gets a default prop.
	 *
	 * @param string $prop Name of prop to get.
	 * @return mixed
	 */
	public static function get_default_prop( $prop ) {
		$default_metadata = self::get_default_props();
		return isset( $default_metadata[ $prop ] ) ? $default_metadata[ $prop ] : null;
	}

	/**
	 * Get the cached form field post instance.
	 * 
	 * @param int $field_post_id
	 * @return false|WP_Post
	 */
	public static function maybe_get_post( $field_post_id ) {
		if ( ! $field_post_id || ! is_numeric( $field_post_id ) ) {
			return false;
		}

		if ( ! isset( self::$post[ $field_post_id ] ) ) {
			$post = get_post( $field_post_id );

			if ( ! $post || self::POST_TYPE !== $post->post_type ) {
				return false;
			}

			if ( ! array_key_exists( $post->post_status, _qts_get_form_field_statuses() ) ) {
				return false;
			}

			self::$post[ $field_post_id ] = $post;
		}

		return self::$post[ $field_post_id ];
	}

	/**
	 * Check whether the form field exists?
	 * 
	 * @param int $field_post_id
	 * @return bool
	 */
	public static function exists( $field_post_id ) {
		$post = self::maybe_get_post( $field_post_id );
		return is_a( $post, 'WP_Post' );
	}

	/**
	 * Check whether the given field is default field?
	 * 
	 * @param int $field_post_id
	 * @return bool
	 */
	public static function is_default_field( $field_post_id ) {
		return in_array( self::get_prop( $field_post_id, 'name' ), self::$default_field_keys );
	}

	/**
	 * Prepare the name key.
	 * 
	 * @param int $field_post_id
	 * @return string
	 */
	public static function prepare_name( $field_post_id ) {
		return "qts_field_{$field_post_id}";
	}

	/**
	 * Gets a prop.
	 * Gets the value from either cached data or from db itself.
	 *
	 * @param int $field_post_id
	 * @param string $prop Name of prop to get.
	 * @param bool $default
	 * @return mixed
	 */
	public static function get_prop( $field_post_id, $prop, $default = false ) {
		if ( isset( self::$metadata[ $field_post_id ][ $prop ] ) ) {
			return self::$metadata[ $field_post_id ][ $prop ];
		}

		if ( ! self::exists( $field_post_id ) || is_null( self::get_default_prop( $prop ) ) ) {
			return $default ? self::get_default_prop( $prop ) : null;
		}

		switch ( $prop ) {
			case 'type':
				self::$metadata[ $field_post_id ][ $prop ] = self::$post[ $field_post_id ]->post_excerpt;
				break;
			case 'label':
				self::$metadata[ $field_post_id ][ $prop ] = self::$post[ $field_post_id ]->post_title;
				break;
			case 'status':
				self::$metadata[ $field_post_id ][ $prop ] = self::$post[ $field_post_id ]->post_status;
				break;
			case 'description':
				self::$metadata[ $field_post_id ][ $prop ] = self::$post[ $field_post_id ]->post_content;
				break;
			case 'priority':
				self::$metadata[ $field_post_id ][ $prop ] = self::$post[ $field_post_id ]->menu_order;
				break;
			default:
				$metadata                                  = get_post_meta( $field_post_id );

				if ( isset( $metadata[ "_$prop" ][ 0 ] ) ) {
					self::$metadata[ $field_post_id ][ $prop ] = maybe_unserialize( $metadata[ "_$prop" ][ 0 ] );
				}
				break;
		}

		return isset( self::$metadata[ $field_post_id ][ $prop ] ) ? self::$metadata[ $field_post_id ][ $prop ] : ( $default ? self::get_default_prop( $prop ) : '' );
	}

	/**
	 * Create default form fields on demand.
	 */
	public static function create_default_form_fields() {
		if ( is_null( WC()->countries ) ) {
			return;
		}

		$country           = WC()->countries->get_base_country();
		$allowed_countries = WC()->countries->get_allowed_countries();

		if ( ! array_key_exists( $country, $allowed_countries ) ) {
			$country = current( array_keys( $allowed_countries ) );
		}

		$default_fields                        = WC()->countries->get_address_fields( $country, 'billing_' );
		$default_fields[ 'qts_customer_note' ] = array(
			'label'       => __( 'Customer Note', 'quote-request-for-woocommerce' ),
			'type'        => 'textarea',
			'class'       => array( 'form-row-wide' ),
			'placeholder' => esc_attr__( 'Notes about your quote.', 'quote-request-for-woocommerce' ),
		);

		$priority = 1;
		foreach ( $default_fields as $field_key => $value ) {
			if ( ! in_array( $field_key, self::$default_field_keys ) ) {
				continue;
			}

			$args               = $value;
			$args[ 'name' ]     = $field_key;
			$args[ 'priority' ] = $priority;

			switch ( $field_key ) {
				case 'billing_first_name':
					$args[ 'status' ]   = 'enabled';
					$args[ 'required' ] = 'yes';
					break;
				case 'billing_email':
					$args[ 'status' ]   = 'enabled';
					$args[ 'required' ] = 'yes';
					$args[ 'default' ]  = 'user_email';
					break;
				case 'billing_address_2':
					$args[ 'status' ]   = 'disabled';
					$args[ 'label' ]    = __( 'Street address 2', 'quote-request-for-woocommerce' );
					$args[ 'required' ] = false;
					break;
				case 'billing_last_name':
				case 'qts_customer_note':
					$args[ 'status' ]   = false === get_option( QTS_PREFIX . $field_key . '_enabled' ) || 'yes' === get_option( QTS_PREFIX . $field_key . '_enabled' ) ? 'enabled' : 'disabled';
					$args[ 'required' ] = 'yes' === get_option( QTS_PREFIX . $field_key . '_is_mandatory' ) ? true : false;
					break;
				default:
					$args[ 'status' ]   = 'yes' === get_option( QTS_PREFIX . $field_key . '_enabled' ) ? 'enabled' : 'disabled';
					$args[ 'required' ] = 'yes' === get_option( QTS_PREFIX . $field_key . '_is_mandatory' ) ? true : false;
			}

			_qts_create_form_field( $args );

			$priority ++;
		}
	}

	/**
	 * Get an array of form fields.
	 *
	 * @return array
	 */
	public static function get_fields() {
		if ( is_array( self::$fields ) ) {
			return self::$fields;
		}

		$all_field_posts = _qts_get_all_form_fields();

		if ( empty( $all_field_posts ) ) {
			return array();
		}

		self::$fields = array();
		foreach ( $all_field_posts as $field_post_id ) {
			$field_key = self::get_prop( $field_post_id, 'name' );

			if ( empty( $field_key ) ) {
				continue;
			}

			self::$fields[ $field_key ][ 'status' ] = _qts_trim_post_status( self::get_prop( $field_post_id, 'status' ) );

			foreach ( self::$allowed_attribute_data as $attribute_key => $default_value ) {
				if ( 'name' === $attribute_key ) {
					continue;
				}

				$attribute_value = self::get_prop( $field_post_id, $attribute_key, true );

				if ( 'default' === $attribute_key ) {
					if ( 'user_email' === $attribute_value ) {
						self::$fields[ $field_key ][ $attribute_key ] = wp_get_current_user()->user_email;
					} else {
						/**
						 * Get the form field default value.
						 * 
						 * @since 1.0
						 */
						self::$fields[ $field_key ][ $attribute_key ] = apply_filters( 'qts_form_field_default_value', $attribute_value, $field_key );
					}
				} else if ( 'billing_address_2' === $field_key && 'label' === $attribute_key ) {
					self::$fields[ $field_key ][ $attribute_key ] = '';
				} else {
					self::$fields[ $field_key ][ $attribute_key ] = $attribute_value;
				}
			}
		}

		return self::$fields;
	}

	/**
	 * Get an array of default form fields.
	 *
	 * @return array
	 */
	public static function get_default_fields() {
		$fields = self::get_fields();

		if ( ! empty( $fields ) ) {
			$default_fields = array_filter( $fields, function( $field, $field_key ) {
				return in_array( $field_key, self::$default_field_keys ) ? $field : false;
			}, ARRAY_FILTER_USE_BOTH );
		} else {
			$default_fields = array();
		}

		return $default_fields;
	}

	/**
	 * Get an array of available form fields.
	 *
	 * @return array
	 */
	public static function get_available_fields() {
		if ( is_array( self::$available_fields ) ) {
			return self::$available_fields;
		}

		$fields = self::get_fields();
		if ( ! empty( $fields ) ) {
			self::$available_fields = array_filter( $fields, function( $field ) {
				return 'enabled' === $field[ 'status' ] ? $field : false;
			} );
		}

		return self::$available_fields;
	}

	/**
	 * Validates the posted data based on field properties.
	 *
	 * @param  array    $data   An array of posted data.
	 * @param  WP_Error $errors Validation error.
	 */
	public static function validate_posted_data( &$data, &$errors ) {
		foreach ( self::get_available_fields() as $field_key => $field ) {
			if ( ! isset( $field[ 'label' ] ) || ! $field[ 'required' ] ) {
				continue;
			}

			if ( ! isset( $data[ $field_key ] ) || '' === $data[ $field_key ] ) {
				if ( empty( $field[ 'label' ] ) ) {
					/* translators: 1: field name */
					$errors->add( 'required-field', esc_html__( 'Street address 2 is a required field.', 'quote-request-for-woocommerce' ) );
				} else {
					/* translators: 1: field name */
					$errors->add( 'required-field', sprintf( esc_html__( '%1$s is a required field.', 'quote-request-for-woocommerce' ), '<strong>' . esc_html( $field[ 'label' ] ) . '</strong>' ) );
				}
			}
		}
	}

	/**
	 * Gets the value from POST. Sets the default values in form fields.
	 *
	 * @param string $input Name of the input we want to grab data for. e.g. first_name.
	 * @return string The default value.
	 */
	public static function get_value( $input ) {
		$posted = $_REQUEST;

		if (
				isset( $posted[ QTS_PREFIX . 'nonce' ] ) &&
				wp_verify_nonce( sanitize_key( wp_unslash( $posted[ QTS_PREFIX . 'nonce' ] ) ), 'qts-quote-request' ) &&
				! empty( $posted[ $input ] )
		) {
			return wc_clean( wp_unslash( $posted[ $input ] ) );
		}

		/**
		 * Allow 3rd parties to short circuit the logic and return their own default value.
		 * 
		 * @since 1.0
		 */
		$value           = apply_filters( 'qts_form_field_default_value', null, $input );
		$customer_object = false;

		if ( is_user_logged_in() ) {
			$customer_object = new WC_Customer( get_current_user_id(), true );
		}

		if ( ! $customer_object ) {
			$customer_object = WC()->customer;
		}

		if ( is_callable( array( $customer_object, "get_$input" ) ) ) {
			$value = $customer_object->{"get_$input"}();
		} elseif ( $customer_object->meta_exists( $input ) ) {
			$value = $customer_object->get_meta( $input, true );
		}

		if ( '' === $value ) {
			$value = null;
		}

		return $value;
	}

	/**
	 * Enqueue WC scripts on demand.
	 */
	public static function enqueues() {
		wp_enqueue_script( 'wc-country-select' );
		wp_enqueue_script( 'wc-address-i18n' );
	}

	/**
	 * Prepare the fieldset.
	 * 
	 * @param string $field_body
	 * @param array $args
	 * @return string
	 */
	protected static function prepare_fieldset( $field_body, $args ) {
		if ( $args[ 'required' ] ) {
			$required = '&nbsp;<abbr class="required" title="' . esc_attr__( 'required', 'quote-request-for-woocommerce' ) . '">*</abbr>';
		} else {
			$required = '&nbsp;<span class="optional">(' . esc_html__( 'optional', 'quote-request-for-woocommerce' ) . ')</span>';
		}

		$field_container = '<p class="form-row %1$s" id="%2$s">%3$s</p>';
		$field_html      = '';

		if ( $args[ 'label' ] ) {
			$field_html .= '<label for="' . esc_attr( $args[ 'id' ] ) . '" class="' . esc_attr( implode( ' ', $args[ 'label_class' ] ) ) . '">' . wp_kses_post( $args[ 'label' ] ) . $required . '</label>';
		}

		$field_html .= $field_body;

		if ( $args[ 'description' ] ) {
			$field_html .= '<span class="description" id="' . esc_attr( $args[ 'id' ] ) . '-description" aria-hidden="true">' . wp_kses_post( $args[ 'description' ] ) . '</span>';
		}

		$container_class = esc_attr( implode( ' ', $args[ 'class' ] ) );
		$container_id    = esc_attr( $args[ 'id' ] ) . '_field';
		$field           = sprintf( $field_container, $container_class, $container_id, $field_html );

		return $field;
	}

	/**
	 * Prepare and get the multiselect fieldset.
	 */
	public static function prepare_multiselect_fieldset( $field, $key, $args, $value ) {
		if ( empty( $args[ 'options' ] ) ) {
			return $field;
		}

		$field_body        = '';
		$options           = '';
		$custom_attributes = array();

		if ( ! empty( $args[ 'custom_attributes' ] ) && is_array( $args[ 'custom_attributes' ] ) ) {
			foreach ( $args[ 'custom_attributes' ] as $attribute => $attribute_value ) {
				$custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $attribute_value ) . '"';
			}
		}

		foreach ( $args[ 'options' ] as $option_key => $option_text ) {
			if ( '' === $option_key ) {
				// If we have a blank option, select2 needs a placeholder.
				if ( empty( $args[ 'placeholder' ] ) ) {
					$args[ 'placeholder' ] = $option_text ? $option_text : __( 'Choose an option', 'woocommerce' );
				}
				$custom_attributes[] = 'data-allow_clear="true"';
			}
			$options .= '<option value="' . esc_attr( $option_key ) . '" ' . selected( $value, $option_key, false ) . '>' . esc_html( $option_text ) . '</option>';
		}

		$field_body .= '<select multiple="multiple" name="' . esc_attr( $key ) . '[]" id="' . esc_attr( $args[ 'id' ] ) . '" class="select wc-enhanced-select ' . esc_attr( implode( ' ', $args[ 'input_class' ] ) ) . '" ' . implode( ' ', $custom_attributes ) . ' data-placeholder="' . esc_attr( $args[ 'placeholder' ] ) . '">
							' . $options . '
						</select>';

		return self::prepare_fieldset( $field_body, $args );
	}

	/**
	 * Prepare and get the customer note to quote input fieldset.
	 */
	public static function prepare_customer_note_fieldset( $field, $key, $args, $value ) {
		$custom_attributes = array();

		if ( ! empty( $args[ 'custom_attributes' ] ) && is_array( $args[ 'custom_attributes' ] ) ) {
			foreach ( $args[ 'custom_attributes' ] as $attribute => $attribute_value ) {
				$custom_attributes[] = esc_attr( $attribute ) . '="' . esc_attr( $attribute_value ) . '"';
			}
		}

		if ( is_null( $value ) ) {
			$value = $args[ 'default' ];
		}

		$field_body = '<textarea name="' . esc_attr( $key ) . '" class="input-text ' . esc_attr( implode( ' ', $args[ 'input_class' ] ) ) . '" id="' . esc_attr( $args[ 'id' ] ) . '" placeholder="' . esc_attr( $args[ 'placeholder' ] ) . '" ' . ( empty( $args[ 'custom_attributes' ][ 'rows' ] ) ? ' rows="2"' : '' ) . ( empty( $args[ 'custom_attributes' ][ 'cols' ] ) ? ' cols="5"' : '' ) . implode( ' ', $custom_attributes ) . '>' . esc_textarea( $value ) . '</textarea>';
		$field_body .= '<a href="#" class="button qts-save-note">' . esc_html__( 'Save', 'quote-request-for-woocommerce' ) . '</a>&nbsp;<a href="#" class="button qts-clear-note">' . esc_html__( 'Clear', 'quote-request-for-woocommerce' ) . '</a>&nbsp;';
		$field_body .= '<span class="qts-notice"></span>';

		return self::prepare_fieldset( $field_body, $args );
	}

	/**
	 * Prepare and get the billing email input fieldset for checkout.
	 */
	public static function prepare_billing_email_checkout_fieldset( $field, $key, $args, $value ) {
		if ( 'billing_email' === $key && isset( $args[ 'qtsquote_request' ][ 'email' ] ) ) {
			$field_body = '<input type="hidden" name="' . esc_attr( $key ) . '" value="' . esc_attr( $args[ 'qtsquote_request' ][ 'email' ] ) . '">';
			$field_body .= '<span>' . esc_html( $args[ 'qtsquote_request' ][ 'email' ] ) . '</span>';
			$field      = self::prepare_fieldset( $field_body, $args );
		}

		return $field;
	}

}
