<?php

defined( 'ABSPATH' ) || exit;

/**
 * Manage order activities for the quote request.
 * 
 * @class QTS_Order_Manager
 * @package Class
 */
class QTS_Order_Manager {

	/**
	 * Data stored in meta keys, prefixed with QTS_PREFIX
	 *
	 * @var array
	 */
	protected static $internal_meta_keys = array(
		'quote_request_id',
		'is_quote_request_order',
		'quote_request_completed',
	);

	/**
	 * Init QTS_Order_Manager.
	 */
	public static function init() {
		// Mode 1&2
		add_action( 'woocommerce_order_status_on_hold', __CLASS__ . '::payment_pending', 99 );
		add_action( 'woocommerce_order_status_completed', __CLASS__ . '::complete_the_quote_request', 99 );
		add_action( 'woocommerce_order_status_processing', __CLASS__ . '::complete_the_quote_request', 99 );

		// Mode 1
		add_action( 'woocommerce_checkout_order_processed', __CLASS__ . '::save_quote_request_to_order', 5 );

		// Mode 2
		add_filter( 'woocommerce_cancel_unpaid_order', __CLASS__ . '::prevent_unpaid_order_from_cancel', 99, 2 );
		add_action( 'qts_quote_request_status_accepted', __CLASS__ . '::prepare_to_pay_for_order', -1 );
		add_action( 'qts_quote_request_status_approved', __CLASS__ . '::prepare_to_pay_for_order', -1 );
		add_action( 'qts_quote_request_status_rejected', __CLASS__ . '::quote_request_failed', -1 );
		add_action( 'qts_quote_request_status_expired', __CLASS__ . '::quote_request_failed', -1 );
		add_action( 'woocommerce_thankyou_' . QTS_PREFIX . 'quote_request', __CLASS__ . '::quote_request_submitted_success' );
	}

	/**
	 * Get the meta data for the order.
	 * 
	 * @param WC_Order $order
	 * @param array $internal_meta_key
	 * @return mixed
	 */
	public static function get_meta( $order, $internal_meta_key ) {
		if ( ! $order || ! in_array( $internal_meta_key, self::$internal_meta_keys ) ) {
			return null;
		}

		return $order->get_meta( QTS_PREFIX . $internal_meta_key, true );
	}

	/**
	 * Save the meta data to the order.
	 * 
	 * @param WC_Order $order
	 * @param array $internal_meta_key
	 * @param mixed $value
	 */
	public static function save_meta( $order, $internal_meta_key, $value ) {
		if ( ! $order || ! in_array( $internal_meta_key, self::$internal_meta_keys ) ) {
			return;
		}

		$order->update_meta_data( QTS_PREFIX . $internal_meta_key, $value );
		$order->save_meta_data();
	}

	/*
	  |--------------------------------------------------------------------------
	  | Mode 1&2 Combined Funtions
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Notice the admin saying quote request payment is on hold.
	 * 
	 * @param int $order_id
	 */
	public static function payment_pending( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		if ( 'yes' === self::get_meta( $order, 'quote_request_completed' ) ) {
			return;
		}

		$quote_request_id = self::get_meta( $order, 'quote_request_id' );
		$quote_request    = _qts_get_quote_request( $quote_request_id );

		if ( ! $quote_request ) {
			return;
		}

		if ( $quote_request->needs_payment() ) {
			/* translators: %s: order id */
			$quote_request->add_order_note( sprintf( __( 'Payment is on hold for the order #%1$s.', 'quote-request-for-woocommerce' ), $order_id ) );

			/**
			 * Trigger after quote request awaiting for payment.
			 * 
			 * @param QTS_Quote_Request $quote_request 
			 * @param int $order_id 
			 * @since 1.0
			 */
			do_action( 'qts_quote_request_waiting_for_payment', $quote_request, $order_id );
		}
	}

	/**
	 * Complete the quote request only if the order status changed to completed or processing.
	 * 
	 * @param int $order_id
	 */
	public static function complete_the_quote_request( $order_id ) {
		$order = wc_get_order( $order_id );
		if ( ! $order ) {
			return;
		}

		if ( 'yes' === self::get_meta( $order, 'quote_request_completed' ) ) {
			return;
		}

		$quote_request_id = self::get_meta( $order, 'quote_request_id' );
		$quote_request    = _qts_get_quote_request( $quote_request_id );

		if ( ! $quote_request ) {
			return;
		}

		// Complete the quote request.
		$quote_request->payment_complete();
		self::save_meta( $order, 'quote_request_completed', 'yes' );

		/**
		 * Trigger after quote request completed.
		 * 
		 * @param QTS_Quote_Request $quote_request 
		 * @param int $order_id 
		 * @since 1.0
		 */
		do_action( 'qts_quote_request_completed', $quote_request, $order_id );
	}

	/*
	  |--------------------------------------------------------------------------
	  | Mode 1 Funtions
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Save the quote request to order meta.
	 * 
	 * @param int $order_id
	 */
	public static function save_quote_request_to_order( $order_id ) {
		if ( is_null( WC()->cart ) ) {
			return;
		}

		$added_quote = QTS_Cart::awaiting_for_payment();
		if ( ! $added_quote || ! is_numeric( $added_quote[ 'payment' ] ) ) {
			return;
		}

		$order            = wc_get_order( $order_id );
		$quote_request_id = absint( $added_quote[ 'payment' ] );
		$quote_request    = _qts_get_quote_request( $quote_request_id );

		if ( $quote_request && $quote_request->is_created_via( 'quote' ) ) {
			$quote_request->set_order_id( $order_id );
			$quote_request->save();

			// Save the quote request in order meta.
			self::save_meta( $order, 'is_quote_request_order', 'yes' );
			self::save_meta( $order, 'quote_request_id', $quote_request_id );
		}
	}

	/*
	  |--------------------------------------------------------------------------
	  | Mode 2 Funtions
	  |--------------------------------------------------------------------------
	 */

	/**
	 * Prevent the pending order from cancel which is created for quote.
	 * 
	 * @param bool $cancel_order
	 * @param WC_Order $order
	 * @return bool
	 */
	public static function prevent_unpaid_order_from_cancel( $cancel_order, $order ) {
		if ( 'yes' === self::get_meta( $order, 'is_quote_request_order' ) ) {
			$cancel_order = false;
		}

		return $cancel_order;
	}

	/**
	 * Prepare the order to pay when the quote request accepted/approved for mode 2.
	 * 
	 * @param int $quote_request_id
	 */
	public static function prepare_to_pay_for_order( $quote_request_id ) {
		$quote_request = _qts_get_quote_request( $quote_request_id );

		if ( ! $quote_request || ! $quote_request->is_created_via( 'checkout' ) ) {
			return;
		}

		$pay_order = wc_get_order( $quote_request->get_order_id() );

		if ( $pay_order && $pay_order->has_status( 'qts_new_quote' ) ) {
			$quote_request_items = $quote_request->get_items( array( 'line_item', 'tax', 'shipping', 'fee' ) );

			if ( empty( $quote_request_items ) ) {
				return;
			}

			$pay_order->set_payment_method( '' );
			$pay_order->remove_order_items();
			$pay_order->save();

			// Update the admin offered price in the order.
			foreach ( $quote_request_items as $from_item ) {
				$to_item = false;

				switch ( $from_item->get_type() ) {
					case 'line_item':
						$to_item = new WC_Order_Item_Product();
						$to_item->set_props( array(
							'name'         => $from_item->get_name(),
							'product_id'   => $from_item->get_product_id(),
							'variation_id' => $from_item->get_variation_id(),
							'quantity'     => $from_item->get_quantity(),
							'tax_class'    => $from_item->get_tax_class(),
							'subtotal'     => $from_item->get_subtotal(),
							'total'        => $from_item->get_total(),
							'taxes'        => $from_item->get_taxes(),
						) );
						break;
					case 'shipping':
						$to_item = new WC_Order_Item_Shipping();
						$to_item->set_props( array(
							'name'      => $from_item->get_name(),
							'method_id' => $from_item->get_method_id(),
							'total'     => $from_item->get_total(),
							'taxes'     => $from_item->get_taxes(),
						) );
						break;
					case 'tax':
						$to_item = new WC_Order_Item_Tax();
						$to_item->set_props( array(
							'name'               => $from_item->get_name(),
							'rate_id'            => $from_item->get_rate_id(),
							'label'              => $from_item->get_label(),
							'compound'           => $from_item->get_compound(),
							'tax_total'          => $from_item->get_tax_total(),
							'shipping_tax_total' => $from_item->get_shipping_tax_total(),
							'rate_percent'       => $from_item->get_rate_percent(),
						) );
						break;
					case 'fee':
						$to_item = new WC_Order_Item_Fee();
						$to_item->set_props( array(
							'name'       => $from_item->get_name(),
							'tax_class'  => $from_item->get_tax_class(),
							'tax_status' => $from_item->get_tax_status(),
							'total'      => $from_item->get_total(),
							'taxes'      => $from_item->get_taxes(),
						) );
						break;
				}

				if ( $to_item ) {
					foreach ( $from_item->get_meta_data() as $meta_data ) {
						$to_item->update_meta_data( $meta_data->key, $meta_data->value );
					}

					$pay_order->add_item( $to_item );
				}
			}

			// Calculate the order total.
			$pay_order->calculate_totals();
			/* translators: %s: quote request status */
			$pay_order->update_status( 'pending', sprintf( __( 'Quote Request status set to %s. By paying this order quote request gets completed.', 'quote-request-for-woocommerce' ), _qts_get_quote_request_status_name( $quote_request->get_status() ) ) );
		}
	}

	/**
	 * Change the order status to qts_quote_failed when the quote request rejected/expired for mode 2.
	 * 
	 * @param int $quote_request_id
	 */
	public static function quote_request_failed( $quote_request_id ) {
		$quote_request = _qts_get_quote_request( $quote_request_id );

		if ( ! $quote_request ) {
			return;
		}

		if ( $quote_request->is_created_via( 'checkout' ) ) {
			$order = wc_get_order( $quote_request->get_order_id() );

			//Update the order status from New Quote to Quote Failed.
			if ( $order && $order->has_status( 'qts_new_quote' ) ) {
				/* translators: %s: quote request status */
				$order->update_status( 'qts_quote_failed', sprintf( __( 'Quote Request status set to %s.', 'quote-request-for-woocommerce' ), _qts_get_quote_request_status_name( $quote_request->get_status() ) ) );
			}
		}
	}

	/**
	 * 
	 * Print the quote request submitted notice.
	 *
	 * @param int $order_id
	 */
	public static function quote_request_submitted_success( $order_id ) {
		$order = wc_get_order( $order_id );

		if ( 'yes' !== self::get_meta( $order, 'is_quote_request_order' ) ) {
			return;
		}

		$quote_request_id = self::get_meta( $order, 'quote_request_id' );
		$quote_request    = _qts_get_quote_request( $quote_request_id );

		if ( $quote_request ) {
			if ( $quote_request->get_user_id() ) {
				wc_print_notice( trim( str_replace( array( '[quote_request_number]', '[view_quote_request_url]' ), array( esc_html( $quote_request->get_id() ), esc_url( $quote_request->get_view_order_url() ) ), get_option( QTS_PREFIX . 'quote_submitted_success_msg_for_logged_in_users' ) ) ), 'success' );
			} else {
				wc_print_notice( trim( str_replace( '[quote_request_number]', esc_html( $quote_request->get_id() ), get_option( QTS_PREFIX . 'quote_submitted_success_msg_for_guests' ) ) ), 'success' );
			}
		}
	}

}

QTS_Order_Manager::init();
