<?php

defined( 'ABSPATH' ) || exit;

/**
 * Quote Request Data Store: Stored in Custom Order Tables.
 *
 * Extends OrdersTableDataStore to make sure quote request related meta data is read/updated.
 */
class QTS_Orders_Table_Quote_Request_Data_Store extends \Automattic\WooCommerce\Internal\DataStores\Orders\OrdersTableDataStore {

	/**
	 * Define quote request specific data which augments the meta of an order.
	 *
	 * The meta keys here determine the prop data that needs to be manually set. We can't use
	 * the $internal_meta_keys property from OrdersTableDataStore because we want its value
	 * too, so instead we create our own and merge it into $internal_meta_keys in __construct.
	 *
	 * @var array
	 */
	protected $quote_request_internal_meta_keys = array(
		'_order_id',
		'_price_negotiations_started',
		'_date_expiry',
		'_form_fields',
	);

	/**
	 * Array of quote request specific data which augments the meta of an order in the form meta_key => prop_key
	 *
	 * Used to read/update props on the quote request.
	 *
	 * @var array
	 */
	protected $quote_request_meta_keys_to_props = array(
		'_order_id'                   => 'order_id',
		'_price_negotiations_started' => 'price_negotiations_started',
		'_date_expiry'                => 'date_expiry',
		'_form_fields'                => 'form_fields',
	);

	/**
	 * Table column to QTS_Quote_Request mapping for wc_operational_data table.
	 *
	 * For quote requests, all columns are inherited from orders except for the following columns:
	 *
	 * - cart_hash
	 * - new_order_email_sent
	 * - order_stock_reduced
	 * - date_paid_gmt
	 * - recorded_sales
	 * - date_completed_gmt
	 *
	 * @var \string[][]
	 */
	protected $operational_data_column_mapping = array(
		'id'                          => array( 'type' => 'int' ),
		'order_id'                    => array( 'type' => 'int' ),
		'created_via'                 => array(
			'type' => 'string',
			'name' => 'created_via',
		),
		'woocommerce_version'         => array(
			'type' => 'string',
			'name' => 'version',
		),
		'prices_include_tax'          => array(
			'type' => 'bool',
			'name' => 'prices_include_tax',
		),
		'coupon_usages_are_counted'   => array(
			'type' => 'bool',
			'name' => 'recorded_coupon_usage_counts',
		),
		'download_permission_granted' => array(
			'type' => 'bool',
			'name' => 'download_permissions_granted',
		),
		'order_key'                   => array(
			'type' => 'string',
			'name' => 'order_key',
		),
		'shipping_tax_amount'         => array(
			'type' => 'decimal',
			'name' => 'shipping_tax',
		),
		'shipping_total_amount'       => array(
			'type' => 'decimal',
			'name' => 'shipping_total',
		),
		'discount_tax_amount'         => array(
			'type' => 'decimal',
			'name' => 'discount_tax',
		),
		'discount_total_amount'       => array(
			'type' => 'decimal',
			'name' => 'discount_total',
		),
	);

	/**
	 * Constructor.
	 */
	public function __construct() {
		// Exclude the quote request related meta data we set and manage manually from the objects "meta" data.
		$this->internal_meta_keys = array_merge( $this->internal_meta_keys, $this->quote_request_internal_meta_keys );
	}

	/**
	 * Returns data store object to use backfilling.
	 *
	 * @return \QTS_Quote_Request_Data_Store_CPT
	 */
	protected function get_post_data_store_for_backfill() {
		return new \QTS_Quote_Request_Data_Store_CPT();
	}

	/**
	 * Gets amount refunded for all related orders.
	 *
	 * @param \QTS_Quote_Request $quote_request
	 *
	 * @return string
	 */
	public function get_total_refunded( $quote_request ) {
		return 0;
	}

	/**
	 * Gets the total tax refunded for all related orders.
	 *
	 * @param \QTS_Quote_Request $quote_request
	 *
	 * @return float
	 */
	public function get_total_tax_refunded( $quote_request ) {
		return 0;
	}

	/**
	 * Gets the total shipping refunded for all related orders.
	 *
	 * @param \QTS_Quote_Request $quote_request The quote request object.
	 *
	 * @return float
	 */
	public function get_total_shipping_refunded( $quote_request ) {
		return 0;
	}

	/**
	 * Returns count of quote requests with a specific status.
	 *
	 * @param string $status Quote Request status. The _qts_get_quote_request_statuses() function returns a list of valid statuses.
	 *
	 * @return int The number of quote requests with a specific status.
	 */
	public function get_order_count( $status ) {
		global $wpdb;
		$_wpdb        = &$wpdb;
		$orders_table = self::get_orders_table_name();
		return absint( $_wpdb->get_var( $_wpdb->prepare( "SELECT COUNT(*) FROM {$orders_table} WHERE type = 'qts_quote_request' AND status = %s", $status ) ) ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Get all quote requests matching the passed in args.
	 *
	 * @param array $args
	 * @return array of orders
	 */
	public function get_orders( $args = array() ) {
		$args = wp_parse_args( $args, array(
			'type'   => 'qts_quote_request',
			'return' => 'objects'
				) );

		$parent_args = $args;

		// We only want IDs from the parent method
		$parent_args[ 'return' ] = 'ids';

		$quote_requests = wc_get_orders( $parent_args );

		if ( isset( $args[ 'paginate' ] ) && $args[ 'paginate' ] ) {
			if ( 'objects' === $args[ 'return' ] ) {
				$return = array_map( '_qts_get_quote_request', $quote_requests->orders );
			} else {
				$return = $quote_requests->orders;
			}

			return ( object ) array(
						'orders'        => $return,
						'total'         => $quote_requests->total,
						'max_num_pages' => $quote_requests->max_num_pages
			);
		} else {
			if ( 'objects' === $args[ 'return' ] ) {
				$return = array_map( '_qts_get_quote_request', $quote_requests );
			} else {
				$return = $quote_requests;
			}

			return $return;
		}
	}

	/**
	 * Creates a new quote request in the database.
	 *
	 * @param \QTS_Quote_Request $quote_request Quote Request object.
	 */
	public function create( &$quote_request ) {
		$this->persist_save( $quote_request );
		/**
		 * New quote request created.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_new_quote_request', $quote_request->get_id() );
	}

	/**
	 * Updates a quote request in the database.
	 *
	 * @param \QTS_Quote_Request $quote_request Quote Request object.
	 */
	public function update( &$quote_request ) {
		if ( null === $quote_request->get_date_created( 'edit' ) ) {
			$quote_request->set_date_created( time() );
		}

		$quote_request->set_version( QTS_VERSION );

		// Fetch changes.
		$changes = $quote_request->get_changes();
		$this->persist_updates( $quote_request );

		// Update download permissions if necessary.
		if ( array_key_exists( 'billing_email', $changes ) || array_key_exists( 'customer_id', $changes ) ) {
			$data_store = \WC_Data_Store::load( 'customer-download' );
			$data_store->update_user_by_order_id( $quote_request->get_id(), $quote_request->get_customer_id(), $quote_request->get_billing_email() );
		}

		// Mark user account as active.
		if ( array_key_exists( 'customer_id', $changes ) ) {
			wc_update_user_last_active( $quote_request->get_customer_id() );
		}

		$quote_request->apply_changes();
		$this->clear_caches( $quote_request );

		/**
		 * Quote request updated.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_update_quote_request', $quote_request->get_id(), $quote_request );
	}

	/**
	 * Helper method responsible for persisting new data to order table.
	 *
	 * This should not contain and specific meta or actions, so that it can be used other quote request types safely.
	 *
	 * @param QTS_Quote_Request $quote_request The quote request to save.
	 * @param bool      $force_all_fields Force update all fields, instead of calculating and updating only changed fields.
	 * @param bool      $backfill Whether to backfill data to post datastore.
	 *
	 * @return void
	 *
	 * @throws \Exception When unable to save data.
	 */
	protected function persist_save( &$quote_request, $force_all_fields = false, $backfill = true ) {
		$quote_request->set_version( QTS_VERSION );
		$quote_request->set_date_created( time() );
		$quote_request->set_currency( $quote_request->get_currency() ? $quote_request->get_currency() : get_woocommerce_currency()  );
		$quote_request->set_order_key( wc_generate_order_key() );

		$this->update_order_meta( $quote_request );
		$this->persist_order_to_db( $quote_request, $force_all_fields );

		$quote_request->save_meta_data();
		$quote_request->apply_changes();

		if ( $backfill ) {
			$data_sync = wc_get_container()->get( \Automattic\WooCommerce\Internal\DataStores\Orders\DataSynchronizer::class );

			if ( $data_sync->data_sync_is_enabled() ) {
				$this->backfill_post_record( $quote_request );
			}
		}

		$this->clear_caches( $quote_request );
	}

	/**
	 * Saves a quote request to the database.
	 *
	 * @param QTS_Quote_Request $quote_request The quote request to save.
	 * @param bool            $force_all_fields Optional. Whether to force all fields to be saved. Default false.
	 */
	protected function persist_order_to_db( &$quote_request, $force_all_fields = false ) {
		$is_update = ( 0 !== absint( $quote_request->get_id() ) );

		parent::persist_order_to_db( $quote_request, $force_all_fields );

		if ( $force_all_fields && $is_update ) {
			$props_to_save = $this->quote_request_meta_keys_to_props;
		} else {
			$props_to_save = $this->get_props_to_update( $quote_request, $this->quote_request_meta_keys_to_props );
		}

		$quote_request_meta_data = array_column( $this->data_store_meta->read_meta( $quote_request ), null, 'meta_key' );

		foreach ( $props_to_save as $meta_key => $prop ) {
			$meta_value = $quote_request->{"get_$prop"}( 'edit' );

			if ( 'date_expiry' === $prop ) {
				$meta_value = $meta_value ? $meta_value->getTimestamp() : null;
			}

			$existing_meta_data = isset( $quote_request_meta_data[ $meta_key ] ) ? $quote_request_meta_data[ $meta_key ] : false;
			$new_meta_data      = array(
				'key'   => $meta_key,
				'value' => $meta_value,
			);

			if ( empty( $existing_meta_data ) ) {
				$this->data_store_meta->add_meta( $quote_request, ( object ) $new_meta_data );
			} elseif ( $existing_meta_data->meta_value !== $new_meta_data[ 'value' ] ) {
				$new_meta_data[ 'id' ] = $existing_meta_data->meta_id;
				$this->data_store_meta->update_meta( $quote_request, ( object ) $new_meta_data );
			}
		}
	}

	/**
	 * Helper method to initialize quote request object from DB data.
	 *
	 * @param \WC_Abstract_Order $quote_request Quote Request object.
	 * @param int                $quote_request_id Quote Request ID.
	 * @param \stdClass          $quote_request_data Quote Request data fetched from DB.
	 */
	protected function init_order_record( \WC_Abstract_Order &$quote_request, $quote_request_id, \stdClass $quote_request_data ) {
		parent::init_order_record( $quote_request, $quote_request_id, $quote_request_data );

		if ( empty( $quote_request_data->meta_data ) ) {
			return;
		}

		// Flag the quote request as still being read from the database while we set our quote request properties.
		$quote_request->set_object_read( false );

		// Set quote request specific properties that we store in meta.
		$meta_data    = wp_list_pluck( $quote_request_data->meta_data, 'meta_value', 'meta_key' );
		$props_to_set = array();

		foreach ( $this->quote_request_meta_keys_to_props as $meta_key => $prop_key ) {
			$is_internal_meta = in_array( $meta_key, $this->internal_meta_keys, true );

			// We only need to set props that are internal meta keys or dates. Everything else is treated as meta.
			if ( ! $is_internal_meta ) {
				continue;
			}

			// If there's no meta data, we don't need to set anything.
			if ( ! isset( $meta_data[ $meta_key ] ) ) {
				continue;
			}

			if ( 'date_expiry' === $prop_key ) {
				$props_to_set[ $prop_key ] = $meta_data[ $meta_key ];
			} else {
				$props_to_set[ $prop_key ] = maybe_unserialize( $meta_data[ $meta_key ] );
			}
		}

		$quote_request->set_props( $props_to_set );

		// Flag the quote request as read.
		$quote_request->set_object_read( true );
	}

	/**
	 * Get the quote request status to save to the post object.
	 *
	 * @param  QTS_Quote_Request $quote_request object.
	 * @return string
	 */
	protected function get_post_status( $quote_request ) {
		$post_status = $quote_request->get_status( 'edit' );

		if ( ! $post_status ) {
			/**
			 * Get default quote request status.
			 * 
			 * @since 1.0
			 */
			$post_status = apply_filters( 'qts_default_quote_request_status', 'new' );
		}

		if ( in_array( QTS_PREFIX . $post_status, $quote_request->get_valid_statuses() ) ) {
			$post_status = QTS_PREFIX . $post_status;
		}

		return $post_status;
	}

	/**
	 * Searches quote request data for a term and returns quote request IDs.
	 *
	 * @param string $term Term to search.
	 *
	 * @return array A list of quote requests IDs that match the search term.
	 */
	public function search_quote_requests( $term ) {
		add_filter( 'woocommerce_order_table_search_query_meta_keys', array( $this, 'get_quote_request_order_table_search_fields' ) );

		$quote_request_ids = wc_get_orders( array(
			's'      => $term,
			'type'   => 'qts_quote_request',
			'status' => array_keys( _qts_get_quote_request_statuses() ),
			'return' => 'ids',
			'limit'  => -1,
				) );

		remove_filter( 'woocommerce_order_table_search_query_meta_keys', array( $this, 'get_quote_request_order_table_search_fields' ) );

		/**
		 * Get quote request search results.
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'woocommerce_qts_quote_request_search_results', $quote_request_ids, $term, $this->get_quote_request_order_table_search_fields() );
	}

	/**
	 * Gets the quote request search fields.
	 *
	 * This function is hooked onto the 'woocommerce_order_table_search_query_meta_keys' filter.
	 *
	 * @param array The default order search fields.
	 *
	 * @return array The quote request search fields.
	 */
	public function get_quote_request_order_table_search_fields( $search_fields = array() ) {
		/**
		 * Get quote request search fields.
		 * 
		 * @since 1.0
		 */
		return array_map( 'wc_clean', apply_filters( 'woocommerce_qts_quote_request_search_fields', array(
			'_billing_address_index',
			'_shipping_address_index',
			'_order_id'
				) ) );
	}

	/**
	 * Gets user IDs for customers who have a quote request.
	 *
	 * @return array An array of user IDs.
	 */
	public function get_quote_request_customer_ids() {
		global $wpdb;
		$_wpdb      = &$wpdb;
		$table_name = self::get_orders_table_name();
		return $_wpdb->get_col( "SELECT DISTINCT customer_id FROM {$table_name} WHERE type = 'qts_quote_request'" ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
	}

	/**
	 * Deletes all rows in the postmeta table with the given meta key.
	 *
	 * @param string $meta_key The meta key to delete.
	 */
	public function delete_all_metadata_by_key( $meta_key ) {
		global $wpdb;
		$_wpdb = &$wpdb;
		$_wpdb->delete( self::get_meta_table_name(), array( 'meta_key' => $meta_key ), array( '%s' ) ); // phpcs:ignore WordPress.DB.SlowDBQuery.slow_db_query_meta_key
	}

	/**
	 * Count quote requests by status.
	 *
	 * @return array
	 */
	public function get_quote_requests_count_by_status() {
		global $wpdb;
		$_wpdb   = &$wpdb;
		$table   = self::get_orders_table_name();
		$results = $_wpdb->get_results( "SELECT status, COUNT(*) AS cnt FROM {$table} WHERE type = 'qts_quote_request' GROUP BY status", ARRAY_A ); // phpcs:ignore WordPress.DB.PreparedSQL.InterpolatedNotPrepared
		return $results ? array_combine( array_column( $results, 'status' ), array_map( 'absint', array_column( $results, 'cnt' ) ) ) : array();
	}

}
