<?php

defined( 'ABSPATH' ) || exit;

/**
 * Load Payment Gateways
 * 
 * @class QTS_Payment_Gateways
 * @package Class
 */
class QTS_Payment_Gateways {

	/**
	 * Get our payment gateways to load in to the WC checkout
	 *
	 * @var array 
	 */
	protected static $load_gateways = array();

	/**
	 * The single instance of the class.
	 */
	protected static $instance = null;

	/**
	 * Get the disabled payment gateways in checkout
	 * 
	 * @var array 
	 */
	protected static $disabled_payment_gateways;

	/**
	 * Create instance for QTS_Payment_Gateways.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Init QTS_Payment_Gateways.
	 */
	public function init() {
		add_action( 'plugins_loaded', array( $this, 'load_payment_gateways' ), 20 );
		add_filter( 'woocommerce_payment_gateways', array( $this, 'add_payment_gateways' ) );
		add_filter( 'woocommerce_available_payment_gateways', array( $this, 'set_payment_gateways' ), 999 );
	}

	/**
	 * Get the disabled payment gateways in checkout to pay for the quote request order.
	 * 
	 * @return array 
	 */
	public function get_disabled_payment_gateways() {
		if ( is_array( self::$disabled_payment_gateways ) ) {
			return self::$disabled_payment_gateways;
		}

		self::$disabled_payment_gateways = get_option( QTS_PREFIX . 'disabled_payment_gateways_while_making_payment', array() );
		return self::$disabled_payment_gateways;
	}

	/**
	 * Get payment gateways to load in to the WC checkout
	 */
	public function load_payment_gateways() {
		if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
			return;
		}

		self::$load_gateways[] = include_once 'gateways/quote-request/class-qts-quote-request-gateway.php';
	}

	/**
	 * Add payment gateways awaiting to load
	 * 
	 * @param object $gateways
	 * @return array
	 */
	public function add_payment_gateways( $gateways ) {
		if ( empty( self::$load_gateways ) ) {
			return $gateways;
		}

		foreach ( self::$load_gateways as $gateway ) {
			$gateways[] = $gateway;
		}
		return $gateways;
	}

	/**
	 * Check whether specific payment gateway is needed in checkout
	 * 
	 * @param WC_Payment_Gateway $gateway
	 * @return bool
	 */
	public function need_payment_gateway( $gateway ) {
		$need = true;

		if ( is_checkout_pay_page() ) {
			if ( $gateway->supports( 'qtsquote_request' ) ) {
				$need = false;
			} else if ( in_array( $gateway->id, ( array ) self::get_disabled_payment_gateways() ) ) {
				$need = false;
			}
		} elseif ( QTS_Cart::asking_for_quote() ) {
			if ( ! $gateway->supports( 'qtsquote_request' ) ) {
				$need = false;
			}
		} else if ( QTS_Cart::awaiting_for_payment() ) {
			if ( $gateway->supports( 'qtsquote_request' ) ) {
				$need = false;
			} else if ( in_array( $gateway->id, ( array ) self::get_disabled_payment_gateways() ) ) {
				$need = false;
			}
		} elseif ( $gateway->supports( 'qtsquote_request' ) ) {
				$need = false;
		}

		/**
		 * Need the payment gateway to be available on checkout?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_need_payment_gateway', $need, $gateway );
	}

	/**
	 * Handle payment gateways in checkout
	 * 
	 * @param array $_available_gateways
	 * @return array
	 */
	public function set_payment_gateways( $_available_gateways ) {
		if ( is_admin() || is_null( WC()->cart ) ) {
			return $_available_gateways;
		}

		foreach ( $_available_gateways as $gateway_name => $gateway ) {
			if ( ! isset( $gateway->id ) ) {
				continue;
			}

			if ( ! self::need_payment_gateway( $gateway ) ) {
				unset( $_available_gateways[ $gateway_name ] );
			}
		}
		return $_available_gateways;
	}
}
