<?php

defined( 'ABSPATH' ) || exit ;

include_once 'qts-conditional-functions.php' ;
include_once 'qts-time-functions.php' ;
include_once 'qts-form-field-functions.php' ;
include_once 'qts-quote-functions.php' ;
include_once 'qts-quote-request-functions.php' ;
include_once 'qts-user-functions.php' ;
include_once 'qts-account-functions.php' ;
include_once 'qts-template-functions.php' ;
include_once 'queue/qts-queue-functions.php' ;

/**
 * Get our templates.
 *
 * @param string $template_name
 * @param array $args (default: array())
 * @param string $template_path (default: 'QTS_BASENAME_DIR')
 * @param string $default_path (default: QTS_TEMPLATE_PATH)
 */
function _qts_get_template( $template_name, $args = array() ) {
	if ( ! $template_name ) {
		return ;
	}

	wc_get_template( $template_name, $args, QTS_BASENAME_DIR, QTS_TEMPLATE_PATH ) ;
}

/**
 * Generates a random key.
 * 
 * @param string $prefix
 * @param int $length
 * @return string
 */
function _qts_generate_key( $prefix = QTS_PREFIX, $length = 13 ) {
	if ( ! $prefix || ! is_string( $prefix ) ) {
		$prefix = '' ;
	}

	$length = absint( $length ) ;
	$key    = $prefix ;
	$key    .= wp_generate_password( $length, false ) ;

	return trim( $key ) ;
}

/**
 * Get Number Suffix to Display.
 * 
 * @param int $number
 * @return string
 */
function _qts_get_number_suffix( $number ) {
	// Special case 'teenth'
	if ( ( $number / 10 ) % 10 != 1 ) {
		// Handle 1st, 2nd, 3rd
		switch ( $number % 10 ) {
			case 1:
				return $number . 'st' ;
			case 2:
				return $number . 'nd' ;
			case 3:
				return $number . 'rd' ;
		}
	}
	// Everything else is 'nth'
	return $number . 'th' ;
}

/**
 * Get the type in which the array is sorted by.
 * 
 * @param array $array
 * @return boolean|string
 */
function _qts_array_sorted_by( $array ) {
	$o_array = $array ;

	$asc = $o_array ;
	sort( $asc ) ;
	if ( $o_array === $asc ) {
		return 'asc' ;
	}

	$desc = $o_array ;
	rsort( $desc ) ;
	if ( $o_array === $desc ) {
		return 'desc' ;
	}

	return false ;
}

/**
 * Trim our post status without prefix.
 * 
 * @param string $status
 * @return string
 */
function _qts_trim_post_status( $status ) {
	$status = QTS_PREFIX === substr( $status, 0, 5 ) ? substr( $status, 5 ) : $status ;
	return $status ;
}

/**
 * Return the step value for the given number of decimals.
 * 
 * @return float
 */
function _qts_get_number_input_step_value( $decimals = null ) {
	$decimals   = is_numeric( $decimals ) ? absint( $decimals ) : wc_get_price_decimals() ;
	$step_value = 1 ;

	for ( $i = 0 ; $i < $decimals ; $i++ ) {
		$step_value *= 10 ;
	}

	return ( 1 / $step_value ) ;
}

/**
 * Return the available payment gateways.
 * 
 * @return array
 */
function _qts_get_valid_payment_gateways() {
	$valid              = array() ;
	$available_gateways = WC()->payment_gateways->get_available_payment_gateways() ;

	foreach ( $available_gateways as $key => $gateway ) {
		$valid[ $key ] = $gateway->get_title() ;
	}

	return $valid ;
}

/**
 * Helper method to return the home endpoint.
 *
 * @return string home page.
 */
function _qts_get_home_endpoint() {
	$endpoint = home_url() ;

	if ( false === strpos( $endpoint, '?' ) ) {
		$endpoint = trailingslashit( $endpoint ) ;
	}

	return $endpoint ;
}

/**
 * Return the My Account > My Quote Requests url.
 * 
 * @return string
 */
function _qts_get_my_quote_requests_url() {
	return wc_get_endpoint_url( 'quote-requests', '', wc_get_page_permalink( 'myaccount' ) ) ;
}

/**
 * Calculate the preferred price from the requested quote.
 * 
 * @param array|stdclass $quote_item
 * @return float|null
 */
function _qts_calculate_preferred_price( $quote_item ) {
	$quote_item = ( object ) $quote_item ;

	if ( isset( $quote_item->requested_price ) && is_numeric( $quote_item->requested_price ) ) {
		$preferred_price = $quote_item->requested_price ;
	} else if ( isset( $quote_item->requested_price_percent ) && is_numeric( $quote_item->requested_price_percent ) ) {
		$preferred_price = ( $quote_item->original_price / 100 ) * floatval( $quote_item->requested_price_percent ) ;
	} else if ( isset( $quote_item->requested_discount_percent ) && is_numeric( $quote_item->requested_discount_percent ) ) {
		$preferred_price = max( 0, ( $quote_item->original_price - ( ( $quote_item->original_price / 100 ) * floatval( $quote_item->requested_discount_percent ) ) ) ) ;
	} else {
		$preferred_price = null ;
	}

	return $preferred_price ;
}
