<?php

defined( 'ABSPATH' ) || exit ;

/**
 * Parse about any English textual datetime description into a Unix timestamp
 * 
 * @param string $date
 * @param int|string $base
 * @return int
 */
function _qts_maybe_strtotime( $date, $base = null ) {
	if ( ! $date ) {
		return time() ;
	}

	if ( is_numeric( $date ) ) {
		return absint( $date ) ;
	}

	if ( is_string( $date ) ) {
		if ( $base ) {
			$base = _qts_maybe_strtotime( $base ) ;
		}

		return $base ? strtotime( $date, $base ) : strtotime( $date ) ;
	}
	return time() ;
}

/**
 * Prepare the given time in GMT/UTC date/time
 * 
 * @param int $time
 * @param string $format
 * @return string
 */
function _qts_prepare_datetime( $time = null, $format = 'Y-m-d H:i:s' ) {
	if ( $time ) {
		$time = _qts_maybe_strtotime( $time ) ;
		return gmdate( $format, $time ) ;
	}

	return gmdate( $format ) ;
}

/**
 * Get the time formatted in GMT/UTC 0 or +/- offset
 * 
 * @param string $type Type of time to retrieve. Accepts 'mysql', 'timestamp', or PHP date format string (e.g. 'Y-m-d').
 * @param array $args Accepted values are [
 *              'time' => Optional. A valid date/time string. If null then it returns the current time
 *              'base' => Optional. The timestamp which is used as a base for the calculation of relative dates
 *              'gmt'  => Optional. By default it will consider the WP offset
 *              ]
 * @return mixed
 */
function _qts_get_time( $type = 'mysql', $args = array() ) {
	$args = wp_parse_args( $args, array(
		'time' => null,
		'base' => null,
		'gmt'  => false,
			) ) ;

	$time = _qts_maybe_strtotime( $args[ 'time' ], $args[ 'base' ] ) ;

	if ( 'mysql' === $type || 'timestamp' === $type ) {
		$format = 'Y-m-d H:i:s' ;
	} else {
		$format = $type ;
	}

	$time = ( $args[ 'gmt' ] ) ? _qts_prepare_datetime( $time, $format ) : _qts_prepare_datetime( ( $time + ( get_option( 'gmt_offset' ) * HOUR_IN_SECONDS ) ), $format ) ;

	if ( 'timestamp' === $type ) {
		$time = _qts_maybe_strtotime( $time ) ;
	}

	return $time ;
}

/**
 * Format the date to output.
 * 
 * @param mixed $time A valid date/time string
 * @param bool $allow_time Need HH:MM:SS ?
 * @param bool $maybe_human Need human time in short period ?
 * @return string GMT/UTC +/- offset
 */
function _qts_format_datetime( $time, $allow_time = true, $maybe_human = false ) {
	if ( ! $time ) {
		return '-' ;
	}

	$wp_date_format = '' !== get_option( 'date_format' ) ? get_option( 'date_format' ) : 'F j, Y' ;
	$wp_time_format = '' !== get_option( 'time_format' ) ? get_option( 'time_format' ) : 'g:i a' ;
	$time           = _qts_maybe_strtotime( $time ) ;
	$time_diff      = _qts_get_time( 'timestamp' ) - $time ;

	if ( $maybe_human && $time_diff > 0 && $time_diff < DAY_IN_SECONDS ) {
		/* translators: 1: time */
		$displ_time = sprintf( __( '%1$s ago', 'quote-request-for-woocommerce' ), human_time_diff( $time, _qts_get_time( 'timestamp' ) ) ) ;
	} else {
		$displ_time = $allow_time ? date_i18n( "{$wp_date_format} {$wp_time_format}", $time ) : date_i18n( "{$wp_date_format}", $time ) ;
	}

	return $displ_time ;
}

/**
 * Get the human time difference for any time to output.
 * 
 * @param mixed $time A valid date/time string
 * @return string
 */
function _qts_get_human_time_diff( $time ) {
	if ( ! $time ) {
		return '-' ;
	}

	if ( is_string( $time ) ) {
		if ( _qts_maybe_strtotime( $time ) < _qts_get_time( 'timestamp' ) ) {
			$time = null ;
		}
	} elseif ( is_numeric( $time ) ) {
		if ( absint( $time ) < _qts_get_time( 'timestamp' ) ) {
			$time = null ;
		}
	}

	if ( $time ) {
		$fromDate = new DateTime( _qts_prepare_datetime( _qts_get_time( 'timestamp' ) ) ) ;
		$toDate   = new DateTime( _qts_prepare_datetime( $time ) ) ;
		return $toDate->diff( $fromDate )->format( '<b>%a</b> day(s), <b>%H</b> hour(s), <b>%I</b> minute(s), <b>%S</b> second(s)' ) ;
	}

	return 'now' ;
}

/**
 * Retrieve the array of dates between the given dates.
 * 
 * @param mixed $start_time A valid date/time string
 * @param mixed $end_time A valid date/time string
 * @param array $days_count Should be either array( '1','2','3',... ) or array( ...,'3','2','1' )
 * @param string $default_sortby asc|desc
 * @return array
 */
function _qts_get_dates( $start_time, $end_time, $days_count, $default_sortby = 'asc' ) {
	$dates = array() ;

	if ( empty( $days_count ) || ! is_array( $days_count ) ) {
		return $dates ;
	}

	$start_time = _qts_maybe_strtotime( $start_time ) ;
	$end_time   = _qts_maybe_strtotime( $end_time ) ;
	$sortby     = 1 === count( $days_count ) ? $default_sortby : _qts_array_sorted_by( $days_count ) ;

	foreach ( $days_count as $day_count ) {
		$day_count = absint( $day_count ) ;

		if ( $day_count ) {
			if ( 'asc' === $sortby ) {
				$datetime = _qts_get_time( 'timestamp', array( 'time' => "+{$day_count} days", 'base' => $start_time ) ) ;

				if ( $datetime <= $end_time ) {
					$dates[] = $datetime ;
				}
			} else {
				$datetime = _qts_get_time( 'timestamp', array( 'time' => "-{$day_count} days", 'base' => $end_time ) ) ;

				if ( $datetime >= $start_time ) {
					$dates[] = $datetime ;
				}
			}
		}
	}

	if ( $dates ) {
		$dates = array_unique( $dates ) ;
		sort( $dates ) ;
	}

	return $dates ;
}
