<?php

/**
 * A class for integrating with WooCommerce Blocks scripts.
 * 
 * @since 3.0.0
 */
use Automattic\WooCommerce\Blocks\Integrations\IntegrationInterface;

/**
 * Blocks integration class.
 *
 * @since 3.0.0
 * @class  QTS_Blocks_Integration
 * @implements IntegrationInterface
 * @package Class
 */
class QTS_Blocks_Integration implements IntegrationInterface {

	/**
	 * The single instance of the class.
	 *
	 * @since 3.0.0
	 * @var QTS_Blocks_Integration
	 */
	protected static $instance = null;

	/**
	 * Main QTS_Blocks_Integration instance. Ensures only one instance of QTS_Blocks_Integration is loaded or can be loaded.
	 *
	 * @since 3.0.0 
	 * @return QTS_Blocks_Integration
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}

		return self::$instance;
	}

	/**
	 * The name of the integration.
	 *
	 * @since 3.0.0
	 * @return string
	 */
	public function get_name() {
		return 'quote-request-for-woocommerce';
	}

	/**
	 * When called invokes any initialization/setup for the integration.
	 * 
	 * @since 3.0.0
	 */
	public function initialize() {      
		add_action( 'enqueue_block_editor_assets', array( $this, 'enqueue_block_editor_assets' ) );
		add_action( 'enqueue_block_assets', array( $this, 'enqueue_block_assets' ) );
	}

	/**
	 * Returns an array of script handles to enqueue in the frontend context.
	 *
	 * @since 3.0.0
	 * @return string[]
	 */
	public function get_script_handles() {
		return array( 'qts-quote-blocks-integration' );
	}

	/**
	 * Returns an array of script handles to enqueue in the editor context.
	 *
	 * @since 3.0.0
	 * @return string[]
	 */
	public function get_editor_script_handles() {
		return array( 'qts-quote-blocks-integration' );
	}

	/**
	 * An array of key, value pairs of data made available to the block on the client side.
	 *
	 * @since 3.0.0
	 * @return array
	 */
	public function get_script_data() {
		return array(
			'qts-quote-blocks-integration' => 'active',
		);
	}

	/**
	 * Get the file modified time as a cache buster if we're in dev mode.
	 *
	 * @since 3.0.0
	 * @param string $file Local path to the file.
	 * @return string The cache buster value to use for the given file.
	 */
	protected function get_file_version( $file ) {              
		if ( defined( 'SCRIPT_DEBUG' ) && SCRIPT_DEBUG && file_exists( $file ) ) {
			return filemtime( $file );
		}

		return QTS_VERSION;
	}

	/**
	 * Enqueue block assets for the editor.
	 *
	 * @since 3.0.0
	 */
	public function enqueue_block_editor_assets() {
		$script_path       = 'blocks/admin/index.js';
		$script_url        = QTS_URL . "/assets/{$script_path}";
		$script_asset_path = QTS_DIR . 'assets/blocks/admin/index.asset.php';
		$script_asset      = file_exists( $script_asset_path ) ? require $script_asset_path : array(
			'dependencies' => array(),
			'version'      => $this->get_file_version( $script_asset_path ),
		);
		
		wp_register_script(
				'qts-admin-blocks-integration',
				$script_url,
				$script_asset[ 'dependencies' ],
				$script_asset[ 'version' ],
				true
		);
		
		wp_enqueue_script( 'qts-admin-blocks-integration' );
	}

	/**
	 * Enqueue block assets for both editor and front-end.
	 *
	 * @since 3.0.0
	 */
	public function enqueue_block_assets() {
		$script_path = 'blocks/frontend/index.js';
		$style_path  = 'blocks/frontend/index.css';
		
		$script_url = QTS_URL . "/assets/{$script_path}";
		$style_url  = QTS_URL . "/assets/{$style_path}";
			
		$script_asset_path = QTS_DIR . 'assets/blocks/frontend/index.asset.php';
		$style_asset_path  = QTS_DIR . 'assets/blocks/frontend/index.css';
		
		$script_asset = file_exists( $script_asset_path ) ? require $script_asset_path : array(
			'dependencies' => array(),
			'version'      => $this->get_file_version( $script_asset_path ),
		);
		
		wp_enqueue_style(
				'qts-quote-blocks-integration',
				$style_url,
				'',
				$this->get_file_version( $style_asset_path ),
				'all'
		);
		
		wp_register_script(
				'qts-quote-blocks-integration',
				$script_url,
				$script_asset[ 'dependencies' ],
				$script_asset[ 'version' ],
				true
		);
		
		wp_set_script_translations(
				'qts-quote-blocks-integration',
				'quote-request-for-woocommerce',
				QTS_DIR . 'languages/'
		);

		return array( 'qts-quote-blocks-integration' );
	}
}
