<?php

defined( 'ABSPATH' ) || exit;

/**
 * Installation related functions and actions.
 * 
 * @class QTS_Install
 * @package Class
 */
class QTS_Install {

	/**
	 * Init Install.
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'check_version' ), 9 );
		add_filter( 'plugin_action_links_' . QTS_BASENAME, array( __CLASS__, 'plugin_action_links' ) );
	}

	/**
	 * Check Quote Request for WooCommerce version and run updater
	 */
	public static function check_version() {
		if ( get_option( QTS_PREFIX . 'version' ) !== QTS_VERSION ) {
			self::install();

			/**
			 * Trigger after the plugin is updated.
			 * 
			 * @since 1.0
			 */
			do_action( 'qts_updated' );
		}
	}

	/**
	 * Install Quote Request for WooCommerce.
	 */
	public static function install() {
		if ( ! defined( 'QTS_INSTALLING' ) ) {
			define( 'QTS_INSTALLING', true );
		}

		self::create_options();
		self::create_pages();
		self::maybe_create_default_form_fields();
		self::update_qts_version();

		/**
		 * Trigger after the plugin is installed.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_installed' );
	}

	/**
	 * Is this a brand new Quote Request for WooCommerce install?
	 * A brand new install has no version yet.
	 *
	 * @return bool
	 */
	private static function is_new_install() {
		return is_null( get_option( QTS_PREFIX . 'version', null ) );
	}

	/**
	 * Update Quote Request for WooCommerce version to current.
	 */
	private static function update_qts_version() {
		delete_option( QTS_PREFIX . 'version' );
		add_option( QTS_PREFIX . 'version', QTS_VERSION );
	}

	/**
	 * Default options.
	 *
	 * Sets up the default options used on the settings page.
	 */
	private static function create_options() {
		// Include settings so that we can run through defaults.
		include_once __DIR__ . '/admin/class-qts-admin-settings.php';
		QTS_Admin_Settings::save_default_options();
	}

	/**
	 * Create pages that the plugin relies on, storing page IDs in variables.
	 */
	public static function create_pages() {
		include_once __DIR__ . '/admin/qts-admin-functions.php';

		/**
		 * Get an args to create pages.
		 * 
		 * @since 1.0
		 */
		$pages = apply_filters( 'qts_create_pages', array(
			'qts_quote' => array(
				'name'    => _x( 'qts_quote', 'Page slug', 'quote-request-for-woocommerce' ),
				'title'   => 'Quote Request Page',
				'content' => '<!-- wp:shortcode -->[qts_quote]<!-- /wp:shortcode -->',
			),
				) );

		foreach ( $pages as $key => $page ) {
			_qts_create_page( esc_sql( $page[ 'name' ] ), 'woocommerce_' . $key . '_page_id', $page[ 'title' ], $page[ 'content' ] );
		}
	}

	/**
	 * Maybe create default form fields.
	 */
	public static function maybe_create_default_form_fields() {
		if ( 'yes' === get_option( QTS_PREFIX . 'default_form_fields_created' ) ) {
			return;
		}

		QTS_Form_Fields::create_default_form_fields();
		update_option( QTS_PREFIX . 'default_form_fields_created', 'yes' );
	}

	/**
	 * Show action links on the plugin screen.
	 *
	 * @param   mixed $links Plugin Action links
	 * @return  array
	 */
	public static function plugin_action_links( $links ) {
		$setting_page_link = '<a  href="' . admin_url( 'admin.php?page=qts_settings' ) . '">' . esc_html__( 'Settings', 'quote-request-for-woocommerce' ) . '</a>';
		array_unshift( $links, $setting_page_link );
		return $links;
	}
}

QTS_Install::init();
