<?php

defined( 'ABSPATH' ) || exit;

/**
 * Load Payment Gateways
 * 
 * @class QTS_Payment_Gateways
 * @package Class
 */
class QTS_Payment_Gateways {

	/**
	 * Get our payment gateways to load in to the WC checkout
	 *
	 * @var array 
	 */
	protected static $load_gateways = array();

	/**
	 * The single instance of the class.
	 */
	protected static $instance = null;

	/**
	 * Get the disabled payment gateways in checkout
	 * 
	 * @var array 
	 */
	protected static $disabled_payment_gateways;

	/**
	 * Create instance for QTS_Payment_Gateways.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Init QTS_Payment_Gateways.
	 */
	public function init() {
		add_action( 'plugins_loaded', array( $this, 'load_payment_gateways' ), 20 );
		add_filter( 'woocommerce_payment_gateways', array( $this, 'add_payment_gateways' ) );
		add_filter( 'woocommerce_available_payment_gateways', array( $this, 'set_payment_gateways' ), 999 );
		add_action( 'woocommerce_blocks_loaded', array( $this, 'add_gateway_quote_block_support' ) );
	}

	/**
	 * Get the disabled payment gateways in checkout to pay for the quote request order.
	 * 
	 * @return array 
	 */
	public function get_disabled_payment_gateways() {
		if ( is_array( self::$disabled_payment_gateways ) ) {
			return self::$disabled_payment_gateways;
		}

		self::$disabled_payment_gateways = get_option( QTS_PREFIX . 'disabled_payment_gateways_while_making_payment', array() );
		return self::$disabled_payment_gateways;
	}

	/**
	 * Get payment gateways to load in to the WC checkout
	 */
	public function load_payment_gateways() {
		if ( ! class_exists( 'WC_Payment_Gateway' ) ) {
			return;
		}

		self::$load_gateways[] = include_once 'gateways/quote-request/class-qts-quote-request-gateway.php';
	}

	/**
	 * Add payment gateways awaiting to load
	 * 
	 * @param object $gateways
	 * @return array
	 */
	public function add_payment_gateways( $gateways ) {
		if ( empty( self::$load_gateways ) ) {
			return $gateways;
		}

		foreach ( self::$load_gateways as $gateway ) {
			$gateways[] = $gateway;
		}
		return $gateways;
	}

	/**
	 * Check whether specific payment gateway is needed in checkout
	 * 
	 * @param WC_Payment_Gateway $gateway
	 * @return bool
	 */
	public function need_payment_gateway( $gateway ) {
		$need = true;

		if ( is_checkout_pay_page() ) {
			if ( $gateway->supports( 'qtsquote_request' ) ) {
				$need = false;
			} else if ( in_array( $gateway->id, ( array ) self::get_disabled_payment_gateways() ) ) {
				$need = false;
			}
		} elseif ( QTS_Cart::asking_for_quote() ) {
			if ( ! $gateway->supports( 'qtsquote_request' ) ) {
				$need = false;
			}
		} else if ( QTS_Cart::awaiting_for_payment() ) {
			if ( $gateway->supports( 'qtsquote_request' ) ) {
				$need = false;
			} else if ( in_array( $gateway->id, ( array ) self::get_disabled_payment_gateways() ) ) {
				$need = false;
			}
		}

		/**
		 * Need the payment gateway to be available on checkout?
		 * 
		 * @since 1.0
		 */
		return apply_filters( 'qts_need_payment_gateway', $need, $gateway );
	}

	/**
	 * Handle payment gateways in checkout
	 * 
	 * @param array $_available_gateways
	 * @return array
	 */
	public function set_payment_gateways( $_available_gateways ) {
		if ( is_admin() || is_null( WC()->cart ) ) {
			return $_available_gateways;
		}

		foreach ( $_available_gateways as $gateway_name => $gateway ) {
			if ( ! isset( $gateway->id ) ) {
				continue;
			}

			if ( ! self::need_payment_gateway( $gateway ) ) {
				unset( $_available_gateways[ $gateway_name ] );
			}
		}
		return $_available_gateways;
	}

	/**
	 * Hook in Blocks integration. This action is called in a callback on plugins loaded, so current quote gateway class
	 */
	public function add_gateway_quote_block_support() {
		if ( class_exists( 'Automattic\WooCommerce\Blocks\Payments\Integrations\AbstractPaymentMethodType' ) ) {
			include_once 'gateways/quote-request/class-qts-quote-request-gateway-blocks-support.php';

			add_action( 'woocommerce_blocks_payment_method_type_registration', array( self::instance(), 'register_gateway_quote_block_support' ), 5 );
		}
	}

	/**
	 * Priority is important here because this ensures this integration is registered before the WooCommerce Blocks built-in quote gateway registration.
	 * Blocks code has a check in place to only register if quote gateway is not already registered.
	 */
	public function register_gateway_quote_block_support( Automattic\WooCommerce\Blocks\Payments\PaymentMethodRegistry $payment_method_registry ) {
		if ( ! class_exists( 'QTS_Quote_Request_Gateway' ) ) {
			return;
		}

		$container = Automattic\WooCommerce\Blocks\Package::container();
		$container->register( QTS_Quote_Request_Gateway_Blocks_Support::class, function () {
			return new QTS_Quote_Request_Gateway_Blocks_Support();
		} );
		$payment_method_registry->register( $container->get( QTS_Quote_Request_Gateway_Blocks_Support::class ) );
	}
}
