<?php

defined( 'ABSPATH' ) || exit;

/**
 * Initiate Plugin Core class.
 * 
 * @class WC_Quote_Request
 * @package Class
 */
final class WC_Quote_Request {

	/**
	 * Plugin version.
	 * 
	 * @var string 
	 */
	public $version = '3.4.0';

	/**
	 * Plugin prefix.
	 * 
	 * @var string 
	 */
	public $prefix = '_qts_';

	/**
	 * Get Query instance.
	 * 
	 * @var QTS_Query 
	 */
	public $query;

	/**
	 * QTS Quote instance.
	 *
	 * @var QTS_Quote
	 */
	public $quote = null;

	/**
	 * Collect the queued jobs.
	 * 
	 * @var QTS_Queue[] 
	 */
	public $queue = array();

	/**
	 * The single instance of the class.
	 */
	protected static $instance = null;

	/**
	 * WC_Quote_Request constructor.
	 */
	public function __construct() {
		$this->init_plugin_dependencies();

		if ( true !== $this->plugin_dependencies_met() ) {
			return; // Return to stop the existing function to be call 
		}

		$this->define_constants();
		$this->include_files();
		$this->init_hooks();
	}

	/**
	 * Cloning is forbidden.
	 */
	public function __clone() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Cloning is forbidden.', 'quote-request-for-woocommerce' ), '1.2' );
	}

	/**
	 * Unserializing instances of this class is forbidden.
	 */
	public function __wakeup() {
		_doing_it_wrong( __FUNCTION__, esc_html__( 'Unserializing instances of this class is forbidden.', 'quote-request-for-woocommerce' ), '1.2' );
	}

	/**
	 * Auto-load in-accessible properties on demand.
	 *
	 * @param mixed $key Key name.
	 * @return mixed
	 */
	public function __get( $key ) {
		if ( in_array( $key, array( 'gateways', 'mailer' ), true ) ) {
			return $this->$key();
		}
	}

	/**
	 * Main WC_Quote_Request Instance.
	 * Ensures only one instance of WC_Quote_Request is loaded or can be loaded.
	 * 
	 * @return WC_Quote_Request - Main instance.
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}

	/**
	 * Init plugin dependencies.
	 */
	private function init_plugin_dependencies() {
		include_once ABSPATH . 'wp-admin/includes/plugin.php' ; //Prevent fatal error by load the files when you might call init hook.
		add_action( 'init', array( $this, 'prevent_header_sent_problem' ), 1 );
		add_action( 'admin_notices', array( $this, 'plugin_dependencies_notice' ) );
	}

	/**
	 * Prevent header problem while plugin activates.
	 */
	public function prevent_header_sent_problem() {
		ob_start();
	}

	/**
	 * Check whether the plugin dependencies met.
	 * 
	 * @return bool|string True on Success
	 */
	private function plugin_dependencies_met( $return_dep_notice = false ) {
		$return = false;

		if ( is_multisite() && is_plugin_active_for_network( 'woocommerce/woocommerce.php' ) && is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
			$is_wc_active = true;
		} else if ( is_plugin_active( 'woocommerce/woocommerce.php' ) ) {
			$is_wc_active = true;
		} else {
			$is_wc_active = false;
		}

		// WC check.
		if ( ! $is_wc_active ) {
			if ( $return_dep_notice ) {
				$return = 'Quote Request for WooCommerce Plugin requires WooCommerce Plugin should be Active !!!';
			}

			return $return;
		}

		return true;
	}

	/**
	 * Output a admin notice when plugin dependencies not met.
	 */
	public function plugin_dependencies_notice() {
		$return = $this->plugin_dependencies_met( true );

		if ( true !== $return && current_user_can( 'activate_plugins' ) ) {
			$dependency_notice = $return;
			printf( '<div class="error"><p>%s</p></div>', wp_kses_post( $dependency_notice ) );
		}
	}

	/**
	 * Define constants.
	 */
	private function define_constants() {
		$this->define( 'QTS_DIR', plugin_dir_path( QTS_FILE ) );
		$this->define( 'QTS_BASENAME', plugin_basename( QTS_FILE ) );
		$this->define( 'QTS_BASENAME_DIR', trailingslashit( dirname( QTS_BASENAME ) ) );
		$this->define( 'QTS_TEMPLATE_PATH', QTS_DIR . 'templates/' );
		$this->define( 'QTS_URL', untrailingslashit( plugins_url( '/', QTS_FILE ) ) );
		$this->define( 'QTS_VERSION', $this->version );
		$this->define( 'QTS_PREFIX', $this->prefix );
	}

	/**
	 * Define constant if not already set.
	 *
	 * @param string      $name  Constant name.
	 * @param string|bool $value Constant value.
	 */
	public function define( $name, $value ) {
		if ( ! defined( $name ) ) {
			define( $name, $value );
		}
	}

	/**
	 * Is frontend request ?
	 *
	 * @return bool
	 */
	private function is_frontend() {
		return ( ! is_admin() || defined( 'DOING_AJAX' ) ) && ! defined( 'DOING_CRON' );
	}

	/**
	 * Include required core files used in admin and on the frontend.
	 */
	private function include_files() {
		/**
		 * Class autoloader.
		 */
		include_once 'class-qts-autoload.php';

		/**
		 * Abstract classes.
		 */
		include_once 'abstracts/abstract-qts-settings.php';

		/**
		 * Core functions.
		 */
		include_once 'qts-core-functions.php';

		/**
		 * Core classes.
		 */
		include_once 'class-qts-post-types.php';
		include_once 'class-qts-install.php';
		include_once 'class-qts-ajax.php';
		include_once 'class-qts-enqueues.php';
		include_once 'privacy/class-qts-privacy.php';
		include_once 'class-qts-shortcode-quote.php';
		include_once 'class-qts-order-manager.php';
		include_once 'class-qts-blocks-compatibility.php';

		if ( is_admin() ) {
			include_once 'admin/class-qts-admin.php';
		}

		if ( $this->is_frontend() ) {
			$this->frontend_includes();
		}

		$this->gateways->init();
		$this->query = new QTS_Query();
	}

	/**
	 * Include required frontend files.
	 */
	private function frontend_includes() {
		include_once 'class-qts-add-to-quote.php';
		include_once 'class-qts-cart.php';
		include_once 'class-qts-quote.php';
		include_once 'qts-template-hooks.php';
		include_once 'class-qts-form-handler.php';
	}

	/**
	 * Hook into actions and filters.
	 */
	private function init_hooks() {
		register_activation_hook( QTS_FILE, array( 'QTS_Install', 'install' ) );
		register_deactivation_hook( QTS_FILE, array( $this, 'upon_deactivation' ) );
		add_filter( 'woocommerce_qts_quote_request_data_store', array( $this, 'get_orders_table_data_store' ), 10, 2 );
		add_filter( 'woocommerce_data_stores', array( $this, 'add_data_stores' ) );
		add_action( 'plugins_loaded', array( $this, 'on_plugins_loaded' ), 5 );
		add_action( 'init', array( $this, 'init' ), 5 );
	}

	/**
	 * Fire upon deactivating Quote Request for WooCommerce
	 */
	public function upon_deactivation() {
		delete_option( 'qts_flush_rewrite_rules' );
		_qts_cancel_all_queue_process();
	}

	/**
	 * When the custom_order_tables feature is enabled, return the quote request datastore class.
	 *
	 * @param string $default_data_store The data store class name.
	 * @return string
	 */
	public function get_orders_table_data_store( $default_data_store ) {
		if ( ! _qts_is_custom_order_tables_usage_enabled() ) {
			return $default_data_store;
		}

		$data_store = new QTS_Orders_Table_Quote_Request_Data_Store();
		$data_store->init(
				wc_get_container()->get( \Automattic\WooCommerce\Internal\DataStores\Orders\OrdersTableDataStoreMeta::class ),
				wc_get_container()->get( \Automattic\WooCommerce\Internal\Utilities\DatabaseUtil::class ),
				wc_get_container()->get( \Automattic\WooCommerce\Proxies\LegacyProxy::class )
		);

		return $data_store;
	}

	/**
	 * Add our data stores to WC.
	 * 
	 * @param array $data_stores
	 * @return array
	 */
	public static function add_data_stores( $data_stores ) {
		$data_stores[ 'qts_quote_request' ] = 'QTS_Quote_Request_Data_Store_CPT';
		return $data_stores;
	}

	/**
	 * When WP has loaded all plugins, trigger the `qts_loaded` hook.
	 */
	public function on_plugins_loaded() {
		QTS_Form_Fields::init();

		/**
		 * Trigger after the plugin is loaded.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_loaded' );
	}

	/**
	 * Load Localization files.
	 */
	public function load_plugin_textdomain() {
		if ( function_exists( 'determine_locale' ) ) {
			$locale = determine_locale();
		} else {
			$locale = is_admin() ? get_user_locale() : get_locale();
		}

		/**
		 * Get the plugin text domain.
		 * 
		 * @since 1.0
		 */
		$locale = apply_filters( 'plugin_locale', $locale, 'quote-request-for-woocommerce' );

		unload_textdomain( 'quote-request-for-woocommerce' );
		load_textdomain( 'quote-request-for-woocommerce', WP_LANG_DIR . '/quote-request-for-woocommerce/quote-request-for-woocommerce-' . $locale . '.mo' );
		load_textdomain( 'quote-request-for-woocommerce', WP_LANG_DIR . '/plugins/quote-request-for-woocommerce-' . $locale . '.mo' );
		load_plugin_textdomain( 'quote-request-for-woocommerce', false, QTS_BASENAME_DIR . 'languages' );
	}

	/**
	 * Init WC_Quote_Request when WordPress Initializes. 
	 */
	public function init() {
		/**
		 * Trigger before init the plugin.
		 * 
		 * @since 1.0
		 */
		do_action( 'before_qts_init' );

		$this->load_plugin_textdomain();
		$this->maybe_load_quote();
		$this->mailer->init(); //Load mailer
		$this->init_background_process();

		/**
		 * Trigger after init the plugin.
		 * 
		 * @since 1.0
		 */
		do_action( 'qts_init' );
	}

	/**
	 * Maybe initialize and load the quote functionality.
	 */
	private function maybe_load_quote() {
		if ( ! $this->is_frontend() ) {
			return;
		}

		// Initialize the quote.
		if ( WC()->cart && ( is_null( $this->quote ) || ! $this->quote instanceof QTS_Quote ) ) {
			$this->quote = new QTS_Quote();
		}
	}

	/**
	 * Init the background process.
	 */
	protected function init_background_process() {
		include_once 'background-process/qts-background-functions.php';
		include_once 'background-process/abstract-qts-background-process.php';
		include_once 'queue/abstract-qts-queue.php';

		/**
		 * Get the queue classes.
		 * 
		 * @since 1.0
		 */
		$queues = apply_filters( 'qts_queue_classes', array(
			'quote_request' => 'QTS_Quote_Request_Queue',
				) );

		foreach ( $queues as $name => $class_name ) {
			if ( ! class_exists( $class_name ) ) {
				continue;
			}

			$this->queue[ $name ] = new $class_name();
			$this->queue[ $name ]->dispatch();
		}
	}

	/**
	 * Include classes for plugin support.
	 */
	private function other_plugin_support_includes() {
	}

	/**
	 * Get gateways class.
	 *
	 * @return QTS_Payment_Gateways
	 */
	public function gateways() {
		return QTS_Payment_Gateways::instance();
	}

	/**
	 * Email Class.
	 *
	 * @return QTS_Emails
	 */
	public function mailer() {
		return QTS_Emails::instance();
	}
}
