<?php

defined( 'ABSPATH' ) || exit;

/**
 * Return the array of form field types.
 *
 * @return array
 */
function _qts_get_form_field_types() {
	/**
	 * Get form field types.
	 * 
	 * @since 1.0
	 */
	return apply_filters( 'qts_form_field_types', array(
		'text'            => __( 'Text', 'quote-request-for-woocommerce' ),
		'textarea'        => __( 'Textarea', 'quote-request-for-woocommerce' ),
		'checkbox'        => __( 'Checkbox', 'quote-request-for-woocommerce' ),
		'number'          => __( 'Number', 'quote-request-for-woocommerce' ),
		'email'           => __( 'Email', 'quote-request-for-woocommerce' ),
		'tel'             => __( 'Phone', 'quote-request-for-woocommerce' ),
		'password'        => __( 'Password', 'quote-request-for-woocommerce' ),
		'date'            => __( 'Date Picker', 'quote-request-for-woocommerce' ),
		'time'            => __( 'Time', 'quote-request-for-woocommerce' ),
		'country'         => __( 'Country', 'quote-request-for-woocommerce' ),
		'state'           => __( 'State', 'quote-request-for-woocommerce' ),
		'radio'           => __( 'Radio', 'quote-request-for-woocommerce' ),
		'select'          => __( 'Select', 'quote-request-for-woocommerce' ),
		'qts_multiselect' => __( 'Multiselect', 'quote-request-for-woocommerce' ),
			) );
}

/**
 * Get the form field statuses.
 * 
 * @return array
 */
function _qts_get_form_field_statuses() {
	$form_field_statuses = array(
		QTS_PREFIX . 'enabled'  => __( 'Enabled', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'disabled' => __( 'Disabled', 'quote-request-for-woocommerce' ),
	);

	return $form_field_statuses;
}

/**
 * Get the form field status name.
 * 
 * @param string $status
 * @return string
 */
function _qts_get_form_field_status_name( $status ) {
	$statuses = _qts_get_form_field_statuses();
	$status   = _qts_trim_post_status( $status );
	$status   = isset( $statuses[ QTS_PREFIX . $status ] ) ? $statuses[ QTS_PREFIX . $status ] : $status;
	return $status;
}

/**
 * Get the form field type name.
 * 
 * @param string $type
 * @return string
 */
function _qts_get_form_field_type_name( $type ) {
	$types = _qts_get_form_field_types();
	$type  = isset( $types[ $type ] ) ? $types[ $type ] : ucwords( str_replace( array( '-', '_', 'qts' ), ' ', $type ) );
	return $type;
}

/**
 * Retrieve active form fields.
 * 
 * @return array
 */
function _qts_get_active_form_fields() {
	global $wpdb;

	$wpdb_ref           = &$wpdb;
	$active_form_fields = $wpdb_ref->get_col(
			$wpdb_ref->prepare( "SELECT DISTINCT ID FROM {$wpdb_ref->posts}
                            WHERE post_type='%s' AND post_status = '_qts_enabled' ORDER BY menu_order ASC"
					, 'qts_form_field'
			) );

	return $active_form_fields;
}

/**
 * Retrieve all form fields.
 * 
 * @return array
 */
function _qts_get_all_form_fields() {
	global $wpdb;

	$wpdb_ref        = &$wpdb;
	$all_form_fields = $wpdb_ref->get_col(
			$wpdb_ref->prepare( "SELECT DISTINCT ID FROM {$wpdb_ref->posts}
                            WHERE post_type='%s' AND post_status IN ('" . implode( "','", array_map( 'esc_sql', array_keys( _qts_get_form_field_statuses() ) ) ) . "') ORDER BY menu_order ASC"
					, 'qts_form_field'
			) );

	return $all_form_fields;
}

/**
 * Create the form field from the requested args.
 * 
 * @param array $args
 * @return int
 */
function _qts_create_form_field( $args = array() ) {
	$args          = wp_parse_args( $args, QTS_Form_Fields::get_default_props() );
	$field_post_id = wp_insert_post( array(
		'post_type'    => 'qts_form_field',
		'post_status'  => QTS_PREFIX . _qts_trim_post_status( $args[ 'status' ] ),
		'post_title'   => empty( $args[ 'label' ] ) ? esc_html__( 'New Field', 'quote-request-for-woocommerce' ) : esc_html( $args[ 'label' ] ),
		'post_excerpt' => empty( $args[ 'type' ] ) ? 'text' : sanitize_title( $args[ 'type' ] ),
		'post_content' => sanitize_textarea_field( $args[ 'description' ] ),
		'menu_order'   => absint( $args[ 'priority' ] ),
		'ping_status'  => 'closed',
		'post_parent'  => 0,
		'post_author'  => 1,
			) );

	update_post_meta( $field_post_id, '_name', '' !== $args[ 'name' ] ? sanitize_title( $args[ 'name' ] ) : QTS_Form_Fields::prepare_name( $field_post_id )  );
	update_post_meta( $field_post_id, '_required', ! empty( $args[ 'required' ] ) ? '1' : ''  );
	update_post_meta( $field_post_id, '_maxlength', ! empty( $args[ 'maxlength' ] ) ? '1' : ''  );
	update_post_meta( $field_post_id, '_autocomplete', ! empty( $args[ 'autocomplete' ] ) ? '1' : ''  );

	if ( '' !== $args[ 'placeholder' ] ) {
		update_post_meta( $field_post_id, '_placeholder', wc_clean( $args[ 'placeholder' ] ) );
	}

	if ( '' !== $args[ 'default' ] ) {
		update_post_meta( $field_post_id, '_default', wc_clean( $args[ 'default' ] ) );
	}

	if ( '' !== $args[ 'autofocus' ] ) {
		update_post_meta( $field_post_id, '_autofocus', wc_clean( $args[ 'autofocus' ] ) );
	}

	if ( is_array( $args[ 'options' ] ) && ! empty( $args[ 'options' ] ) ) {
		update_post_meta( $field_post_id, '_options', $args[ 'options' ] );
	}

	if ( is_array( $args[ 'class' ] ) && ! empty( $args[ 'class' ] ) ) {
		update_post_meta( $field_post_id, '_class', $args[ 'class' ] );
	}

	if ( is_array( $args[ 'label_class' ] ) && ! empty( $args[ 'label_class' ] ) ) {
		update_post_meta( $field_post_id, '_label_class', $args[ 'label_class' ] );
	}

	if ( is_array( $args[ 'input_class' ] ) && ! empty( $args[ 'input_class' ] ) ) {
		update_post_meta( $field_post_id, '_input_class', $args[ 'input_class' ] );
	}

	if ( is_array( $args[ 'validate' ] ) && ! empty( $args[ 'validate' ] ) ) {
		update_post_meta( $field_post_id, '_validate', $args[ 'validate' ] );
	}

	return $field_post_id;
}
