<?php

defined( 'ABSPATH' ) || exit;

/**
 * Get the add to quote url.
 *
 * @param WC_Product $product
 * @param string $mode Get the add to quote url based upon mode.
 * @param string $default_permalink 
 * @return string
 */
function _qts_add_to_quote_url( $product, $mode = 'approx', $default_permalink = '' ) {
	if ( ! $default_permalink ) {
		$default_permalink = get_permalink( $product->get_id() );
	}

	if ( ! $product->is_type( array( 'simple', 'variable', 'variation' ) ) || ! $product->is_purchasable() ) {
		return $default_permalink;
	}

	if ( 'accurate' === $mode ) {
		if ( $product->is_type( 'variation' ) ) {
			$url = add_query_arg( 'adding_qtsquote', true, $product->add_to_cart_url() );
		} else {
			$url = remove_query_arg( 'added-to-cart', add_query_arg( array( 'add-to-cart' => $product->get_id(), 'adding_qtsquote' => true ), $default_permalink ) );
		}
	} else if ( 'approx' === $mode ) {
		if ( $product->is_type( 'variation' ) ) {
			$url = add_query_arg( array( 'variation_id' => $product->get_id(), 'add-to-qtsquote' => $product->get_parent_id() ), $default_permalink );
		} else if ( $product->is_type( 'variable' ) ) {
			$url = add_query_arg( array( 'add-to-qtsquote' => $product->get_id() ), $default_permalink );
		} else {
			$url = $product->is_in_stock() ? add_query_arg( array( 'add-to-qtsquote' => $product->get_id() ), $default_permalink ) : $default_permalink;
		}
	} else {
		$url = $default_permalink;
	}

	/**
	 * Get product add to quote url.
	 * 
	 * @since 1.0
	 */
	return apply_filters( 'qts_product_add_to_quote_url', $url, $product );
}

/**
 * Gets the url to the quote page.
 *
 * @param string $mode Get the quote url based upon mode.
 * @return string Url to quote page
 */
function _qts_get_quote_url( $mode = 'approx' ) {
	if ( 'accurate' === $mode ) {
		return wc_get_cart_url();
	}

	return wc_get_page_permalink( 'qts_quote' );
}

/**
 * Gets the url to remove an item from the quote.
 *
 * @param string $quote_item_key
 * @return string url to page
 */
function _qts_get_quote_remove_url( $quote_item_key ) {
	$quote_page_url = _qts_get_quote_url();

	return $quote_page_url ? wp_nonce_url( add_query_arg( 'remove_item', $quote_item_key, $quote_page_url ), 'qts-quote', QTS_PREFIX . 'nonce' ) : '';
}

/**
 * Gets the url to re-add an item into the quote.
 *
 * @param  string $quote_item_key
 * @return string url to page
 */
function _qts_get_quote_undo_url( $quote_item_key ) {
	$quote_page_url = _qts_get_quote_url();
	$query_args     = array( 'undo_item' => $quote_item_key );

	return $quote_page_url ? wp_nonce_url( add_query_arg( $query_args, $quote_page_url ), 'qts-quote', QTS_PREFIX . 'nonce' ) : '';
}

/**
 * Clears the quote session when called.
 * 
 * @param string $mode empty the quote based upon mode.
 */
function _qts_empty_quote( $mode = 'approx' ) {
	if ( 'accurate' === $mode ) {
		if ( is_null( WC()->cart ) ) {
			return;
		}

		array_map( array( WC()->cart, 'remove_cart_item' ), array_keys( QTS_Cart::browse_added_quote_list() ) );
	} else if ( 'approx' === $mode ) {
		if ( ! isset( _qts()->quote ) || '' === _qts()->quote ) {
			_qts()->quote = new QTS_Quote();
		}

		_qts()->quote->empty_quote();
	}
}

/**
 * Prevent password protected products being added to the quote.
 *
 * @param  bool $passed
 * @param  int  $product_id Product ID.
 * @return bool
 */
function _qts_protected_product_add_to_quote( $passed, $product_id ) {
	if ( post_password_required( $product_id ) ) {
		wc_add_notice( __( 'This product is protected and cannot be purchased.', 'quote-request-for-woocommerce' ), 'error' );
		return false;
	}

	return $passed;
}

add_filter( 'qts_add_to_quote_validation', '_qts_protected_product_add_to_quote', 10, 2 );

/**
 * Add to quote messages.
 *
 * @param int|array $products
 * @param bool $show_qty
 * @param bool $return
 *
 * @return mixed
 */
function _qts_add_to_quote_message( $products, $show_qty = false, $return = false ) {
	$titles = array();
	$count  = 0;

	if ( ! is_array( $products ) ) {
		$products = array( $products => 1 );
		$show_qty = false;
	}

	if ( ! $show_qty ) {
		$products = array_fill_keys( array_keys( $products ), 1 );
	}

	foreach ( $products as $product_id => $qty ) {
		/* translators: %s: product name */
		$titles[] = ( $qty > 1 ? absint( $qty ) . ' &times; ' : '' ) . sprintf( _x( '&ldquo;%s&rdquo;', 'Item name in quotes', 'quote-request-for-woocommerce' ), strip_tags( get_the_title( $product_id ) ) );
		$count    += $qty;
	}

	$titles     = array_filter( $titles );
	$added_text = wp_kses_post( get_option( QTS_PREFIX . 'added_to_quote_msg', __( 'You have successfully added {product(s)} to your quote.', 'quote-request-for-woocommerce' ) ) );
	$added_text = str_replace( '{product(s)}', wc_format_list_of_items( $titles ), $added_text );

	// Output success messages.
	if ( 'yes' === get_option( QTS_PREFIX . 'quote_redirect_after_add' ) ) {
		$message = sprintf( '<a href="%s" tabindex="1" class="button wc-forward">%s</a> %s', esc_url( wc_get_page_permalink( 'shop' ) ), esc_html__( 'Continue shopping', 'quote-request-for-woocommerce' ), esc_html( $added_text ) );
	} else {
		$message = sprintf( '<a href="%s" tabindex="1" class="button wc-forward">%s</a> %s', esc_url( _qts_get_quote_url() ), esc_html__( 'View quote', 'quote-request-for-woocommerce' ), esc_html( $added_text ) );
	}

	if ( $return ) {
		return $message;
	} else {
		wc_add_notice( $message, 'success' );
	}
}

/**
 * Gets and formats a list of quote item data + variations for display on the frontend.
 *
 * @param array $quote_item Quote item object.
 * @param bool  $flat Should the data be returned flat or in a list.
 * @return string
 */
function _qts_get_formatted_quote_item_data( $quote_item, $flat = false ) {
	$item_data = array();

	// Variation values are shown only if they are not found in the title as of WC 3.0.
	// This is because variation titles display the attributes.
	if ( $quote_item[ 'data' ]->is_type( 'variation' ) && is_array( $quote_item[ 'variation' ] ) ) {
		foreach ( $quote_item[ 'variation' ] as $name => $value ) {
			$taxonomy = wc_attribute_taxonomy_name( str_replace( 'attribute_pa_', '', urldecode( $name ) ) );

			if ( taxonomy_exists( $taxonomy ) ) {
				// If this is a term slug, get the term's nice name.
				$term = get_term_by( 'slug', $value, $taxonomy );
				if ( ! is_wp_error( $term ) && $term && $term->name ) {
					$value = $term->name;
				}
				$label = wc_attribute_label( $taxonomy );
			} else {
				/**
				 * If this is a custom option slug, get the options name.
				 * 
				 * @since 2.9.0
				 */
				$value = apply_filters( 'woocommerce_variation_option_name', $value, null, $taxonomy, $quote_item[ 'data' ] );
				$label = wc_attribute_label( str_replace( 'attribute_', '', $name ), $quote_item[ 'data' ] );
			}

			// Check the nicename against the title.
			if ( '' === $value || wc_is_attribute_in_product_name( $value, $quote_item[ 'data' ]->get_name() ) ) {
				continue;
			}

			$item_data[] = array(
				'key'   => $label,
				'value' => $value,
			);
		}
	}

	/**
	 * Filter item data to allow 3rd parties to add more to the array.
	 * 
	 * @since 1.0
	 */
	$item_data = apply_filters( 'qts_get_item_data', $item_data, $quote_item );

	// Format item data ready to display.
	foreach ( $item_data as $key => $data ) {
		// Set hidden to true to not display meta on quote.
		if ( ! empty( $data[ 'hidden' ] ) ) {
			unset( $item_data[ $key ] );
			continue;
		}

		$item_data[ $key ][ 'key' ]     = ! empty( $data[ 'key' ] ) ? $data[ 'key' ] : $data[ 'name' ];
		$item_data[ $key ][ 'display' ] = ! empty( $data[ 'display' ] ) ? $data[ 'display' ] : $data[ 'value' ];
	}

	if ( count( $item_data ) > 0 ) {
		ob_start();

		if ( $flat ) {
			foreach ( $item_data as $data ) {
				echo esc_html( $data[ 'key' ] ) . ': ' . wp_kses_post( $data[ 'display' ] ) . "\n";
			}
		} else {
			_qts_get_template( 'quote/quote-item-data.php', array( 'item_data' => $item_data ) );
		}

		return ob_get_clean();
	}

	return '';
}

/**
 * Get the original product price to ask for quote by the customer.
 * 
 * @param int|WC_Product $product
 * @return float Admin offered price.
 */
function _qts_get_original_price( $product ) {
	$product = QTS_Add_To_Quote::maybe_get_product_instance( $product );

	if ( ! $product ) {
		return 0;
	}

	/**
	 * Get product original price.
	 * 
	 * @since 1.0
	 */
	return ( float ) apply_filters( 'qts_get_product_original_price', $product->get_price( 'edit' ), $product );
}

/**
 * Get the minimum quote price to ask by the customer.
 * 
 * @param int|WC_Product $product
 * @return int|float
 */
function _qts_get_min_quote_price( $product ) {
	$product = QTS_Add_To_Quote::maybe_get_product_instance( $product );

	if ( ! QTS_Add_To_Quote::is_quote_enabled( $product ) ) {
		return 0;
	}

	// Calculate the minimum quote request price. 
	if ( 'fixed' === QTS_Add_To_Quote::_get( $product->get_id(), 'min_quote_price_type' ) ) {
		$min_quote_price = QTS_Add_To_Quote::_get( $product->get_id(), 'fixed_min_quote_price' );
	} else {
		$min_quote_price = ( _qts_get_original_price( $product ) / 100 ) * QTS_Add_To_Quote::_get( $product->get_id(), 'percentage_min_quote_price' );
	}

	return floatval( $min_quote_price );
}

/**
 * Get the minimum quote price percent to ask by the customer.
 * 
 * @param int|WC_Product $product
 * @return int|float
 */
function _qts_get_min_quote_price_percent( $product ) {
	$product = QTS_Add_To_Quote::maybe_get_product_instance( $product );

	if ( ! QTS_Add_To_Quote::is_quote_enabled( $product ) ) {
		return 0;
	}

	$min_quote_price_percent = QTS_Add_To_Quote::_get( $product->get_id(), 'percentage_min_quote_price' );
	return floatval( $min_quote_price_percent );
}
