<?php

defined( 'ABSPATH' ) || exit;

/**
 * Check whether the given the value is quote request. 
 *
 * @param  mixed $the_quote_request Post object or post ID of the quote request.
 * @return bool True on success.
 */
function _qts_is_quote_request( $the_quote_request ) {
	if ( is_object( $the_quote_request ) && is_a( $the_quote_request, 'QTS_Quote_Request' ) ) {
		$is_quote_request = true;
	} elseif ( is_numeric( $the_quote_request ) && 'qts_quote_request' === WC_Data_Store::load( 'qts_quote_request' )->get_order_type( $the_quote_request ) ) {
		$is_quote_request = true;
	} else {
		$is_quote_request = false;
	}

	return $is_quote_request;
}

/**
 * Main function for returning quote request, uses the wc_get_order function.
 *
 * @param  mixed $the_quote_request Post object or post ID of the quote request.
 * @return bool|WC_Order|WC_Order_Refund
 */
function _qts_get_quote_request( $the_quote_request ) {
	if ( $the_quote_request ) {
		if ( is_a( $the_quote_request, 'QTS_Quote_Request' ) ) {
			$id = $the_quote_request->get_id();
		} else if ( ! empty( $the_quote_request->ID ) ) {
			$id = $the_quote_request->ID;
		} else {
			$id = absint( $the_quote_request );
		}

		$the_quote_request = wc_get_order( $id );

		if ( _qts_is_quote_request( $the_quote_request ) ) {
			return $the_quote_request;
		}
	}

	return false;
}

/**
 * Get the quote request statuses.
 * 
 * @return array
 */
function _qts_get_quote_request_statuses() {
	$quote_request_statuses = array(
		QTS_PREFIX . 'new'             => __( 'New', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'on_hold'         => __( 'On-hold', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'approved'        => __( 'Approved', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'accepted'        => __( 'Accepted', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'rejected'        => __( 'Rejected', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'expired'         => __( 'Expired', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'completed'       => __( 'Completed', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'response_awaitd' => __( 'Response Awaited', 'quote-request-for-woocommerce' ),
		QTS_PREFIX . 'hav_to_respond'  => __( 'Have to Respond', 'quote-request-for-woocommerce' ),
	);

	return $quote_request_statuses;
}

/**
 * Get the quote request status name.
 * 
 * @param string $status
 * @return string
 */
function _qts_get_quote_request_status_name( $status ) {
	$statuses = _qts_get_quote_request_statuses();
	$status   = _qts_trim_post_status( $status );
	$status   = isset( $statuses[ QTS_PREFIX . $status ] ) ? $statuses[ QTS_PREFIX . $status ] : $status;
	return $status;
}

/**
 * Prepare the quote request expiry date to schedule.
 * 
 * @param QTS_Quote_Request $quote_request
 * @return int
 */
function _qts_prepare_quote_request_expiry_date( $quote_request ) {
	if ( $quote_request->needs_payment() ) {
		$expiry_days_count = absint( get_option( QTS_PREFIX . 'due_days_to_make_payment' ) );
	} else {
		$expiry_days_count = absint( get_option( QTS_PREFIX . 'customer_response_time' ) );
	}

	if ( $quote_request->get_date_expiry( 'edit' ) ) {
		$base_timestamp = $quote_request->get_date_expiry( 'edit' )->getTimestamp();
	} else {
		$base_timestamp = null;
	}

	return _qts_get_time( 'timestamp', array( 'time' => "+{$expiry_days_count} days", 'base' => $base_timestamp ) );
}

/**
 * Finds an quote request ID based on an quote request key.
 *
 * @param string $quote_request_key An quote request key has generated by.
 * @return int The ID of an quote request, or 0 if the quote request could not be found.
 */
function _qts_get_quote_request_id_by_quote_request_key( $quote_request_key ) {
	$data_store = WC_Data_Store::load( 'qts_quote_request' );
	return $data_store->get_order_id_by_order_key( $quote_request_key );
}

/**
 * Get an quote request note.
 * 
 * @param  int|WP_Comment $note Note ID (or WP_Comment instance for internal use only).
 * @return array Data with quote request note details or null when does not exists.
 */
function _qts_get_quote_request_note( $note ) {
	if ( is_numeric( $note ) ) {
		$note = get_comment( $note );
	}

	if ( ! is_a( $note, 'WP_Comment' ) ) {
		return null;
	}

	if ( ! get_comment_meta( $note->comment_ID, 'is_qtsquote_request_note', true ) ) {
		return null;
	}

	/**
	 * Get quote request note.
	 * 
	 * @since 1.0
	 */
	return ( object ) apply_filters( 'qts_get_quote_request_note', array(
				'id'            => ( int ) $note->comment_ID,
				'date_created'  => wc_string_to_datetime( $note->comment_date ),
				'content'       => make_clickable( $note->comment_content ),
				'customer_note' => ( bool ) get_comment_meta( $note->comment_ID, 'is_customer_note', true ),
				'added_by'      => get_comment_meta( $note->comment_ID, 'is_qtsquote_request_note_added_by', true ),
				'added_user'    => $note->comment_author,
					), $note );
}

/**
 * Exclude quote request conversation comments from queries and RSS.
 * 
 * Exclude on demand by using add_filter( 'the_comments', '_qts_exclude_quote_request_conversations' )
 * 
 * Since Woocommerce didn't hooked to filter the order notes and so we go with WP hook.
 *
 * @param WP_Comment[] $comments An array of comments.
 * @return array
 */
function _qts_exclude_quote_request_conversations( $comments ) {
	if ( $comments ) {
		foreach ( $comments as $key => $comment ) {
			if ( 'order_note' !== $comment->comment_type || ! get_comment_meta( $comment->comment_ID, 'is_qtsquote_request_note', true ) ) {
				continue;
			}

			if ( get_comment_meta( $comment->comment_ID, 'is_customer_note', true ) ) {
				unset( $comments[ $key ] );
			}
		}
	}

	return $comments;
}

/**
 * Search quote requests.
 *
 * @param  string $term Term to search.
 * @return array List of quote requests ID.
 */
function _qts_quote_request_search( $term ) {
	$data_store = WC_Data_Store::load( 'qts_quote_request' );
	return $data_store->search_quote_requests( str_replace( 'Order #', '', wc_clean( $term ) ) );
}
