<?php

class RPSync_Settings {

	public function __construct() {
		add_action( 'admin_init', array( $this, 'admin_init' ) );
		add_action( 'admin_menu', array( $this, 'admin_menu' ) );
		add_action( 'wp_ajax_rpsync_reset', array( $this, 'ajax_reset' ) );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
	}

	public function admin_init() {
		register_setting( 'rpsync', 'rpsync-settings', array( $this, 'validate' ) );
		add_settings_section( 'rpsync-general', '', '__return_empty_string', 'rpsync' );
		add_settings_field( 'xml_path', "XML file path", array(
			$this,
			'xml_path_callback'
		), 'rpsync', 'rpsync-general' );
		add_settings_field( 'sync_interval', "Background synchronization interval (minutes)", array(
			$this,
			'sync_interval_callback'
		), 'rpsync', 'rpsync-general' );
	}

	public function xml_path_callback() {
		$settings = (array) get_option( 'rpsync-settings' );
		$value    = esc_attr( $settings['xml_path'] );
		?>
		<input type="text" value="<?= $value ?>" name="rpsync-settings[xml_path]">
		<?php
	}

	public function sync_interval_callback() {
		$settings = (array) get_option( 'rpsync-settings' );
		$value    = esc_attr( $settings['sync_interval'] ) ?: 5;
		?>
		<input type="number" min="5" step="1" value="<?= $value ?>" name="rpsync-settings[sync_interval]">
		<?php
	}

	public function validate( $input ) {
		$settings = (array) get_option( 'rpsync-settings' );

		// XML path
		if ( ! file_exists( $input['xml_path'] ) ) {
			add_settings_error( 'rpsync-settings', 'invalid-xml-path', "XML file does not exist." );
			$input['xml_path'] = $settings['xml_path'];
		}

		// Synchronization interval
		if ( $input['sync_interval'] < 5 ) {
			add_settings_error( 'rpsync-settings', 'invalid-sync_interval', "Invalid synchronization interval." );
			$input['sync_interval'] = $settings['sync_interval'];
		} else {
			wp_unschedule_hook( 'rpsync_process_xml' );
		}

		return $input;
	}

	public function admin_menu() {
		add_options_page(
			"Retailpoint Sync",
			'Retailpoint Sync',
			'manage_options',
			'rpsync-settings',
			array( $this, 'settings_page' ),
		);
	}

	public function settings_page() {
		$reset_url = add_query_arg( array(
			'action'   => 'rpsync_reset',
			'security' => wp_create_nonce( 'rpsync_reset' )
		), admin_url( 'admin-ajax.php' ) );
		$check_url = add_query_arg( array(
			'action'   => 'rpsync_check',
			'security' => wp_create_nonce( 'rpsync_check' )
		), admin_url( 'admin-ajax.php' ) );
		?>
		<div id="rpsync-settings" class="wrap">
			<h1>Retailpoint Sync</h1>
			<form action="options.php" method="post">
				<?php
				settings_fields( 'rpsync' );
				do_settings_sections( 'rpsync' );
				submit_button();
				?>
			</form>
			<a href="<?= $reset_url ?>" class="button">
				Réinitialiser la synchronization
			</a>
			<a href="<?= $check_url ?>" class="button">
				Vérifier la base de données
			</a>
		</div>
		<?php
	}

	public function get( $name ) {
		$settings = (array) get_option( 'rpsync-settings' );
		return $settings[ $name ] ?? null;
	}

	public function ajax_reset() {
		check_ajax_referer( 'rpsync_reset', 'security' );
		RPSync::get_instance()->database->reset_md5();
		do_action( 'rpsync_process_xml' );
		?>
		<script>
      window.location = "<?= add_query_arg( array( 'reset' => true ), $_SERVER['HTTP_REFERER'] ) ?>";
		</script>
		<?php
	}

	public function admin_notices() {
		if ( ! empty( $_GET['reset'] ) ) {
			?>
			<div class="notice notice-success is-dismissible">
				<p>
					La base de données RPSync a été réinitialisée. La reconstruction pourrait prendre un certain temps.
				</p>
			</div>
			<?php
		}
	}

}
