<?php

class RPSync_WooCommerce {

	public function __construct() {
		add_action( 'rpsync_process_product_data', array( $this, 'process_product_data' ), 1 );
		add_action( 'delete_post', array( $this, 'delete_post' ), 10, 2 );
		add_action( 'wc_merge_after', array( $this, 'wc_merge_after' ) );
	}

	/**
	 * Process extracted product_data for WooCommerce
	 *
	 * @param $data
	 *
	 * @return void
	 */
	public function process_product_data( $numref ) {
		// Load product data from RPSync database
		if ( empty( $product_data = RPSync::get_instance()->database->load_product_data( $numref ) ) ) {
			return;
		}

		// Process associated WooCommerce products
		foreach ( $product_data['skus'] as &$product ) {
			// Skip empty / invalid product_id
			if ( empty( $product['product_id'] ) ) {
				continue;
			}
			$changes = array();
			if ( empty( get_post_meta( $product['product_id'], '_sku', true ) ) ) {
				update_post_meta( $product['product_id'], '_sku', $product['sku'] );
				$changes['SKU'] = "'' => '${product['sku']}'";
			}
			// Prices
			if ( $product['regular_price'] !== ( $regular_price = get_post_meta( $product['product_id'], '_regular_price', true ) ) ) {
				update_post_meta( $product['product_id'], '_regular_price', $product['regular_price'] );
				update_post_meta( $product['product_id'], '_price', $product['regular_price'] );
				$changes['Prix régulier'] = "'$regular_price' => '${product['regular_price']}'";
			}
			if ( $product['sale_price'] !== ( $sale_price = get_post_meta( $product['product_id'], '_sale_price', true ) ) ) {
				update_post_meta( $product['product_id'], '_sale_price', $product['sale_price'] );
				update_post_meta( $product['product_id'], '_price', $product['sale_price'] );
				$changes['Prix spécial'] = "'$sale_price' => '${product['sale_price']}'";
			}
			// Stock
			if ( $product['stock'] !== ( $stock_quantity = (int) get_post_meta( $product['product_id'], '_stock', true ) ) ) {
				update_post_meta( $product['product_id'], '_stock', $product['stock'] );
				update_post_meta( $product['product_id'], '_stock_status', $product['stock'] ? 'instock' : 'outofstock' );
				update_post_meta( $product['product_id'], '_manage_stock', 'yes' );
				$changes['Inventaire'] = "'$stock_quantity' => '${product['stock']}'";
			}

			// Parent stock status
			if ( sizeof( $product_data['skus'] ) > 1 ) {
				if ( array_sum( array_column( $product_data['skus'], 'stock' ) ) === 0 ) {
					update_post_meta( $product_data['product_id'], '_stock_status', 'outofstock' );
				} else {
					update_post_meta( $product_data['product_id'], '_stock_status', 'instock' );
				}
			}

			// Log changes
			if ( ! empty( $changes ) ) {
				// Log changes
				$changes = implode( "\n", array_map( function ( $label, $change ) {
					return "  $label: $change";
				}, array_keys( $changes ), $changes ) );
				RPSync::get_instance()->info( sprintf( "%s (Numref %s / SKU %s / ID: %d)\n", $product['title'], $product['numref'], $product['sku'], $product['product_id'] ) . $changes );
				// Purge SWCFPC cache
				do_action( 'swcfpc_purge_cache', array( get_the_permalink( $product_data['product_id'] ) ) );
			}
		}
	}

	/**
	 * Find product ID from SKU
	 *
	 * @param $sku string The SKU to find
	 *
	 * @return false|int The product ID if found, false otherwise
	 */
	public static function find_product_by_sku( string $sku ) {
		global $wpdb;
		$query = "SELECT post_id FROM $wpdb->postmeta WHERE meta_key='_sku' AND meta_value='$sku' ORDER BY post_ID DESC LIMIT 1;";
		if ( empty( $results = $wpdb->get_results( $query, ARRAY_A ) ) ) {
			return false;
		}

		return (int) $results[0]['post_id'];
	}

	/**
	 * Create WooCommerce product by numref
	 *
	 * @param string $numref
	 *
	 * @return bool|void
	 * @throws WC_Data_Exception
	 */
	public function create_product( string $numref ) {
		// Load Retailpoint product data
		if ( empty( $product_data = RPSync::get_instance()->database->load_product_data( $numref ) ) ) {
			return false;
		}

		// Check if WC product already exists
		if ( ! empty( $product_data['product_id'] ) ) {
			return false;
		}

		// Create simple / variable product
		if ( sizeof( $product_data['skus'] ) === 1 ) {
			// Create simple product with 'draft' status
			$product    = &$product_data['skus'][ array_key_first( $product_data['skus'] ) ];
			$wc_product = new WC_Product_Simple();
			$wc_product->set_status( 'draft' );
			$wc_product->set_name( $product['title'] );
			$wc_product->set_sku( $product['sku'] );
			if ( ! $wc_product->save() ) {
				return false;
			}
			// Update product_id in Retailpoint tables
			$product_data['product_id'] = $product['product_id'] = $wc_product->get_id();
			if ( ! RPSync::get_instance()->database->save_product_data( $product_data ) ) {
				return false;
			}
		} else {
			// Create variable product with 'draft' status
			$wc_product = new WC_Product_Variable();
			$wc_product->set_status( 'draft' );
			$wc_product->set_name( reset( $product_data['skus'] )['title'] );
			$wc_product->set_attributes( $this->create_variable_attributes( $product_data ) );
			if ( ! $wc_product->save() ) {
				return false;
			}
			$product_data['product_id'] = $wc_product->get_id();
			foreach ( $product_data['skus'] as &$product ) {
				// Create product variation
				$wc_variation = new WC_Product_Variation();
				$wc_variation->set_parent_id( $wc_product->get_id() );
				$wc_variation->set_status( 'publish' );
				$wc_variation->set_sku( $product['sku'] );
				$wc_variation->set_attributes( $product['attributes'] );
				if ( ! $wc_variation->save() ) {
					return false;
				}
				$product['product_id'] = $wc_variation->get_id();
			}
			if ( ! RPSync::get_instance()->database->save_product_data( $product_data ) ) {
				return false;
			}
		}

		// Process newly created product synchronously
		do_action( 'rpsync_process_product_data', $numref, 'rpsync' );
		return true;
	}

	/**
	 * Create variable product attributes
	 *
	 * @param array $product_data
	 *
	 * @return array
	 */
	public function create_variable_attributes( array $product_data ) {
		$attributes = array();
		foreach ( $product_data['skus'] as $product ) {
			foreach ( $product['attributes'] as $name => $value ) {
				$attributes[ $name ][] = $value;
			}
		}

		return array_map( function ( $name, array $options ) {
			$wc_attribute = new WC_Product_Attribute();
			$wc_attribute->set_name( $name );
			$wc_attribute->set_options( $options );
			$wc_attribute->set_visible( true );
			if ( sizeof( $options ) > 1 ) {
				$wc_attribute->set_variation( true );
			}
			return $wc_attribute;
		}, array_keys( $attributes ), array_map( 'array_unique', $attributes ) );
	}

	/**
	 * Update WooCommerce product by numref
	 *
	 * @param string $numref
	 *
	 * @return bool
	 */
	public function update_product( string $numref ) {
		if ( RPSync::get_instance()->database->get_md5( $numref ) === false ) {
			return false;
		}
		do_action( 'rpsync_process_product_data', $numref, 'rpsync' );
		return true;
	}

	/**
	 * Destroy WooCommerce product by numref
	 *
	 * @param string $numref
	 *
	 * @return bool
	 */
	public function destroy_product( string $numref ) {
		if ( empty( $product_data = RPSync::get_instance()->database->load_product_data( $numref ) ) ) {
			return false;
		}
		if ( empty( $product_data['product_id'] ) || ! wp_delete_post( $product_data['product_id'] ) ) {
			return false;
		}
		$product_data['product_id'] = null;
		foreach ( $product_data['skus'] as &$product ) {
			$product['product_id'] = null;
		}

		return RPSync::get_instance()->database->save_product_data( $product_data );
	}

	/**
	 * Invalidate and dispatch numref when saving product
	 *
	 * @param $post_id
	 *
	 * @return void
	 */
	public function save_product( $post_id ) {
		if ( ! empty( $numrefs = explode( ' / ', $_POST["acf"]["field_61d88632b62cd"] ?: '' ) ) ) {
			array_walk( $numrefs, function ( $numref ) {
				RPSync::get_instance()->database->reset_md5( $numref );
				RPSync::get_instance()->dispatch_process_numref( $numref );
			} );
		}
	}

	/**
	 * Delete product association when product is deleted
	 */
	public function delete_post( $post_id, WP_Post $post ) {
		if ( $post->post_type !== 'product' ) {
			return;
		}
		RPSync::get_instance()->database->delete_product_id( $post_id );
	}

	/**
	 * Update product_id in numrefs table after product merge
	 *
	 * @param $product_id
	 *
	 * @return void
	 */
	public function wc_merge_after( $product_id ) {
		RPSync::get_instance()->database->update_product_id( $product_id );
	}
}
