<?php

namespace Vibe\Split_Orders\Addons;

use Vibe\Split_Orders\Order_Numbers;
use Vibe\Split_Orders\Split_Orders;
use WC_Order;
use WC_Seq_Order_Number_Pro;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/**
 * Provides support for Sequential Order Numbers Pro plugin
 *
 * @since 1.2
 */
class Sequential_Order_Numbers_Pro {

	/**
	 * Creates an instance and sets up hooks to integrate with the rest of the extension, only if Sequential Order
	 * Numbers Pro plugin is installed
	 */
	public function __construct() {
		add_filter( Split_Orders::hook_prefix( 'order_number_settings_active' ), '__return_true' );
		add_action( Split_Orders::hook_prefix( 'orders_updated' ), array( __CLASS__, 'orders_updated' ), 10, 2 );
	}

	/**
	 * Assigns a sequential order number to newly created orders in a split
	 *
	 * @param WC_Order $new_order The newly created order
	 */
	public static function orders_updated( WC_Order $new_order, WC_Order $source_order ) {
		if ( ! class_exists( 'WC_Seq_Order_Number_Pro' ) ) {
			return;
		}

		if ( Order_Numbers::suffix_enabled() ) {
			static::assign_order_number( $new_order, $source_order );
		} else {
			WC_Seq_Order_Number_Pro::instance()->set_sequential_order_number( $new_order );
		}
	}

	/**
	 * Generates and saves an order number to the given new order, using the split index as a suffix
	 *
	 * @param WC_Order $new_order    The order to assign a new order number to
	 * @param WC_Order $source_order The original order that the new order was split from
	 */
	private static function assign_order_number( WC_Order $new_order, WC_Order $source_order ) {
		$order_number_formatted = Order_Numbers::generate( $new_order, $source_order );

		if ( $order_number_formatted ) {
			$order_number      = $source_order->get_order_number();
			$order_number_meta = $source_order->get_meta( '_order_number_meta' );

			$new_order->update_meta_data( '_order_number_formatted', $order_number_formatted );
			$new_order->update_meta_data( '_order_number', $order_number );
			$new_order->update_meta_data( '_order_number_meta', $order_number_meta );
			$new_order->save_meta_data();
		}
	}
}
