<?php

namespace Vibe\Split_Orders\Upgrades;

use Vibe\Split_Orders\Split_Orders;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/**
 * Manages upgrade routines
 *
 * @since 1.4.0
 */
class Upgrades {

	// Upgrade routines in order from lowest to highest
	private static $upgrades = array(
		'1.4.0' => Upgrade_140::class,
		'1.8.0' => Upgrade_180::class
	);

	/**
	 * Creates an instance and sets up the hooks to check if upgrade routines need running
	 */
	public function __construct() {
		add_action( 'admin_init', array( __CLASS__, 'maybe_schedule' ) );

		add_action( Split_Orders::hook_prefix('upgrade'), array( __CLASS__, 'upgrade' ) );
	}

	/**
	 * Returns true if an upgrade action is already scheduled
	 *
	 * @return bool True if an upgrade action is already scheduled, false otherwise
	 */
	public static function scheduled() {
		return as_has_scheduled_action( Split_Orders::hook_prefix( 'upgrade' ) );
	}

	/**
	 * Schedules an upgrade action if one is not already scheduled and an upgrade is required
	 *
	 * @param bool $force Whether to force the action to be scheduled without checking if one is already scheduled
	 */
	public static function maybe_schedule( $force = false ) {
		// Only schedule if an upgrade is required and nothing is already scheduled
		if ( $force || ( ! static::scheduled() && static::required_upgrade() ) ) {
			as_enqueue_async_action( Split_Orders::hook_prefix( 'upgrade' ) );
		}
	}

	/**
	 * Returns the key of the next upgrade that must be installed
	 *
	 * @return false|string Returns the key of the next upgrade or false if none is required
	 */
	public static function required_upgrade() {
		$db_version = self::db_version();
		$versions = array_keys( self::$upgrades );
		$required_upgrade = false;

		foreach ( $versions as $version ) {

			if ( version_compare( $db_version, $version, '<' ) ) {
				$required_upgrade = $version;
				break;
			}
		}

		return $required_upgrade;
	}

	/**
	 * Runs a batch of the next required upgrade routine and schedules another batch if there is more to complete
	 */
	public static function upgrade() {
		$required_upgrade = static::required_upgrade();

		if ( $required_upgrade ) {
			$upgrade = self::$upgrades[ $required_upgrade ];
			$upgrade = new $upgrade();

			if ( $upgrade->run() ) {
				self::update_db_version( $required_upgrade );

				// If there are no more upgrades left then set the version number to the current version (if it is a proper number)
				if ( empty( static::required_upgrade() ) && strpos( Split_Orders::instance()->get_version(), '%' ) === false ) {
					self::update_db_version( Split_Orders::instance()->get_version() );
				}
			} else {
				// We need to force the scheduling because the existing upgrade will still be in-progress
				self::maybe_schedule( true );
			}
		}
	}

	/**
	 * Returns the current version as stored in the database
	 *
	 * @return string the database version in the format x.x.x
	 */
	public static function db_version() {
		return get_option( Split_Orders::hook_prefix( 'version' ), '0.0.0' );
	}

	/**
	 * Overwrites the version number in the database
	 *
	 * @param string $version The version number to set or null if the current plugin version should be used
	 */
	public static function update_db_version( $version = null ) {
		update_option( Split_Orders::hook_prefix( 'version' ), is_null( $version ) ? Split_Orders::instance()->get_version() : $version );
	}
}
