<?php

namespace Vibe\Split_Orders;

use Error;
use Exception;
use WC_Data_Exception;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/**
 * AJAX request handlers
 *
 * @since 1.0.0
 */
class AJAX {

	/**
	 * Creates an instance and sets up the AJAX actions
	 */
	public function __construct() {
		$action = Split_Orders::hook_prefix( 'popup_line_items' );
		add_action( "wp_ajax_{$action}", array( __CLASS__, 'get_popup' ) );

		$action = Split_Orders::hook_prefix( 'split_order' );
		add_action( "wp_ajax_{$action}", array( __CLASS__, 'split_order' ) );
	}

	/**
	 * Handles an AJAX request to fetch line item HTML in the splitting modal
	 *
	 * Sends a JSON response containing a success flag and the HTML to be used for the modal
	 */
	public static function get_popup() {
		$response = array(
			'success' => false
		);

		$nonce = isset( $_REQUEST['nonce'] ) ? wc_clean( $_REQUEST['nonce'] ) : false;

		if ( ! empty( $nonce ) && wp_verify_nonce( $nonce, Split_Orders::hook_prefix( 'popup-nonce' ) ) ) {
			$order_id = isset( $_REQUEST['order_id'] ) ? absint( $_REQUEST['order_id'] ) : false;

			if ( Orders::can_split( $order_id ) ) {
				$response['html']    = Admin::get_splitting_popup( $order_id );
				$response['success'] = ! empty( $response['html'] );
			}
		}

		wp_send_json( $response );
	}

	/**
	 * Handles an AJAX request to split an order
	 *
	 * Sends a JSON response containing a success flag
	 */
	public static function split_order() {
		$response = array(
			'success' => false,
		);

		$nonce = isset( $_REQUEST['nonce'] ) ? wc_clean( $_REQUEST['nonce'] ) : false;

		if ( ! empty( $nonce ) && wp_verify_nonce( $nonce, Split_Orders::hook_prefix( 'splitting-nonce' ) ) ) {
			$order_id = isset( $_REQUEST['order_id'] ) ? absint( $_REQUEST['order_id'] ) : false;

			if ( Orders::can_split( $order_id ) ) {
				try {
					$items          = isset( $_REQUEST['items'] ) ? wc_clean( $_REQUEST['items'] ) : array();
					$shipping_items = isset( $_REQUEST['shipping'] ) ? wc_clean( $_REQUEST['shipping'] ) : array();
					$items          = $items + $shipping_items;

					$meta_fields = Settings::meta_fields();

					$result = Orders::split( $order_id, $items, $meta_fields );
					$response['success'] = ! empty( $result );
				} catch ( Error $error ) {
					$plugin = Utilities::identify_plugin($error->getFile());

					// Unexpected fatal error occurred during the split - probably originating in another plugin
					$response['message']      = __( '<p>An error has occurred while splitting this order, and splitting could not be completed.</p>', 'split-orders' );
					$response['error_type']   = 'error';
					$response['log']          = $error->getMessage() . ' in ' . $error->getFile() . ', line ' . $error->getLine() . "\n" . $error->getTraceAsString();
					$response['error_origin'] = isset( $plugin['Name'] ) ? $plugin['Name'] : null;
				} catch ( WC_Data_Exception $e ) {
					$response['message']    = __( '<p>An error has occurred while splitting this order, and splitting could not be completed.</p>', 'split-orders' );
					$response['error_type'] = 'exception';
					$response['log']        = $e->getMessage() . ' in ' . $e->getFile() . ', line ' . $e->getLine() . "\n" . $e->getTraceAsString();
				} catch ( Exception $e ) {
					$response['message']    = __( '<p>An error has occurred while splitting this order, and splitting could not be completed.</p>', 'split-orders' );
					$response['error_type'] = 'exception';
					$response['log']        = $e->getMessage() . ' in ' . $e->getFile() . ', line ' . $e->getLine() . "\n" . $e->getTraceAsString();
				}
			}
		}

		wp_send_json( $response );
	}
}
