<?php

namespace Vibe\Split_Orders;

use Automattic\WooCommerce\Admin\Schedulers\OrdersScheduler as OldOrdersScheduler;
use Automattic\WooCommerce\Internal\Admin\Schedulers\OrdersScheduler;
use Exception;
use WC_Data_Exception;
use WC_Order;
use WC_Order_Item_Product;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/**
 * Handles order functions such as splitting orders and adding notes
 *
 * @since 1.0
 */
class Orders {

	/**
	 * Returns true if the given order can be split by the current user
	 *
	 * @param int $order_id The ID of the order to check
	 *
	 * @return bool True if the order can be split and false otherwise
	 */
	public static function can_split( $order_id ) {
		$user_can = current_user_can( 'edit_shop_orders', $order_id );

		return apply_filters( Split_Orders::hook_prefix( 'can_split' ), $user_can, wc_get_order( $order_id ) );
	}

	/**
	 * Splits an order into two orders with the provided array of items moved to a new order.
	 *
	 * The split order will be assigned the following information to match the original order:
	 *
	 * - Billing address
	 * - Shipping address
	 * - Date created
	 * - Currency
	 * - Customer IP
	 * - Customer user agent
	 * - Customer note
	 * - Date paid
	 * - Date completed
	 * - Payment method
	 * - Payment method title
	 * - Transaction ID
	 *
	 * Shipping methods are also copied to the new order, but with the shipping price set to zero.
	 *
	 * @param int   $order_id    The ID of the order to split
	 * @param array $items       An array of items to split to the new order, with line item ID as the key and the quantity
	 *                           as the value. All line item IDs must match line items from the given order and the quantity
	 *                           must not exceed the quantity of the line item on the existing order.
	 * @param array $meta_fields Additional meta data fields to copy if they exist on the source order
	 *
	 * @return WC_Order The new order that was created
	 * @throws WC_Data_Exception Exception thrown if products or meta data cannot be assigned to the new order
	 * @throws Exception Exception thrown if unable to set line item meta on the new order line items
	 */
	public static function split( $order_id, array $items, array $meta_fields = array() ) {
		$source_order = wc_get_order( $order_id );

		// Disable emails temporarily
		self::maybe_disable_emails( $source_order );

		// Create the new target order
		/* @var WC_Order $target_order */
		$target_order = wc_create_order( array(
			'status'      => self::order_status( $source_order, $items ),
			'customer_id' => $source_order->get_customer_id(),
			'created_via' => __( 'Order split', 'split-orders' )
		) );

		do_action( Split_Orders::hook_prefix( 'order_created' ), $target_order, $source_order, $items );

		// Restore emails
		self::maybe_restore_emails( $source_order );

		// Copy basic order details from source order to target order
		$target_order->set_address( $source_order->get_address( 'billing' ), 'billing' );
		$target_order->set_address( $source_order->get_address( 'shipping' ), 'shipping' );
		$target_order->set_currency( $source_order->get_currency() );
		$target_order->set_prices_include_tax( $source_order->get_prices_include_tax() );
		$target_order->set_customer_ip_address( $source_order->get_customer_ip_address() );
		$target_order->set_customer_user_agent( $source_order->get_customer_user_agent() );
		$target_order->set_customer_note( $source_order->get_customer_note() );
		$target_order->set_date_paid( $source_order->get_date_paid() );
		$target_order->set_date_completed( $source_order->get_date_completed() );
		$target_order->set_order_stock_reduced( $source_order->get_order_stock_reduced() );
		$target_order->set_payment_method( $source_order->get_payment_method() );
		$target_order->set_payment_method_title( $source_order->get_payment_method_title() );
		$target_order->set_transaction_id( $source_order->get_transaction_id() );

		// Only copy the order date if enabled by a filter
		if ( apply_filters( Split_Orders::hook_prefix( 'clone_date_created' ), false, $target_order, $source_order, $items ) ) {
			$target_order->set_date_created( $source_order->get_date_created() );
		}

		// Copy the address indexes to retain search - WooCommerce doesn't set these up automatically when setting address
		if ( Admin::is_hpos_enabled() ) {
			$target_order->update_meta_data( '_billing_address_index', $source_order->get_meta( '_billing_address_index' ) );
			$target_order->update_meta_data( '_shipping_address_index', $source_order->get_meta( '_shipping_address_index' ) );
		} else {
			$billing_address_index  = get_post_meta( $source_order->get_id(), '_billing_address_index', true );
			$shipping_address_index = get_post_meta( $source_order->get_id(), '_shipping_address_index', true );

			update_post_meta( $target_order->get_id(), '_billing_address_index', $billing_address_index );
			update_post_meta( $target_order->get_id(), '_shipping_address_index', $shipping_address_index );
		}

		// Copy requested meta fields as well as any defaults
		$meta_fields = array_unique( array_merge( $meta_fields, static::default_meta_fields_to_copy() ) );
		$meta_fields = apply_filters( Split_Orders::hook_prefix( 'meta_fields' ), $meta_fields, $target_order, $source_order );

		foreach ( $meta_fields as $meta_field ) {
			if ( $source_order->meta_exists( $meta_field ) ) {
				$meta_values = $source_order->get_meta( $meta_field, false, 'edit' );

				foreach ( $meta_values as $meta_value ) {
					$target_order->update_meta_data( $meta_field, $meta_value->value );
				}
			}
		}

		/**
		 * Filters whether to update the source order, by removing the specified quantity of items
		 *
		 * @param bool     $update       True if the source order should be updated, false otherwise. Defaults to true.
		 * @param WC_Order $source_order The original order that is being split
		 * @param array    $items        The line items being split
		 *
		 * @since 1.5.0
		 */
		$update_source_order   = apply_filters( Split_Orders::hook_prefix( 'update_source_order' ), true, $source_order, $items );
		$source_items          = array();
		$source_shipping_items = array();

		// Separate by item type
		$items_by_type = array();
		foreach ( $items as $item_id => $amount_to_split ) {
			$item = $source_order->get_item( $item_id );
			$type = $item->get_type();

			if ( ! isset( $items_by_type[ $type ] ) ) {
				$items_by_type[ $type ] = array();
			}

			$items_by_type[ $type ][ $item_id ] = $amount_to_split;
		}

		// Add product line items to the target order with correct quantity and totals
		foreach ( $items_by_type['line_item'] as $item_id => $item_split_qty ) {
			/* @var WC_Order_Item_Product $source_item */
			$source_item = $source_order->get_item( $item_id );
			$subtotal    = $source_order->get_item_subtotal( $source_item, false, false );
			$total       = $source_order->get_item_total( $source_item, false, false );

			// Sanitize as a stock amount
			$item_split_qty         = wc_stock_amount( $item_split_qty );
			$original_qty           = $source_item->get_quantity( 'edit' );
			$original_reduced_stock = $source_item->get_meta( '_reduced_stock' );
			$updated_qty            = $original_qty - $item_split_qty;

			// Reduced stock value should be split between the items, prioritising the original order
			$updated_reduced_stock = ( $updated_qty >= $original_reduced_stock ) ? $original_reduced_stock : $updated_qty;
			$reduced_stock         = $original_reduced_stock ? ( $original_reduced_stock - $updated_reduced_stock ) : null;

			// Create and save the new line item before updating it, otherwise some changes can not be applied correctly
			$target_item = clone $source_item;
			$target_item->set_id( 0 );
			$target_item->set_order_id( 0 );
			$target_item->save();

			$subtotal_price = max( 0.0, (float) $subtotal );
			$total_price    = max( 0.0, (float) $total );
			$qty            = max( 0.0, (float) $item_split_qty );
			$subtotal       = $subtotal_price * $qty;
			$total          = $total_price * $qty;

			$target_item->set_quantity( $qty );
			$target_item->set_subtotal( $subtotal );
			$target_item->set_total( $total );
			$target_item->save();

			// Only add reduced stock meta for non-0 value
			if ( $reduced_stock ) {
				$target_item->update_meta_data( '_reduced_stock', $reduced_stock );
			} else {
				$target_item->delete_meta_data( '_reduced_stock' );
			}

			$target_order->add_item( $target_item );

			// Update the source item, but delay saving the changes until after the target items have all been added
			if ( $update_source_order ) {

				if ( 0 == $updated_qty ) {
					$source_order->remove_item( $item_id );
				} else {
					$subtotal = $subtotal_price * $updated_qty;
					$total    = $total_price * $updated_qty;

					$source_item->set_quantity( $updated_qty );
					$source_item->set_subtotal( $subtotal );
					$source_item->set_total( $total );

					// We don't want to add reduced stock meta for a 0 value.
					if ( $updated_reduced_stock ) {
						$source_item->update_meta_data( '_reduced_stock', $updated_reduced_stock );
					}

					// Keep a record of these items, so they can be saved after the target items have all been added
					$source_items[] = $source_item;
				}
			}
		}

		if ( $items_by_type['shipping'] ) {

			// Add shipping line items to the target order with the correct totals
			foreach ( $items_by_type['shipping'] as $item_id => $item_split_total ) {
				/* @var \WC_Order_Item_Shipping $source_shipping_item */
				$source_shipping_item = $source_order->get_item( $item_id );
				$total                = $source_order->get_item_total( $source_shipping_item, false, false );

				// Don't allow the split total to be more than the total
				$item_split_total = min( $item_split_total, $total );

				// Create and save the new line item before updating it, otherwise some changes can not be applied correctly
				$target_shipping_item = clone $source_shipping_item;
				$target_shipping_item->set_id( 0 );
				$target_shipping_item->set_total( $item_split_total );

				// Attempt to split the items meta data
				$target_items_meta = static::shipping_items_metadata( $target_shipping_item->get_meta( 'Items' ), $target_order->get_items() );
				$target_shipping_item->update_meta_data( 'Items', $target_items_meta );

				$target_order->add_item( $target_shipping_item );

				// Update the source item, but delay saving the changes until after the target items have all been added
				if ( $update_source_order ) {
					// Get a full list of source items, using the updated ones where they exist
					$source_product_items = array_filter( $source_order->get_items(), function ( $source_product_item ) use ( $items_by_type ) {
						return ! ( isset( $items_by_type['line_item'][ $source_product_item->get_id() ] ) );
					} );
					$source_product_items = array_merge( $source_product_items, $source_items );
					$source_items_meta    = static::shipping_items_metadata( $source_shipping_item->get_meta( 'Items' ), $source_product_items );

					$source_shipping_item->set_total( $total - $item_split_total );
					$source_shipping_item->update_meta_data( 'Items', $source_items_meta );

					// Keep a record of these items, so they can be saved after the target items have all been added
					$source_shipping_items[] = $source_shipping_item;
				}
			}
		}

		// Make sure all changes to the target order have been saved, before saving changes to the source items
		$target_order->calculate_totals();

		// Save the changes to all the source items
		if ( $update_source_order ) {
			foreach ( $source_items as $source_item ) {
				// Replace the item on the source order, so it is up-to-date with all the changes
				$source_order->add_item( $source_item );
				$source_item->save();
			}

			foreach ( $source_shipping_items as $source_shipping_item ) {
				// Replace the item on the source order, so it is up-to-date with all the changes
				$source_order->add_item( $source_shipping_item );
				$source_shipping_item->save();
			}
		}

		// Add metadata about the split itself
		$origin = static::origin( $source_order );
		$origin = $origin ? $origin : $source_order;

		$origin_count = $origin->get_meta( '_vibe_split_orders_origin_split_count' );
		$origin_count = $origin_count ? ( intval( $origin_count ) + 1 ) : 1;

		$origin->update_meta_data( '_vibe_split_orders_origin_split_count', $origin_count );
		$origin->save_meta_data();

		$split_count = $source_order->get_meta( '_vibe_split_orders_split_count' );
		$split_count = $split_count ? ( intval( $split_count ) + 1 ) : 1;

		$source_order->update_meta_data( '_vibe_split_orders_split_count', $split_count );

		$target_order->add_meta_data( '_vibe_split_orders_origin_id', $origin->get_id(), true );
		$target_order->add_meta_data( '_vibe_split_orders_split_from', $source_order->get_id(), true );
		$target_order->add_meta_data( '_vibe_split_orders_split_index', $split_count, true );
		$target_order->add_meta_data( '_vibe_split_orders_origin_split_index', $origin_count, true );

		// Make sure all changes are saved and totals updated
		$target_order->calculate_totals();
		$source_order->calculate_totals();

		$source_order = wc_get_order( $source_order->get_id() );
		$target_order = wc_get_order( $target_order->get_id() );

		do_action( Split_Orders::hook_prefix( 'orders_updated' ), $target_order, $source_order, $items );

		// Schedule the updated orders to be re-imported to the analytics, to update the stored data
		self::possibly_schedule_import( $source_order, $target_order );

		self::add_splitting_notes( $target_order, $source_order );

		do_action( Split_Orders::hook_prefix( 'after_order_split' ), $target_order, $source_order, $items );

		return $target_order;
	}

	/**
	 * Extracts the provided items meta and removes any that are not present in the product items
	 *
	 * @param string $existing_items_meta A correctly formatted items meta data value
	 * @param array $product_items An array of product line items
	 *
	 * @return string A revised items meta data value with missing products removed
	 */
	protected static function shipping_items_metadata( $existing_items_meta, $product_items ) {
		$new_items_meta = array();

		preg_match_all('/\b[^,]*&times; \d+\b/', $existing_items_meta, $items_meta );
		$items_meta = $items_meta[0];

		$items = array();
		foreach ( $product_items as $product_item ) {
			$items[ $product_item->get_name() ] = $product_item->get_quantity();
		}

		foreach ( $items_meta as $items_meta_item ) {
			preg_match('/^(.*) &times; (\d+)$/', $items_meta_item, $matches);
			$item_name = $matches[1];
			$item_qty = $matches[2];

			if ( isset( $items[ $item_name ] ) ) {
				$new_items_meta[] = $item_name . ' &times; ' . min( $item_qty, $items[ $item_name ] );
			}
		}

		return implode( ', ', $new_items_meta );
	}

	/**
	 * Returns an array of the default meta fields to be copied to new orders on split
	 *
	 * @return array An array of the order meta fields to copy
	 */
	public static function default_meta_fields_to_copy() {
		// Add VAT exemption field
		$default_meta_fields = array( 'is_vat_exempt' );

		// Add attribution meta to those to copy
		$default_meta_fields = array_merge( $default_meta_fields, array(
			'_wc_order_attribution_source_type',
			'_wc_order_attribution_device_type',
			'_wc_order_attribution_referrer',
			'_wc_order_attribution_session_count',
			'_wc_order_attribution_session_entry',
			'_wc_order_attribution_session_pages',
			'_wc_order_attribution_session_start_time',
			'_wc_order_attribution_user_agent',
			'_wc_order_attribution_utm_creative_format',
			'_wc_order_attribution_utm_marketing_tactic',
			'_wc_order_attribution_utm_source',
			'_wc_order_attribution_utm_source_platform'
		));

		return $default_meta_fields;
	}

	/**
	 * Schedule the import of the updated orders into the WooCommerce Admin
	 *
	 * @param WC_Order $source_order The source order to import
	 * @param WC_Order $target_order The target order to import
	 *
	 * @return void
	 */
	protected static function possibly_schedule_import( WC_Order $source_order, WC_Order $target_order ) {
		$scheduler_function       = array(
			'Automattic\WooCommerce\Internal\Admin\Schedulers\OrdersScheduler',
			'possibly_schedule_import'
		);
		$scheduler_function_pre64 = array(
			'Automattic\WooCommerce\Admin\Schedulers\OrdersScheduler',
			'possibly_schedule_import'
		);

		if ( ! method_exists( ...$scheduler_function ) ) {
			$scheduler_function = $scheduler_function_pre64;
		}

		if ( method_exists( ...$scheduler_function ) ) {
			$scheduler_function( $source_order->get_id() );
			$scheduler_function( $target_order->get_id() );
		}
	}

	/**
	 * Disable customer and admin emails unless vibe_split_orders_disable_emails filter returns false for this order
	 *
	 * Emails that will be disabled are:
	 *
	 * - Customer - Processing order
	 * - Customer - Completed order
	 * - Customer - Refunded order
	 * - Customer - On-hold order
	 * - Admin - New order
	 * - Admin - Cancelled order
	 * - Admin - Failed order
	 *
	 * @param WC_Order $order The order that is being split used for filtering
	 */
	public static function maybe_disable_emails( WC_Order $order ) {
		if ( apply_filters( Split_Orders::hook_prefix( 'disable_emails' ), true, $order ) ) {
			add_action( 'woocommerce_email_enabled_customer_processing_order', '__return_false' );
			add_action( 'woocommerce_email_enabled_customer_completed_order', '__return_false' );
			add_action( 'woocommerce_email_enabled_customer_refunded_order', '__return_false' );
			add_action( 'woocommerce_email_enabled_customer_on_hold_order', '__return_false' );
			add_action( 'woocommerce_email_enabled_cancelled_order', '__return_false' );
			add_action( 'woocommerce_email_enabled_failed_order', '__return_false' );
			add_action( 'woocommerce_email_enabled_new_order', '__return_false' );

			do_action( Split_Orders::hook_prefix( 'emails_disabled' ), $order );
		}
	}

	/**
	 * Restore customer and admin emails unless vibe_split_orders_disable_emails filter returns false for this order
	 *
	 * Emails that will be restored are:
	 *
	 * - Customer - Processing order
	 * - Customer - Completed order
	 * - Customer - Refunded order
	 * - Customer - On-hold order
	 * - Admin - New order
	 * - Admin - Cancelled order
	 * - Admin - Failed order
	 *
	 * @param WC_Order $order The order that is being split used for filtering
	 */
	public static function maybe_restore_emails( WC_Order $order ) {
		if ( apply_filters( Split_Orders::hook_prefix( 'disable_emails' ), true, $order ) ) {
			remove_action( 'woocommerce_email_enabled_customer_processing_order', '__return_false' );
			remove_action( 'woocommerce_email_enabled_customer_completed_order', '__return_false' );
			remove_action( 'woocommerce_email_enabled_customer_refunded_order', '__return_false' );
			remove_action( 'woocommerce_email_enabled_customer_on_hold_order', '__return_false' );
			remove_action( 'woocommerce_email_enabled_cancelled_order', '__return_false' );
			remove_action( 'woocommerce_email_enabled_failed_order', '__return_false' );
			remove_action( 'woocommerce_email_enabled_new_order', '__return_false' );

			do_action( Split_Orders::hook_prefix( 'emails_restored' ), $order );
		}
	}

	/**
	 * Adds a note to the given orders to record a split
	 *
	 * @param WC_Order $new_order      The new order that was split from the original order
	 * @param WC_Order $original_order The original order that was split
	 */
	public static function add_splitting_notes( WC_Order $new_order, WC_Order $original_order ) {
		if ( ! apply_filters( Split_Orders::hook_prefix( 'add_splitting_notes' ), true ) ) {
			return;
		}

		$message = sprintf(
		/* translators: 1: Link to the order split from 2: The order number of the order split from */
			__( 'Order split from <a href="%1$s">#%2$s</a>.', 'split-orders' ),
			$original_order->get_edit_order_url(),
			$original_order->get_order_number()
		);

		$new_order->add_order_note( $message, 0, false );

		$message = sprintf(
		/* translators: 1: Link to the order 2: The order number of the order split to */
			__( 'Order split into <a href="%1$s">#%2$s</a>.', 'split-orders' ),
			$new_order->get_edit_order_url(),
			$new_order->get_order_number()
		);

		$original_order->add_order_note( $message, 0, false );
	}

	/**
	 * Checks whether the given order is one part of a split order
	 *
	 * @param WC_Order $order The order to check
	 *
	 * @return bool True if the order is either part of a split, false otherwise
	 */
	public static function is_split( WC_Order $order ) {
		return static::is_split_parent( $order ) || static::is_split_child( $order );
	}

	/**
	 * Checks whether the given order is the original order of a split
	 *
	 * @param WC_Order $order The order to check
	 *
	 * @return bool True if the order is an original order that has been split, false otherwise
	 */
	public static function is_split_origin( WC_Order $order ) {
		return $order->meta_exists( '_vibe_split_orders_origin_split_count' );
	}

	/**
	 * Checks whether the given order is the parent order in at least one split operation
	 *
	 * Note that an order may be both a parent of a split and a child of a split.
	 *
	 * @param WC_Order $order The order to check
	 *
	 * @return bool True if the order has been split, false otherwise
	 */
	public static function is_split_parent( WC_Order $order ) {
		return $order->meta_exists( '_vibe_split_orders_split_count' );
	}

	/**
	 * Checks whether the given order was created as part of a split operation
	 *
	 * Note that an order may be both a parent of a split and a child of a split.
	 *
	 * @param WC_Order $order The order to check
	 *
	 * @return bool True if the order was created by a split, false otherwise
	 */
	public static function is_split_child( WC_Order $order ) {
		return $order->meta_exists( '_vibe_split_orders_split_from' );
	}

	/**
	 * Returns the original order the given order was split from
	 *
	 * @param WC_Order $order The order to get the origin for
	 *
	 * @return WC_Order|null The origin order or null if the given order was not split or the origin no-longer exists
	 */
	public static function origin( WC_Order $order ) {
		$origin_id = $order->get_meta( '_vibe_split_orders_origin_id' );

		return $origin_id ? wc_get_order( $origin_id ) : null;
	}

	/**
	 * Returns the order the given order was split from or null if the order was not split or the parent no-longer
	 * exists
	 *
	 * @param WC_Order $order The order to get the parent of
	 *
	 * @return WC_Order|null The order
	 */
	public static function parent( WC_Order $order ) {
		$parent_id = $order->get_meta( '_vibe_split_orders_split_from' );

		return $parent_id ? wc_get_order( $parent_id ) : null;
	}

	/**
	 * Returns all orders split from the given order
	 *
	 * @param WC_Order $order The order to fetch the children for
	 *
	 * @return WC_Order[] Orders split from the given order
	 */
	public static function children( WC_Order $order) {
		return wc_get_orders(
			array(
				'limit' => -1,
				'meta_key' => '_vibe_split_orders_split_from',
				'meta_value' => $order->get_id()
			)
		);
	}

	/**
	 * Returns all orders that have been split directly or indirectly from the given order
	 *
	 * @param WC_Order $origin The order to get the descendants of
	 *
	 * @return WC_Order[] An array of orders that have been split directly or indirectly from the given order
	 */
	public static function descendants( WC_Order $order ) {
		$parts = array();
		$queue = array( $order );

		while ( ! empty( $queue ) ) {
			// Get the next item in the queue to process
			$order = array_shift( $queue );

			// Get its children
			$children = static::children( $order );

			// Add the children to the list of parts and also add it to the queue to process
			$parts = array_merge( $parts, $children );
			$queue = array_merge( $queue, $children );
		}

		return $parts;
	}

	/**
	 * Returns the order status to apply to an order split from the given source order
	 *
	 * @param WC_Order $source_order The original order that is being split
	 * @param array    $items        The line items being split
	 *
	 * @return string Order status
	 */
	private static function order_status( WC_Order $source_order, array $items ) {
		$order_status_option = Settings::split_order_status();
		$order_status        = $order_status_option ? $order_status_option : $source_order->get_status();

		return apply_filters( Split_Orders::hook_prefix( 'split_order_status' ), $order_status, $source_order, $items );
	}
}
