<?php

namespace Vibe\Split_Orders;

use WC_Order;
use Automattic\WooCommerce\Utilities\OrderUtil;
use WP_Post;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/**
 * Sets up Admin modifications
 *
 * @since 1.0.0
 */
class Admin {

	/**
	 * Creates an instance and sets up the hooks to integrate with the admin
	 */
	public function __construct() {
		add_action( 'woocommerce_order_item_add_action_buttons', array( __CLASS__, 'output_split_button' ) );
		add_action( 'add_meta_boxes', array( __CLASS__, 'add_meta_boxes' ), 10, 2 );
		add_action( 'in_admin_footer', array( __CLASS__, 'output_modal' ) );

		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'enqueue_scripts' ) );

		// Delay setting up hooks for column so HPOS status can be checked
		add_action( 'plugins_loaded', array( __CLASS__, 'setup_split_from_column' ) );
	}

	public static function setup_split_from_column() {
		$list_table = static::is_hpos_enabled() ? 'woocommerce_page_wc-orders' : 'shop_order_posts';

		add_filter( "manage_{$list_table}_columns", array( __CLASS__, 'add_split_from_column' ), 20 );
		add_action( "manage_{$list_table}_custom_column", array( __CLASS__, 'split_from_column' ), 20, 2 );
	}

	/**
	 * Outputs an action button to split an order if not creating a new order and if the given order can be split
	 *
	 * @param WC_Order $order The order that would be split
	 */
	public static function output_split_button( $order ) {
		if ( Orders::can_split( $order->get_id() ) ) {
			printf( '<button type="button" class="button split-order" data-id="%d">%s</button>', esc_attr__( $order->get_id(), 'split-orders' ), esc_html__( 'Split order', 'split-orders' ) );
		}
	}

	/**
	 * Registers admin meta boxes
	 */
	public static function add_meta_boxes( $post_type = '', $order = null ) {
		if ( 'shop_order' === $post_type ) {
			$order = wc_get_order( $order );
		}

		if ( static::is_edit_order_screen() && Orders::is_split( $order ) ) {
			$origin = Orders::is_split_origin( $order ) ? $order : Orders::origin( $order );

			// We don't have anything to display unless there is an origin - it could have been deleted
			if ( $origin ) {
				// Add meta box for related split orders
				add_meta_box(
					'related_split_orders',
					__( 'Split Order Parts', 'split-orders' ),
					array( __CLASS__, 'related_orders_meta_box' ),
					array( wc_get_page_screen_id( 'shop-order' ), 'shop_order' ),
					'advanced',
					'high'
				);
			}
		}
	}

	/**
	 * Renders the related order parts meta box
	 *
	 * @param WC_Order|WP_Post $order The current order to display the related orders for
	 */
	public static function related_orders_meta_box( $order ) {
		// If HPOS is disabled then the order will be a WP_Post
		$order = wc_get_order( $order );
		$origin = Orders::is_split_origin( $order ) ? $order : Orders::origin( $order );
		$descendants = Orders::descendants( $origin );

		// Sort the descendants so they are displayed in a sensible order
		usort( $descendants, function( $a, $b ) {
			return ( $a->get_id() < $b->get_id() ) ? -1 : 1;
		} );

		$related_orders = array_merge( array( $origin ), $descendants );
		?>

		<div id="split-orders-related-orders">
		<table>
		<thead>
		<tr>
			<th><?php esc_html_e( 'Order Number', 'split-orders' ); ?></th>
			<th><?php esc_html_e( 'Relationship', 'split-orders' ); ?></th>
			<th><?php esc_html_e( 'Date', 'split-orders' ); ?></th>
			<th><?php esc_html_e( 'Status', 'split-orders' ); ?></th>
			<th><?php echo esc_html_x( 'Total', 'table heading', 'split-orders' ); ?></th>
		</tr>
		</thead>
		<tbody>
			<?php foreach ( $related_orders as $index => $related_order ) : ?>
				<tr>
					<td>
						<?php if ( $related_order->get_id() === $order->get_id() ) : ?>
							<?php
								// translators: placeholder is an order number.
								echo sprintf( esc_html_x( '#%s (this order)', 'hash before order number and current order indicator', 'split-orders' ), esc_html( $related_order->get_order_number() ) );
							?>
						<?php else : ?>
							<a href="<?php echo esc_url( $related_order->get_edit_order_url() ); ?>">
								<?php
								// translators: placeholder is an order number.
								echo sprintf( esc_html_x( '#%s', 'hash before order number', 'split-orders' ), esc_html( $related_order->get_order_number() ) );
								?>
							</a>
						<?php endif; ?>
					</td>
					<td>
						<?php
						if ( 0 === $index ) {
							esc_html_e( 'Original Order', 'split-orders' );
						} else {
							esc_html_e( 'Split Part', 'split-orders' );
						}
						?>
					</td>
					<td>
						<?php
						$date_created = $related_order->get_date_created();

						if ( $date_created ) {
							$date_created_timestamp = $date_created->getTimestamp();
							$t_time                 = $related_order->get_date_created()->date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) );
							$date_to_display        = $t_time;

							if ( $date_created_timestamp > strtotime( '-1 day', time() ) && $date_created_timestamp <= time() ) {
								$date_to_display = sprintf(
									/* translators: %s: human-readable time difference */
									_x( '%s ago', '%s = human-readable time difference', 'split-orders' ),
									human_time_diff( $date_created_timestamp, time() )
								);
							}
						} else {
							$t_time          = __( 'Unpublished', 'split-orders' );
							$date_to_display = $t_time;
						}
						?>
						<abbr title="<?php echo esc_attr( $t_time ); ?>">
							<?php echo esc_html( $date_to_display ); ?>
						</abbr>
					</td>
					<td>
						<?php
						$classes = array(
							'order-status',
							sanitize_html_class( 'status-' . $related_order->get_status() ),
						);

						$status_name = wc_get_order_status_name( $related_order->get_status() );

						printf( '<mark class="%s"><span>%s</span></mark>', esc_attr( implode( ' ', $classes ) ), esc_html( $status_name ) );
						?>
					</td>
					<td>
						<span class="amount"><?php echo wp_kses( $related_order->get_formatted_order_total(), array( 'small' => array(), 'span' => array( 'class' => array() ), 'del' => array(), 'ins' => array() ) ); ?></span>
					</td>
				</tr>
			<?php endforeach; ?>

		</tbody>
		</table>

		<?php
	}

	/**
	 * Checks whether the current screen is one to display split functionality on
	 *
	 * By default splittable screens would be any for the shop_order post type, except adding a new post
	 *
	 * @return bool True if the current screen is one that should include a split button, false otherwise
	 */
	public static function is_splittable_screen() {
		$is_splittable = static::is_edit_order_screen();

		return apply_filters( Split_Orders::hook_prefix( 'is_splittable_screen' ), $is_splittable, get_current_screen() );
	}

	public static function is_edit_order_screen() {
		$screen        = get_current_screen();
		$screen_base   = isset( $screen->base ) ? $screen->base : '';
		$screen_id     = isset( $screen->id ) ? $screen->id : '';
		$screen_action = isset( $screen->action ) ? $screen->action : '';

		$order_screen      = function_exists( 'wc_get_page_screen_id' ) && wc_get_page_screen_id( 'shop-order' ) == $screen_base;
		$post_order_screen = ( 'post' === $screen_base && 'shop_order' === $screen_id && 'add' !== $screen_action );

		return $order_screen || $post_order_screen;
	}

	/**
	 * Outputs the HTML for a modal to be used for splitting an order
	 */
	public static function output_modal() {
		if ( static::is_splittable_screen() ) {
			?>
			<script type="text/template" id="tmpl-wc-modal-split-order">
				<div class="wc-backbone-modal">
					<div class="wc-backbone-modal-content split-orders-modal-content">
						<section class="wc-backbone-modal-main" role="main">
							<header class="wc-backbone-modal-header">
								<h1><?php esc_html_e( 'Split order', 'split-orders' ); ?></h1>
								<button class="modal-close modal-close-link dashicons dashicons-no-alt">
									<span class="screen-reader-text"><?php esc_html_e( 'Close modal panel', 'split-orders' ); ?></span>
								</button>
							</header>
							<article id="modal-split-order-line-items">
								<?php // Will be populated by AJAX when opened ?>
							</article>
							<footer>
								<div class="inner">
									<button id="btn-ok" class="button button-primary button-large"><?php esc_html_e( 'Complete split', 'split-orders' ); ?></button>
								</div>
							</footer>
						</section>
					</div>
				</div>
				<div class="wc-backbone-modal-backdrop modal-close"></div>
			</script>

			<script type="text/template" id="tmpl-wc-modal-split-order-alert">
				<div class="wc-backbone-modal">
					<div class="wc-backbone-modal-content split-orders-modal-content">
						<section class="wc-backbone-modal-main" role="main">
							<header class="wc-backbone-modal-header">
								<h1><?php esc_html_e( 'Error - Split not completed', 'split-orders' ); ?></h1>
								<button class="modal-close modal-close-link dashicons dashicons-no-alt">
									<span class="screen-reader-text"><?php esc_html_e( 'Close modal panel', 'split-orders' ); ?></span>
								</button>
							</header>
							<article id="modal-split-order-alert-message">
								<div class="notice notice-error">
									{{{ data.message }}}
								</div>

								<# if ( data.error_type == "error" ) { #>
									<h2><?php esc_html_e( 'Why has this happened?', 'split-orders' ); ?></h2>

									<p><?php esc_html_e( 'This usually occurs because another active plugin on the website is attempting to perform an action in response to the split, but is failing with an error.', 'split-orders' ); ?></p>

									<# if ( data.error_origin ) { #>
										<p><?php echo wp_kses_post( __( 'The error appears to have originated in the <strong>{{ data.error_origin }}</strong> plugin.', 'split-orders' ) ); ?></p>
									<# } #>

									<h2><?php esc_html_e( 'What to do now?', 'split-orders' ); ?></h2>

									<p><?php esc_html_e( 'First, try the split again to check it was not a one-off glitch.', 'split-orders' ); ?></p>

									<# if ( data.error_origin ) { #>
										<p><?php echo wp_kses_post( __( 'If the <strong>{{ data.error_origin }}</strong> plugin is no-longer required, deactivating it may solve the issue.', 'split-orders' ) ); ?></p>
									<# } #>

									<p><?php echo wp_kses_post( __( 'If the error still occurs, <a href="https://woocommerce.com/my-account/contact-support/" target="_blank">open a support ticket</a> on your woo.com account, so we can investigate a solution for you. Please include the full error from the box below in your message.', 'split-orders' ) ); ?></p>
								<# } #>

								<# if ( data.error_type == "exception" ) { #>
									<h2><?php esc_html_e( 'Why has this happened?', 'split-orders' ); ?></h2>

									<p><?php esc_html_e( 'This usually indicates that the split was not able to create a new order. It may be caused by a conflict with another plugin, or a glitch with the Split Orders plugin itself.', 'split-orders' ); ?></p>

									<h2><?php esc_html_e( 'What to do now?', 'split-orders' ); ?></h2>

									<p><?php esc_html_e( 'The split may have partially completed before the error. To keep your order history clean, confirm that the original order is still complete and that a new order did not get created by the split.', 'split-orders' ); ?></p>

									<p><?php esc_html_e( 'Try the split again to check it was not a one-off glitch.', 'split-orders' ); ?></p>

									<p><?php echo wp_kses_post( __( 'If the error still occurs, please <a href="https://woocommerce.com/my-account/contact-support/" target="_blank">open a support ticket</a> on your woo.com account, and include the full error details displayed below.', 'split-orders' ) ); ?></p>
								<# } #>

								<# if ( data.log ) { #>
									<textarea id="split-order-alert-log" rows="4" style="width: 100%;">{{ data.log }}</textarea>

									<button id="split-order-alert-copy-to-clipboard" class="button button-small"><?php esc_html_e( 'Copy to clipboard', 'split-orders' ); ?></button>
								<# } #>
							</article>
							<footer>
								<div class="inner">
									<button id="btn-ok" class="button button-primary button-large"><?php esc_html_e( 'Close', 'split-orders' ); ?></button>
								</div>
							</footer>
						</section>
					</div>
				</div>
				<div class="wc-backbone-modal-backdrop modal-close"></div>
			</script>
			<?php
		}
	}

	/**
	 * Returns the HTML to populate the order splitting modal, with line items from the given order
	 *
	 * @param int $order_id The ID of the order to generate the output for
	 *
	 * @return string An HTML string with controls for splitting the given order
	 */
	public static function get_splitting_popup( $order_id ) {
		$order = wc_get_order( $order_id );
		$items = $order ? $order->get_items() : false;
		$shipping = $order ? $order->get_items( 'shipping' ) : false;

		if ( ! $items ) {
			return '';
		}

		/**
		 * Filters the notices to display prior to splitting the given order
		 *
		 * The notices array should be arranged with the key being one of the following notice types and the values
		 * being either a string to be used as a message, or an array of strings to use as messages.
		 *
		 * - info
		 * - success
		 * - warning
		 * - error
		 */
		$notices = apply_filters( Split_Orders::hook_prefix( 'pre_split_notices' ), array(), $order );

		ob_start();

		include vibe_split_orders()->path( 'includes/partials/popup.php' );

		return ob_get_clean();
	}

	/**
	 * Enqueues scripts and styles on the order admin pages
	 */
	public static function enqueue_scripts() {
		if ( ! static::is_splittable_screen() ) {
			return;
		}

		$handle = Split_Orders::hook_prefix( 'js' );

		wp_register_script(
			$handle,
			vibe_split_orders()->uri( 'assets/js/vibe-split-orders.min.js' ),
			array( 'jquery' ),
			vibe_split_orders()->get_version()
		);
		wp_localize_script( $handle, 'vibe_split_orders_data', static::script_data() );

		wp_enqueue_script( $handle );

		$handle = Split_Orders::hook_prefix( 'css' );
		wp_enqueue_style(
			$handle,
			vibe_split_orders()->uri( 'assets/css/vibe-split-orders.min.css' ),
			array(),
			vibe_split_orders()->get_version(),
			'all'
		);
	}

	/**
	 * Sets up data to be passed to front end via script localisation
	 *
	 * @return array An array of data items
	 */
	public static function script_data() {
		$script_data['ajaxurl']         = admin_url( 'admin-ajax.php' );
		$script_data['popup_nonce']     = wp_create_nonce( Split_Orders::hook_prefix( 'popup-nonce' ) );
		$script_data['splitting_nonce'] = wp_create_nonce( Split_Orders::hook_prefix( 'splitting-nonce' ) );

		return apply_filters( Split_Orders::hook_prefix( 'script_data' ), $script_data );
	}

	/**
	 * Adds Split For column to the array of columns after the existing order status column
	 *
	 * @param array $columns The existing array of columns
	 *
	 * @return array The updated array of columns
	 */
	public static function add_split_from_column( $columns ) {
		$updated_columns = array();

		foreach ( $columns as $key => $column ) {
			$updated_columns[ $key ] = $column;

			// Insert after order status column
			if ( 'order_status' === $key ) {
				$updated_columns['split_from'] = __( 'Split From', 'split-orders' );
			}
		}
		return $updated_columns;
	}

	/**
	 * Renders the content of the Split From column for one row, for the given order
	 *
	 * @param string $column The name of the column to render
	 * @param int|WC_Order $order The order or order_id (depending on if using HPOS or not) to render the column for
	 */
	public static function split_from_column( $column, $order ) {
		if ( 'split_from' === $column ) {
			$order = wc_get_order( $order );
			$origin = wc_get_order( $order->get_meta( '_vibe_split_orders_origin_id' ) );

			if ( $origin ) {
				?>

				<a href="<?php echo esc_url( $origin->get_edit_order_url() ); ?>">
					<?php
					// translators: placeholder is an order number.
					echo sprintf( esc_html_x( '#%s', 'hash before order number', 'split-orders' ), esc_html( $origin->get_order_number() ) );
					?>
				</a>

				<?php
			} else {
				echo '&ndash;';
			}
		}
	}

	/**
	 * Outputs the given array of notices
	 *
	 * The notices array should be arranged with the key being one of the following notice types and the values
	 * being either a string to be used as a message, or an array of strings to use as messages.
	 *
	 * - info
	 * - success
	 * - warning
	 * - error
	 *
	 * @param array $notices
	 */
	public static function output_notices( array $notices ) {
		foreach ( $notices as $notice_type => $type_notices ) {
			if ( ! static::is_valid_notice_type( $notice_type ) ) {
				continue;
			}

			foreach ( $type_notices as $message ) {
				?>

				<div class="notice notice-<?php echo esc_attr( $notice_type ); ?>">
					<?php echo wp_kses_post( $message ); ?>
				</div>

				<?php
			}
		}
	}

	public static function add_notice( array $notices, $message, $type = 'info' ) {
		if ( ! static::is_valid_notice_type( $type ) ) {
			$type = 'info';
		}

		if ( ! isset( $notices[ $type ] ) ) {
			$notices[ $type ] = array();
		} elseif ( ! is_array( $notices[ $type ] ) ) {
			$notices[ $type ] = array( $notices[ $type ] );
		}

		$notices[ $type ][] = strval( $message );

		return $notices;
	}

	protected static function is_valid_notice_type( $type ) {
		return in_array( $type, array( 'info', 'success', 'warning', 'error' ) );
	}

	public static function is_hpos_enabled() {
		if ( ! did_action( 'plugins_loaded' ) && ! doing_action( 'plugins_loaded' ) ) {
			_doing_it_wrong( __FUNCTION__, esc_html__( 'is_hpos_enabled should not be called before the init action.', 'split-orders' ), '1.7.0' );
		}

		return class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) && OrderUtil::custom_orders_table_usage_is_enabled();
	}
}
