<?php

namespace Vibe\Split_Orders;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/**
 * Sets up Settings page and provides access to setting values
 *
 * @since 1.4.0
 */
class Settings {

	protected static $default = false;

	/**
	 * Creates an instance and sets up the hooks to integrate with the admin
	 */
	public function __construct() {
		add_filter( 'woocommerce_get_settings_pages', array( __CLASS__, 'add_orders_settings_tab' ) );

		add_filter( 'woocommerce_get_sections_vibe-orders', array( __CLASS__, 'add_settings_page' ) );
		add_filter( 'woocommerce_get_settings_vibe-orders', array( __CLASS__, 'add_settings' ), 10, 2 );
		add_filter( 'plugin_action_links', array( __CLASS__, 'add_settings_link' ), 10, 2 );

		add_filter( 'woocommerce_get_sections_advanced', array( __CLASS__, 'add_old_settings_page' ) );
		add_filter( 'woocommerce_get_settings_advanced', array( __CLASS__, 'add_old_settings' ), 10, 2 );

		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'add_settings_scripts' ), 11 );
	}

	/**
	 * Adds an Orders setting tab if it has not already been added (by one of our other order management plugins)
	 *
	 * @param array $settings An array of the existing settings
	 *
	 * @return array The settings array with our Orders tab added
	 */
	public static function add_orders_settings_tab( $settings ) {
		$settings_ids = array_map( function( $settings_page ) {
			return $settings_page->get_id();
		}, $settings );

		if ( ! in_array( 'vibe-orders', $settings_ids ) ) {
			// Load split orders as the default section for the Orders tab, only if it created the Orders tab
			self::$default = true;

			$settings[] = new Settings_Tab_Orders();
		}

		return $settings;
	}

	/**
	 * Adds a section to the orders tab
	 *
	 * @param array $sections The existing settings sections on the orders tab
	 *
	 * @return array The sections with the split-orders settings section added
	 */
	public static function add_settings_page( array $sections ) {
		$sections['split-orders'] = __( 'Split orders', 'split-orders' );

		return $sections;
	}

	/**
	 * Adds setting fields to the split-orders section of the settings
	 *
	 * @param array  $settings        The current settings
	 * @param string $current_section The name of the current section of settings
	 *
	 * @return array The settings fields including split orders settings if the current section is 'split-orders'
	 */
	public static function add_settings( array $settings, $current_section ) {
		if ( 'split-orders' != $current_section && ! ( self::$default && '' === $current_section ) ) {
			return $settings;
		}

		$remove_old = isset( $_REQUEST['remove-old'] ) ? boolval( $_REQUEST['remove-old'] ) : false;

		if ( $remove_old ) {
			update_option( 'vibe_split_orders_remove_old_settings', $remove_old );
		}

		$settings[] = array(
			'name' => __( 'Split orders', 'split-orders' ),
			'type' => 'title',
			'desc' => __( 'The following options are used to configure the Split Orders extension.', 'split-orders' )
		);

		$settings[] = array(
			'name'     => __( 'Custom meta fields', 'split-orders' ),
			'desc_tip' => __( 'Any fields added here will be copied to the new order when splitting. Use this to add support for custom fields added by other plugins.<br /><br />
							   Input each field on a new line, or separated by a comma.', 'split-orders' ),
			'id'       => Split_Orders::hook_prefix( 'meta_fields' ),
			'type'     => 'textarea',
			'css'      => 'min-width: 50%; height: 100px;'
		);

		$settings[] = array(
			'name'     => __( 'Order status', 'split-orders' ),
			'desc_tip' => __( 'This option will set the order status for the new split order.', 'split-orders' ),
			'id'       => Split_Orders::hook_prefix( 'order_status' ),
			'type'     => 'select',
			'css'      => 'min-width: 50%;',
			'options'  => self::order_status_options(),
			'default'  => '0'
		);

		$settings = apply_filters( Split_Orders::hook_prefix( 'settings' ), $settings );

		$settings[] = array( 'type' => 'sectionend', 'id' => 'split-orders' );

		return $settings;
	}

	protected static function display_old_settings() {
		return ! get_option( 'vibe_split_orders_remove_old_settings', false );
	}

	/**
	 * Adds a section to the advanced tab where our settings used to be
	 *
	 * @param array $sections The existing settings sections on the advanced tab
	 *
	 * @return array The sections with a section added
	 */
	public static function add_old_settings_page( $sections ) {
		if ( static::display_old_settings() ) {
			$sections['split-orders-old'] = __( 'Split orders', 'split-orders' );
		}

		return $sections;
	}

	/**
	 * Adds setting fields to the old split-orders section of the settings
	 *
	 * @param array  $settings        The current settings
	 * @param string $current_section The name of the current section of settings
	 *
	 * @return array The settings fields including settings to explain the movement of the settings
	 */
	public static function add_old_settings( array $settings, $current_section ) {
		if ( 'split-orders-old' != $current_section || ! static::display_old_settings() ) {
			return $settings;
		}

		global $hide_save_button;
		$hide_save_button = true;

		$settings_link = add_query_arg( 'remove-old', '1', static::settings_link() );

		$settings[] = array(
			'name' => __( 'Split orders', 'split-orders' ),
			'type' => 'title',
			/* translators: %s: URL to new settings page */
			'desc' => sprintf( __( 'The settings for Split Orders have now moved to <a href="%s">Settings > Orders > Split Orders</a>.', 'split-orders' ), $settings_link )
		);

		$settings[] = array( 'type' => 'sectionend', 'id' => 'split-orders' );

		return $settings;
	}

	/**
	 * Fetches and returns the meta fields setting after cleaning it and splitting into an array
	 *
	 * @return array The meta fields setting cleaned up
	 */
	public static function meta_fields() {
		$option = get_option( Split_Orders::hook_prefix( 'meta_fields' ), '' );

		// Split at commas and new-line characters
		$fields = preg_split( '/[\n,]/', $option );

		// Trim
		$fields = array_map( 'trim', $fields );

		// Remove blanks
		$fields = array_filter( $fields );

		// Remove any duplicates
		$fields = array_unique( $fields );

		return array_values( $fields );
	}

	/**
	 * Adds a settings link to the plugin's action links
	 *
	 * @return array The plugin's action links with a link to the plugin settings added
	 */
	public static function add_settings_link( $plugin_actions, $plugin_file ) {
		$new_actions = array();

		if ( 'split-orders.php' === basename( $plugin_file ) ) {
			/* translators: %s: Settings */
			$new_actions['settings'] = sprintf( __( '<a href="%s">Settings</a>', 'split-orders' ), esc_url( static::settings_link() ) );
		}

		return array_merge( $new_actions, $plugin_actions );
	}

	public static function settings_link() {
		return admin_url( 'admin.php?page=wc-settings&tab=vibe-orders&section=split-orders' );
	}

	/**
	 * Returns the possible options for the order status field
	 *
	 * @return array An array of the available order statuses for selection
	 */
	private static function order_status_options() {
		$order_statuses = wc_get_order_statuses();

		return array_merge( array( '0' => __( 'Same as the original', 'split-orders' ) ), $order_statuses );
	}

	/**
	 * Fetches and returns the order status setting to be applied to new orders
	 *
	 * @return string|null The order status if set or null otherwise
	 */
	public static function split_order_status() {
		$option_field = get_option( Split_Orders::hook_prefix( 'order_status' ) );

		// Check if the option field order status is valid
		return array_key_exists( $option_field, self::order_status_options() ) ? $option_field : null;
	}

	/**
	 * Fetches and returns the order number suffix setting
	 *
	 * @return bool True if order number suffix is enabled, false otherwise
	 */
	public static function order_number_suffix_enabled() {
		return Order_Numbers::suffix_enabled();
	}

	/**
	 * Fetches the meta fields found in the last 10 orders
	 *
	 * @return array An array of meta fields with duplicates removed
	 */
	private static function meta_fields_from_recent_orders() {
		$meta_data_array = array();

		$args   = array(
			'type'   => 'shop_order',
			'status' => array( 'wc-completed', 'wc-processing', 'wc-on-hold' ),
			'meta_key' => '_vibe_split_orders_split_from',
			'meta_compare' => 'NOT EXISTS',
			'limit'  => 20,
		);
		$orders = wc_get_orders( $args );

		foreach ( $orders as $order ) {
			$get_meta = $order->get_meta_data();

			foreach ( $get_meta as $meta ) {
				$meta_get_data     = $meta->get_data();
				$meta_data_array[] = $meta_get_data['key'];
			}
		}

		return array_unique( array_filter( $meta_data_array ) );
	}

	/**
	 * Returns an array of suggested meta fields, excluding any already in use
	 *
	 * @return array An array of suggested meta fields
	 */
	public static function suggested_meta_fields() {
		// Get the default meta fields to remove from the suggestions
		$default_meta_fields = Orders::default_meta_fields_to_copy();

		// Other known fields that should not be copied
		$exclude = array(
			'_edit_lock',
			'_order_number',
			'_vibe_split_orders_origin_id',
			'_vibe_split_orders_origin_split_index',
			'_vibe_split_orders_origin_split_count',
			'_vibe_split_orders_split_from',
			'_vibe_split_orders_split_index',
			'_vibe_split_orders_split_count',
			'_vibe_merge_orders_merged_into',
			'_billing_address_index',
			'_shipping_address_index'
		);

		$exclude = array_merge( $exclude, $default_meta_fields );

		// Allow additions/subtractions from the array of fields to exclude from suggestions
		$exclude = apply_filters( Split_Orders::hook_prefix( 'excluded_suggested_meta_fields' ), $exclude );

		// Remove any meta fields already saved in the field
		$exclude = array_merge( $exclude, self::meta_fields() );

		return array_values( array_diff( self::meta_fields_from_recent_orders(), $exclude ) );
	}

	/**
	 * Enqueues scripts and styles on the split order settings page
	 */
	public static function add_settings_scripts() {
		if (! self::is_current_screen()) {
			return;
		}

		$handle = Split_Orders::hook_prefix( 'settings_js' );

		wp_register_script(
			$handle,
			vibe_split_orders()->uri( 'assets/js/vibe-split-orders-settings.min.js' ),
			array( 'jquery' ),
			vibe_split_orders()->get_version(),
			true
		);
		wp_localize_script( $handle, 'vibe_split_orders_data', self::script_data() );

		wp_enqueue_script( $handle );

		$handle = Split_Orders::hook_prefix( 'settings_css' );

		wp_enqueue_style(
			$handle,
			vibe_split_orders()->uri( 'assets/css/admin.min.css' ),
			array(),
			vibe_split_orders()->get_version(),
			'all'
		);
	}

	/**
	 * Sets up data to be passed to front end via script localisation
	 *
	 * @return array An array of data items
	 */
	public static function script_data() {
		$script_data['suggested_meta_fields'] = self::suggested_meta_fields();
		$script_data['recent_fields_label'] = __( 'Meta fields detected from recent orders:', 'split-orders' );

		return apply_filters( Split_Orders::hook_prefix( 'settings_script_data' ), $script_data );
	}

	/**
	 * Returns true if the current screen is the plugin settings page
	 *
	 * @return bool True if the current screen is the settings page, false otherwise
	 */
	public static function is_current_screen() {
		$screen      = get_current_screen();
		$screen_base = isset( $screen->base ) ? $screen->base : '';

		if ( 'woocommerce_page_wc-settings' !== $screen_base ) {
			return false;
		}

		$tab = isset( $_REQUEST['tab'] ) ? wc_clean( $_REQUEST['tab'] ) : '';

		if ( 'vibe-orders' !== $tab ) {
			return false;
		}

		$tab_section = isset( $_REQUEST['section'] ) ? wc_clean( $_REQUEST['section'] ) : '';

		return ( 'split-orders' == $tab_section ) || ( self::$default && '' == $tab_section );
	}
}
