<?php

namespace Vibe\Split_Orders\Upgrades;

use Vibe\Split_Orders\Admin;
use Vibe\Split_Orders\Orders;
use WC_Order;

defined( 'ABSPATH' ) || exit; // Exit if accessed directly

/**
 * Runs required upgrades for the 1.8.0 release
 *
 * @since 1.8.0
 */
class Upgrade_180 extends Upgrade {

	/**
	 * Add metadata to batch of split orders about origin
	 */
	public function run() {
		$origins = $this->origins_to_process();

		// No origins to process, so the upgrade must be complete
		if ( empty( $origins ) ) {
			return true;
		}

		foreach ( $origins as $origin ) {
			$descendants = Orders::descendants( $origin );

			usort( $descendants, function( $a, $b ) {
				return ( $a->get_id() < $b->get_id() ) ? -1 : 1;
			} );

			foreach ( $descendants as $index => $descendant ) {
				$descendant->update_meta_data( '_vibe_split_orders_origin_id', $origin->get_id() );
				$descendant->update_meta_data( '_vibe_split_orders_origin_split_index', $index + 1 );
				$descendant->save_meta_data();
			}

			$origin_split_count = count( $descendants );
			$split_count = $origin->get_meta( '_vibe_split_orders_split_count' );

			// Don't allow the count to be less than the existing split count, because it might mess up order number suffixes
			if ( $origin_split_count < $split_count ) {
				$origin_split_count = $split_count;
			}

			$origin->update_meta_data( '_vibe_split_orders_origin_split_count', $origin_split_count );
			$origin->save_meta_data();
		}

		// Assume there's more to process
		return false;
	}

	/**
	 * Gets an array of orders that are unprocessed origins
	 *
	 * @param int $count The maximum number of origins to get
	 *
	 * @return WC_Order[] An array of orders
	 */
	public function origins_to_process( $count = 10 ) {
		if ( Admin::is_hpos_enabled() ) {
			// Get only the origins of a split that have not already had the origin split count meta added
			$origins = wc_get_orders(
				array(
					'type' => 'shop_order',
					'limit' => $count,
					'meta_query' => array(
						array(
							'key' => '_vibe_split_orders_split_count',
							'compare' => 'EXISTS',
						),
						array(
							'key' => '_vibe_split_orders_split_from',
							'compare' => 'NOT EXISTS',
						),
						array(
							'key' => '_vibe_split_orders_origin_split_count',
							'compare' => 'NOT EXISTS',
						)
					)
				)
			);
		} else {
			// When not using HPOS the meta query needs to be added via a filter on the post query
			add_filter( 'woocommerce_order_data_store_cpt_get_orders_query', array( __CLASS__, 'origins_query_meta' ), 10, 2 );

			$origins = wc_get_orders(
				array(
					'type' => 'shop_order',
					'split-order-origins' => true,
					'limit' => $count
				)
			);

			remove_filter( 'woocommerce_order_data_store_cpt_get_orders_query', array( __CLASS__, 'origins_query_meta' ), 10 );
		}

		return $origins;
	}

	public static function origins_query_meta( array $query, array $query_vars ) {
		if ( ! empty( $query_vars['split-order-origins'] ) && $query_vars['split-order-origins'] ) {
			$query['meta_query'][] = array(
				'key' => '_vibe_split_orders_split_count',
				'compare' => 'EXISTS',
			);
			$query['meta_query'][] = array(
				'key' => '_vibe_split_orders_split_from',
				'compare' => 'NOT EXISTS',
			);
			$query['meta_query'][] = array(
				'key' => '_vibe_split_orders_origin_split_count',
				'compare' => 'NOT EXISTS',
			);
		}

		return $query;
	}
}
