<?php

namespace Vibe\Split_Orders;

use WC_Order;

/**
 * Handles customising order numbers
 *
 * @since 1.7
 */
class Order_Numbers {

	/**
	 * Creates an instance and sets up hooks to integrate with the rest of the extension to provide features for
	 * customising order numbers
	 */
	public function __construct() {
		add_filter( Split_Orders::hook_prefix( 'settings' ), array( __CLASS__, 'add_settings' ) );
		add_filter( Split_Orders::hook_prefix( 'system_status_items' ), array( __CLASS__, 'add_system_status_items' ) );
	}

	/**
	 * Order number settings are off by default and will normally only be turned on if a supported plugin is active
	 *
	 * @return mixed|null
	 */
	public static function settings_active() {
		return apply_filters( Split_Orders::hook_prefix( 'order_number_settings_active' ), false );
	}

	/**
	 * Fetches and returns the order number suffix setting
	 *
	 * @return bool True if order number suffix is enabled, false otherwise
	 */
	public static function suffix_enabled() {
		return static::settings_active() && get_option( Split_Orders::hook_prefix( 'order_number_suffix' ) ) === 'yes';
	}

	/**
	 * Adds settings for managing order numbers
	 *
	 * @param array $settings An array of the existing settings
	 *
	 * @return array The updated settings array
	 */
	public static function add_settings( $settings ) {
		if ( static::settings_active() ) {

			$settings[] = array(
				'name'     => __( 'Order numbers', 'split-orders' ),
				'desc'     => __( 'Assign the same order number on split', 'split-orders' ),
				'desc_tip' => __( 'If enabled, orders will be assigned the same order number as the order they were split from. An index will be appended to differentiate orders. For example: #123-1', 'split-orders' ),
				'id'       => Split_Orders::hook_prefix( 'order_number_suffix' ),
				'type'     => 'checkbox'
			);
		}

		return $settings;
	}

	/**
	 * Adds information to the system status report
	 *
	 * @param array $debug_data An array of the existing data to be output in the system status report
	 *
	 * @return array The updated array of data for the system status report
	 */
	public static function add_system_status_items( $debug_data ) {
		$is_order_number_suffix_enabled = static::suffix_enabled();

		$debug_data['append_order_number_suffix'] = array(
			'name'      => _x( 'Append order number suffix', 'system status item', 'split-orders' ),
			'label'     => 'Append order number suffix',
			'note'    => $is_order_number_suffix_enabled ? __( 'Yes', 'split-orders' ) : __( 'No', 'split-orders' ),
			'success' => $is_order_number_suffix_enabled
		);

		return $debug_data;
	}

	/**
	 * Generates an order number for the given new order by appending the index of the split
	 *
	 * @param WC_Order $new_order The order to generate an order number for
	 * @param WC_Order $source_order The original order the new_order was split from
	 *
	 * @return bool|string An order number or false if no order number could be generated in the expected format
	 */
	public static function generate( WC_Order $new_order, WC_Order $source_order ) {
		$order_number_formatted = false;
		$order_number           = $source_order->get_order_number();
		$split_index            = $new_order->get_meta( '_vibe_split_orders_split_index' );

		if ( $split_index ) {
			$order_number_formatted = sprintf( '%s-%d', $order_number, $split_index );
		}

		return apply_filters( Split_Orders::hook_prefix( 'split_order_number' ), $order_number_formatted, $new_order, $source_order, $split_index );
	}
}
