<?php
/**
 * Plugin Name: WooCommerce Merge
 * Description: This plugin allows merging WooCommerce products.
 * Version: 0.1.0
 * Author: Le Web simple
 * Author URI: https://lewebsimple.ca
 * Text Domain: wc-merge
 */

if ( ! defined( 'WPINC' ) ) {
	die;
}

const WC_MERGE_VERSION = '0.1.0';

class WC_Merge {

	/**
	 * Singleton instance
	 * @var WC_Merge
	 */
	private static $instance = null;

	public static function get_instance() {
		if ( self::$instance === null ) {
			self::$instance = new self;
		}
		return self::$instance;
	}

	public function __construct() {
		add_action( 'plugins_loaded', array( $this, 'init' ) );
	}

	/**
	 * Initialize WooCommerce Merge plugin
	 *
	 * @return void
	 */
	public function init() {
		load_plugin_textdomain( 'wc-merge', false, '/wc-merge/languages' );
		add_filter( 'bulk_actions-edit-product', array( $this, 'bulk_actions_register' ) );
		add_filter( 'handle_bulk_actions-edit-product', array( $this, 'bulk_actions_handle' ), 10, 3 );
		add_action( 'admin_notices', array( $this, 'admin_notices' ) );
	}

	/**
	 * Register bulk actions
	 *
	 * @param $bulk_actions
	 *
	 * @return mixed
	 */
	public function bulk_actions_register( $bulk_actions ) {
		$bulk_actions['wc-merge'] = __( "Merge products", 'wc-merge' );
		return $bulk_actions;
	}

	/**
	 * Handle bulk actions
	 *
	 * @param $redirect_url
	 * @param $action
	 * @param $post_ids
	 *
	 * @return mixed
	 */
	public function bulk_actions_handle( $redirect_url, $action, $post_ids ) {
		if ( sizeof( $post_ids ) <= 1 ) {
			return add_query_arg( array(
				'wc_merge_status' => 'error',
			), $redirect_url );
		}

		$obsolete_variations = array();
		$wc_products         = array();
		$attributes          = array();
		foreach ( $post_ids as $post_id ) {
			if ( empty( $wc_product = wc_get_product( $post_id ) ) ) {
				continue;
			}
			switch ( $wc_product->get_type() ) {
				case 'simple':
					$wc_products[] = $wc_product;
					break;
				case 'variable':
					$wc_products           = array_merge( $wc_products, $wc_product->get_available_variations( 'objects' ) );
					$obsolete_variations[] = $wc_product;
					break;
				default:
					continue 2;
			}

			/**
			 * @var WC_Product_Attribute $wc_attribute
			 */
			foreach ( $wc_product->get_attributes() as $slug => $wc_attribute ) {
				if ( empty( $attributes[ $slug ] ) ) {
					$attributes[ $slug ] = new WC_Product_Attribute();
					if ( $wc_attribute->is_taxonomy() ) {
						$attributes[ $slug ]->set_id( $wc_attribute->get_id() );
					}
					$attributes[ $slug ]->set_name( $wc_attribute->get_name() );
					$attributes[ $slug ]->set_visible( true );
					$attributes[ $slug ]->set_variation( true );
				}
				$options = array_unique( array_merge( $attributes[ $slug ]->get_options(), $wc_attribute->get_options() ) );
				$attributes[ $slug ]->set_options( $options );
			}
		}

		$attributes['options'] = new WC_Product_Attribute();
		$attributes['options']->set_name( 'Options' );
		$attributes['options']->set_visible( true );
		$attributes['options']->set_variation( true );
		$attributes['options']->set_options( array_map( function ( $wc_product ) {
			return $wc_product->get_name();
		}, $wc_products ) );

		// Create draft variable product from $post_ids
		$merged_product = new WC_Product_Variable();
		$merged_product->set_status( 'draft' );
		$merged_product->set_name( $this->find_common_title( $post_ids ) );
		$merged_product->set_attributes( $attributes );
		$merged_product->save();

		/**
		 * Update product variation
		 *
		 * @var WC_Product $wc_product
		 */
		foreach ( $wc_products as &$wc_product ) {
			if ( $wc_product->get_type() === 'variation' ) {
				wp_insert_post( array(
					'ID'          => $wc_product->get_id(),
					'post_parent' => $merged_product->get_id(),
					'post_status' => 'publish',
				) );
				$wc_product = new WC_Product_Variation( $wc_product->get_id() );
				$wc_product->set_parent_id( $merged_product->get_id() );
				$wc_product->set_attributes( array( 'options' => $wc_product->get_name() ) );
				$wc_product->save();
			} else {
				$product_attributes = array( 'options' => $wc_product->get_name() );
				foreach ( $wc_product->get_attributes() as $slug => $attribute ) {
					if ( $attribute->is_taxonomy() ) {
						$product_attributes[ $slug ] = get_term_by( 'term_id', reset( $attribute->get_options() ), $attribute->get_taxonomy() )->slug;
					} else {
						$product_attributes[ $slug ] = reset( $attribute->get_options() );
					}
				}
				wp_insert_post( array(
					'ID'          => $wc_product->get_id(),
					'post_type'   => 'product_variation',
					'post_status' => 'publish',
					'post_parent' => $merged_product->get_id(),
				) );
				$wc_product = new WC_Product_Variation( $wc_product->get_id() );
				$wc_product->set_attributes( $product_attributes );
				$wc_product->save();
			}
		}

		// Delete obsolete variations
		foreach ( $obsolete_variations as $obsolete_variation ) {
			$obsolete_variation->set_children( array() );
			$obsolete_variation->set_visible_children( array() );
			$obsolete_variation->save();
			wp_delete_post( $obsolete_variation->get_id(), true );
		}

		return add_query_arg( array(
			'wc_merge_status' => 'success',
			'product_id'      => $merged_product->get_id(),
		), $redirect_url );
	}

	/**
	 * Display admin notices
	 *
	 * @return void
	 */
	public function admin_notices() {
		if ( empty( $status = $_GET['wc_merge_status'] ) ) {
			return;
		}
		if ( $status === 'success' && ! empty( $product_id = $_GET['product_id'] ) ) {
			?>
			<div id="message" class="notice notice-success is-dismissable">
				<p>
					<?= __( "Products merged successfully.", 'wc-merge' ) ?>
					<?= sprintf( __( '<a href="%s">Edit product</a>', 'wc-merge' ), get_edit_post_link( $product_id ) ) ?>
				</p>
			</div>
			<?php
		} else {
			?>
			<div id="message" class="notice notice-error is-dismissable">
				<p>
					<?= __( "Could not merge products.", 'wc-merge' ) ?>
				</p>
			</div>
			<?php
		}
	}

	/**
	 * Find common title from multiple products
	 *
	 * @param array $post_ids
	 *
	 * @return string
	 */
	protected function find_common_title( array $post_ids ) {
		$match       = "";
		$title_parts = array_map( function ( $post_id ) {
			return explode( ' ', get_the_title( $post_id ) );
		}, $post_ids );
		for ( $i = 0; $i < sizeof( $title_parts[0] ); $i ++ ) {
			if ( sizeof( $parts = array_column( $title_parts, $i ) ) !== sizeof( $title_parts ) ) {
				break;
			}
			if ( sizeof( array_unique( $parts ) ) === 1 ) {
				$match = trim( "${match} ${parts[0]}" );
			} else {
				break;
			}
		}

		return $match ?: __( "Merged product", 'wc-merge' );
	}

}

function WC_Merge() {
	return WC_Merge::get_instance();
}

WC_Merge();
