/**
 * @preserve
 * JS Scripts for Pick List for WooCommerce
 * by Consortia AS
 * @package PickList/js
 * @endpreserve
 */

(function($) {

	const url = new URL(window.location.href);
	const tab = url.searchParams.get('tab');
	
	// LocalStorage keys
	const storageKey = 'picklist-sorting';

	$(document).ready(function ( $ ) {
	
		// Check if Bootstrap's tooltip is not present or not properly initialized
		if (typeof $.fn.tooltip === 'undefined' || !$.fn.tooltip.Constructor || !$.fn.tooltip.Constructor.VERSION) {
			// Apply jQuery UI's tooltip if Bootstrap's tooltip is not loaded
			$(document).tooltip();
		}

		const body    = $('body');
		var dark_mode = cas_object.use_dark_mode && cas_object.use_dark_mode === 'yes' ? true : false;

		if (dark_mode) {
			body.addClass('dark-mode');
		} else {
			body.removeClass('dark-mode');
		}

		//Only on orders tab
		if ( null === tab || 'main' === tab || 'unpicked' === tab ) {
	
			$('.progressbar').progressbar({
				value: false,
				change: function() {
					progressLabel.text( progressbar.progressbar( "value" ) + "%" );
				},
				complete: function() {
					progressLabel.text( "100%" );
				}
			});
			
			$('.progressbar').each( function() {
				var currentValue  = parseInt( this.dataset.progress );
				var working       = this.dataset.working;
				var progressLabel = this.childNodes[0].childNodes[0];

				if ( working === 'working' && currentValue === 0 ) {

					$(this).progressbar({
						value: false
					});
					progressLabel.textContent = ''; 
					
				} else {

					$(this).progressbar({
						value: currentValue,
						change: function() {
							progressLabel.textContent = $(this).progressbar('value') + '%';
						},
						complete: function() {
							progressLabel.textContent = '100%';
						}
					});
				}
				
			});
	
			$('#cas-select-order-filter').multiselect({
				noneSelectedText: cas_object.select_status,
				selectedList: 4 // 0-based index
			});
	
			$('#cas-pick-priority-filter').multiselect({
				noneSelectedText: cas_object.select_priority,
				selectedList: 4
			});
	
			$('#cas-pick-progress-filter').multiselect({
				noneSelectedText: cas_object.select_progress,
				selectedList: 4
			});
	
			$('#cas-pick-users-filter').multiselect({
				noneSelectedText: cas_object.select_users,
				selectedList: 4
			});
	
			casFilterpick();
			casFilterProgress();
			casFilterUsers();
	
			//Assign user if change
			$('select[name="pick_assigned_user"]').on( 'change', casAssignListUser );
	
			//Refresh every minute(s)?
			var timer_show    = $('span[name="cas-timer-display"]').attr('data-timer-show');
			var timer_minutes = $('span[name="cas-timer-display"]').attr('data-timer-minutes');
			
			if ( timer_minutes ) {
				if ( 'yes' === timer_show ) {
					$('.cas-timer').toggle();
				}
				casReloadPage( timer_minutes, timer_show );
			}
			
			//Check updates and new orders
			casCheckPickList();

			// Use select2 for status change in all tabs
			$('select[name="cas-order-status-select"]').select2({
				placeholder: cas_object.select_default,
				allowClear: true,
				width: '100%'
				//dropdownParent: $('#cas-change-status-dialog')
			});

			$('#cas-pick-order-table tbody').on('click', 'tr', function(e) {
				// Avoid double trigger if the button was clicked
				if ($(e.target).is('a, i, input, select')) return;

				const url = $(this).data('url');
				if (url) {
					window.location.href = url;
				}
			});

			// Summarize table data
			summarizeTableData('#cas-pick-order-table');
			
	
		}
	
		//Pick Order tab
		if ( 'single' === tab ) {
	
			// Barcode Input Handling
			$('#cas-barcode-input').on('keypress', function(e) {
				if ( e.which === 13 ) {
					var barcode = $(this).val();
					var orderId = $(this).data('order_id');
					if ( barcode.length > 0 ) {
						casHandleBarcodeScan(barcode,orderId);
						$(this).val('');
					}
				}
			});

			// Barcode lost focus
			$('#cas-barcode-input').on('focusout', function (e) {
				$('#cas-barcode-focus-btn').css('background', 'initial');
				$('#cas-barcode-icon').hide();
			});

			$('#cas-barcode-focus-btn').on('click', function(e) {
				casBarcodeFocus();
			});
		
			// Barcode input has focus initially
			$('#cas-barcode-focus-btn').css('background','greenyellow');

			//If assigned user changes
			$('select[name="pick_assigned_user"]').on('change', casAssignUser );
	
			//Check if we use barcode
			var use_barcode = $('#cas-pick-table').attr('data-use-barcode');
			if ( '1' === use_barcode ) {
	
				var settings = {
					barHeight: 40,
					fontSize: 11
				}
	
				$('[name="cas-barcode"]').each(function(){
					var id       = '#' + this.id;
					var barValue = this.dataset.barcodeValue;
					$(id).barcode(
						barValue, 
						cas_object.barcode_type,
						settings
					);
				});
				
			}

			//Bulk pick qty
			$('#cas-bulk-pick-chk').on('change', function(){
				var value = this.checked ? 'yes' : 'no';
				$('.cas-barcode[name="cas-barcode"]').attr('data-scan-all', value);
				if (this.checked) {
					$('input[name="pick-qty"]').hide();
				} else {
					$('input[name="pick-qty"]').show();
				} 
			});

			// Initialize the dialog
			var casDialogStatus = $('#cas-change-status-dialog').dialog({
				autoOpen: false,
				modal: false,
    			height: 'auto',
				buttons: {
					"Save": function() {
						// Get the selected status
						var selectedStatus = $('#cas-order-status-select').val();
						var order_id       = this.dataset.order_id;
						var nonce_val      = $( '#_wpnonce' ).val();
						
						var json_data = { 
							action: 'pick_change_order_status', 
							nonce: nonce_val, 
							orderId: order_id, 
							status: selectedStatus 
						};
					
						$.ajax(
							{
								type: 'POST',
								url: ajaxurl,
								datatype: 'json',
								data: json_data,
								success: function( response ) {
									//console.log( response );
									$('#cas-current-order-status').text(response.status_name).removeClass().addClass('order-status status-'+response.status);
									$('#cas-order-status-changed').show();
								},
								error: function( response ){
									console.log( response );
								}
							}
						);
						
						// For now, just close the dialog
						casDialogStatus.dialog("close");
					},
					"Cancel": function() {
						casDialogStatus.dialog("close");
					}
				}
			});
		
			// Position the dialog to the right of the icon and open it on icon click
			$('#cas-change-status-icon').on('click', function() {
				
				if (window.innerWidth <= 768) {
					// Mobile / small screen: center it
					casDialogStatus.dialog("option", "position", {
						my: "center",
						at: "center",
						of: window
					});
				} else {
					// Desktop: next to the icon
					casDialogStatus.dialog("option", "position", {
						my: "left top",
						at: "right+10 top",
						of: this
					});
				}

				casDialogStatus.dialog("open");
			});

			// Open the slider when the button is clicked
			$("#cas-open-slider-btn, #cas-open-slider-btn-none").on('click', function() {
				$("#cas-next-order-slider").css("right", "0");
			});
		
			// Close the slider when the close button is clicked
			$("#cas-close-slider, cas-open-slider-btn-none").on('click', function() {
				$("#cas-next-order-slider").css("right", "-100%");
			});

			$('.cas-slider-order-item').on('click', function() {
				
				var nextHref = $(this).find('a').attr('href');
    
				// Check if a valid href was found, then navigate to the address
				if (nextHref) {
					window.location.href = nextHref;
				} 
			});

			
			// Use select2
			$('.cas-pick-user-select').select2({
                placeholder: cas_object.select_default,
                allowClear: true,
				width: '97%'
            });

			// Use select2 for status change
			$('select[name="cas-order-status-select"]').select2({
				placeholder: cas_object.select_default,
				allowClear: true,
				width: '100%'
				//dropdownParent: $('#cas-change-status-dialog')
			});

			 // Initialize Select2 for sorting
			 $('#picklist-sort').select2({
				placeholder: cas_object.sort_default,
				allowClear: true,
				multiple: true,
				width: '100%'
			});
			
			// Load saved sorting preference from LocalStorage
			let savedSorting = localStorage.getItem(storageKey);
			if (savedSorting) {
				savedSorting = JSON.parse(savedSorting);
				$('#picklist-sort').val(savedSorting).trigger('change'); // Restore saved sorting order
			} else {
				$('#picklist-sort').val(null).trigger('change'); // Clear Select2 selection
			}

			// Store the default row order when the page loads (WooCommerce default sorting)
			let defaultRows = $('#cas-picking-table tbody tr').get();

			// Reset sorting when all sorting criteria are removed
			$('#picklist-sort').on('select2:unselect', function (e) {
				// Reset sorting to WooCommerce default
				if ( $('#picklist-sort').val().length === 0 ) {
					$('#cas-picking-table tbody').empty().append(defaultRows); // Restore original order
					localStorage.removeItem(storageKey); // Clear stored sorting preferences
					var priceIndex  = $('#cas-picking-table th[data-index="price"]').index();
					var stockIndex  = $('#cas-picking-table th[data-index="stock"]').index();
					var weightIndex = $('#cas-picking-table th[data-index="weight"]').index();
					$( '#p' + priceIndex ).removeClass( 'fa-sort-down' ).removeClass( 'fa-sort-up' ).addClass( 'fa-sort' );
					$( '#p' + stockIndex ).removeClass( 'fa-sort-down' ).removeClass( 'fa-sort-up' ).addClass( 'fa-sort' );
					$( '#p' + weightIndex ).removeClass( 'fa-sort-down' ).removeClass( 'fa-sort-up' ).addClass( 'fa-sort' );
				}
			});
		
			// Trigger sorting when select changes
			$('#picklist-sort').on('change', function() {
				sortPickList();
			});
			
	
			// Handle show 3D box dimensions
			$(document).on('click', '#cas-picking-table .show-box', function () {
				const $cell = $(this).closest('td');
				let dimText = ($cell.data('dimensions') || '').toString().trim();

				if (!dimText || dimText.toUpperCase() === 'N/A') {
					alert('No dimensions available.');
					return;
				}

				// Parse "L x W x H" and strip units like "cm", "mm"
				// Accept 'x' or '×', optional spaces, optional units after each number
				const parts = dimText.split(/\s*[×x]\s*/).map(s => {
					const m = s.match(/([\d.,]+)/);
					return m ? parseFloat(m[1].replace(',', '.')) : NaN;
				});

				if (parts.length !== 3 || parts.some(isNaN)) return;

				const [length, width, height] = parts;

				// px-per-unit scale
				const scale = 3;
				const L = length * scale;
				const W = width  * scale;
				const H = height * scale;


				// Isometric angles
				const COS30 = Math.cos(Math.PI / 6); // ~0.866
				const SIN30 = Math.sin(Math.PI / 6); // 0.5

				// Foreshortening: reduce apparent depth without changing angles
				const depthF = 0.65; // try 0.6–0.8 to taste

				// Receding offsets (use W = width*scale)
				const dx = W * COS30 * depthF;
				const dy = W * SIN30 * depthF;

				// Start point (front-bottom-left of the front face)
				const startX = 10;
				const startY = 120;

				// Compute extents for viewBox
				const minX = startX;
				const minY = Math.min(startY - H - dy, startY - H); // top-most point
				const maxX = startX + L + dx;
				const maxY = startY; // bottom edge of front
				const vbX = Math.floor(minX - 4);
				const vbY = Math.floor(minY - 4);
				const vbW = Math.ceil((maxX - minX) + 8);
				const vbH = Math.ceil((maxY - minY) + 8);

				const svg = `
					<svg class="dimension-box" viewBox="${vbX} ${vbY} ${vbW} ${vbH}">
					<style>
						.box-front{fill:#cfe4ff;stroke:#4a7abf}
						.box-top{fill:#e8f2ff;stroke:#4a7abf}
						.box-side{fill:#d7e9ff;stroke:#4a7abf}
						.box-line{stroke:#4a7abf;stroke-dasharray:4 3;fill:none}
						.dimension-box *{vector-effect:non-scaling-stroke;stroke-width:1}
					</style>

					<!-- Front face -->
					<rect class="box-front"
						x="${startX}" y="${startY - H}"
						width="${L}" height="${H}" />

					<!-- Top face -->
					<polygon class="box-top" points="
						${startX},${startY - H}
						${startX + dx},${startY - H - dy}
						${startX + L + dx},${startY - H - dy}
						${startX + L},${startY - H}
					" />

					<!-- Side face -->
					<polygon class="box-side" points="
						${startX + L},${startY - H}
						${startX + L + dx},${startY - H - dy}
						${startX + L + dx},${startY - dy}
						${startX + L},${startY}
					" />

					<!-- Optional guide -->
					<line class="box-line"
						x1="${startX}" y1="${startY}"
						x2="${startX + dx}" y2="${startY - dy}" />
					</svg>
				`;

				const $existing = $cell.find('.box-svg');
					if ($existing.length) {
						$existing.toggle();
					} else {
						$cell.append('<div class="box-svg">' + svg + '</div>');
					}
				});


			// Shipping Label: Show/hide custom input for priority selection
			$('#cas-shipping-priority').on('change', function () {
				if ($(this).val() === 'custom') {
					$('#cas-custom-priority').show();
				} else {
					$('#cas-custom-priority').hide();
				}
			});

			// Shipping Label: Show/hide barcode input when checkbox is checked
			$('#cas-include-barcode').on('change', function () {
				if ($(this).is(':checked')) {
					$('#cas-barcode-value').show();
					$('#cas-barcode-type').show();
				} else {
					$('#cas-barcode-value').hide();
					$('#cas-barcode-type').hide();
				}
			});
			  
		}

		// Customers tab
		if ( 'customers' === tab ) {
			$('.progressbar').progressbar({
				value: false,
				change: function() {
					progressLabel.text( progressbar.progressbar( "value" ) + "%" );
				},
				complete: function() {
					progressLabel.text( "100%" );
				}
			});
			
			$('.progressbar').each( function() {
				var currentValue  = parseInt( this.dataset.progress );
				var working       = this.dataset.working;
				var progressLabel = this.childNodes[0].childNodes[0];

				if ( working === 'working' && currentValue === 0 ) {

					$(this).progressbar({
						value: false
					});
					progressLabel.textContent = ''; 
					
				} else {

					$(this).progressbar({
						value: currentValue,
						change: function() {
							progressLabel.textContent = $(this).progressbar('value') + '%';
						},
						complete: function() {
							progressLabel.textContent = '100%';
						}
					});
				}
				
			});

			$('#cas_customer_email_select').select2({
				allowClear: true,
				multiple: false,
				placeholder: cas_object.select_customer,
				width: 'resolve'
			});

			$('#cas_customer_email_select').on('select2:clear', function (e) {
				const $select = $(this);
				$select.val(null).trigger('change');
			});

			//Assign user if change
			$('select[name="pick_assigned_user"]').on( 'change', casAssignListUser );

			$('#cas-pick-customer-table tbody').on('click', 'tr', function(e) {
				// Avoid double trigger if the button was clicked
				if ($(e.target).is('a, i, input, select')) return;

				const url = $(this).data('url');
				if (url) {
					window.location.href = url;
				}
			});

			//Summarize table data
			summarizeTableData('#cas-pick-customer-table');
		}

	});

	// Open the stock quantity dialog, edit and save new quantity
	window.casOpenStockQuantityDialog = function(productId, currentQty, triggerElement = null) {
		$('#cas-stock-qty-product-id').val(productId);
		$('#cas-stock-qty-value').val(currentQty);

		$('#cas-stock-qty-dialog').dialog({
			modal: false,
			 position: {
				my: 'left top',
				at: 'right top',
				of: triggerElement || window, // Fallback to center if no trigger
				collision: 'fit fit'
			},
			buttons: {
				"Save": function () {
					const pid     = $('#cas-stock-qty-product-id').val();
					const qty     = $('#cas-stock-qty-value').val();
					var nonce_val = $( '#_wpnonce' ).val();

					$.ajax({
						type: 'POST',
						url: ajaxurl,
						data: {
							action: 'pick_update_stock_quantity',
							product_id: pid,
							quantity: qty,
							nonce: nonce_val
						},
						success: function (response) {
							if (response.success) {
								//alert('Stock updated');
								$('#stock-qty-product-'+response.id).text(response.quantity); // Update the displayed stock quantity
							} else {
								alert(response.data || 'Failed to update stock');
							}
						}
					});

					$(this).dialog("close");
				},
				"Cancel": function () {
					$(this).dialog("close");
				}
			}
		});
	}


	// Toggle the visibility of the 3D boxes
	window.casToggleBoxes = function() {
		
		$('#cas-picking-table tbody tr').each(function() {
			$(this).find('.show-box').click();
		});

		if ($('#cas-eye-toggle-boxes-icon').hasClass('fa-eye')) {
			$('#cas-eye-toggle-boxes-icon').removeClass('fa-eye').addClass('fa-eye-slash');
		} else {
			$('#cas-eye-toggle-boxes-icon').removeClass('fa-eye-slash').addClass('fa-eye');
		}
	}

	function escapeRegex(string) {
		return string.replace(/[.*+?^${}()|[\]\\]/g, '\\$&');
	}

	window.summarizeTableData = function(tableSelector) {
        const { currency_symbol, currency_pos, decimal_sep, thousand_sep, decimals } = cas_object;

		const $table = $(tableSelector);
		const $thead = $table.find('thead');
		const $tbody = $table.find('tbody');
		const $tfoot = $table.find('tfoot');

		const thMap = {
			'cas-th-items': null,
			'cas-th-total': null,
			'cas-th-shipping': null
		};

		// Determine column indexes
		$thead.find('th').each(function(idx) {
			const id = $(this).attr('id');
			if (id in thMap) thMap[id] = idx;
		});

		const totals = { 'cas-th-items': 0, 'cas-th-total': 0, 'cas-th-shipping': 0 };
		let rowCount = 0;

		// Sum each column
		$tbody.find('tr:visible').each(function() {
			const $cells = $(this).find('td');
			Object.entries(thMap).forEach(([thId, idx]) => {
				if (idx === null) return;

				let raw = $cells.eq(idx).text().trim();

				// Remove currency symbol and spaces
				raw = raw.replace(currency_symbol, '').trim();

				// Remove thousand separators, replace decimal
				const regexThousand = new RegExp(escapeRegex(thousand_sep), 'g');
				raw = raw.replace(regexThousand, '');
				if (decimal_sep !== '.') {
					const regexDecimal = new RegExp(escapeRegex(decimal_sep), 'g');
					raw = raw.replace(regexDecimal, '.');
				}

				const num = parseFloat(raw) || 0;
				totals[thId] += num;
			});
			rowCount++;
		});

		// Format output
		const $tfootRow = $tfoot.find('tr').first();
		Object.entries(thMap).forEach(([thId, idx]) => {
			if (idx === null) return;

			let output = '';
			if (thId === 'cas-th-items') {
				output = totals[thId].toFixed(0);
			} else {
				let numStr = totals[thId].toFixed(decimals);
				if (decimal_sep !== '.') {
					numStr = numStr.replace(/\./g, decimal_sep);
				}
				// Reinstate thousand separators
				const parts = numStr.split(decimal_sep);
				parts[0]    = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thousand_sep);
				numStr      = parts.join(decimal_sep);

				output = (currency_pos === 'left') ? `${currency_symbol}${numStr}` : `${numStr}${currency_symbol}`;
			}

			$tfootRow.find('td').eq(idx).text(output);
		});

		
		// Update external summary (if present)
		const average = rowCount > 0 ? totals['cas-th-total'] / rowCount : 0;
		const avship = rowCount > 0 ? totals['cas-th-shipping'] / rowCount : 0;

		// Helper
		function formatCurrency(val) {
			let str = val.toFixed(decimals).replace('.', decimal_sep);
			const parts = str.split(decimal_sep);
			parts[0] = parts[0].replace(/\B(?=(\d{3})+(?!\d))/g, thousand_sep);
			str = parts.join(decimal_sep);
			return (currency_pos === 'left') ? currency_symbol + str : str + currency_symbol;
		}

		$('#total-items').text(totals['cas-th-items'].toFixed(0));
		$('#total-value').text(formatCurrency(totals['cas-th-total']));
		$('#average-value').text(formatCurrency(average));
		$('#order-rows').text(rowCount);
		$('#shipping-value').text(formatCurrency(totals['cas-th-shipping']));
		$('#shipping-average-value').text(formatCurrency(avship));
    }


    // Sorting function
    window.sortPickList = function() {
        let sortBy = $('#picklist-sort').val(); // Get selected sorting criteria (array)
        let rows   = $('#cas-picking-table tbody tr').get(); // Get all table rows

        if (!sortBy || sortBy.length === 0) {
            return; // If no sorting criteria, do nothing
        }

        rows.sort(function(a, b) {
            for (let i = 0; i < sortBy.length; i++) {
                let key    = sortBy[i];
                let aValue = $(a).data(`item-${key}`).toString().toLowerCase();
                let bValue = $(b).data(`item-${key}`).toString().toLowerCase();

                if (aValue < bValue) return -1; // Ascending order
                if (aValue > bValue) return 1;
            }
            return 0; // If equal, move to the next sorting key
        });

        // Append sorted rows back to the table
        $.each(rows, function(index, row) {
            $('#cas-picking-table tbody').append(row);
        });

		// Save sorting preference to LocalStorage
        localStorage.setItem(storageKey, JSON.stringify(sortBy));
    }

	//Change order status in tables
	function casChangeOrderStatus(obj, orderId) {

		var casDialogStatus = $('#cas-change-status-dialog').dialog({
			autoOpen: false,
			modal: false,
			height: 'auto',
			buttons: {
				"Save": function() {
					// Get the selected status
					var selectedStatus = $('#cas-order-status-select').val();
					var order_id       = orderId;
					var nonce_val      = $( '#_wpnonce' ).val();
					var json_data      = { action: 'pick_change_order_status', nonce: nonce_val, orderId: order_id, status: selectedStatus };
				
					$.ajax(
						{
							type: 'POST',
							url: ajaxurl,
							datatype: 'json',
							data: json_data,
							success: function( response ) {
								//console.log( response );
								$('#cas-current-order-status-'+orderId).text(response.status_name).removeClass().addClass('order-status status-'+response.status);
							},
							error: function( response ){
								console.log( response );
							}
						}
					);
					
					// For now, just close the dialog
					casDialogStatus.dialog("close");
				},
				"Cancel": function() {
					casDialogStatus.dialog("close");
				}
			}
		});

		
		if (window.innerWidth <= 768) {
			// Mobile / small screen: center it
			casDialogStatus.dialog("option", "position", {
				my: "center+40 top+20",
				at: "center center",
				of: obj,
				collision: 'fit fit'
			});
		} else {
			// Desktop: next to the trigger, but don't go off-screen
			casDialogStatus.dialog('option', 'position', {
				my: 'left top',
				at: 'right+10 top',
				of: obj,
				collision: 'flipfit'
			});
		}

		casDialogStatus.dialog("open");

	}
	window.casChangeOrderStatus = function(obj,orderId) {
		casChangeOrderStatus(obj,orderId);
	}

	function casGetPostItNote(id) {
		const $textarea = $('#'+id);
		$textarea.toggle();
	}
	window.casGetPostItNote = function(id) {
		casGetPostItNote(id);
	}

	function casSavePostItNote(order,item) {
		const $textarea   = $('#cas-postit-'+item);
		const txt_comment = $textarea.val();
		var nonce_val     = $( '#_wpnonce' ).val();
		var json_data     = { action: 'pick_update_pick_orderitem_comment_data', nonce: nonce_val, orderId: order, itemId: item, comment: txt_comment };
	
		if (txt_comment.length > 0){
			$.ajax(
				{
					type: 'POST',
					url: ajaxurl,
					datatype: 'json',
					data: json_data,
					success: function( response ) {
						//console.log( response );
					},
					error: function( response ){
						console.log( response );
					}
				}
			);
		}

	}
	window.casSavePostItNote = function(order,item) {
		casSavePostItNote(order,item);
	}

	function casToggleAllComments() {
		var isHidden = $('#cas-eye-toggle-icon').hasClass('fa-eye');
		if (isHidden) {
			$('#cas-eye-toggle-icon').removeClass('fa-eye').addClass('fa-eye-slash');
			$('#cas-pick-table .cas-post-it-note').show();
		} else {
			$('#cas-eye-toggle-icon').removeClass('fa-eye-slash').addClass('fa-eye');
			$('#cas-pick-table .cas-post-it-note').hide();
		}
	}
	window.casToggleAllComments = casToggleAllComments;
	
	function casGetTotals_OLD() {
	
		var itemCount   = 0;
		var totCount    = 0;
		var rowCount    = 0;
		var totshipping = 0;
	
		$('#cas-pick-order-table tbody tr:visible').each( function() {
			
			// Get data-items attribute and text content
			var itemText  = $(this).find('td[data-items]').text();
			var totalText = $(this).find('td[data-total]').text();
    		var shipping  = $(this).find('td[data-shipping-cost]').data('shipping-cost');

			if (shipping) {
				totshipping += parseFloat( shipping );
			}
		
			// Increment itemCount using the cell's text content
			if (itemText) {
				itemCount += parseInt(itemText); // Parse the visible number in the cell
			}
		
			// Increment totCount by parsing the total value
			if (totalText) {
				totCount += parseFloat(totalText.replace(/[^0-9.,]+/, '')); // Remove non-numeric characters
			}
		
			rowCount++; // Count the visible rows
		});
		
		var side    = $('#total-value').data('currency-pos');
		var symbol  = $('#total-value').data('currency-symbol');
		var average = (rowCount > 0) ? totCount / rowCount : 0;
		var avship  = (rowCount > 0 ) ? totshipping / rowCount : 0;
	
		if ( 'left' == side || 'left_space' === side ) {
			totCount    = symbol + '' + totCount.toFixed(2);
			average     = symbol + '' + average.toFixed(2);
			avship      = symbol + '' + avship.toFixed(2);
			totshipping = symbol + '' + totshipping.toFixed(2);
		} else {
			totCount    = totCount.toFixed(2) + '' + symbol;
			average     = average.toFixed(2) + '' + symbol;
			avship      = avship.toFixed(2) + '' + symbol;
			totshipping = totshipping.toFixed(2) + '' + symbol;
		}
		
		$('#total-items').text( itemCount );
		$('#total-value').text( totCount );
		$('#average-value').text( average );
		$('#order-rows').text( rowCount );
		$('#shipping-value').text( totshipping );
		$('#shipping-average-value').text( avship );
	}
	window.casGetTotals_OLD = casGetTotals_OLD;

	
	function casReloadPage( msec, showTimer ) {
	
		let countdownDuration = msec * 60; // Countdown duration in seconds
		  let remainingTime     = countdownDuration;
		var timer_message     = jQuery('span[name="cas-timer-display"]').attr('data-timer-message');
	
		function formatTime(seconds) {
			const minutes          = Math.floor(seconds / 60);
			const remainingSeconds = seconds % 60;
			const formattedTime    = `${minutes}:${remainingSeconds < 10 ? '0' : ''}${remainingSeconds}`;
			return formattedTime;
		}
	
		function updateProgressBar() {
			const percentage = ((countdownDuration - remainingTime) / countdownDuration) * 100;
			$('span[name="cas-timer-display"]').text( formatTime(remainingTime) + ' ' + timer_message );
		}
	
		function startCountdown() {
			const interval = setInterval(function() {
			remainingTime--;
			if ( 'yes' === showTimer ) {
				updateProgressBar();
			}
	
			if (remainingTime <= 0) {
				clearInterval(interval);
				location.reload();
			}
			}, 1000); // Update every 1 second
		}
	
		startCountdown();
	
	}
	window.casReloadPage = function( msec, showTimer) {
		casReloadPage(msec,showTimer);
	}

	function casCopyCustomerName() {
		// Create a temporary input element
		var $temp = $("<input>");
		$("body").append($temp);

		// Set the value of the input to the customer name
		$temp.val($('#cas-customer').text().trim()).select();

		// Copy the text inside the input
		document.execCommand("copy");

		// Remove the temporary input element
		$temp.remove();
		
		// Optionally, provide feedback to the user
		$('#cas-customer-name-copied').show();
	}
	window.casCopyCustomerName = casCopyCustomerName;

	function casAddressToggle(address){
		var tglClass = '.cas-shipping-tgl';
		var current  = '#tgl-shipping';
		if('billing' === address){
			tglClass = '.cas-billing-tgl';
			current  = '#tgl-billing';
		}

		var $icon = $(current).find('i');
		if ($icon.hasClass('fa-plus')) {
            $icon.removeClass('fa-plus').addClass('fa-minus');
        } else {
            $icon.removeClass('fa-minus').addClass('fa-plus');
        }

		$(tglClass).toggle();

	}
	window.casAddressToggle = function(address){
		casAddressToggle(address);
	}

	//Get the totals in the top of the tabs (orders/pending)
	function casGetOrderTotals( tab ) {

		var nonce_val = $( '#_wpnonce' ).val();
		var json_data = { action: 'pick_get_order_tabs_totals', nonce: nonce_val, query: tab };
		
		if ('orders' === tab){
			$('#cas-get-order-totals').find('i').addClass('cas-blink');
		} else {
			$('#cas-get-pending-totals').find('i').addClass('cas-blink');
		}
	
		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
					casTotalsSuccess(response);
				},
				error: function( response ){
					console.log( response );
				}
			}
		);

		function casTotalsSuccess(response){
			if ('orders' === tab){
				$('#cas-get-order-totals').find('i').removeClass('cas-blink');
			} else {
				$('#cas-get-pending-totals').find('i').removeClass('cas-blink');
			}

			var background = 'yes' === cas_object.use_dark_mode ? 'black' : '#f8dda7';
			
			$('#total-items').text( response.items ).css('background',background);
			$('#total-value').html( response.total ).css('background',background);
			$('#average-value').html( response.average ).css('background',background);
			$('#order-rows').text( response.orders ).css('background',background);
			$('#shipping-value').html( response.shipping ).css('background',background);
			$('#shipping-average-value').html( response.average_shipping ).css('background',background);
			
		}

	}
	window.casGetOrderTotals = function(tab) {
		casGetOrderTotals(tab);
	}
	
	function casFilterpick( visible = '' ) {
		
		var thePriorities = [];
		$('#cas-pick-priority-filter option:selected').each(function(){
			thePriorities.push( this.value );
		});
	
		//Filter Priority
		$(visible+'select[name="row-priority"]').each(function() {
			if ( thePriorities.length === 0 ) {
				$(this).closest('tr').show();
			} else {
				if ( thePriorities.includes( this.value ) ) {
					$(this).closest('tr').show();
				} else {
					$(this).closest('tr').hide();
				}
			}
	
		});
	
		//Check other filters
		var progress_arr = casProgressFilterOn( visible );
		var user_arr     = casUsersFilterOn( visible );
	
		//Update paging with filters
		casUpdatePagingUrls( thePriorities, progress_arr, user_arr );
	
		//Totals
		summarizeTableData('#cas-pick-order-table');
	}
	window.casFilterpick = function( visible = '' ){
		casFilterpick(visible);
	}
	
	function casFilterProgress( visible = '' ) {
	
		var progress_arr = [];
		$('#cas-pick-progress-filter option:selected').each(function(){
			progress_arr.push( this.value );
		});
	
		$(visible+'.progressbar[name="cas-progress"]').each(function() {
			if ( progress_arr.length === 0 ) {
				$(this).closest('tr').show();
			} else {
				var current_progress = parseInt( this.dataset.progress );
				var the_progress     = '';
				if ( 0 === current_progress ) {
					the_progress = 'null';
				} else if ( 100 === current_progress ) {
					the_progress = 'done';
				} else {
					the_progress = 'ongoing';
				}
	
				if ( progress_arr.includes( the_progress ) ) {
					$(this).closest('tr').show();
				} else {
					$(this).closest('tr').hide();
				}
				
			}
	
		});
	
		//Check other filters
		var thePriorities = casPriorityFilterOn( visible ); 
		var user_arr      = casUsersFilterOn( visible );
	
		//Update paging with filters
		casUpdatePagingUrls( thePriorities, progress_arr, user_arr );
	
		//Totals
		summarizeTableData('#cas-pick-order-table');
	}
	window.casFilterProgress = function(visible = '') {
		casFilterProgress(visible);
	}
	
	function casFilterUsers( visible = '' ) {
		var users_arr = [];
		$('#cas-pick-users-filter option:selected').each(function(){
			users_arr.push( this.value );
		});
	
		//Filter users
		$(visible+'select[name="pick_assigned_user"]').each(function() {
			if ( users_arr.length === 0 ) {
				$(this).closest('tr').show();
			} else {
				if ( users_arr.includes( this.value ) ) {
					$(this).closest('tr').show();
				} else {
					$(this).closest('tr').hide();
				}
			}
		});
	
		//Check other filters
		var progress_arr = casProgressFilterOn( visible );
		var priority_arr = casPriorityFilterOn( visible );
		
		//Update paging with filters
		casUpdatePagingUrls( priority_arr, progress_arr, users_arr );
	
		//Totals
		summarizeTableData('#cas-pick-order-table');
	}
	window.casFilterUsers = function(visible = ''){
		casFilterUsers(visible);
	}
	
	//Check if progress filter is on - then update list
	function casProgressFilterOn( visible = '' ) {
		
		var progress_arr = [];
		$('#cas-pick-progress-filter option:selected').each(function(){
			progress_arr.push( this.value );
		});
	
		if ( visible.length === 0 && progress_arr.length > 0 ) {
			casFilterProgress( 'tr:visible ');
		}
	
		return progress_arr;
	
	}
	window.casProgressFilterOn = function(visible=''){
		casProgressFilterOn(visible);
	}
	
	//Check if the priority filter is on - then update list
	function casPriorityFilterOn( visible = '' ) {
		
		var thePriorities = [];
		$('#cas-pick-priority-filter option:selected').each(function(){
			thePriorities.push( this.value );
		});
	
		if ( visible.length === 0 && thePriorities.length > 0 ) {
			casFilterpick( 'tr:visible ' );
		}
	
		return thePriorities;
	}
	window.casPriorityFilterOn = function(visible='') {
		casPriorityFilterOn(visible);
	}
	
	//Check if the users filter is on - then update list
	function casUsersFilterOn( visible = '' ) {
		
		var user_arr = [];
		$('#cas-pick-users-filter option:selected').each(function(){
			user_arr.push( this.value );
		});
	
		if ( visible.length === 0 && user_arr.length > 0 ) {
			casFilterUsers( 'tr:visible ' );
		}
	
		return user_arr;
	}
	window.casUsersFilterOn = function(visible='') {
		casUsersFilterOn(visible);
	}
	
	function casUpdatePagingUrls( priorities_arr, progress_arr, users_arr ) {
		//Update URLs
		$('.cas-pagination').find('a').each(function(){
			var url     = $(this).attr('href');
			var cleared = url.replace( /\&priority\=.*/i, '' ).replace( /\&progress\=.*/i, '' ).replace( /\&users\=.*/i, '' );
			$(this).attr( 'href', cleared + '&priority=' + priorities_arr.toString() + '&progress=' + progress_arr.toString() + '&users=' + users_arr.toString());
		});
	}
	window.casUpdatePagingUrls = function( priorities_arr, progress_arr, users_arr ) {
		casUpdatePagingUrls(priorities_arr,progress_arr,users_arr);
	}
	
	function casResetFilters() {
		
		//Order statuses
		var status_found = false;
		$('#cas-select-order-filter option:selected').each( function() {
			this.value   = '';
			status_found = true;
		});
		$('#cas-selected-filters').val('');
		
		//Priority
		$('#cas-pick-priority-filter').multiselect('uncheckAll');
		$('#cas-pick-priority-filter').multiselect('refresh');
		$('#cas-selected-priorities').val('');
	
		//Progress
		$('#cas-pick-progress-filter').multiselect('uncheckAll');
		$('#cas-pick-progress-filter').multiselect('refresh');
		$('#cas-selected-progress').val('');
	
		//Users
		$('#cas-pick-users-filter').multiselect('uncheckAll');
		$('#cas-pick-users-filter').multiselect('refresh');
		$('#cas-selected-users').val('');
	
		//Loop filters to original
		casFilterpick();
	
		//Submit status if used
		if (status_found) {
			$('#cas-order-filter-form')[0].submit();
		}
	
	}
	window.casResetFilters = casResetFilters;
	
	//Filter on change
	function casFilterByStatus() {
		
		var items = ''; 
		$('#cas-select-order-filter option:selected').each( function() {
			items += this.value + ',';
		});
		$('#cas-selected-filters').val(items.slice(0,-1));
	
	}
	window.casFilterByStatus = casFilterByStatus;

	//Filter on change
	function casFilterByPriority() {
		
		var items = ''; 
		$('#cas-pick-priority-filter option:selected').each( function() {
			items += this.value + ',';
		});
		$('#cas-selected-priorities').val(items.slice(0,-1));
	}
	window.casFilterByPriority = casFilterByPriority;
	
	//Filter on change
	function casFilterByProgress() {
		var items = ''; 
		$('#cas-pick-progress-filter option:selected').each( function() {
			items += this.value + ',';
		});
		$('#cas-selected-progress').val(items.slice(0,-1));
	}
	window.casFilterByProgress = casFilterByProgress;
	
	//Filter on change
	function casFilterByUser() {
		var items = ''; 
		$('#cas-pick-users-filter option:selected').each( function() {
			items += this.value + ',';
		});
		$('#cas-selected-users').val(items.slice(0,-1));
	}
	window.casFilterByUser = casFilterByUser;
	
	function casDoOrderFilter( obj ) {
	
		event.preventDefault();
		var form = $( '#' + obj.id )[0];
		form['order-filter'].remove();
		form.submit();
	
	}
	window.casDoOrderFilter = function(obj){
		casDoOrderFilter(obj);
	}
	
	function casDoPriorityFilter() {
		casFilterpick();
	}
	window.casDoPriorityFilter = casDoPriorityFilter;
	
	function casToggleGrid() {
		$( '#cas-pick-grid' ).toggle( 'blind' );
		$( '#cas-pick-table' ).toggle( 'blind' );
	}	
	window.casToggleGrid = casToggleGrid;

	function casToggleOrderInfo() {
		$('.cas-single-info').toggle();
		$('.cas-single-grid-table').toggleClass('cas-horizontal-scroll'); // Toggle class for horizontal scroll
	}
	window.casToggleOrderInfo = casToggleOrderInfo;

	function casToggleBarcodeInput(){

		var $element = $('#cas-barcode-input');

		if ($element.is(':visible')) {
			$element.hide(); // Hide input for scanner use
			$element.val(''); // Clear value if necessary
		} else {
			$element.show(); // Show input for manual entry
		}

	}
	window.casToggleBarcodeInput = casToggleBarcodeInput;
	
	function cas_progress(id, value) {
	
		var progressbar = jQuery('#'+id);
	
		progressbar.progressbar( 'value', value + 2 );
	
		if ( val < 99 ) {
			setTimeout( cas_progress, 80 );
		}
	}
	window.cas_progress = function(id,value){
		cas_progress(id,value);
	}
	
	function casChangepickPriority( obj, order_id ) {
	
		var selected  = $(obj).find('option:selected').val();
		var nonce_val = $( '#_wpnonce' ).val();
		var json_data = { action: 'pick_set_pick_order_priority_data', nonce: nonce_val, orderid: order_id, priority: selected };
		
		switch(selected){
			case 'high':
				$('#'+obj.id).next('i').css('color','red');
				break;
			case 'low':
				$('#'+obj.id).next('i').css('color','cornflowerblue');
				break;
			case 'medium':
				$('#'+obj.id).next('i').css('color','orange');
				break;
			case 'on_hold':
				$('#'+obj.id).next('i').css('color','black');
				break;
		}
	
		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
				},
				error: function( response ){
					console.log( response );
				}
			}
		);
	
	}
	window.casChangepickPriority = function(obj,order_id) {
		casChangepickPriority(obj,order_id);
	}
	
	// Function to detect if the user is on a mobile device
	function casIsMobileDevice() {
		return /Mobi|Android/i.test(navigator.userAgent);
	}
	window.casIsMobileDevice = casIsMobileDevice;

	//Set focus on barcode input
	function casBarcodeFocus() {

		// Provide visual feedback for focus
		$('#cas-barcode-focus-btn').css('background', 'greenyellow');
		$('#cas-barcode-icon').show();
	
		var input = $('#cas-barcode-input');

		// Ensure no conflicting event handlers
		input.off('focus');
	
		if (casIsMobileDevice()) {
			// Apply readonly for mobile devices to prevent keyboard from showing
			input.attr('readonly', true);
		} else {
			// Ensure readonly is not applied for desktop users
			input.attr('readonly', false);
		}
	
		// Brief timeout to ensure readonly is applied before focus
		setTimeout(function () {
			// Using native focus with preventScroll
			input[0].focus({ preventScroll: true });

			// Trigger jQuery event handling as needed
			input.trigger('focus');

			input.addClass('highlight');
			
			// Highlight removal
			setTimeout(function () {
				input.removeClass('highlight');
			}, 200);
	
			// Remove readonly if needed after focus is set
			input.attr('readonly', false);
	
		}, 50); // Small delay to handle timing issues, especially on mobile devices
	}
	window.casBarcodeFocus = casBarcodeFocus;
	
	//Use scanner to validate and pick items
	function casHandleBarcodeScan( barcode, orderId ) {
		
		var nonce_val = $( '#_wpnonce' ).val();
		$('#cas-barcode-notice-error').hide();
		$('#cas-barcode-notice-item-picked').hide();
		$('#pick_error_barcode').text('');
		$('#pick_error_picked').text('');
		
		var pick_info        = $('#cas-pick-table tbody tr .cas-barcode');
		var barcode_exist    = false;
		var barcodeItemId    = ''; // Order item id
		var scan_all         = '';
		var pick_qty         = 1;
		var bulk_pick        = $('#cas-bulk-pick-chk').prop('checked') ? 'yes' : 'no';
		var barcode_picked   = false;
		var total_qty_picked = 0; // Track the total number of picked items
		var max_qty          = 0; // Maximum quantity of this item

		if ( barcode.length > 0 ) {
			// Check if barcode exists in pick_info array
			pick_info.each( function() {
				
				var barcodeValue = this.dataset.barcodeValue; 
				var pick_done    = ('1' === this.dataset.pickDone) ? true : false;
				var barcodeQty   = this.dataset.barcodeQty ? parseInt(this.dataset.barcodeQty) : 1; // Get total quantity for the product in the order

				// Compare barcodes and check if the item is not yet picked fully
				if ( barcodeValue.toString() === barcode.toString() ) {
					max_qty = barcodeQty;

					if ( ! pick_done && total_qty_picked < barcodeQty ) {
						barcodeItemId = this.dataset.barcodeItemid;
						scan_all      = bulk_pick;
						barcode_exist = true;
						product_id    = this.dataset.prodId;
						pick_qty      = (scan_all === 'no') ? $('#pick-qty-' + barcodeItemId).val() : barcodeQty;

						total_qty_picked += parseInt(pick_qty); // Track total picked quantities
						barcode_picked    = false;

						if (max_qty === total_qty_picked) {
							this.dataset.pickDone = '1';
						}

						//Stop loop and pick this item
						return false;
					} else {
						barcode_picked = true; // If fully picked
					}
				}
			});
		}


		if ( ! barcode_exist || barcode_picked ) {
			if ( barcode_picked ) {
				//Picked
				$('#pick_error_picked').text(barcode);
				$('#cas-barcode-notice-item-picked').show();
			} else {
				//Not found
				$('#pick_error_barcode').text(barcode);
				$('#cas-barcode-notice-error').show();
			}
			casPlaySound('cas-error-sound');
			//casBarcodeFocus();
		} else {

			$.ajax({
				url: ajaxurl,
				type: 'POST',
				datatype: 'json',
				data: {
					action: 'pick_list_find_product_by_barcode_id',
					barcode: barcode,
					order: orderId,
					itemId: barcodeItemId,
					scanAll: scan_all,
					nonce: nonce_val,
					qty: pick_qty,
				},
				success: function(response) {
					if (response.success) {
						if (response.isPicked && !response.isDone) {
							
							$('#tbl-items-picked-'+response.itemId).text(response.item_qty + '/' + response.sum_qty);
							$('#print-items-picked-'+response.itemId).text(response.item_qty + '/' + response.sum_qty);
							$('#cas_picked_count').text(response.count_tot);
	
							if ( parseInt( response.item_qty ) === parseInt( response.sum_qty ) ) {
	
								$('#btn-pick-'+response.itemId).prop('disabled', true);
								$('#btn-pick-tbl-'+response.itemId).prop('disabled', true);
								$('#btn-unpick-'+response.itemId).prop('disabled', false);
								$('#btn-unpick-tbl-'+response.itemId).prop('disabled', false);
								$('#cas-assign-me-button').prop('disabled',true);
								$('#cas-pack-check-id-'+response.itemId).show();
								$('#table-img-'+response.itemId).prepend('<i class="fas fa-check-circle cas-icon"></i>');
								$('#grid-img-'+response.itemId).prepend('<i class="fas fa-check-circle cas-icon"></i>');
								
								$('#cas-barcode-'+response.itemId).attr('data-pick-done', '1').trigger('change');

							}

							$('#progress-order-'+response.orderId).val(response.percent).attr('title',response.percent.toFixed(2)+'%');

							if ( 100 === parseInt( response.percent ) ) {
								$('#cas-barcode-notice-order-picked').show();
							}
	
							casPlaySound('cas-success-sound');
	
						} else {
							if (response.isDone){
								//Already picked
								$('#cas-barcode-notice-item-picked').show();
							} else {
								//Not found
								$('#cas-barcode-notice-error').show();
							}
							casPlaySound('cas-error-sound');
						}	
					} 
				},
				error: function(response) {
					console.log(response);
					casPlaySound('cas-error-sound');
				}
			});
		}
		//Set focus back to scanner input
		casBarcodeFocus();

	}
	window.casHandleBarcodeScan = function(barcode, orderid){
		casHandleBarcodeScan(barcode,orderid);
	}
	
	function casPlaySound(soundId) {
		if ( 'yes' === cas_object.scan_sounds_off ) {
			return;
		}
		/*
		var sound = document.getElementById(soundId);
		if (sound) {
			sound.play().catch(e => console.error("Audio play failed:", e));
		}*/
		if ( 'cas-success-sound' === soundId ) {
			playPingSound();
		} else {
			playErrorSound();
		}
	}
	window.casPlaySound = function(soundId){
		casPlaySound(soundId);
	}

	function playPingSound() {
		var audioCtx = new (window.AudioContext || window.webkitAudioContext)();
		var oscillator = audioCtx.createOscillator();
		oscillator.type = 'square';//'sine'; // Pure tone
		oscillator.frequency.setValueAtTime(1000, audioCtx.currentTime); // Frequency in hertz
		oscillator.connect(audioCtx.destination);
		oscillator.start();
		oscillator.stop(audioCtx.currentTime + 0.2); // Play for 200ms
	}
	
	function playErrorSound() {
		var audioCtx = new (window.AudioContext || window.webkitAudioContext)();
		var oscillator = audioCtx.createOscillator();
		oscillator.type = 'square'; // Harsher tone
		oscillator.frequency.setValueAtTime(200, audioCtx.currentTime); // Lower frequency
		oscillator.connect(audioCtx.destination);
		oscillator.start();
		oscillator.stop(audioCtx.currentTime + 0.5); // Play for 500ms
	}
	
	//Update picklist
	function caspickItem( itemId, orderId, qty, orderItemId ) {
	
		var nonce_val = $( '#_wpnonce' ).val();
		var bulk_pick = $('#cas-bulk-pick-chk').prop('checked') ? 'yes' : 'no';
		var pick_qty  = ( 'no' === bulk_pick ) ? $('#pick-qty-'+orderItemId).val() : qty;
		var json_data = { action: 'pick_update_pick_orderitem_data', nonce: nonce_val, orderid: orderId, itemid: itemId, quantity: pick_qty, add: 1, orderItemId: orderItemId, bulkPick: bulk_pick };
	
		$('.cas-barcode-notice').hide();

		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
					casDisableItem(response);
				},
				error: function( response ){
					console.log( response );
				}
			}
		);
	
	}
	window.caspickItem = function(itemId,orderId,qty,orderItemId){
		caspickItem(itemId,orderId,qty,orderItemId);
	}

	//Unpick item
	function casUnPickItem( itemId, orderId, qty, orderItemId ) {
		
		var nonce_val = $( '#_wpnonce' ).val();
		var json_data = { action: 'pick_update_pick_orderitem_data', nonce: nonce_val, orderid: orderId, itemid: itemId, quantity: qty, add: 0, orderItemId: orderItemId };
	
		$('.cas-barcode-notice').hide();

		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
					casDisableItem(response);
				},
				error: function( response ){
					console.log( response );
				}
			}
		);
	}
	window.casUnPickItem = function(itemId,orderId,qty,orderItemId){
		casUnPickItem(itemId,orderId,qty,orderItemId);
	}
	
	function casAssignMe( orderId ) {
	
		var nonce_val = $( '#_wpnonce' ).val();
		var json_data = { action: 'pick_assign_user_to_pick', nonce: nonce_val, orderid: orderId, user: '' };
	
		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
					var items = response;
					jQuery('select[name="pick_assigned_user"]').val(items['key']);
					jQuery('#cas-assign-me-button').prop('disabled',true);
				},
				error: function( response ){
					console.log( response );
				}
			}
		);
	}
	window.casAssignMe = function(orderId) {
		casAssignMe(orderId);
	}
	
	function casAssignUser() {
	
		var nonce_val = $( '#_wpnonce' ).val();
		var selected  = $('select[name="pick_assigned_user"] :selected').val();
		var order_id  = $('select[name="pick_assigned_user"]').attr('data-order-id');
		var json_data = { action: 'pick_assign_user_to_pick', nonce: nonce_val, orderid: order_id, user: selected };
	
		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
					var items = response;
					$('select[name="pick_assigned_user"]').val(items['key']);
					$('#cas-assign-me-button').prop('disabled',true);
					$('#cas-assigned-changed').show();
				},
				error: function( response ){
					console.log( response );
				}
			}
		);
	}
	window.casAssignUser = casAssignUser;
	
	function casAssignListUser() {
	
		var nonce_val = $( '#_wpnonce' ).val();
		var selected  = $(this).find(':selected').val();
		var order_id  = $(this).attr('data-order-id');
		var json_data = { action: 'pick_assign_user_to_pick', nonce: nonce_val, orderid: order_id, user: selected };
	
		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
					var items = response;
					jQuery('select[data-order-id="'+order_id+']"').val(items['key']);
				},
				error: function( response ){
					console.log( response );
				}
			}
		);
	}
	window.casAssignListUser = casAssignListUser;
	
	function casCheckPickList() {
	
		var nonce_val = jQuery( '#_wpnonce' ).val();
		var json_data = { action: 'pick_list_check_changes', nonce: nonce_val };
		
		jQuery.ajax({
			url: ajaxurl,
			datatype: 'json',
			data: json_data,
			success: function(response) {
				var data = response;
				if ( 'yes' === cas_object.auto_refresh && 'true' === data['pickChanged'] || 'true' === data['newOrder'] ) {
					location.reload();
				} else {			
					if ( 'yes' === cas_object.notify_changes && 'true' === data['pickChanged'] ) {
						jQuery('.cas-notice1').toggle();
					}
					if ( 'yes' === cas_object.notify_orders && 'true' === data['newOrder'] ) {
						jQuery('.cas-notice2').toggle();
					}
				}
			},
			error: function( response ) {
				//console.log(response);
				location.reload(); //Nonce has timed out
			},
			complete: function() {
				// Schedule the next request after a delay
				setTimeout(casCheckPickList, 20000); // Repeat every 20 seconds
			}
		});
	}
	window.casCheckPickList = casCheckPickList;
	
	function casBulkPick( orderId ) {
	
		var nonce_val = $( '#_wpnonce' ).val();
		var json_data = { action: 'pick_bulk_update_orderitem_data', nonce: nonce_val, orderid: orderId };
	
		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
					location.reload();
				},
				error: function( response ){
					console.log( response );
				}
			}
		);
	}
	window.casBulkPick = function(orderId){
		casBulkPick(orderId);
	}
	
	// Pick order shipped
	window.casPickShipped = function( order_id ) {
		
		var nonce_val = $( '#_wpnonce' ).val();
		var json_data = { action: 'pick_set_order_status_shipped', nonce: nonce_val, orderid: order_id };
		var percent   = $('#progress-order-'+order_id).val();

		// Set status shipped 
		if ( parseInt(percent) === 100 || confirm(cas_object.ship_message) ) {
			$.ajax({
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function(response) {
					location.reload();
				},
				error: function(error) {
					console.log('Order status update failed:', error);
				}
			});
		}
	}

	//Print picklist, picking slip and shipping labels
	window.casPrintList = function(type) {
		// Remove any previously set print classes
		$('body').removeClass('cas-printing-pick cas-printing-slip cas-printing-label');

		//QR Code
		var order_id    = $('div[name="cas-qr-code"]').attr('data-order_id');
		var order_url   = $('div[name="cas-qr-code"]').attr('data-order_url');
		var account_url = $('div[name="cas-qr-code"]').attr('data-account_url');
	
		$('#cas-qr-code-table-'+order_id).empty();
	
		if (type === 'pick') {
			
			$('body').addClass('cas-printing-pick');

			$('#cas-print-table-'+order_id+' .cas-qty-several').css({'background-color':'lightgreen'});
			$('#cas-print-table-'+order_id+' .cas-var-bg').css({'background-color':'lightgreen'});
			$('.cas-type-of-print').show();
			$('input[name="cas-print-input-check"]').show();
			$('.cas-print-checked').hide();
			$('#cas-packing-message-'+order_id).hide();
			$('.cas-barcode-print').show();
			$('.cas-print-items-picked').hide();
			$('.cas-custom-fields-print').show();
	
			//pick list QR code
			casMakeQr( order_id, order_url );

		} else if (type === 'pack') {
			
			$('body').addClass('cas-printing-slip');

			$('#cas-print-table-'+order_id+' .cas-qty-several').css('background-color','white');
			$('#cas-print-table-'+order_id+' .cas-var-bg').css({'background-color':'white'});
			$('.cas-type-of-print').hide();
			$('input[name="cas-print-input-check"]').hide();
			$('.cas-print-checked').show();
			$('.cas-print-items-picked').show();
			if ( $('#cas-check-slip-note-'+order_id +' :checked') ) {
				var message  = $('#cas-order-slip-note-'+order_id).val();
				var formated = message.replace(/\n/g, '<br>');
				$('#cas-packing-message-'+order_id).html(formated).show();
			}
			$('.cas-barcode-print').hide();
			$('.cas-custom-fields-print').hide();
			
			//packing slip QR code
			casMakeQr( order_id, account_url );

		} else if (type === 'shipping_label') {
			
			openShippingLabelDialog(order_id);
			return;
	
		}
	
		window.print();
	}

	// Generate the shipping label
	window.generateShippingLabel = function(order_id) {
		closeShippingLabelDialog();

		$('#cas-shipping-spinner').show(); 

		let priority = $('#cas-shipping-priority').val();
		if (priority === 'custom') {
			priority = $('#cas-custom-priority').val();
		}

		let weight       = $('#cas-shipping-weight').val();
		let barcode      = $('#cas-include-barcode').is(':checked') ? $('#cas-barcode-value').val() : '';
		let barcode_type = $('#cas-include-barcode').is(':checked') ? $('#cas-barcode-type').val() : '';
		let order_url    = $('div[name="cas-qr-code"]').attr('data-order_url');
		let order_note   = $('#cas-shipping-note').val();
		let is_return    = $('#cas-is-return').is(':checked') ? 'yes' : '';

		let selectedIcons = [];
		if ($('#cas-icon-fragile').is(':checked')) selectedIcons.push('fragile');
		if ($('#cas-icon-keep-dry').is(':checked')) selectedIcons.push('keep-dry');
		if ($('#cas-icon-this-side-up').is(':checked')) selectedIcons.push('this-side-up');
		if ($('#cas-icon-snow').is(':checked')) selectedIcons.push('snow');

		// Set the print size dynamically before the AJAX request
		let printSize = cas_object.ship_label_size === '4x6' ? "4in 6in" : ''; // Default, update based on settings
		if (cas_object.ship_label_size === '4x8') printSize = "4in 8in";
		if (cas_object.ship_label_size === '6x8') printSize = "6in 8in";
		if (cas_object.ship_label_size === 'A6') printSize = "105mm 148mm";
		if (cas_object.ship_label_size === 'A5') printSize = "148mm 210mm";

		// Fetch shipping label dynamically before printing
		$.get(ajaxurl, { 
			action: 'cas_generate_shipping_label', 
			order_id: order_id,
			priority: priority,
			weight: weight,
			barcode: barcode,
			barcode_type: barcode_type,
			order_note: order_note,
			is_return: is_return,
			icons: JSON.stringify(selectedIcons)
		})
		.done(function(response) {
			if (response.success) {

				let labelContainer = $('#shipping-label-content-' + order_id);

				// Inject the label into the visible area
				labelContainer.html(response.data.html);

				// Remove old print style block (if any)
				$('#cas-print-size-style').remove();

				// Create new style block for the label print area
				let style = document.createElement('style');
				style.id  = "cas-print-size-style";
				style.innerHTML = `
					@page { size: ${printSize}; margin: 0 !important; }
					body {
						width: ${printSize.split(' ')[0]};
						height: ${printSize.split(' ')[1]};
						margin: 0 !important;
						padding: 0 !important;
						box-sizing: border-box;
						overflow: hidden;
					}
					* {
						box-sizing: border-box;
					}
				`;
				document.head.appendChild(style);

				labelContainer.show(); // Ensure it's visible

				if ('yes' === cas_object.use_ship_order_qr) {
					casMakeShippingLabelQr(order_id, order_url);
				}

				// Wait a short delay for rendering, then trigger print
				setTimeout(function () {
					printShippingLabel(order_id, printSize, is_return);
				}, 500);
			} else {
				alert('Failed to generate shipping label.');
			}
		})
		.fail(function(error) {
			console.log('Error fetching shipping label:', error);
			$('#cas-shipping-spinner').hide(); 
		});

		return; // Stop execution here to wait for AJAX response before printing
	}
	
	function printShippingLabel(order_id, printSize, is_return) {
		let labelContainer = $('#shipping-label-content-' + order_id);
	
		if (labelContainer.length === 0) {
			alert('Shipping label not found.');
			return;
		}
	
		const element = labelContainer.get(0);
	
		// Extract print size
		let dimensions = printSize.split(' ');
		let widthStr   = dimensions[0];
		let heightStr  = dimensions[1];
	
		let unit   = widthStr.includes('in') ? 'in' : 'mm';
		let width  = parseFloat(widthStr);
		let height = parseFloat(heightStr);
	
		// Convert everything to mm for jsPDF consistency
		const inchToMm = 25.4;
		let pdfWidth   = unit === 'in' ? width * inchToMm : width;
		let pdfHeight  = unit === 'in' ? height * inchToMm : height;
	
		const { jsPDF } = window.jspdf;
		let pdf = new jsPDF({
			orientation: 'portrait',
			unit: 'mm',
			format: [pdfWidth, pdfHeight]
		});
	
		generateShippingBarcode(order_id, function () {
			setTimeout(() => {
				html2canvas(element, {
					scale: 3,   // High-res
					useCORS: true,
					backgroundColor: null
				}).then(canvas => {
					const imgData = canvas.toDataURL('image/png');
	
					// Center the image or fit exactly to PDF
					pdf.addImage(
						imgData,
						'PNG',
						0, 0, // x, y
						pdfWidth, // force full width
						pdfHeight // force full height
					);
	
					const filename = is_return === 'yes'
						? `picklist-return-label-order-${order_id}.pdf`
						: `picklist-shipping-label-order-${order_id}.pdf`;
	
					pdf.save(filename);

					// Cleanup after printing
					setTimeout(() => {
						labelContainer.html('').hide();

						// Remove the print styling from the head
						$('#cas-print-size-style').remove();

						// Restore normal scrolling and layout
						$('body').css({
							width: '',
							height: '',
							overflow: '',
							boxSizing: ''
						});

						// Hide spinner
						$('#cas-shipping-spinner').hide();

					}, 500);

				}).catch(error => {
					console.error('html2canvas error:', error);
					// Hide spinner
					$('#cas-shipping-spinner').hide();

					alert('Could not generate label PDF.');
				});
			}, 700);
		});
	}
	
	function generateShippingBarcode(order_id, callback) {
		
		let barcodeElement = $('#track-' + order_id);
		let barcodeValue   = barcodeElement.attr('data-barcode-value');
		let barcodeFormat  = barcodeElement.attr('data-barcode-format');
	
		if (!barcodeValue || !barcodeFormat) {
			callback(); // Proceed without barcode if missing
			return;
		}
	
		if (barcodeFormat === 'qr') {
			new QRCode(barcodeElement[0], {
				text: barcodeValue,
				width: 80,
				height: 80,
				colorDark: "#000",
				colorLight: "#fff",
				correctLevel: QRCode.CorrectLevel.M
			});
		} else if (barcodeFormat === 'code128' || barcodeFormat === 'code39') {
			barcodeElement.barcode(barcodeValue, barcodeFormat, { barHeight: 40, fontSize: 11, color: 'black' });
		}
	
		// Ensure barcode renders before proceeding
		setTimeout(() => {
			callback(); // Now generate the PDF
		}, 500);
	}
	
	function casMakeShippingLabelQr( order_id, url ) {
		
		new QRCode(document.getElementById("shipping-label-qr-code-"+order_id), {
			text: url+order_id,
			width: 60,
			height: 60,
			colorDark : "#000",
			colorLight : "#fff",
			correctLevel : QRCode.CorrectLevel.M
		});

	}

	window.openShippingLabelDialog = function(order_id) {

		let weight = 0;

		$('#cas-picking-table tbody td').each(function(){
			let itemWeight = parseFloat($(this).attr('data-item-weight')) || 0; // Convert to float, default to 0 if null
			weight += itemWeight;
		});

		$('#cas-shipping-weight').val(weight);
		$('#cas-shipping-label-dialog').show();
	}
	
	window.closeShippingLabelDialog = function() {
		$('#cas-shipping-label-dialog').hide();
	}

	
	function casResetpick( order_id, message ) {
	
		if ( confirm( message ) ) {
	
			var nonce_val = $( '#_wpnonce' ).val();
			var json_data = { action: 'pick_reset_pick_orderitem_data', nonce: nonce_val, orderid: order_id };
			
			jQuery.ajax(
				{
					type: 'POST',
					url: ajaxurl,
					datatype: 'json',
					data: json_data,
					success: function( response ) {
						//console.log( response );
						location.reload();
					},
					error: function( response ){
						console.log( response );
					}
				}
			);
		}
	
	}
	window.casResetpick = function(order_id,message){
		casResetpick(order_id,message);
	}
	
	function casAddOrderNote( textbox_id, order_id ) {
		
		var text      = $(textbox_id).val();
		var nonce_val = $( '#_wpnonce' ).val();
		var json_data = { action: 'pick_add_pick_order_note', nonce: nonce_val, orderid: order_id, message: text };
		
		if ( '' === text ) {
			return 0;
		}
	
		jQuery.ajax(
			{
				type: 'POST',
				url: ajaxurl,
				datatype: 'json',
				data: json_data,
				success: function( response ) {
					//console.log( response );
					location.reload();
				},
				error: function( response ){
					console.log( response );
				}
			}
		);
	
		
	
	}
	window.casAddOrderNote = function(textbox_id,order_id){
		casAddOrderNote(textbox_id,order_id);
	}
	
	function casDisableItem( obj ) {
	
		var items      = obj;
		var picked_txt = items['tot_pick'] + '/' + items['sum_qty'];
	
		$('#tbl-items-picked-'+items['item-pick']).text(picked_txt);
		$('#grid-items-picked-'+items['item-pick']).text(picked_txt);
		$('#print-items-picked-'+items['item-pick']).text(picked_txt);

		if (items['add']) {
			if ( parseInt( items['sum_qty'] ) === parseInt( items['tot_pick'] ) ) {

				$('#btn-pick-'+items['item-pick']).prop('disabled', true);
				$('#btn-pick-tbl-'+items['item-pick']).prop('disabled', true);
				$('#btn-unpick-'+items['item-pick']).prop('disabled', false);
				$('#btn-unpick-tbl-'+items['item-pick']).prop('disabled', false);
				$('#cas-assign-me-button').prop('disabled',true);
				$('#cas-pack-check-id-'+items['item-pick']).show();
				$('#table-img-'+items['item-pick']).prepend('<i class="fas fa-check-circle cas-icon"></i>');
				$('#grid-img-'+items['item-pick']).prepend('<i class="fas fa-check-circle cas-icon"></i>');

				$('#cas-barcode-'+items['item-pick']).attr('data-pick-done', 1).trigger('change');

				//Add style to image
				$('#table-img-'+items['item-pick'] + ' img').addClass('cas-img-pick-done');
			}
		} else {
			$('#btn-pick-'+items['item-pick']).prop('disabled', false);
			$('#btn-pick-tbl-'+items['item-pick']).prop('disabled', false);
			$('#cas-pack-check-id-'+items['item-pick']).hide();
			$('#table-img-'+items['item-pick']+' i').remove();
			$('#grid-img-'+items['item-pick']+' i').remove();
			$('#cas-barcode-'+items['item-pick']).attr('data-pick-done', '').trigger('change');
		
			//Remove style to image
			$('#table-img-'+items['item-pick'] + ' img').removeClass('cas-img-pick-done');
		}
	
		$('#progress-order-'+items['orderid']).val(parseInt(items['percent'])).attr('title',parseInt(items['percent'])+'%');;
		$('select[name="pick_assigned_user"]').val(items['user']);
		$('#cas_picked_count').text(items['count_tot']);
		$('#cas-current-order-status').text(items['status']).removeClass().addClass('order-status status-'+items['status'].toLowerCase());

		if ( 100 === parseInt( items['percent'] ) ) {
			$('#cas-barcode-notice-order-picked').show();
		}

		if ( $('#cas-barcode-input').length ) {
			casBarcodeFocus();
		}

		/*
		if ( 100.00 === parseFloat( items['percent'] ) ) {
			$('#cas-barcode-notice-order-picked').show();
		}*/
		
	
	}
	window.casDisableItem = function(obj){
		casDisableItem(obj);
	}
	
	function casSortCol( column, type, prefix, table ) {
	
		casDoSort(column, type, '#'+table);
		
		if ( $( '#' + prefix + column ).hasClass( 'fa-sort' ) ) {
			$( '#' + prefix + column ).removeClass( 'fa-sort' ).addClass( 'fa-sort-up' );
			resetSortClass( column, prefix );
			return;
		}
		if ( $( '#' + prefix + column ).hasClass( 'fa-sort-up' ) ) {
			$( '#' + prefix + column ).removeClass( 'fa-sort-up' ).addClass( 'fa-sort-down' );
			resetSortClass( column, prefix );
			return;
		}
		if ( $( '#' + prefix + column ).hasClass( 'fa-sort-down' ) ) {
			$( '#' + prefix + column ).removeClass( 'fa-sort-down' ).addClass( 'fa-sort-up' );
			resetSortClass( column, prefix );
			return;
		}
	
		function resetSortClass( column, prefix ){
			for ( var i = 1; i < 13; i++ ) { 
				if ( i != column ) {
					$( '#' + prefix + i ).removeClass( 'fa-sort-down' ).removeClass( 'fa-sort-up' ).addClass( 'fa-sort' );
				}
			}
		}
	
	}
	window.casSortCol = function( column, type, prefix, table ){
		casSortCol( column, type, prefix, table );
	}
	
	function casDoSort( column, type, table ){
	
		var switching = true;
		var shouldSwitch;
		var switchcount = 0;
		var i;
		var dir = 'asc';
		var rows;
		var table = $(table +' > tbody')[0];
	

		while ( switching ) {
	
			switching = false;
			rows = table.rows;
	
			for ( i = 0; i < (rows.length - 1); i++ ) {
			
				shouldSwitch = false;
				
				var x = rows[i].cells[column];
				var y = rows[i + 1].cells[column];
	
				var node_type = ( undefined !== x.children[0] ) ? x.children[0].nodeName : '';
				switch(type) {
					case 'number':
						if( node_type.toLowerCase() == 'input' ){
							x = (x.childNodes[1].value == '' ) ? 0 : parseFloat(x.childNodes[1].value.replace(/[^0-9.,]+/, ''));
							y = (y.childNodes[1].value == '' ) ? 0 : parseFloat(y.childNodes[1].value.replace(/[^0-9.,]+/, ''));
						} else {
							x = (undefined !== x.innerText)
								? x.innerText.replace(/^[^0-9.,]+|[^0-9.,]+$/g, '')
								: '';
							y = (undefined !== y.innerText)
								? y.innerText.replace(/^[^0-9.,]+|[^0-9.,]+$/g, '')
								: '';
						}
						break;
					case 'progress':
						if( node_type.toLowerCase() == 'div' ){
							x = (x.childNodes[1].ariaValueNow == '' ) ? 0 : parseFloat(x.childNodes[1].ariaValueNow.replace(/[^0-9.,]+/, ''));
							y = (y.childNodes[1].ariaValueNow == '' ) ? 0 : parseFloat(y.childNodes[1].ariaValueNow.replace(/[^0-9.,]+/, ''));
						}
						break;
					case 'text':
						if( node_type.toLowerCase() == 'input' || node_type.toLowerCase() == 'select' ) {
							x = x.childNodes[1].value.replace(/^\s+|\s+$/g, '');
							y = y.childNodes[1].value.replace(/^\s+|\s+$/g, '');
						} else {
							x = ( undefined !== x.innerText ) ? x.innerText.replace( /^\s+|\s+$/g, '' ) : '';
							y = ( undefined !== y.innerText ) ? y.innerText.replace( /^\s+|\s+$/g, '' ) : '';
						}
						break;
					case 'date':
						if( node_type.toLowerCase() == 'input' ){
							x = (x.childNodes[1].value == '' ) ? new Date(-25920000000) : new Date(x.childNodes[1].value);
							y = (y.childNodes[1].value == '' ) ? new Date(-25920000000) : new Date(y.childNodes[1].value);
						} else {
							x = (x.innerText == '' ) ? new Date( -25920000000 ) : new Date( x.innerText );
							y = (y.innerText == '' ) ? new Date( -25920000000 ) : new Date( y.innerText );
						}
						break;
					case 'percent':
						if( node_type.toLowerCase() == 'input' ){
							x = (x.childNodes[0].value == '' ) ? 0 : parseFloat(x.childNodes[0].value.replace(/%\s/g, ''));
							y = (y.childNodes[0].value == '' ) ? 0 : parseFloat(y.childNodes[0].value.replace(/%\s/g, ''));
						} else {
							x = x.innerText.replace(/%\s/g, '');
							y = y.innerText.replace(/%\s/g, '');
						}
						break;
				}
	
				if ( x === y ) {
					continue;
				}
	
				if ( dir === "asc" ) {
					if( 'number' === type || 'percent' === type || 'progress' === type ) {
						if( parseFloat( x ) > parseFloat( y ) ) {
							shouldSwitch = true;
							break;
						}
					} else {
						if ( x > y ) {
							shouldSwitch = true;
							break;
						}
					}
				} else if ( dir === "desc" ) {
					if( 'number' === type || 'percent' === type || 'progress' === type ) {
						if( parseFloat( x ) < parseFloat( y ) ) {
							shouldSwitch = true;
							break;
						}
					} else {
						if ( x < y ) {
							shouldSwitch = true;
							break;
						}
					}
				}
			}
	
			if ( shouldSwitch ) {
				rows[i].parentNode.insertBefore( rows[i + 1], rows[i] );
				switching = true;
				switchcount++;
			} else {
				if (switchcount === 0 && dir === "asc") {
					dir       = "desc";
					switching = true;
				}
			}
	
		}
	
	}
	window.casDoSort = function(column,type,table){
		casDoSort(column,type,table);
	}

	function casExportPickList( view ) {
	
		const dateTime = new Date();
		const day      = dateTime.getDate();
		const month    = dateTime.getMonth() + 1;
		const year     = dateTime.getFullYear();
		const hour     = dateTime.getHours();
		const minute   = dateTime.getMinutes();
		const postfix  = `${day}.${month}.${year}_${hour}.${minute}`;
	
		var t     = view;
		let table = document.getElementById(t);
	
		TableToExcel.convert(table, { 
			name: postfix +'_export.xlsx', 
			sheet: {
				name: 'Pick List',
			}
		});
	
	
	}
	window.casExportPickList = function(view){
		casExportPickList(view);
	}
	
	function casMakeQr( order_id, url ) {
		new QRCode(document.getElementById("cas-qr-code-table-"+order_id), {
			text: url+order_id,
			width: 150,
			height: 150,
			colorDark : "#000",
			colorLight : "#fff",
			correctLevel : QRCode.CorrectLevel.M
		});
	}
	window.casMakeQr = function(order_id, url){
		casMakeQr(order_id,url);
	}
	
	/*
	window.casOpenNextOrders = function(){
		var slider = $('#cas-order-slider');
		var test = slider;
        
		// Check current position and toggle
		if (slider.css('right') === '-100%') {
			slider.css('right', '0');
		} else {
			slider.css('right', '-100%');
		}
	}*/

})(jQuery);