<?php
/**
 * Pick List
 *
 * @package picklist/includes
 */

defined( 'ABSPATH' ) || exit;

const CAS_PICK_ORDER = 'Cas_Pick_Order';

/**
 * Class for pick order
 */
class Cas_Pick_Order {

	/**
	 * Get value for barcode
	 */
	public static function cas_get_barcode_data( $product ) {
		
		$options  = get_option( 'picklist_options' );
		$bar_data = ( ! empty( $options['cas_pick_product_c1'] ) ) ? $product->get_meta( $options['cas_pick_product_c1'] ) : '';
		
		if ( empty( $bar_data ) ) {
			$bar_field = ( ! empty( $options['cas_pick_product_c1_alt'] ) ) ? $options['cas_pick_product_c1_alt'] : '';
			switch ( $bar_field ) {
				case '':
					$bar_data = '';
					break;
				case 'id':
					$bar_data = ( $product->get_variation_id() > 0 ) ? $product->get_variation_id() : $product->get_product_id();
					break;
				case 'sku':
					$bar_data = $product->get_sku();
					break;
				case '_global_unique_id':
					$bar_data = get_post_meta( $product->get_id(), '_global_unique_id', true );
					break;
			}
		}
		return $bar_data;
	}

	/**
	 * Get shipping class name
	 */
	public static function cas_get_shipping_class_name( $product ) {
		$slug = $product->get_shipping_class();
		if ( $slug ) {
			$term = get_term_by( 'slug', $slug, 'product_shipping_class' );
			return ( $term && ! is_wp_error( $term ) ) ? $term->name : '';
		}
		return '';
	}


	/**
	 * Get regular product fields
	 * 
	 * @since vs 2.5.9
	 * @param object $product The product object.
	 */
	public static function cas_get_regular_product_fields( $product ) {
		$options     = get_option( 'picklist_options' );
		$weight_unit = get_option( 'woocommerce_weight_unit' );
		
		if ( isset( $options['cas_pick_regular_product_fields'] ) && is_array( $options['cas_pick_regular_product_fields'] ) ) {
			foreach ( $options['cas_pick_regular_product_fields'] as $field_slug ) {
				if ( ! empty( $field_slug ) ) {
					$label = '';

					switch ( $field_slug ) {
						case 'shipping_class':
							$label = __( 'Shipping Class', 'woo-picklist' );
							$value = self::cas_get_shipping_class_name( $product ); 
							break;
						case 'weight':
							$label = __( 'Weight', 'woo-picklist' );
							$value = $product->get_weight() ? $product->get_weight() . ' ' . esc_html( $weight_unit ) : '';
							break;
						case 'dimensions':
							$label      = __( 'Dimensions', 'woo-picklist' );
							$dimensions = $product->get_dimensions( false );

							if ( ! empty( $dimensions['length'] ) || ! empty( $dimensions['width'] ) || ! empty( $dimensions['height'] ) ) {
								$value = wc_format_dimensions( $dimensions );
							} else {
								$value = '';
							}
							break;
						case 'price':
							$label = __( 'Price', 'woo-picklist' );
							$value = $product->get_regular_price() >= 0 ? wc_price( $product->get_regular_price() ) : '';
							if ( $product->is_on_sale() ) {
								$value = sprintf(
									/* translators: %s: Regular price, %s: Sale price */
									'<s title="%s">%s</s> %s',
									esc_attr__( 'Regular price', 'woo-picklist' ) . ': ' . wp_strip_all_tags( wc_price( $product->get_regular_price() ) ),
									wc_price( $product->get_regular_price() ),
									wc_price( $product->get_sale_price() )
								);
							}
							break;
						case 'sale_price':
							$label = __( 'Sale Price', 'woo-picklist' );
							$value = $product->get_sale_price() > 0 ? '<s>' . wc_price( $product->get_regular_price() ) . '</s> ' . wc_price( $product->get_sale_price() ) : '';
							break;
						case 'description':
							$label = __( 'Description', 'woo-picklist' );
							$value = $product->get_description();
							break;
						case 'short_description':
							$label = __( 'Short Description', 'woo-picklist' );
							// If product is a variation, get parent
							if ( $product->is_type( 'variation' ) ) {
								$parent_id = $product->get_parent_id();
								if ( $parent_id ) {
									$parent_product = wc_get_product( $parent_id );
									$value          = $parent_product ? $parent_product->get_short_description() : '';
								} else {
									$value = '';
								}
							} else {
								$value = $product->get_short_description();
							}
							break;
						default:
							$label = __( 'Custom Field', 'woo-picklist' ) . ' ' . ucfirst( $field_slug );
							$value = get_post_meta( $product->get_id(), $field_slug, true );
							break;
					}
					if ( $value ) {
						?>
							<div class="cas-product-field">
								<span class="cas-product-field-label"><?php echo esc_html( $label ); ?>:</span>
								<span class="cas-product-field-value"><?php echo wp_kses( $value, Cas_Pick_Functions::cas_allowed_html_tags() ); ?></span>
							</div>
						<?php
					}
				}
			}
		}

	}


	/**
	 * Get all custom fields from settings
	 */
	public static function cas_pl_get_custom_fields_array() {

		$options       = get_option( 'picklist_options' );
		$custom_fields = array();

		for ( $i = 1; $i <= 5; $i++ ) {
			
			$custom_field       = new stdClass();
			$custom_field->name = isset( $options["cas_pick_custom_product_field_{$i}_lbl"] ) ? $options["cas_pick_custom_product_field_{$i}_lbl"] : '';
			$custom_field->slug = isset( $options["cas_pick_custom_product_field_{$i}"] ) ? $options["cas_pick_custom_product_field_{$i}"] : '';
			$custom_field->use  = isset( $options["cas_pick_custom_product_field_{$i}_use"] ) ? $options["cas_pick_custom_product_field_{$i}_use"] : '';

			if ( 'yes' === $custom_field->use ) {
				$custom_fields[] = $custom_field;
			}
		}

		/**
		 * Apply a filter to allow custom fields to be added dynamically
		 *  
		 * @since vs 2.4.38
		 * 
		 * @param array $custom_fields Return array of: name, slug and use(yes/no)
		 * @return array
		 */
		$custom_fields = apply_filters( 'cas_picklist_custom_product_fields', $custom_fields );

		// Ensure all custom fields are converted to stdClass
		$custom_fields = array_map( function( $field ) {
			
			if ( is_array( $field ) ) {
				$custom_field       = new stdClass();
				$custom_field->name = isset( $field['name'] ) ? $field['name'] : '';
				$custom_field->slug = isset( $field['slug'] ) ? $field['slug'] : '';
				$custom_field->use  = isset( $field['use'] ) ? $field['use'] : '';
				return $custom_field;
			}
	
			// If already stdClass, return as is
			return $field;

		}, $custom_fields );

		return $custom_fields;
	}

	/**
	 * Get readable stock status
	 */
	private static function cas_get_readable_stock_status( $status ) {
		switch ( $status ) {
			case 'instock':
				return __( 'In Stock', 'woocommerce' );
			case 'outofstock':
				return __( 'Out of stock', 'woocommerce' );
			case 'onbackorder':
				return __( 'On backorder', 'woocommerce' );
			default:
				return ucfirst( $status );
		}
	}

	/**
	 * Get page notices
	 */
	private static function cas_get_page_notices() {
		?>
		<div id="cas-barcode-notice-error" class="notice notice-error is-dismissible cas-barcode-notice" style="display:none;">
			<p>
			<?php echo esc_attr__( 'Product not found for this barcode.', 'woo-picklist' ); ?>
			<?php echo esc_attr__( 'Try again!', 'woo-picklist' ); ?>
			<?php echo esc_attr__( 'Barcode:', 'woo-picklist' ); ?> <span id="pick_error_barcode" style="font-weight:600;"></span>
			</p>
		</div>

		<div id="cas-barcode-notice-item-picked" class="notice notice-info is-dismissible cas-barcode-notice" style="display:none;">
			<p>
			<?php echo esc_attr__( 'Item is already picked.', 'woo-picklist' ); ?>
			<?php echo esc_attr__( 'Pick another item!', 'woo-picklist' ); ?>
			<?php echo esc_attr__( 'Barcode:', 'woo-picklist' ); ?> <span id="pick_error_picked" style="font-weight:600;"></span>
			</p>
		</div>

		<div id="cas-barcode-notice-order-picked" class="notice notice-info is-dismissible cas-barcode-notice" style="display:none;">
			<p>
			<?php echo esc_attr__( 'The order is 100% picked.', 'woo-picklist' ); ?>
			</p>
		</div>

		<div id="cas-customer-name-copied" class="notice notice-info is-dismissible cas-barcode-notice" style="display:none;">
			<p>
			<?php echo esc_attr__( 'Customer name copied', 'woo-picklist' ); ?>
			</p>
		</div>

		<div id="cas-assigned-changed" class="notice notice-info is-dismissible cas-barcode-notice" style="display:none;">
			<p>
			<?php echo esc_attr__( 'Assigned user has changed', 'woo-picklist' ); ?>
			</p>
		</div>

		<div id="cas-order-status-changed" class="notice notice-info is-dismissible cas-barcode-notice" style="display:none;">
			<p>
			<?php echo esc_attr__( 'Order status changed', 'woo-picklist' ); ?>
			</p>
		</div>

		<?php
	}

	/**
	 * Get change status dialog
	 */
	private static function cas_get_status_dialog( $order_id ) {
		?>
		<div id="cas-change-status-dialog" data-order_id="<?php echo esc_attr( $order_id ); ?>" title="<?php echo esc_attr__( 'Change Order Status', 'woo-picklist' ); ?>" style="display:none;">
			<form id="change-status-form">
				<label for="cas-order-status-select"><?php echo esc_html__( 'Select status:', 'woo-picklist' ); ?></label>
				<select name="cas-order-status-select" id="cas-order-status-select" style="width: 100%;">
					<?php
					$statuses = wc_get_order_statuses();
					asort( $statuses ); // Sort ASC
					foreach ( $statuses as $status_slug => $status_name ) {
						echo '<option value="' . esc_attr( $status_slug ) . '">' . esc_html( $status_name ) . '</option>';
					}
					?>
				</select>
			</form>
		</div>
		<?php
	}

	/**
	 * Get refunded information
	 */
	private static function cas_get_refunded_order_items( $order ) {
		
		$refunds   = $order->get_refunds(); // Returns array of refund objects
		$items_arr = array();

		if ( ! empty( $refunds ) ) {
			foreach ( $refunds as $refund ) {
				foreach ( $refund->get_items() as $item_id => $item ) {
					$items_arr[$item->get_meta('_refunded_item_id')] = $item->get_quantity();
				}
			}
		}

		return $items_arr;
	}

	/**
	 * Pick List single page
	 */
	public static function cas_tab_single() {

		$order_id            = ( ! empty( filter_input( 1, 'orderid', FILTER_VALIDATE_INT ) ) ) ? filter_input( 1, 'orderid', FILTER_VALIDATE_INT ) : '';
		$order               = ( ! empty( $order_id ) ) ? wc_get_order( $order_id ) : '';
		$date_format         = get_option( 'date_format' );
		$time_format         = get_option( 'time_format' );
		$weight_unit         = get_option( 'woocommerce_weight_unit' );
		$options             = get_option( 'picklist_options' );
		$url                 = admin_url( 'admin.php?page=picklist' );
		$user_url            = admin_url( 'user-edit.php?user_id=' );
		$url_customer_orders = admin_url( 'admin.php?page=picklist&tab=customers&customer=' );
		$account_url         = trailingslashit( get_permalink( wc_get_page_id( 'myaccount' ) ) ) . 'view-order/';
		$hide_categories     = ( ! empty( $options['cas_pick_order_hide_category'] ) && 'yes' === $options['cas_pick_order_hide_category'] ) ? true : false;
		$from_tab            = ( ! empty( filter_input( 1, 'tbl', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) ) ) ? '&tab=' . filter_input( 1, 'tbl', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) : '';
		$show_grid_view      = ( isset( $options['cas_include_grid_pick_view'] ) && 'yes' === $options['cas_include_grid_pick_view'] ) ? true : false;
		$custom_fields       = self::cas_pl_get_custom_fields_array();
		$hide_next_orders    = ( isset( $options['cas_hide_next_unpicked_btn'] ) && 'yes' === $options['cas_hide_next_unpicked_btn'] ) ? true : false;
		$hide_price_col      = ( isset( $options['cas_hide_th_price'] ) && 'yes' === $options['cas_hide_th_price'] ) ? true : false;
		$hide_stock_col	     = ( isset( $options['cas_hide_th_stock'] ) && 'yes' === $options['cas_hide_th_stock'] ) ? true : false;
		$hide_weight_col     = ( isset( $options['cas_hide_th_weight'] ) && 'yes' === $options['cas_hide_th_weight'] ) ? true : false;
		$show_dimensions_col = ( isset( $options['cas_show_th_dimensions'] ) && 'yes' === $options['cas_show_th_dimensions'] ) ? true : false;
		$dimension_unit	     = get_option( 'woocommerce_dimension_unit' );
		$is_dark_mode        = isset( $options['cas_use_dark_mode'] ) && 'yes' === $options['cas_use_dark_mode'];
		$ship_lbl_enabled    = isset( $options['cas_hide_shipping_label_button'] ) && 'yes' !== $options['cas_hide_shipping_label_button'];

		/**
		 * Filter whether to hide prices for the current user.
		 *
		 * This filter checks if the current user should have prices hidden
		 * based on the 'picklist_options' settings in the database.
		 * 
		 * @since vs 2.4.25
		 * 
		 * @param bool $hide Whether to hide prices for the user (default: false).
		 * @return bool
		 */
		$hide_prices = apply_filters( 'cas_pick_hide_prices_for_user', false );
		
		?>
		<?php if ( ! empty( $order ) ) : ?>
			<div class="cas-div-bulk-pick-btn">
				<div class="cas-bulk-pick-btn-top" title="<?php echo esc_attr__( 'Unchecked adds quantity selector for manual input (Only in table view)', 'woo-picklist' ); ?>">
					<input type="checkbox" id="cas-bulk-pick-chk" <?php echo esc_attr( 'yes' === $options['cas_pick_scan_all_qty'] ? 'checked' : '' ); ?>> <?php echo esc_attr__( 'Bulk Pick Quantity', 'woo-picklist' ); ?>
				</div>	
			</div>
		<?php endif; ?>


		<div class="cas-single-main-container">

		<?php self::cas_get_page_notices(); ?>
	
		<?php if ( empty( $order ) ) : ?>
			<div>
			<h2 class="cas-h2-no-order"><?php echo esc_attr__( 'No order selected. Please select from: Orders or Pending Pickup Orders', 'woo-picklist' ); ?></h2><br/>
			<a href="<?php echo esc_attr( $url . $from_tab ); ?>" class="button"><i class="fa-solid fa-circle-left"></i> <?php echo esc_attr__( 'Back', 'woo-picklist' ); ?></a>
			<?php if ( ! $hide_next_orders ) : ?>
				<!-- Open Slider -->
				<button class="button" id="cas-open-slider-btn-none" title="<?php echo esc_attr__( 'See the next 7 orders to pick.', 'woo-picklist' ); ?>"><?php echo esc_attr__( 'Next Unpicked Orders', 'woo-picklist' ); ?></button>
			<?php endif; ?>
			</div>
		<?php endif; ?>

		<?php 
		
		if ( ! empty( $order ) ) : 

			$customer_name   = ( null !== $order->get_billing_first_name() && null !== $order->get_billing_last_name() ) ? $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() : '';
			$pick_status     = $order->get_meta( '_cas_pick_order_status', true );
			$pick_percent    = $order->get_meta( '_cas_pick_order_status_percent', true );
			$pick_priority   = $order->get_meta( '_cas_pick_order_priority', true );
			$pick_admin_usr  = $order->get_meta( '_cas_pick_admin_user', true );
			$shipped_count   = ( ! empty( $order->get_meta( '_cas_pick_order_shipped_count', true ) ) && intval( $order->get_meta( '_cas_pick_order_shipped_count', true ) ) > 0 ) ? intval( $order->get_meta( '_cas_pick_order_shipped_count', true ) ) : 0;
			$url_order       = admin_url( 'post.php?post=' . $order->get_id() . '&action=edit' );
			$priority_color  = Cas_Pick_Functions::cas_get_priority_status_color( $pick_priority );
			$pick_status_arr = ( is_array( $pick_status ) ) ? $pick_status : array();
			$order_notes     = wc_get_order_notes( array( 'limit' => '6', 'order_id' => $order->get_id() ) );
			$use_barcode     = ( isset( $options['cas_pick_product_c1_is_bar'] ) && 'yes' === $options['cas_pick_product_c1_is_bar'] ) ? true : false;
			$picked_items    = $order->get_meta( '_cas_pick_order_status_qty', true );
			$picked_obj      = $order->get_meta( '_cas_pick_order_object_items', true );
			$refunds_arr     = self::cas_get_refunded_order_items( $order );
			$have_refunds    = count( $refunds_arr ) > 0;

			if ( '' === $pick_percent ) {
				$pick_percent = 0;
			}

			$in_pick_progress = false;
			$picked_count     = 0;
			$picked_arr_stats = ( is_array( $picked_obj ) ) ? $picked_obj : $picked_items; // Use new object if values

			if ( is_array( $picked_arr_stats ) ) {
				foreach ( array_values( $picked_arr_stats ) as $value ) {
					if ( ! empty( $value ) ) {
						$in_pick_progress = true;
						$picked_count    += intVal( $value );
					}	
				}
			} else {
				$picked_count = $picked_items;
			}			

			/**
			 * Check and count the items that is not filtered out
			 * 
			 * Replaces the $order->get_item_count() function 
			 * 
			 * @since vs 2.5.8
			 */
			$order_item_count = Cas_Pick_Functions::cas_get_filtered_item_count( $order );

			?>
			<div class="cas-single-info">

			<div name="cas-qr-code" data-order_id="<?php echo esc_attr( $order->get_id() ); ?>" data-order_url="<?php echo esc_attr( $url ); ?>&tab=single&orderid=" data-account_url="<?php echo esc_attr( $account_url ); ?>"></div>
			
			<h1 class="cas-pick-h1">
				<a href="<?php echo esc_url( $url_order ); ?>">
				<?php echo esc_attr__( 'Order:', 'woo-picklist' ); ?> <?php echo esc_attr( $order->get_order_number() ); ?></a>
			</h1>
			<div class="cas-span-div">
				<span class="cas-span-div-left"><?php echo esc_attr__( 'Customer:', 'woo-picklist' ); ?></span>
				<span id="cas-customer"><strong>
					<?php if ( $order->get_customer_id() > 0 ) : ?>
						<a href="<?php echo esc_attr( $user_url . $order->get_customer_id() ); ?>" target="_new" title="<?php echo esc_attr__( 'Edit user', 'woo-picklist' ); ?>"><?php echo esc_attr( $customer_name ); ?></a>
					<?php else : ?>
						<?php echo esc_attr( $customer_name ); ?>
					<?php endif; ?>
					</strong> <i class="fa-solid fa-copy" onclick="casCopyCustomerName();" title="<?php echo esc_attr__( 'Copy customer name', 'woo-picklist' ); ?>" style="padding-left:4px;"></i>
				</span>
			</div>

			<div class="cas-span-div">
				<span class="cas-span-div-left"></span>
				<span id="cas-customer" style="font-size:small;"><strong><a href="<?php echo esc_url( $url_customer_orders . ( ( $order->get_customer_id() > 0 ) ? $order->get_customer_id() : $order->get_billing_email() ) ); ?>" title="<?php echo esc_attr__( 'Click to view other orders from customer', 'woo-picklist' ); ?>"><?php echo esc_attr__( 'View Other Orders →', 'woo-picklist' ); ?></a></strong> </span>
			</div>

			<div class="cas-span-div">
				<span class="cas-span-div-left"><?php echo esc_attr__( 'Priority:', 'woo-picklist' ); ?></span>
				<span style="font-variant-caps: all-small-caps;font-size:large;"><?php echo esc_attr( $pick_priority ); ?>
					<i class="fa-solid fa-fire" style="padding-left:5px; font-size:1rem;color:<?php echo esc_attr( $priority_color ); ?>;" title="<?php echo esc_attr( strtoupper( $pick_priority ) ); ?>"></i>	
				</span>
				<?php if ( $in_pick_progress && 100 !== intval( $pick_percent ) ) : ?>
					<i class="fa-solid fa-people-carry-box" style="color:black;font-size:medium;margin-left:10px;" title="<?php echo esc_attr__( 'Working', 'woo-picklist' ); ?>"></i>
				<?php endif; ?>
			</div>
			<div class="cas-span-div">
				<span class="cas-span-div-left"><?php echo esc_attr__( 'Status:', 'woo-picklist' ); ?></span>
				<span name="pick-order-status"><span id="cas-current-order-status" class="order-status status-<?php echo esc_attr( $order->get_status() ); ?>"><?php echo esc_attr( wc_get_order_status_name( $order->get_status() ) ); ?></span>
				<i class="fa-solid fa-pen-to-square" id="cas-change-status-icon" style="color:black;font-size:medium;margin-left:10px;" title="<?php echo esc_attr__( 'Change Order Status', 'woo-picklist' ); ?>"></i>
				<?php self::cas_get_status_dialog( $order->get_id() ); ?>
				<?php if ( 'yes' === $order->get_meta( '_cas_pick_order_shipped', true ) ) : ?>
					<i class="fa-solid fa-truck-arrow-right" style="color:green; font-size:large;margin-left:10px;" title="<?php echo esc_attr__( 'Order is shipped', 'woo-picklist' ); ?>"></i>
				<?php endif; ?>
				
			</div>
			<div class="cas-span-div">
				<span class="cas-span-div-left"><?php echo esc_attr__( 'Date:', 'woo-picklist' ); ?></span>
				<span><?php echo esc_attr( date_format( new DateTime( $order->get_date_created() ), $date_format . ' ' . $time_format ) ); ?></span>
			</div>
			<div class="cas-span-div">
				<span class="cas-span-div-left"><?php echo esc_attr__( 'Products:', 'woo-picklist' ); ?></span>
				<span><?php echo esc_attr( $order_item_count ); ?></span>
			</div>
			<?php if ( ! $hide_prices ) : ?>
				<div class="cas-span-div">
					<span class="cas-span-div-left"><?php echo esc_attr__( 'Total:', 'woo-picklist' ); ?></span>
					<span><?php echo wp_kses( wc_price( $order->get_total() ), 'post' ); ?></span>
				</div>
			<?php endif; ?>
			<?php if ( $have_refunds ) : ?>
				<div class="cas-span-div">
					<span class="cas-span-div-left"></span>
					<span style="color:red;" title="<?php echo esc_attr__( 'View Order for more details', 'woo-picklist' ); ?>">- <?php echo esc_attr__( 'Refunded Items', 'woo-picklist' ); ?> -</span>
				</div>
			<?php endif; ?>

			<div class="cas-single-shipping-info">
				<h2  class="cas-addr" style="display:inline-block;"><?php echo esc_attr__( 'Shipping', 'woo-picklist' ); ?></h2><button class="cas-address-toggle" id="tgl-shipping" onclick="casAddressToggle('shipping');"><i class="fa-solid fa-plus"></i></button>
				<div class="cas-shipping-tgl">
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'Customer:', 'woo-picklist' ); ?></span>
						<span id="cas-customer"><strong><a href="<?php echo esc_attr( $user_url . $order->get_customer_id() ); ?>" target="_new"><?php echo esc_attr( $customer_name ); ?></a></strong> <i class="fa-solid fa-copy" onclick="casCopyCustomerName();" title="<?php echo esc_attr__( 'Copy customer name', 'woo-picklist' ); ?>" style="padding-left:4px;"></i></span>
					</div>
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'Phone:', 'woo-picklist' ); ?></span>
						<span><strong><?php echo esc_attr( $order->get_billing_phone() ); ?></a></strong></span>
					</div>
					<?php if ( 'yes' === $options['cas_pick_show_company'] ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Company:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_billing_company() ? $order->get_billing_company() : ''  ); ?></span>
						</div>
					<?php endif; ?>
					<?php if ( 'yes' === $options['cas_pick_show_role'] ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Role:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_customer_id() ? Cas_Pick_Functions::cas_get_customer_role( $order->get_customer_id() ) : ''  ); ?></span>
						</div>
					<?php endif; ?>
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'Address 1:', 'woo-picklist' ); ?></span>
						<span><?php echo esc_attr( $order->get_shipping_address_1() ); ?></span>
					</div>
					<?php if ( ! empty( $order->get_shipping_address_2() ) ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Address 2:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_shipping_address_2() ); ?></span>
						</div>
					<?php endif; ?>
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'City:', 'woo-picklist' ); ?></span>
						<span><?php echo esc_attr( $order->get_shipping_city() ); ?></span>
					</div>
					<?php if ( ! empty( $order->get_shipping_state() ) ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'State:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_shipping_state() ); ?></span>
						</div>
					<?php endif; ?>
					<?php if ( ! empty( $order->get_shipping_postcode() ) ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Postcode / ZIP:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_shipping_postcode() ); ?></span>
						</div>
					<?php endif; ?>
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'Country:', 'woo-picklist' ); ?></span>
						<span><?php echo esc_attr( ( ! empty( $order->get_shipping_country() ) ) ? WC()->countries->countries[ $order->get_shipping_country() ] : '' ); ?></span>
					</div>
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'Shipping Method:', 'woo-picklist' ); ?></span>
						<span><?php echo esc_attr( $order->get_shipping_method() ); ?></span>
					</div>
					<?php if ( ! $hide_prices ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Shipping Total:', 'woo-picklist' ); ?></span>
							<span><?php echo wp_kses( wc_price( $order->get_shipping_total() ), 'post' ); ?></span>
						</div>
					<?php endif; ?>
				</div>
				<br/>
				<h2 class="cas-addr" style="display:inline-block;"><?php echo esc_attr__( 'Billing', 'woo-picklist' ); ?></h2><button class="cas-address-toggle"  id="tgl-billing" onclick="casAddressToggle('billing');"><i class="fa-solid fa-plus"></i></button>
				<div class="cas-billing-tgl">
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'Customer:', 'woo-picklist' ); ?></span>
						<span><strong><?php echo esc_attr( $customer_name ); ?></strong></span>
					</div>
					<?php if ( 'yes' === $options['cas_pick_show_company'] ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Company:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_billing_company() ? $order->get_billing_company() : ''  ); ?></span>
						</div>
					<?php endif; ?>

					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'Address 1:', 'woo-picklist' ); ?></span>
						<span><?php echo esc_attr( $order->get_billing_address_1() ); ?></span>
					</div>
					<?php if ( ! empty( $order->get_billing_address_2() ) ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Address 2:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_billing_address_2() ); ?></span>
						</div>
					<?php endif; ?>
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'City:', 'woo-picklist' ); ?></span>
						<span><?php echo esc_attr( $order->get_billing_city() ); ?></span>
					</div>
					<?php if ( ! empty( $order->get_billing_state() ) ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'State:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_billing_state() ); ?></span>
						</div>
					<?php endif; ?>
					<?php if ( ! empty( $order->get_billing_postcode() ) ) : ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Postcode / ZIP:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_billing_postcode() ); ?></span>
						</div>
					<?php endif; ?>
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr__( 'Country:', 'woo-picklist' ); ?></span>
						<span><?php echo esc_attr( ( ! empty( $order->get_billing_country() ) ) ? WC()->countries->countries[ $order->get_billing_country() ] : '' ); ?></span>
					</div>
				</div>
			</div>

			<!-- CUSTOM ORDER FIELDS --->
			<?php if ( 'yes' === $options['cas_pick_order_custom1_order'] && strlen( $options['cas_pick_order_custom1'] ) > 0 ) : ?>
				<div class="cas-span-div">
					<span class="cas-span-div-left"><?php echo esc_attr( $options['cas_pick_order_custom1_lbl'] . ':' ); ?></span>
					<span><?php echo esc_attr( $order->get_meta( $options['cas_pick_order_custom1'], true )  ); ?></span>
				</div>
			<?php endif; ?>
			<?php if ( 'yes' === $options['cas_pick_order_custom2_order'] && strlen( $options['cas_pick_order_custom2'] ) > 0 ) : ?>
				<div class="cas-span-div">
					<span class="cas-span-div-left"><?php echo esc_attr( $options['cas_pick_order_custom2_lbl'] . ':' ); ?></span>
					<span><?php echo esc_attr( $order->get_meta( $options['cas_pick_order_custom2'], true )  ); ?></span>
				</div>
			<?php endif; ?>
		
			<div class="cas-span-div2">
				<span class="cas-span-div-left"><?php echo esc_attr__( 'Assigned to:', 'woo-picklist' ); ?></span>
				<div style="max-width:95%;margin-top:10px;margin-bottom:10px;"><?php Cas_Pick_Functions::cas_get_pick_list_shop_managers( $pick_admin_usr, $order->get_id() ); ?></div>
			</div>
			<div class="cas-span-div">
				<span style="font-size:0.85rem;"><?php echo esc_attr__( 'Customer Provided Note:', 'woo-picklist' ); ?></span>
			</div>
			<div>
				<textarea id="cas_customer_order_note" type="text" class="input-txt" rows="3" cols="42" style="background-color:floralwhite;border-left:4px solid orangered;font-size:0.75rem;"><?php echo wp_kses( $order->get_customer_note(), 'post' ); ?></textarea>
			</div>
			<div class="cas-span-div">
				<span class="cas-span-div-left"><?php echo esc_attr__( 'Order Notes:', 'woo-picklist' ); ?></span>
				<span></span>
			</div>
			<div>
				<textarea id="cas_order_notes" type="text" class="input-txt" rows="10" cols="42" style="background-color:floralwhite;border-left:4px solid lightgreen;font-size:0.75rem;">
				<?php 
				if ( is_array( $order_notes ) ) {
					foreach ( $order_notes as $note ) {
						?>
							<?php echo wp_kses( $note->content, Cas_Pick_Functions::cas_allowed_html_tags() ); ?>
							<?php echo esc_attr( date_format( new DateTime( $note->date_created ), $date_format . ' ' . $time_format ) ); ?>  
						<?php 
					}
				}
				?>
				</textarea>
			</div>
			<div>
				<textarea id="cas-order-note-<?php echo esc_attr( $order->get_id() ); ?>" type="text" class="input-txt" rows="4" cols="42" style="border-left:4px solid lightgreen;font-size:0.75rem;"></textarea><br/>
				<button class="button" onclick="casAddOrderNote('#cas-order-note-<?php echo esc_attr( $order->get_id() ); ?>', '<?php echo esc_attr( $order->get_id() ); ?>');"><i class="fa-solid fa-note-sticky"></i> <?php echo esc_attr__( 'Add Order Note', 'woo-picklist' ); ?></button>
				<br/>
			</div>
			<div class="cas-span-div" style="margin-top:20px;">
				<span><?php echo esc_attr__( 'Packing Slip Note:', 'woo-picklist' ); ?></span>
				<span></span>
			</div>
			<div>
				<textarea id="cas-order-slip-note-<?php echo esc_attr( $order->get_id() ); ?>" type="text" class="input-txt" rows="12" cols="42" style="border-left:4px solid lightgreen;font-size:0.75rem;"><?php echo wp_kses( $options['cas_pick_message_slip'], 'post' ); ?></textarea>
				<div class="cas-bulk-pick-btn" style="float:left;">
					<input type="checkbox" id="cas-check-slip-note-<?php echo esc_attr( $order->get_id() ); ?>" <?php echo esc_attr( isset( $options['cas_pick_message_slip_on'] ) && 'yes' === $options['cas_pick_message_slip_on'] ? 'checked="checked"' : '' ); ?>><?php echo esc_attr__( 'Include Note in Print', 'woo-picklist' ); ?>
				</div>
			</div>
			
			
			<!-- End left side -->
			</div>

			<?php wp_nonce_field( 'cas_pick_id' ); ?>

			
			<div class="cas-single-grid-table">
				
				<div style="display: flex;">
					<span class="cas-button-row-left">
						<button class="button" id="cas-toggle-order-info-button" onclick="casToggleOrderInfo();" title="<?php echo esc_attr__( 'Show or Hide Order Info', 'woo-picklist' ); ?>"> <i class="fa-solid fa-address-book"></i> </button>
						<a href="<?php echo esc_attr( $url . $from_tab ); ?>" class="button cas-btn-back" title="<?php echo esc_attr__( 'Back to Order List', 'woo-picklist' ); ?>"><i class="fa-solid fa-circle-left"></i> <?php echo esc_attr__( 'Back', 'woo-picklist' ); ?></a> 
						<button id="cas-shipping-button" class="button cas-notification-button" onclick="casPickShipped('<?php echo esc_attr( $order->get_id() ); ?>');" title="<?php echo esc_attr__( 'Click if item(s) shipped to customer', 'woo-picklist' ); ?>"><i class="fa-solid fa-truck-arrow-right"></i> 
							<?php if ( $shipped_count > 0 ) : ?>
								<span class="cas-count-badge" title="<?php echo esc_attr__( 'Shipped count', 'woo-picklist' ); ?>"><?php echo esc_attr( $shipped_count ); ?></span>
							<?php endif; ?>
						</button>
						<?php if ( $ship_lbl_enabled ) : ?>
							<button id="cas-shipping-label-button" class="button" onclick="casPrintList('shipping_label');" title="<?php echo esc_attr__( 'Print Shipping Label', 'woo-picklist' ); ?>" style="<?php echo esc_attr( $is_dark_mode ? 'color:lime;' : 'color:black;'); ?>"> <i class="fa-solid fa-tag"></i> </button>
						<?php endif; ?>
						<button id="cas-assign-me-button" class="button" onclick="casAssignMe('<?php echo esc_attr( $order->get_id() ); ?>');" <?php echo esc_attr( ( strlen( $pick_admin_usr ) > 0 ) ? 'disabled="disabled"' : '' ); ?> title="<?php echo esc_attr__( 'Assign me to Pick this Order', 'woo-picklist' ); ?>"><i class="fa-solid fa-user-plus"></i></button>
						
						<?php if ( $use_barcode ) : ?>
							<!-- BARCODE INPUT -->
							<input type="text" id="cas-barcode-input" name="cas-barcode-input" class="cas-barcode-input" data-order_id="<?php echo esc_attr( $order->get_id() ); ?>" placeholder="barcode + enter" autofocus="autofocus"> 
						<?php endif; ?>
					</span>
					
					<span style="direction:rtl;margin-bottom:5px;">
						
						<?php if ( ! $hide_next_orders ) : ?>
							<!-- Open Slider -->
							<button class="button" id="cas-open-slider-btn" title="<?php echo esc_attr__( 'See the next 7 orders to pick.', 'woo-picklist' ); ?>"><?php echo esc_attr__( 'Next Orders', 'woo-picklist' ); ?></button>
						<?php endif; ?>
						
						<!-- Other buttons -->
						<button class="button" onclick="casToggleGrid();" title="<?php echo esc_attr__( 'Toggle Grid and Table View', 'woo-picklist' ); ?>"><i class="fa-solid fa-table-cells-large"></i></button>
						<button class="button" style="background-color:lightgreen;color:black;" onclick="casPrintList('pick');" title="<?php echo esc_attr__( 'Print Pick List', 'woo-picklist' ); ?>"><i class="fa-solid fa-print"></i></button>
						<button class="button" style="background-color:deepskyblue;color:black;" onclick="casPrintList('pack');" title="<?php echo esc_attr__( 'Print Packing Slip', 'woo-picklist' ); ?>"><i class="fa-solid fa-print"></i></button>
						<button class="button" onclick="casResetpick('<?php echo esc_attr( $order->get_id() ); ?>', '<?php echo esc_attr__( 'Please confirm to reset the order pick!', 'woo-picklist' ); ?>');" title="<?php echo esc_attr__( 'Reset pick - undo all picked items', 'woo-picklist' ); ?>"><i class="fa-solid fa-rotate-left"></i></button>
						<button class="button" onclick="casBulkPick('<?php echo esc_attr( $order->get_id() ); ?>');" title="<?php echo esc_attr__( 'Bulk pick all items', 'woo-picklist' ); ?>"><i class="fa-solid fa-list-check"></i></button>
						
						<?php if ( $use_barcode ) : ?>
							<button class="button" id="cas-barcode-focus-btn" title="<?php echo esc_attr__( 'Barcode scanner (input focus)', 'woo-picklist' ); ?>"><i class="fa-solid fa-barcode"></i></button>
							<img id="cas-barcode-icon" src="<?php echo esc_attr( plugins_url( '../assets/images/picklist-scan-icon-small.webp', __FILE__ ) ); ?>" style="width:28px;" title="<?php echo esc_attr__( 'Scan ready', 'woo-picklist' ); ?>">
						<?php endif; ?>
					</span>
				</div>
				
				<div>
					<progress id="progress-order-<?php echo esc_attr( $order->get_id() ); ?>" value="<?php echo esc_attr( $pick_percent ); ?>" max="100" title="<?php echo esc_attr( $pick_percent ); ?>%"> <?php echo esc_attr( $pick_percent ); ?>% </progress>
				</div>
				
				<!-- GRID -->
				<?php
				if ( $show_grid_view ) {
					self::cas_get_pick_grid_view( $order, $picked_items, $pick_status, $use_barcode, $picked_obj );
				} else {
					?>
					<div id="cas-pick-grid" class="cas-grid-container cas-hide">
						<h3><i class="fa-solid fa-circle-info"></i> <?php echo esc_attr__( 'Grid View is not enabled. Enable in Pick List settings (Pick Order).', 'woo-picklist' ); ?></h3>
					</div>
					<?php
				}

				$th_index = 2;
				?>

				<!-- TABLE -->
				<div id="cas-pick-table" class="cas-table-container" data-use-barcode="<?php echo esc_attr( $use_barcode ); ?>">
					<table class="widefat striped posts cas-pick-table" style="width: 100%;" id="cas-picking-table">
						<thead>
						<tr>
							<th style="width: 10%;"></th>
							<th>
								<select id="picklist-sort" title="<?php echo esc_attr__( 'Sort by...', 'woo-picklist' ); ?>">
									<option value="name"><?php echo esc_attr__( 'Name', 'woo-picklist' ); ?></option>
									<option value="sku"><?php echo esc_attr__( 'SKU', 'woo-picklist' ); ?></option>
									<option value="category"><?php echo esc_attr__( 'Category', 'woo-picklist' ); ?></option>
								</select>
							</th>
							<th style="width: 20px;"><i id="cas-eye-toggle-icon" class="fa-solid fa-eye" onclick="casToggleAllComments();" title="<?php echo esc_attr__( 'Toggle Comments (automatically saves entered item comments)', 'woo-picklist' ); ?>"></i></th>
							<?php if ( ! $hide_stock_col ) : ?>
								<?php $th_index++; ?>
								<th style="width: 10%;" data-index="stock"><a href="#" onclick="casSortCol(<?php echo esc_attr( $th_index ); ?>,'number','p','cas-picking-table');">
									<i id="p<?php echo esc_attr( $th_index ); ?>" class="fas fa-sort cas-i-space"></i><?php echo esc_attr__( 'Stock', 'woo-picklist' ); ?></a></th>
							<?php endif; ?>
							<?php if ( ! $hide_prices && ! $hide_price_col ) : ?>
								<?php $th_index++; ?>
								<th style="width: 9%;" data-index="price"><a href="#" onclick="casSortCol(<?php echo esc_attr( $th_index ); ?>,'number','p','cas-picking-table');">
									<i id="p<?php echo esc_attr( $th_index ); ?>" class="fas fa-sort cas-i-space"></i><?php echo esc_attr__( 'Price', 'woo-picklist' ); ?></a></th>
							<?php endif; ?>
							<?php $th_index++; ?>
							<th style="width: 120px;"><?php echo esc_attr__( 'Quantity', 'woo-picklist' ); ?></th>
							<?php if ( ! $hide_weight_col ) : ?>
								<?php $th_index++; ?>
								<th style="width: 7%;" data-index="weight"><a href="#" onclick="casSortCol(<?php echo esc_attr( $th_index ); ?>,'number','p','cas-picking-table');">
									<i id="p<?php echo esc_attr( $th_index ); ?>" class="fas fa-sort cas-i-space"></i><?php echo esc_attr__( 'Weight', 'woo-picklist' ); ?></a></th>
							<?php endif; ?>
							<?php if ( $show_dimensions_col ) : ?>
								<?php $th_index++; ?>
								<th style="width: 12%;"><i id="cas-eye-toggle-boxes-icon" class="fa-solid fa-eye" onclick="casToggleBoxes();" title="<?php echo esc_attr__( 'Toggle Boxes in 3D', 'woo-picklist' ); ?>"></i> <?php echo esc_attr__( 'Dimensions', 'woo-picklist' ); ?></th>
							<?php endif; ?>
							<th style="width: 160px;text-align:center;background-color:gainsboro;">
								<?php echo esc_attr__( 'Picked:', 'woo-picklist' ); ?> <span id="cas_picked_count" style="font-weight:600;color:red;font-family:monospace;font-size:15px;padding:4px;"><?php echo esc_attr( $picked_count ? $picked_count : 0 ); ?>/<?php echo esc_attr( $order_item_count ); ?></span>
							</th>
						</tr>
						</thead>
						<tbody>
						<?php
						foreach ( $order->get_items() as $item ) : 

							$item_id        = $item->get_id();
							$product_id     = ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id();
							$product        = wc_get_product( $product_id );
							$desc           = '';
							$item_price     = ( $item->get_subtotal() && $item->get_quantity() ) ? floatval( $item->get_subtotal() ) / $item->get_quantity() : 0;
							$using_pick_obj = empty( $picked_obj ) ? false : true;
							$comment        = $item->get_meta( '_cas_pick_item_comment', true );

							/**
							 * Filter to include or exclude order item from pick list.
							 * 
							 * @since 2.5.8
							 */
							$include_item = apply_filters( 'cas_picklist_include_order_item', true, $item, $product, $order );

							if ( ! $include_item ) {
								continue;
							}

							// Make dummy product if not found
							if ( ! $product ) {
								$product = self::cas_make_dummy_product( $item->get_id(), $item );
								// Fallback ID
								$product_id = $item->get_id();
							}

							if ( 'variation' === $product->get_type() ) {
								$desc = $product->get_attribute_summary();
							}
							
							$qty_css    = ( $item->get_quantity() > 1 ) ? 'cas-qty-several' : 'cas-qty-normal';
							$picked_qty = ( is_array( $picked_items ) && array_key_exists( $product_id, $picked_items ) && ! empty( $picked_items[$product_id] ) ) ? $picked_items[$product_id] . '/' . $item->get_quantity() : '';  
							$item_qty   = ( is_array( $picked_items ) && array_key_exists( $product_id, $picked_items ) ) ? $picked_items[$product_id] : 0;

							if ( ! is_array( $picked_items ) && is_array( $pick_status ) ) {
								$item_qty = ( in_array( $product_id, $pick_status, true) ) ? $item->get_quantity() : 0; 
							}

							// New object to hold picked items (vs 2.4.16)
							if ( $using_pick_obj ) {
								$picked_qty = ( is_array( $picked_obj ) && array_key_exists( $item_id, $picked_obj ) && ! empty( $picked_obj[$item_id] ) ) ? $picked_obj[$item_id] . '/' . $item->get_quantity() : '';
								$item_qty   = ( is_array( $picked_obj ) && array_key_exists( $item_id, $picked_obj ) ) ? $picked_obj[$item_id] : 0;
							}
							
							// Pick completed?
							$pick_done = ( $item_qty === $item->get_quantity() ) ? true : false; 

							// Refund qty
							$qty_refunded = ( $have_refunds && array_key_exists( $item_id, $refunds_arr ) ) ? $refunds_arr[$item_id] : ''; 

							// Categories
							$categories     = wp_get_post_terms( $item->get_product_id(), 'product_cat', array( 'fields' => 'names' ) );
							$category_names = ! empty( $categories ) ? implode( ', ', $categories ) : 'Uncategorized';

							// Product image
							$product_image = $product->get_image_id() ? wp_get_attachment_image_url( $product->get_image_id(), 'medium' ) : wc_placeholder_img_src();

							
							?>
							<tr data-item-name="<?php echo esc_attr( $product->get_name() ); ?>"
								data-item-sku="<?php echo esc_attr( $product->get_sku() ); ?>"
								data-item-category="<?php echo esc_attr( $category_names ); ?>"
								>
								<td data-label="">
								<div class="cas-image-container" id="table-img-<?php echo esc_attr( $item_id ); ?>">
									<?php if ( $pick_done ) : ?>
										<i class="fas fa-check-circle cas-icon"></i>	
									<?php endif; ?>
									<img src="<?php echo wp_kses( $product_image, 'post' ); ?>" class="cas-table-image <?php echo esc_attr( $pick_done ? 'cas-img-pick-done' : '' ); ?>" alt="">
								</div>
								</td>
								<td class="cas-tbl-head" 
									data-label="<?php echo esc_attr__( 'Name:', 'woo-picklist' ); ?>"
									>
									<div class="cas-tbl-h2"><a href="<?php echo esc_url( get_admin_url() . 'post.php?post=' . $item->get_product_id() . '&action=edit' ); ?>" target="new"><?php echo esc_attr( $item->get_name() ); ?></a></div>
									
									<?php self::cas_get_order_item_attributes( $item ); ?>

									<span class="cas-tbl-sku"><?php echo esc_attr__( 'SKU:', 'woo-picklist' ); ?> <?php echo esc_attr( $product->get_sku() ); ?></span> <br/>
									<?php if ( isset( $product->is_dummy ) && $product->is_dummy ) : ?>
										<span style="color:red;font-size:smaller;"><?php echo esc_attr__( '( Not linked to product )', 'woo-picklist' ); ?></span>
									<?php endif; ?>
									<?php self::cas_get_regular_product_fields( $product ); ?>
									<?php foreach ( $custom_fields as $custom ) : ?> 
										<?php
										if ( 0 === $item->get_product_id() || empty( get_post_meta( $item->get_product_id(), $custom->slug, true ) ) ) { 
											continue; 
										}
										?>
										<i class="fa-solid fa-circle-dot cas-custom-fields-dot"></i> 
										<?php echo esc_attr( $custom->name ); ?>: <?php echo esc_attr( get_post_meta( $item->get_product_id(), $custom->slug, true ) ); ?><br/>
									<?php endforeach; ?>	
									<?php if ( ! $hide_categories && 0 > $item->get_product_id() ) : ?>
										<div class="cas-td-scroll">
										<?php echo esc_attr__( 'Category:', 'woo-picklist' ); ?> <?php echo wp_kses( wc_get_product_category_list( $item->get_product_id() ), 'post' ); ?> <br/>
										</div>
									<?php endif; ?>
									
									
									<?php 
									if ( $use_barcode ) : 
										$bar_data = self::cas_get_barcode_data( $product );
										?>
										<div class="cas-barcode" name="cas-barcode" id="cas-barcode-<?php echo esc_attr( $item->get_id() ); ?>" 
											data-barcode-value="<?php echo esc_attr( $bar_data ); ?>"
											data-barcode-itemid="<?php echo esc_attr( $item->get_id() ); ?>"
											data-barcode-qty="<?php echo esc_attr( $item->get_quantity() ); ?>"
											data-scan-all="<?php echo esc_attr( $options['cas_pick_scan_all_qty'] ); ?>"
											data-prod-id="<?php echo esc_attr( $product_id ); ?>"
											data-pick-done="<?php echo esc_attr( $pick_done ); ?>"
										></div>
									<?php endif; ?>	
								</td>
								<td>
									<i class="cas-info-icon fas fa-info-circle" onclick="casGetPostItNote('cas-postit-<?php echo esc_attr( $item->get_id() ); ?>');" title="<?php echo esc_attr__( 'Toggle Comment', 'woo-picklist' ); ?>"></i>
									<textarea class="cas-post-it-note" id="cas-postit-<?php echo esc_attr( $item->get_id() ); ?>" onfocusout="casSavePostItNote('<?php echo esc_attr( $order->get_id() ); ?>','<?php echo esc_attr( $item->get_id() ); ?>');" placeholder="<?php echo esc_attr__( 'Add your comment...', 'woo-picklist' ); ?>"><?php echo esc_attr( $comment ); ?></textarea>
								</td>
								</td>
								<?php if ( ! $hide_stock_col ) : ?>
									<td data-label="<?php echo esc_attr__( 'Stock:', 'woo-picklist' ); ?>">
										<?php echo esc_attr( self::cas_get_readable_stock_status( $product->get_stock_status() ) . ': ' ); ?>
										<span id="stock-qty-product-<?php echo esc_attr( $product_id ); ?>" class="cas-stock-qty" style="<?php echo esc_attr( ( $product->get_stock_quantity() <= 0 ) ? 'color:red;' : '' ); ?>"><?php echo esc_attr( $product->get_stock_quantity() ); ?></span>
										<br/><i class="fa-solid fa-pen-to-square cas-edit-icon"
												title="<?php echo esc_attr__( 'Edit Stock Quantity', 'woo-picklist' ); ?>" 
												onclick="casOpenStockQuantityDialog('<?php echo esc_attr( $product_id ); ?>', <?php echo (int) $product->get_stock_quantity(); ?>, this);"></i>
									</td>
								<?php endif; ?>
								<?php if ( ! $hide_prices && ! $hide_price_col ) : ?>
									<td data-label="<?php echo esc_attr__( 'Price:', 'woo-picklist' ); ?>">
										<?php echo wp_kses( wc_price( $item_price ), 'post' ); ?> 
									</td>
								<?php endif; ?>
								<td data-label="<?php echo esc_attr__( 'Quantity:', 'woo-picklist' ); ?>">
									<span class="<?php echo esc_attr( $qty_css ); ?>"><?php echo esc_attr( $item->get_quantity() ); ?></span> 
									<input type="number" value="1" min="1" max="<?php echo esc_attr( $item->get_quantity() ); ?>" id="pick-qty-<?php echo esc_attr( $item_id ); ?>" name="pick-qty" class="cas-qty-input-number" style="<?php echo esc_attr( ( ( 'yes' === $options['cas_pick_scan_all_qty'] ) || ( 1 === $item->get_quantity() ) ) ? 'display:none;' : '' ); ?>">
									<div id="tbl-items-picked-<?php echo esc_attr( $item_id ); ?>" style="margin-top:10px;font-weight:600;color:red;font-family: monospace;" title="<?php echo esc_attr__( 'Picked items', 'woo-picklist' ); ?>">
										<?php echo esc_attr( $picked_qty ); ?>
									</div>
									<?php if ( ! empty( $qty_refunded ) ) : ?>
										<div style="margin-top:10px;color:red;font-size:larger;">
											<span title="<?php echo esc_attr__( 'Refunded Items', 'woo-picklist' ); ?>"><i class="fa-solid fa-arrow-rotate-left"></i> <?php echo esc_attr( $qty_refunded ); ?></span>
										</div>
									<?php endif; ?>
								</td>
								<?php if ( ! $hide_weight_col ) : ?>
									<td data-label="<?php echo esc_attr__( 'Total Weight:', 'woo-picklist' ); ?>" data-item-weight="<?php echo esc_attr( $item->get_quantity() * floatval( $product->get_weight() ) ); ?>"><?php echo esc_attr( $item->get_quantity() * floatval( $product->get_weight() ) . ' ' . $weight_unit ); ?></td>
								<?php endif; ?>
								<?php if ( $show_dimensions_col ) : ?>
									<td data-label="<?php echo esc_attr__( 'Dimensions:', 'woo-picklist' ); ?>" data-dimensions="<?php echo esc_attr( wc_format_dimensions( $product->get_dimensions( false ) ) ); ?>" class="cas-dim-font">
										<?php 
										// Show dimensions
										echo wp_kses( self::cas_get_formatted_product_dimensions_with_quantity( $product, $item->get_quantity() ), 'post' ); 

										$dimensions = wc_format_dimensions( $product->get_dimensions( false ) );
										$dims       = ! empty( $dimensions ) ? explode( ' ', $dimensions ) : array();
										
										// Ensure we have (Width, Length, Height)
										if ( 6 === count( $dims ) ) {
											echo wp_kses( '<div><button class="show-box button">' . __( 'Visualize Box', 'woo-picklist' ) . '</button></div>', 'post' );
										}

										?>
									</td>
								<?php endif; ?>
								<td class="cas-td-right" data-label="">
									<button <?php echo esc_attr( $pick_done ? 'disabled="disabled"' : '' ); ?> data-btn-disabled="<?php echo esc_attr__( 'Done', 'woo-picklist' ); ?>" id="btn-pick-tbl-<?php echo esc_attr( $item_id ); ?>" onclick="caspickItem('<?php echo esc_attr( $product_id ); ?>', '<?php echo esc_attr( $order->get_id() ); ?>', '<?php echo esc_attr( $item->get_quantity() ); ?>', '<?php echo esc_attr( $item->get_id() ); ?>');" class="cas-btn"><i class="fa-solid fa-circle-check"></i> <?php echo esc_attr__( 'Pick', 'woo-picklist' ); ?></button>
									<button id="btn-unpick-tbl-<?php echo esc_attr( $item_id ); ?>" title="<?php echo esc_attr__( 'Undo Pick', 'woo-picklist' ); ?>" onclick="casUnPickItem('<?php echo esc_attr( $product_id ); ?>', '<?php echo esc_attr( $order->get_id() ); ?>', '<?php echo esc_attr( $item->get_quantity() ); ?>', '<?php echo esc_attr( $item->get_id() ); ?>');" class="cas-ubtn"><i class="fa-solid fa-arrow-rotate-left"></i></button>
								</td>
							</tr>

						<?php endforeach; ?>
						</tbody>
						<tfoot>

						</tfoot>
					</table>
				</div>

			</div>
		<?php endif; ?>
		</div>

		<!-- Edit Stock Quantity Dialog -->
		<div id="cas-stock-qty-dialog" title="<?php esc_attr_e( 'Edit Stock Quantity', 'woo-picklist' ); ?>" style="display:none;">
			<form id="cas-stock-qty-form">
				<input type="hidden" name="product_id" id="cas-stock-qty-product-id">
				<label for="cas-stock-qty-value"><?php esc_html_e( 'New Quantity:', 'woo-picklist' ); ?></label>
				<input type="number" id="cas-stock-qty-value" name="quantity" min="0">
			</form>
		</div>

		<!-- Spinner -->
		<div id="cas-shipping-spinner" style="display:none; position:fixed; top:50%; left:50%; transform:translate(-50%,-50%); z-index:9999; background:rgba(255,255,255,0.8); padding:30px 40px; border-radius:8px; font-weight:bold; box-shadow:0 0 10px rgba(0,0,0,0.2);">
			<span class="dashicons dashicons-update dashicons-spin" style="font-size: 24px; vertical-align: middle;"></span>
			<span style="margin-left: 10px;"><?php echo esc_attr__( 'Generating label...', 'woo-picklist' ); ?></span>
		</div>

		<!-- NEXT ORDERS -->
		<?php if ( ! $hide_next_orders ) : ?>
			<?php self::cas_get_next_orders(); ?>
		<?php endif; ?>

		<!-- PRINT TABLE (Pick List and Picking Slip) -->
		<?php if ( ! empty( $order ) ) : ?>
			<div id="cas-print-pick-table" class="cas-print-table">
				<?php
				$show_billing = ( isset( $options['cas_show_billing_prints'] ) && 'yes' === $options['cas_show_billing_prints'] ) ? true : false;
				$logo_url     = $options['cas_pick_site_logo'];
				if ( ! empty( $logo_url ) ) {
					?>
					<img src="<?php echo esc_url( $logo_url ); ?>" style="padding-bottom:20px; max-height: 200px; height: auto;" alt="">
				<?php } else { ?>
					<h1 style="padding-bottom:80px;"><?php echo esc_attr( get_bloginfo( 'name' ) ); ?></h1>
				<?php }; ?>

				<h1><?php echo esc_attr__( 'Order:', 'woo-picklist' ); ?> <?php echo esc_attr( $order->get_order_number() ); ?></h1>
				<span id="cas-print-type" class="cas-type-of-print"><?php echo esc_attr__( 'Pick List', 'woo-picklist' ); ?></span>
				<div class="cas-span-div">
					<span class="cas-span-div-left" style="font-size: 19px;"><?php echo esc_attr__( 'Products:', 'woo-picklist' ); ?> <?php echo esc_attr( $order_item_count ); ?></span>	
				</div>
				
				<div class="cas-parent-flex">
					<?php if ( $show_billing ) : ?>
					<div class="cas-billing">

						<div><h2><?php echo esc_attr__( 'Billing', 'woo-picklist' ); ?></h2></div>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Customer:', 'woo-picklist' ); ?></span>
							<span><strong><?php echo esc_attr( $customer_name ); ?></strong></span>
						</div>
						<?php if ( 'yes' === $options['cas_pick_show_company'] ) : ?>
							<div class="cas-span-div">
								<span class="cas-span-div-left"><?php echo esc_attr__( 'Company:', 'woo-picklist' ); ?></span>
								<span><?php echo esc_attr( $order->get_billing_company() ? $order->get_billing_company() : ''  ); ?></span>
							</div>
						<?php endif; ?>

						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Address 1:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_billing_address_1() ); ?></span>
						</div>
						<?php if ( ! empty( $order->get_billing_address_2() ) ) : ?>
							<div class="cas-span-div">
								<span class="cas-span-div-left"><?php echo esc_attr__( 'Address 2:', 'woo-picklist' ); ?></span>
								<span><?php echo esc_attr( $order->get_billing_address_2() ); ?></span>
							</div>
						<?php endif; ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'City:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_billing_city() ); ?></span>
						</div>
						<?php if ( ! empty( $order->get_billing_state() ) ) : ?>
							<div class="cas-span-div">
								<span class="cas-span-div-left"><?php echo esc_attr__( 'State:', 'woo-picklist' ); ?></span>
								<span><?php echo esc_attr( $order->get_billing_state() ); ?></span>
							</div>
						<?php endif; ?>
						<?php if ( ! empty( $order->get_billing_postcode() ) ) : ?>
							<div class="cas-span-div">
								<span class="cas-span-div-left"><?php echo esc_attr__( 'Postcode / ZIP:', 'woo-picklist' ); ?></span>
								<span><?php echo esc_attr( $order->get_billing_postcode() ); ?></span>
							</div>
						<?php endif; ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Country:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( ( ! empty( $order->get_billing_country() ) ) ? WC()->countries->countries[ $order->get_billing_country() ] : '' ); ?></span>
						</div>

					</div>
					<?php endif; ?>
					<div class="cas-shipping">

						<div><h2><?php echo esc_attr__( 'Shipping', 'woo-picklist' ); ?></h2></div>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Customer:', 'woo-picklist' ); ?></span>
							<span><strong><?php echo esc_attr( $customer_name ); ?></strong></span>
						</div>
						<?php if ( 'yes' === $options['cas_pick_show_company'] ) : ?>
							<div class="cas-span-div">
								<span class="cas-span-div-left"><?php echo esc_attr__( 'Company:', 'woo-picklist' ); ?></span>
								<span><?php echo esc_attr( $order->get_shipping_company() ? $order->get_shipping_company() : ''  ); ?></span>
							</div>
						<?php endif; ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Address 1:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_shipping_address_1() ); ?></span>
						</div>
						<?php if ( ! empty( $order->get_shipping_address_2() ) ) : ?>
							<div class="cas-span-div">
								<span class="cas-span-div-left"><?php echo esc_attr__( 'Address 2:', 'woo-picklist' ); ?></span>
								<span><?php echo esc_attr( $order->get_shipping_address_2() ); ?></span>
							</div>
						<?php endif; ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'City:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_shipping_city() ); ?></span>
						</div>
						<?php if ( ! empty( $order->get_shipping_state() ) ) : ?>
							<div class="cas-span-div">
								<span class="cas-span-div-left"><?php echo esc_attr__( 'State:', 'woo-picklist' ); ?></span>
								<span><?php echo esc_attr( $order->get_shipping_state() ); ?></span>
							</div>
						<?php endif; ?>
						<?php if ( ! empty( $order->get_shipping_postcode() ) ) : ?>
							<div class="cas-span-div">
								<span class="cas-span-div-left"><?php echo esc_attr__( 'Postcode / ZIP:', 'woo-picklist' ); ?></span>
								<span><?php echo esc_attr( $order->get_shipping_postcode() ); ?></span>
							</div>
						<?php endif; ?>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Country:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( ( ! empty( $order->get_shipping_country() ) ) ? WC()->countries->countries[ $order->get_shipping_country() ] : '' ); ?></span>
						</div>
						<div class="cas-span-div">
							<span class="cas-span-div-left"><?php echo esc_attr__( 'Shipping method:', 'woo-picklist' ); ?></span>
							<span><?php echo esc_attr( $order->get_shipping_method() ); ?></span>
						</div>

					</div>

				</div>	

				<?php if ( 'yes' === $options['cas_pick_order_custom1_order'] && strlen( $options['cas_pick_order_custom1'] ) > 0 ) : ?>
				<div class="cas-span-div">
					<span class="cas-span-div-left"><?php echo esc_attr( $options['cas_pick_order_custom1_lbl'] . ':' ); ?></span>
					<span><?php echo esc_attr( $order->get_meta( $options['cas_pick_order_custom1'], true )  ); ?></span>
				</div>
				<?php endif; ?>
				<?php if ( 'yes' === $options['cas_pick_order_custom2_order'] && strlen( $options['cas_pick_order_custom2'] ) > 0 ) : ?>
					<div class="cas-span-div">
						<span class="cas-span-div-left"><?php echo esc_attr( $options['cas_pick_order_custom2_lbl'] . ':' ); ?></span>
						<span><?php echo esc_attr( $order->get_meta( $options['cas_pick_order_custom2'], true )  ); ?></span>
					</div>
				<?php endif; ?>

				<div class="cas-qr-code-print" id="cas-qr-code-table-<?php echo esc_attr( $order->get_id() ); ?>"></div>
				<table id="cas-print-table-<?php echo esc_attr( $order->get_id() ); ?>" class="wp-list-table widefat striped posts" style="width: 100%;margin-top:40px;border-left:none;border-right:none;border-bottom:none;">
					<thead>
						<tr>
							<th style="width: 10%;"></th>
							<th><?php echo esc_attr__( 'Name', 'woo-picklist' ); ?></th>
							<?php if ( ! $hide_prices ) : ?>
								<th style="text-align:center;width:130px;"><?php echo esc_attr__( 'Price', 'woo-picklist' ); ?></th>
							<?php endif; ?>
							<th style="text-align:center;width:130px;"><?php echo esc_attr__( 'Quantity', 'woo-picklist' ); ?></th>
							<th style="text-align:center;width:130px;"><?php echo esc_attr__( 'Total Weight', 'woo-picklist' ); ?></th>
							<th style="width:130px;"></th>
						</tr>
					</thead>
					<tbody>
					<?php 
					foreach ( $order->get_items() as $item ) : 
						
						$item_id        = $item->get_id();
						$product_id     = ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id();
						$product        = wc_get_product( $product_id );
						$desc           = '';
						$item_price     = ( $item->get_subtotal() && $item->get_quantity() ) ? floatval( $item->get_subtotal() ) / $item->get_quantity() : 0;
						$is_variation   = false;
						$using_pick_obj = empty( $picked_obj ) ? false : true;

						/**
						 * Filter to include or exclude order item from pick list.
						 * 
						 * @since 2.5.8
						 */
						$include_item = apply_filters( 'cas_picklist_include_order_item', true, $item, $product, $order );

						if ( ! $include_item ) {
							continue;
						}
						
						// Make dummy product if not found
						if ( ! $product ) {
							$product = self::cas_make_dummy_product( $item->get_id(), $item );
							// Fallback ID
							$product_id = $item->get_id();
						}

						if ( 'variation' === $product->get_type() ) {
							$desc         = $product->get_attribute_summary();
							$is_variation = true;
						}

						$qty_css    = ( $item->get_quantity() > 1 ) ? 'cas-qty-several' : 'cas-qty-normal';
						$picked_qty = ( is_array( $picked_items ) && array_key_exists( $product_id, $picked_items ) && ! empty( $picked_items[$product_id] ) ) ? $picked_items[$product_id] . '/' . $item->get_quantity() : '';
						
						if ( ! is_array( $picked_items ) && is_array( $pick_status ) ) {
							$picked_qty = ( in_array( $product_id, $pick_status, true) ) ?  $item->get_quantity() . '/' . $item->get_quantity() : ''; 
						}

						// New object to hold picked items (vs 2.4.16)
						if ( $using_pick_obj ) {
							$picked_qty = ( is_array( $picked_obj ) && array_key_exists( $item_id, $picked_obj ) && ! empty( $picked_obj[$item_id] ) ) ? $picked_obj[$item_id] . '/' . $item->get_quantity() : '';
						}
						
						?>
						<tr>
							<td><img src="<?php echo wp_kses( wp_get_attachment_image_url( $product->get_image_id(), 'thumbnail' ), 'post' ); ?>" class="cas-table-image"></td>
							<td style="font-size: 0.9rem;">
								<h3 style="margin-bottom:1px;margin-top:5px;"><?php echo esc_attr( $item->get_name() ); ?></h3>
								
								<?php self::cas_get_order_item_attributes( $item ); ?>
							
								<?php echo esc_attr__( 'SKU:', 'woo-picklist' ); ?> <?php echo esc_attr( $product->get_sku() ); ?> <br/>
								<?php self::cas_get_regular_product_fields( $product ); ?>
								<span class="cas-custom-fields-print">
								<?php foreach ( $custom_fields as $custom ) : ?> 
									<?php 
									if ( empty( get_post_meta( $item->get_product_id(), $custom->slug, true ) ) ) { 
										continue; 
									}
									?>
									<i class="fa-solid fa-circle-dot" style="color:#2271b1;"></i> <?php echo esc_attr( $custom->name ); ?>: <?php echo esc_attr( get_post_meta( $item->get_product_id(), $custom->slug, true ) ); ?><br/>
								<?php endforeach; ?>
								</span>
								<?php if ( ! $hide_categories ) : ?>	
									<?php echo esc_attr__( 'Category:', 'woo-picklist' ); ?> <?php echo wp_kses( wc_get_product_category_list( $product->get_id() ), 'post' ); ?> <br/>
								<?php endif; ?>
								
								<?php 
								if ( $use_barcode ) : 
									$bar_data = self::cas_get_barcode_data( $product );
									?>
									<div class="cas-barcode-print" name="cas-barcode" id="cas-print-barcode-<?php echo esc_attr( $item_id ); ?>" data-barcode-value="<?php echo esc_attr( $bar_data ); ?>"></div>
								<?php endif; ?>	
							</td>
							<?php if ( ! $hide_prices ) : ?>
								<td style="font-size: 1rem;text-align:center;padding-top:15px;"><?php echo wp_kses( wc_price( $item_price ), 'post' ); ?></td>
							<?php endif; ?>
							<td style="font-size: 1.1rem;text-align:center;padding-top:15px;">
								<span class="<?php echo esc_attr( $qty_css ); ?>"><?php echo esc_attr( $item->get_quantity() ); ?></span>
								<div class="cas-print-items-picked" id="print-items-picked-<?php echo esc_attr( $item_id ); ?>" style="margin-top:10px;font-weight:600;color:red;font-family: monospace;">
									<?php echo esc_attr( $picked_qty ); ?>
								</div>
							</td>
							<td style="font-size: 1.1rem;text-align:center;padding-top:15px;"><?php echo esc_attr( $item->get_quantity() * floatval( $product->get_weight() ) . ' ' . $weight_unit ); ?></td>
							<td style="text-align: center;padding-top:15px;">
							<input name="cas-print-input-check" type="text" style="border: 1px solid silver;width:50px;height:50px;background-color:inherit;" disabled="disabled">
							
							<div class="cas-print-checked">
								<i id="cas-pack-check-id-<?php echo esc_attr( $item_id ); ?>" class="fa-solid fa-check" style="<?php echo esc_attr( in_array( $item_id, $pick_status_arr, true ) ? 'display="block"' : 'display:none' ); ?>;"></i>
							</div>
							
							</td>
						</tr>

					<?php endforeach; ?>
					</tbody>
					<tfoot>

					</tfoot>
				</table>
				<p id="cas-packing-message-<?php echo esc_attr( $order->get_id() ); ?>" style="display:none;font-size:1rem;padding-top:32px;">
				</p>
			</div>
		<?php endif; ?>
		

		<!-- PRINT SHIPPING LABEL -->
		<?php if ( ! empty( $order ) && $ship_lbl_enabled ) : ?>
			<?php self::cas_shipping_label_and_dialog( $order->get_id() ); ?>
		<?php endif; ?>
		
		<?php
		
	}

	/**
	 * Make a dummy product object to ensure we have a product object
	 * 
	 * @param int       $product_id Product ID fallback (e.g., item ID).
	 * @param WC_Order_Item_Product $item The order item object.
	 * @return WC_Product
	 */
	private static function cas_make_dummy_product( $product_id, $item ) {

		$product = new WC_Product();

		$product->set_id( $product_id );
		$product->set_name( $item->get_name() );
		$product->set_sku( '' );
		$product->set_price( $item->get_total() / max( 1, $item->get_quantity() ) ); // average unit price
		$product->set_stock_status( 'na' );
		$product->set_stock_quantity( 0 );

		// Attempt to pull weight & dimensions from item meta if available
		$weight  = $item->get_meta( '_weight', true );
		$length  = $item->get_meta( '_length', true );
		$width   = $item->get_meta( '_width', true );
		$height  = $item->get_meta( '_height', true );

		$product->set_weight( $weight ? $weight : 0 );
		$product->set_length( $length ? $length : 0 );
		$product->set_width( $width ? $width : 0 );
		$product->set_height( $height ? $height : 0 );

		$product->set_image_id( 0 );
		$product->is_dummy = true;

		return $product;
	}


	/**
	 * Shipping label dialog
	 */
	private static function cas_shipping_label_and_dialog( $order_id ) {

		if ( empty( $order_id ) ) {
			return;
		}
		?>
		<!-- Shipping Label Content for Print -->
		<div id="shipping-label-content-<?php echo esc_attr( $order_id ); ?>" class="shipping-label" style="display: none;">
			<!-- Shipping Label Content Injected via AJAX -->
		</div>

		<!-- Shipping Label Dialog -->
		<div id="cas-shipping-label-dialog" class="cas-modal-overlay" style="display: none;">
			<div class="cas-modal-content">
				<span class="cas-close-modal" onclick="closeShippingLabelDialog();">&times;</span>
				<h2 class="cas-h2-no-order"><?php echo esc_attr__( 'Shipping Label Options', 'woo-picklist' ); ?></h2>

				<table class="cas-dialog-table">
					<thead>
						<tr>
							<td style="width:37%"></td>
							<td></td>
						</tr>
					</thead>
					<tbody>
						<tr>
							<td><label for="cas-shipping-priority"><?php echo esc_attr__( 'Shipping Priority:', 'woo-picklist' ); ?></label></td>
							<td>
								<select id="cas-shipping-priority">
									<option value="S"><?php echo esc_attr__( 'Standard (S)', 'woo-picklist' ); ?></option>
									<option value="P"><?php echo esc_attr__( 'Priority (P)', 'woo-picklist' ); ?></option>
									<option value="E"><?php echo esc_attr__( 'Economy (E)', 'woo-picklist' ); ?></option>
									<option value="custom"><?php echo esc_attr__( 'Custom', 'woo-picklist' ); ?></option>
								</select>
							</td>
						</tr>
						<tr>
							<td></td>
							<td>
								<input type="text" id="cas-custom-priority" class="cas-hidden" placeholder="<?php echo esc_attr__( 'Enter custom priority', 'woo-picklist' ); ?>">
							</td>
						</tr>
						<tr>
							<td><label><?php echo esc_attr__( 'Shipping Symbols:', 'woo-picklist' ); ?></label></td>
							<td>
								<label><input type="checkbox" id="cas-icon-fragile"> <i class="fa-solid fa-wine-glass-empty"></i> <?php echo esc_attr__( 'Fragile', 'woo-picklist' ); ?></label><br/>
								<label><input type="checkbox" id="cas-icon-keep-dry"> <i class="fa-solid fa-umbrella"></i> <?php echo esc_attr__( 'Keep Dry', 'woo-picklist' ); ?></label><br/>
								<label><input type="checkbox" id="cas-icon-this-side-up"> <i class="fa-solid fa-up-long"></i> <?php echo esc_attr__( 'This Side Up', 'woo-picklist' ); ?></label><br/>
								<label><input type="checkbox" id="cas-icon-snow"> <i class="fa-solid fa-snowflake"></i> <?php echo esc_attr__( 'Keep Cold', 'woo-picklist' ); ?></label>
							</td>
						</tr>
						<tr>
							<td><label for="cas-shipping-weight"><?php echo esc_attr__( 'Total Weight:', 'woo-picklist' ); ?>  <i class="fa-solid fa-circle-info" title="<?php echo esc_attr__( 'Products total weight, add boxing weight.', 'woo-picklist' ); ?>"></i> </label></label></td>
							<td><input type="number" id="cas-shipping-weight" step="0.01" value="1"></td>
						</tr>
						<tr>
							<td><label for="cas-shipping-note"><?php echo esc_attr__( 'Note:', 'woo-picklist' ); ?></label></td>
							<td><input type="text" id="cas-shipping-note" placeholder="<?php echo esc_attr__( 'Enter custom note', 'woo-picklist' ); ?>"></td>
						</tr>
						<tr>
							<td><label for="cas-is-return"><?php echo esc_attr__( 'Return Label?', 'woo-picklist' ); ?> <i class="fa-solid fa-circle-info" title="<?php echo esc_attr__( 'Check this if you want to create return label.', 'woo-picklist' ); ?>"></i> </label></td>
							<td>
								<input type="checkbox" id="cas-is-return">
							</td>
						</tr>
						<tr>
							<td><label for="cas-include-barcode"><?php echo esc_attr__( 'Include Barcode:', 'woo-picklist' ); ?> <i class="fa-solid fa-circle-info" title="<?php echo esc_attr__( 'Carrier tracking number is used to make barcode or QR code.', 'woo-picklist' ); ?>"></i></label></td>
							<td>
								<input type="checkbox" id="cas-include-barcode">
							</td>
						</tr>
						<tr>
							<td></td>
							<td>
								<input type="text" id="cas-barcode-value" class="cas-hidden" placeholder="<?php echo esc_attr__( 'Enter tracking/reference number', 'woo-picklist' ); ?>">
							</td>
						</tr>
						<tr>
							<td><label for="cas-barcode-type"><?php echo esc_attr__( 'Barcode Type:', 'woo-picklist' ); ?> <i class="fa-solid fa-circle-info" title="<?php echo esc_attr__( 'If none is selected, the selected option in settings is used.', 'woo-picklist' ); ?>"></i> </label></td>
							<td>
								<select id="cas-barcode-type" class="cas-hidden">
									<option value=""><?php echo esc_attr__( 'Carrier Barcode Format', 'woo-picklist' ); ?></option>
									<?php foreach ( self::cas_get_shipping_barcode_format() as $key => $value ) : ?>
										<option value="<?php echo esc_attr( $key ); ?>"><?php echo esc_attr( $value ); ?></option>
									<?php endforeach; ?>
								</select>
							</td>
						</tr>
					</tbody>
					<tfoot>
						<tr>
							<td></td>
							<td>
								<button class="button button-primary" onclick="generateShippingLabel(<?php echo esc_attr( $order_id ); ?>);">
									<?php echo esc_attr__( 'Generate Label', 'woo-picklist' ); ?>
								</button>
							</td>
						</tr>
					</tfoot>
				</table>

				<div class="cas-button-container">
					
				</div>
			</div>
		</div>
		<?php
	}

	/**
	 * Get shipping barcode format
	 */
	public static function cas_get_shipping_barcode_format() {
		return array(
			'code128' => __( 'Code 128 (Default for UPS, FedEx, DHL)', 'woo-picklist' ),
			'code39'  => __( 'Code 39 (Older FedEx, some carriers)', 'woo-picklist' ),
			'qr'      => __( 'QR Code (Amazon, some carriers)', 'woo-picklist' ),
		);
	}

	/**
	 * Get grid view
	 */
	private static function cas_get_pick_grid_view( $order, $picked_items, $pick_status, $use_barcode, $picked_obj ) {
		$weight_unit = get_option( 'woocommerce_weight_unit' );
		
		/**
		 * Filter whether to hide prices for the current user.
		 *
		 * This filter checks if the current user should have prices hidden
		 * based on the 'picklist_options' settings in the database.
		 * 
		 * @since vs 2.4.25
		 * 
		 * @param bool $hide Whether to hide prices for the user (default: false).
		 * @return bool
		 */
		$hide_prices = apply_filters( 'cas_pick_hide_prices_for_user', false );
		?>
		<div id="cas-pick-grid" class="cas-grid-container cas-hide">
			<?php
			foreach ( $order->get_items() as $item ) : 
				
				$item_id        = $item->get_id();
				$product_id     = ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id();
				$product        = wc_get_product( $product_id );
				$desc           = '';
				$item_price     = ( $item->get_subtotal() && $item->get_quantity() ) ? floatval( $item->get_subtotal() ) / $item->get_quantity() : 0;
				$is_variation   = false;
				$using_pick_obj = empty( $picked_obj ) ? false : true;
	
				/**
				 * Filter to include or exclude order item from pick list.
				 * 
				 * @since 2.5.8
				 */
				$include_item = apply_filters( 'cas_picklist_include_order_item', true, $item, $product, $order );

				if ( ! $include_item ) {
					continue;
				}

				if ( ! $product ) {

					$product = self::cas_make_dummy_product( $item->get_id(), $item );
					// Fallback ID
					$product_id = $item->get_id();
				
				}

				if ( 'variation' === $product->get_type() ) {
					$desc         = $product->get_attribute_summary();
					$is_variation = true;
				}

				$qty_css    = ( $item->get_quantity() > 1 ) ? 'cas-qty-several' : 'cas-qty-normal';  
				$item_qty   = ( is_array( $picked_items ) && array_key_exists( $product_id, $picked_items ) ) ? $picked_items[$product_id] : 0;
				$picked_qty = ( is_array( $picked_items ) && array_key_exists( $product_id, $picked_items ) && ! empty( $picked_items[$product_id] ) ) ? $picked_items[$product_id] . '/' . $item->get_quantity() : '';  
							

				if ( ! is_array( $picked_items ) && is_array( $pick_status ) ) {
					$item_qty = ( in_array( $product_id, $pick_status, true) ) ? $item->get_quantity() : 0; 
				}

				// New object to hold picked items (vs 2.4.16)
				if ( $using_pick_obj ) {
					$item_qty   = ( is_array( $picked_obj ) && array_key_exists( $item_id, $picked_obj ) ) ? $picked_obj[$item_id] : 0;
					$picked_qty = ( is_array( $picked_obj ) && array_key_exists( $item_id, $picked_obj ) && ! empty( $picked_obj[$item_id] ) ) ? $picked_obj[$item_id] . '/' . $item->get_quantity() : '';
				}

				// Pick completed?
				$pick_done = ( $item_qty === $item->get_quantity() ) ? true : false; 
				?>
				<div class="cas-grid-col">
					<div class="cas-grid-image-box cas-image-container" id="grid-img-<?php echo esc_attr( $item_id ); ?>">
					<?php if ( $pick_done ) : ?>
						<i class="fas fa-check-circle cas-icon"></i>	
					<?php endif; ?>
					<img src="<?php echo wp_kses( wp_get_attachment_image_url( $product->get_image_id(), 'large' ), 'post' ); ?>" class="cas-grid-image" style="cursor:pointer;"
					onclick="caspickItem('<?php echo esc_attr( $item_id ); ?>', '<?php echo esc_attr( $order->get_id() ); ?>', '<?php echo esc_attr( $item->get_quantity() ); ?>');">
					</div>
					<h2 style="margin-bottom:1px;margin-top:5px;margin-bottom:5px;"><a href="<?php echo esc_url( get_admin_url() . 'post.php?post=' . $item->get_product_id() . '&action=edit' ); ?>" target="new" style="text-decoration:none;font-size:1.5rem;"><?php echo esc_attr( $item->get_name() ); ?></a></h2>
					<span style="height: 25px;display:block;">
					<?php if ( '' !== $desc ) : ?>
						<strong class="cas-var-bg"><?php echo esc_attr( $desc ); ?></strong>
					<?php endif; ?>
					</span>
					<h3><?php echo esc_attr__( 'Quantity:', 'woo-picklist' ); ?> <span class="<?php echo esc_attr( $qty_css ); ?>"><?php echo esc_attr( $item->get_quantity() ); ?></span> ( <?php echo esc_attr( $item->get_quantity() * floatval( $product->get_weight() ) . ' ' . $weight_unit ); ?> )
						<span id="grid-items-picked-<?php echo esc_attr( $item_id ); ?>" style="margin-top:10px;font-weight:600;color:red;font-family: monospace;" title="<?php echo esc_attr__( 'Picked items', 'woo-picklist' ); ?>">
							<?php echo esc_attr( $picked_qty ); ?>
						</span>
					</h3>
					<h4>
						<?php echo esc_attr__( 'SKU:', 'woo-picklist' ); ?> <?php echo esc_attr( $product->get_sku() ); ?> | 
						<?php if ( ! $hide_prices ) : ?>
							<?php echo esc_attr__( 'Price:', 'woo-picklist' ); ?> <?php echo wp_kses( wc_price( $item_price ), 'post' ); ?> |
						<?php endif; ?>
						<?php echo esc_attr( $product->get_stock_status() . ': ' . $product->get_stock_quantity() ); ?>
					</h4>	
						<?php 
						if ( $use_barcode ) : 
							$bar_data = self::cas_get_barcode_data( $product );
							?>
							<div class="cas-grid-barcode">
							<div class="cas-barcode" name="cas-barcode" id="cas-grid-barcode-<?php echo esc_attr( $item_id ); ?>" data-barcode-value="<?php echo esc_attr( $bar_data ); ?>"></div>
							</div>
						<?php endif; ?>	
					
					<div>
						<button <?php echo esc_attr( $pick_done ? 'disabled="disabled"' : '' ); ?> data-btn-disabled="<?php echo esc_attr__( 'Done', 'woo-picklist' ); ?>" id="btn-pick-<?php echo esc_attr( $item_id ); ?>" onclick="caspickItem('<?php echo esc_attr( $item_id ); ?>', '<?php echo esc_attr( $order->get_id() ); ?>', '<?php echo esc_attr( $item->get_quantity() ); ?>','<?php echo esc_attr( $item->get_id() ); ?>');" class="cas-btn"><i class="fa-solid fa-circle-check"></i> <?php echo esc_attr__( 'Pick', 'woo-picklist' ); ?></button>
						<button id="btn-unpick-<?php echo esc_attr( $item_id ); ?>" onclick="casUnPickItem('<?php echo esc_attr( $item_id ); ?>', '<?php echo esc_attr( $order->get_id() ); ?>', '<?php echo esc_attr( $item->get_quantity() ); ?>', '<?php echo esc_attr( $item->get_id() ); ?>');" class="cas-ubtn"><i class="fa-solid fa-arrow-rotate-left"></i> <?php echo esc_attr__( 'Undo', 'woo-picklist' ); ?></button>
					</div>
				</div>
			<?php endforeach; ?>
		</div>
		
		<?php
	}

	/**
	 * Get all order item attributes, including custom attributes.
	 */
	public static function cas_get_order_item_attributes( $order_item ) {
		 
		// Initialize an empty array to hold attributes
		$attributes     = array();
		$processed_keys = array(); // Track processed keys to avoid duplicates
	
		// Get all meta data for the order item
		$item_meta_data = $order_item->get_meta_data();
	
		// Loop through each meta data item to find product attributes
		foreach ( $item_meta_data as $meta ) {
			$meta_key   = strtolower( $meta->get_data()['key'] ); // Convert key to lowercase for consistency
			$meta_value = $meta->get_data()['value'];
	
			// Skip internal WooCommerce metadata and duplicates
			if ( strpos( $meta_key, '_' ) === 0 || in_array( $meta_key, $processed_keys ) ) {
				continue;
			}
	
			// Mark this key as processed
			$processed_keys[] = $meta_key;
	
			// Check if meta key is an attribute, either with 'pa_' prefix or another custom attribute
			if ( strpos( $meta_key, 'pa_' ) === 0 || !empty($meta_value) ) {
				// Use wc_attribute_label to get a readable name, if possible
				$attribute_name = strpos( $meta_key, 'pa_' ) === 0 ? wc_attribute_label( str_replace( 'pa_', '', $meta_key ) ) : ucfirst( $meta_key );
	
				// Add attribute name and value to the attributes array
				$attributes[ strtolower( $attribute_name ) ] = $meta_value;
			}
		}
	
		// Display the attributes (Name: Value)
		if ( ! empty( $attributes ) ) {
			$out_string = '<div class="cas-var-bg">';
			foreach ( $attributes as $name => $value ) {
				if ( is_string( $value ) ) { 
					$out_string .= '<i class="fa-solid fa-square-check"></i><strong> ' . esc_html( ucfirst( $name ) ) . ':</strong> ' . esc_html( ucfirst( $value ) ) . '<br/>';
				}
			}
			$out_string .= '</div>';
			echo wp_kses( $out_string, 'post' );
		}
	}
	
	/**
	 * Get the next orders
	 */
	public static function cas_get_next_orders() {

		$orders      = Cas_Pick_Functions::cas_get_unpicked_orders( array(), 7 );
		$date_format = get_option( 'date_format' );
		$time_format = get_option( 'time_format' );
		$url         = admin_url( 'admin.php?page=picklist' );
		$no_orders   = __( 'No more orders to pick.', 'woo-picklist' )
		
		?>
		<!-- The Slider -->
		<div id="cas-next-order-slider" class="cas-order-slider">
			<button id="cas-close-slider"><?php echo esc_attr__( 'Close', 'woo-picklist' ); ?></button>
			<?php 
			if ( empty( $orders ) ) {
				?>
				<div class="cas-slider-order-item">
					<i class="fa-solid fa-circle-info"></i> <?php echo esc_attr( $no_orders ); ?>
				</div>
				<?php
			}
			foreach ( $orders as $order ) : 
				$customer_name = ( null !== $order->get_billing_first_name() && null !== $order->get_billing_last_name() ) ? $order->get_billing_first_name() . ' ' . $order->get_billing_last_name() : '';
				$pick_percent  = $order->get_meta( '_cas_pick_order_status_percent', true );
				?>
				<div class="cas-slider-order-item">
					<a href="<?php echo esc_url( $url . '&tab=single&orderid=' . $order->get_id() . '&tbl=unpicked' ); ?>" title="<?php echo esc_attr__( 'Pick Order', 'woo-picklist' ); ?>"> 
					<span style="font-size:larger;"><?php echo esc_attr__( 'Order', 'woo-picklist' ); ?> #<?php echo esc_attr( $order->get_order_number() ); ?> </span></a><br/>
					<?php echo esc_attr__( 'Items', 'woo-picklist' ); ?>: <strong><?php echo esc_attr( $order->get_item_count() ); ?></strong> ( <?php echo esc_attr__( 'Status', 'woo-picklist' ); ?>: <strong><?php echo esc_attr( $pick_percent > 0 ? $pick_percent : '0' ); ?>%</strong> )<br/> 
					<span class="order-status status-<?php echo esc_attr( $order->get_status() ); ?>" style="margin-top:4px;margin-bottom:4px;"><?php echo esc_attr( wc_get_order_status_name( $order->get_status() ) ); ?></span><br/>
					<?php echo esc_attr__( 'Customer', 'woo-picklist' ); ?>: <?php echo esc_attr( $customer_name ); ?><br/>
					<span style="font-size:smaller;"><?php echo esc_attr__( 'Order Date', 'woo-picklist' ); ?>: <?php echo esc_attr( date_format( new DateTime( $order->get_date_created() ), $date_format . ' ' . $time_format ) ); ?></span>
				</div>
			<?php endforeach; ?>
			
		</div>
		<?php

	}

	
	/**
	 * Format dimensions, area, and volume with quantity for pick list display.
	 *
	 * @param WC_Product $product
	 * @param int        $quantity
	 * @return string HTML
	 */
	public static function cas_get_formatted_product_dimensions_with_quantity( $product, $quantity = 1 ) {
		
		if ( ! $product instanceof WC_Product ) {
			return esc_html__( 'Invalid product object', 'woo-picklist' );
		}

		$l    = (float) $product->get_length();
		$w    = (float) $product->get_width();
		$h    = (float) $product->get_height();
		$qty  = max( 1, (int) $quantity );
		$unit = get_option( 'woocommerce_dimension_unit', 'cm' );

		// Labels
		$dimensions_label = __( 'Dimensions', 'woo-picklist' );
		$area_label       = __( 'Area', 'woo-picklist' );
		$volume_label     = __( 'Volume', 'woo-picklist' );
		$total_label      = __( 'Total', 'woo-picklist' );
		$quantity_label   = __( 'Quantity', 'woo-picklist' );

		$output = [];

		// All three dimensions present → volume
		if ( $l && $w && $h ) {
			// Convert to meters
			$l_m = (float) wc_get_dimension( $l, 'm', $unit );
			$w_m = (float) wc_get_dimension( $w, 'm', $unit );
			$h_m = (float) wc_get_dimension( $h, 'm', $unit );

			// Volume in m³
			$vol_m3_per   = $l_m * $w_m * $h_m;
			$vol_m3_total = $vol_m3_per * $qty;

			// Native cubic unit
			$native_label = $unit . '³';
			switch ( $unit ) {
				case 'mm':
					$vol_native_per   = $vol_m3_per * 1e9;
					$vol_native_total = $vol_m3_total * 1e9;
					break;
				case 'cm':
					$vol_native_per   = $vol_m3_per * 1e6;
					$vol_native_total = $vol_m3_total * 1e6;
					break;
				case 'in':
					$vol_native_per   = $vol_m3_per * pow( 39.37007874, 3 );
					$vol_native_total = $vol_m3_total * pow( 39.37007874, 3 );
					$native_label     = 'in³';
					break;
				case 'yd':
					$vol_native_per   = $vol_m3_per * pow( 1.0936132983, 3 );
					$vol_native_total = $vol_m3_total * pow( 1.0936132983, 3 );
					$native_label     = 'yd³';
					break;
				case 'm':
				default:
					$vol_native_per   = $vol_m3_per;
					$vol_native_total = $vol_m3_total;
					$native_label     = 'm³';
					break;
			}

			// Liters
			$liters_per   = $vol_m3_per * 1000;
			$liters_total = $vol_m3_total * 1000;

			// Output
			$output[] = sprintf(
				'%s: %s × %s × %s %s',
				$dimensions_label,
				wc_format_localized_decimal( $l ),
				wc_format_localized_decimal( $w ),
				wc_format_localized_decimal( $h ),
				esc_html( $unit )
			);
			$output[] = sprintf(
				'%s (1x): %s %s (%.3f m³ ≈ %s L)',
				$volume_label,
				wc_format_localized_decimal( $vol_native_per, 0 ),
				$native_label,
				$vol_m3_per,
				wc_format_localized_decimal( $liters_per, 0 )
			);
			$output[] = sprintf(
				'%s (%dx): %s %s (%.3f m³ ≈ %s L)',
				$total_label,
				$qty,
				wc_format_localized_decimal( $vol_native_total, 0 ),
				$native_label,
				$vol_m3_total,
				wc_format_localized_decimal( $liters_total, 0 )
			);

		// Length & width only → area
		} elseif ( $l && $w ) {
			// Convert to m
			$l_m = (float) wc_get_dimension( $l, 'm', $unit );
			$w_m = (float) wc_get_dimension( $w, 'm', $unit );

			// Area in m²
			$area_m2_per   = $l_m * $w_m;
			$area_m2_total = $area_m2_per * $qty;

			// Native unit²
			$area_native_per   = $l * $w;
			$area_native_total = $area_native_per * $qty;
			$native_label      = $unit . '²';

			$output[] = sprintf(
				'%s: %s × %s %s',
				$dimensions_label,
				wc_format_localized_decimal( $l ),
				wc_format_localized_decimal( $w ),
				esc_html( $unit )
			);
			$output[] = sprintf(
				'%s (1x): %s %s (%.3f m²)',
				$area_label,
				wc_format_localized_decimal( $area_native_per, 0 ),
				$native_label,
				$area_m2_per
			);
			$output[] = sprintf(
				'%s (%dx): %s %s (%.3f m²)',
				$total_label,
				$qty,
				wc_format_localized_decimal( $area_native_total, 0 ),
				$native_label,
				$area_m2_total
			);

		// Only some dimensions → partial display
		} elseif ( $l || $w || $h ) {
			$output[] = sprintf(
				'%s: %s',
				$dimensions_label,
				trim( sprintf( '%s × %s × %s %s', $l, $w, $h, $unit ), ' × ' )
			);
			$output[] = sprintf( '%s: %d', $quantity_label, $qty );
		}

		return nl2br( esc_html( implode( "\n", $output ) ) );
	}


}

