<?php
/**
 * Pick List
 *
 * @package picklist/includes
 */

defined( 'ABSPATH' ) || exit;

use Automattic\WooCommerce\Internal\DataStores\Orders\CustomOrdersTableController;

require_once dirname( __FILE__ ) . '/class-pick-templates.php';
require_once dirname( __FILE__ ) . '/class-pick-functions.php';
require_once dirname( __FILE__ ) . '/class-pick-order.php';
require_once dirname( __FILE__ ) . '/class-pick-unpicked.php';
require_once dirname( __FILE__ ) . '/class-pick-customers.php';

/**
 * Class for admin
 */
class Cas_Pick_Woo {

	/**
	 * Hooks
	 */
	public static function cas_pick_admin_hooks() {

		add_action( 'admin_notices', array( __CLASS__, 'cas_pick_admin_activation_notice_success' ) );
		add_action( 'admin_menu', array( __CLASS__, 'cas_pick_create_admin_menu' ) );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'cas_pick_header_scripts' ) );
		add_filter( 'admin_footer_text', array( __CLASS__, 'cas_pick_set_footer_text' ) );
		add_filter( 'woocommerce_get_settings_pages', array( __CLASS__, 'cas_pick_settings' ) );
		add_action( 'admin_init', array( __CLASS__, 'cas_redirect_unauthorized_users' ) );
		
		//Language support
		add_action( 'plugins_loaded', array( __CLASS__, 'cas_woo_picklist_load_textdomain' ) );

		//Add metabox in edit order
		add_action( 'add_meta_boxes', array( __CLASS__, 'cas_woo_picklist_order_meta_box' ) );
		
		//Custom metadata
		add_action( 'woocommerce_process_shop_order_meta', array( __CLASS__, 'cas_pick_save_custom_order_metadata' ) );
		add_action( 'woocommerce_new_order', array( CAS_PICK_FUNCTIONS, 'cas_new_order_in_store' ), 10, 2 );
		
		//WooCommerce etc
		add_action( 'woocommerce_order_item_meta_end', array( __CLASS__,  'cas_display_custom_meta_on_order_view' ), 10, 4 );
		add_filter( 'woocommerce_admin_order_item_headers', array( __CLASS__, 'cas_add_custom_order_item_column_header' ) );
		add_action( 'woocommerce_admin_order_item_values', array( __CLASS__, 'cas_display_custom_order_item_meta_admin' ), 10, 3 );
		add_filter( 'woocommerce_order_item_display_meta_key', array( __CLASS__, 'cas_woo_picklist_custom_order_item_meta_key' ), 20, 3 );
		add_filter( 'default_hidden_meta_boxes', array( __CLASS__, 'cas_default_on_meta_box' ), 10, 2 );

		//Add meta for custom attributes 
		add_action( 'woocommerce_checkout_create_order_line_item', array( __CLASS__, 'cas_woo_picklist_save_all_attributes_to_order_item_meta' ), 10, 4 );

		//Hide prices from a user / picker
		add_action( 'cas_pick_hide_prices_for_user', array( __CLASS__, 'cas_pick_hide_prices_for_user_callback' ) );

		//WooCommerce Order list
		add_filter( 'woocommerce_admin_order_actions', array( __CLASS__, 'cas_pick_list_add_pick_order_link' ), 10, 2 );
		add_action( 'admin_enqueue_scripts', array( __CLASS__, 'cas_woo_order_pick_list_icon_styles' ) );

		//Templates
		add_action( 'cas_pick_get_main_page', array( CAS_PICK_TEMPLATES, 'cas_pick_template_get_main_page' ) );
		
		//Functions
		add_filter( 'cas_pick_get_orders', array( CAS_PICK_FUNCTIONS, 'cas_get_orders' ) );

		//AJAX
		add_action( 'wp_ajax_pick_update_pick_orderitem_data', array( CAS_PICK_FUNCTIONS, 'cas_update_pick_orderitem_data' ) );
		add_action( 'wp_ajax_nopriv_pick_update_pick_orderitem_data', array( CAS_PICK_FUNCTIONS, 'cas_update_pick_orderitem_data' ) );
		add_action( 'wp_ajax_pick_reset_pick_orderitem_data', array( CAS_PICK_FUNCTIONS, 'cas_reset_pick_order_data' ) );
		add_action( 'wp_ajax_nopriv_pick_reset_pick_orderitem_data', array( CAS_PICK_FUNCTIONS, 'cas_reset_pick_order_data' ) );
		add_action( 'wp_ajax_pick_set_pick_order_priority_data', array( CAS_PICK_FUNCTIONS, 'cas_set_pick_order_priority' ) );
		add_action( 'wp_ajax_nopriv_pick_set_pick_order_priority_data', array( CAS_PICK_FUNCTIONS, 'cas_set_pick_order_priority' ) );
		add_action( 'wp_ajax_pick_add_pick_order_note', array( CAS_PICK_FUNCTIONS, 'cas_pick_add_order_note' ) );
		add_action( 'wp_ajax_nopriv_pick_add_pick_order_note', array( CAS_PICK_FUNCTIONS, 'cas_pick_add_order_note' ) );
		add_action( 'wp_ajax_pick_bulk_update_orderitem_data', array( CAS_PICK_FUNCTIONS, 'cas_pick_bulk_update_items' ) );
		add_action( 'wp_ajax_nopriv_pick_bulk_update_orderitem_data', array( CAS_PICK_FUNCTIONS, 'cas_pick_bulk_update_items' ) );
		add_action( 'wp_ajax_pick_set_order_status_shipped', array( CAS_PICK_FUNCTIONS, 'cas_pick_set_order_status_shipped' ) );
		add_action( 'wp_ajax_nopriv_pick_set_order_status_shipped', array( CAS_PICK_FUNCTIONS, 'cas_pick_set_order_status_shipped' ) );

		add_action( 'wp_ajax_pick_unpick_orderitem_data', array( CAS_PICK_FUNCTIONS, 'cas_unpick_order_item' ) );
		add_action( 'wp_ajax_nopriv_pick_unpick_orderitem_data', array( CAS_PICK_FUNCTIONS, 'cas_unpick_order_item' ) );
		add_action( 'wp_ajax_pick_assign_user_to_pick', array( CAS_PICK_FUNCTIONS, 'cas_assign_user_to_pick_order' ) );
		add_action( 'wp_ajax_nopriv_pick_assign_user_to_pick', array( CAS_PICK_FUNCTIONS, 'cas_assign_user_to_pick_order' ) );
		
		add_action( 'wp_ajax_pick_list_check_changes', array( CAS_PICK_FUNCTIONS, 'cas_pick_list_check_changes' ) );
		add_action( 'wp_ajax_nopriv_pick_list_check_changes', array( CAS_PICK_FUNCTIONS, 'cas_pick_list_check_changes' ) );

		add_action( 'wp_ajax_pick_get_order_tabs_totals', array( CAS_PICK_FUNCTIONS, 'pick_get_order_tabs_totals_callback' ) );
		add_action( 'wp_ajax_nopriv_pick_get_order_tabs_totals', array( CAS_PICK_FUNCTIONS, 'pick_get_order_tabs_totals_callback' ) );

		add_action( 'wp_ajax_pick_update_pick_orderitem_comment_data', array( CAS_PICK_FUNCTIONS, 'pick_update_pick_orderitem_comment_data_callback' ) );
		add_action( 'wp_ajax_nopriv_pick_update_pick_orderitem_comment_data', array( CAS_PICK_FUNCTIONS, 'pick_update_pick_orderitem_comment_data_callback' ) );

		add_action( 'wp_ajax_pick_update_stock_quantity', array( CAS_PICK_FUNCTIONS, 'cas_pick_update_stock_quantity' ) );
		add_action( 'wp_ajax_nopriv_pick_update_stock_quantity', array( CAS_PICK_FUNCTIONS, 'cas_pick_update_stock_quantity' ) );
		
		//Check barcode, product and order
		add_action( 'wp_ajax_pick_list_find_product_by_barcode', array( CAS_PICK_FUNCTIONS, 'cas_pick_list_find_product_by_barcode' ) );
		add_action( 'wp_ajax_nopriv_pick_list_find_product_by_barcode', array( CAS_PICK_FUNCTIONS, 'cas_pick_list_find_product_by_barcode' ) );

		//Change Order status on a pick
		add_action( 'wp_ajax_pick_change_order_status', array( CAS_PICK_FUNCTIONS, 'cas_pick_change_order_status_callback' ) );
		add_action( 'wp_ajax_nopriv_pick_change_order_status', array( CAS_PICK_FUNCTIONS, 'cas_pick_change_order_status_callback' ) );

		//Update using barcode and id
		add_action( 'wp_ajax_pick_list_find_product_by_barcode_id', array( CAS_PICK_FUNCTIONS, 'cas_pick_list_find_product_by_barcode_id' ) );
		add_action( 'wp_ajax_nopriv_pick_list_find_product_by_barcode_id', array( CAS_PICK_FUNCTIONS, 'cas_pick_list_find_product_by_barcode_id' ) );

		//Generate shipping labels
		add_action( 'wp_ajax_cas_generate_shipping_label', array( __CLASS__, 'cas_generate_shipping_label' ) );
		add_action( 'wp_ajax_nopriv_cas_generate_shipping_label', array( __CLASS__, 'cas_generate_shipping_label' ) );
	
		//Add custom meta box to the WooCommerce product edit page
		add_action( 'add_meta_boxes', array( __CLASS__, 'cas_picklist_add_barcode_meta_box' ) );
		
	}

	/**
	 * Add a custom meta box to the WooCommerce product edit page
	 */
	public static function cas_picklist_add_barcode_meta_box() {
		add_meta_box(
			'wbc_product_barcode',
			__( 'Add Barcodes (Pick List)', 'woo-picklist' ),
			'Cas_Pick_Woo::cas_picklist_render_barcode_meta_box',
			'product',
			'side',
			'default'
		);
	}

	/**
	 * Render the barcode input field
	 */
	public static function cas_picklist_render_barcode_meta_box( $post ) {
		$product     = wc_get_product( $post->ID );
		$is_variable = $product && $product->is_type( 'variable' );
		
		?>
		<button type="button" id="picklist-scanner-focus" class="button" style="background-color:transparent;">
			<img id="cas-barcode-icon" src="<?php echo esc_attr( plugins_url( '../assets/images/picklist-scan-icon-small.webp', __FILE__ ) ); ?>" style="width:32px;padding:5px;padding-top:15px;" title="<?php echo esc_attr__( 'Scanner Focus', 'woo-picklist' ); ?>" />
		</button>
		<div style="font-size:smaller;margin-top:5px;margin-bottom:10px;"><?php echo esc_attr__( 'Click Scanner icon to focus on the Global Unique ID field, scan and then update the product.', 'woo-picklist' ); ?></div>
	
		<script>
			jQuery(document).ready(function($) {
				$('#picklist-scanner-focus').on('click', function() {
					// Open the Inventory tab
					$('a[href="#inventory_product_data"]').trigger('click');
	
					// Small delay before focusing on the input field
					setTimeout(function() {
						$('#_global_unique_id').focus();
					}, 300);
				});
			});
		</script>
		<?php if ( $is_variable ) : ?>
			<p><strong><?php echo esc_attr__( 'Variations missing barcode:', 'woo-picklist' ); ?></strong></p>
			<ul id="cas-picklist-missing-barcodes">
				<?php
				$missing_variations = [];
				foreach ( $product->get_children() as $variation_id ) {
					$variation = wc_get_product( $variation_id );
					$barcode   = get_post_meta( $variation_id, '_global_unique_id', true );

					if ( empty( $barcode ) ) {
						$variation_name       = implode( ', ', $variation->get_attributes() );
						$missing_variations[] = [
							'id'   => $variation_id,
							'name' => $variation_name ? $variation_name : __( 'Unnamed Variation', 'woo-picklist' )
						];
					}
				}

				if ( ! empty( $missing_variations ) ) {
					foreach ( $missing_variations as $variation ) {
						echo wp_kses( '<li><a href="#" class="cas-picklist-open-variation" data-id="' . esc_attr( $variation['id'] ) . '"><i class="fa-solid fa-pen-to-square"></i> #' . esc_html( $variation['id'] ) . ' - ' . __( 'Scan / enter barcode', 'woo-picklist' ) . '</a></li>', 'post' );
					}
				} else {
					echo wp_kses( '<li><i class="fa-solid fa-circle-info"></i> ' . __( 'All variations have barcodes.', 'woo-picklist' ) . '</li>', 'post' );
				}
				?>
			</ul>
			<div style="font-size:smaller;margin-top:20px;"><a href="<?php echo esc_attr( admin_url( 'admin.php?page=picklist' ) ); ?>"><?php echo esc_attr__( 'Go To Pick List Orders →', 'woo-picklist' ); ?></a></div>
	

			<script>
				jQuery(document).ready(function($) {
					$('.cas-picklist-open-variation').on('click', function(e) {
						e.preventDefault();

						// Open the "Variations" tab
						$('a[href="#variable_product_options"]').trigger('click');

						var variationID = $(this).data('id');

						setTimeout(function() {
							// Find the correct variation panel
							var variationPanel = $('.woocommerce_variation.wc-metabox').has('input.variable_post_id[value="' + variationID + '"]');

							if (variationPanel.length) {
								// Check if the panel is closed
								if (variationPanel.hasClass('closed')) {
									variationPanel.find('.edit_variation').trigger('click'); // Click "Edit" to open it
								}

								// Wait for animation, then focus on the correct input
								setTimeout(function() {
									// Try targeting the correct input field dynamically
									var inputField = variationPanel.find('input[name^="variable_global_unique_id"]'); 

									if (inputField.length) {
										inputField.focus(); // Set focus
										//console.log('Focused on:', inputField);
									} else {
										console.log('Input field not found!');
									}
								}, 500);
							}
						}, 500);
					});
				});
			</script>
		<?php endif; ?>
		<?php
	}

	/**
	 * Get translations
	 */
	public static function cas_woo_picklist_load_textdomain() {
		//unload_textdomain( 'woo-picklist' );
		load_plugin_textdomain( 'woo-picklist', false, dirname( dirname( plugin_basename( __FILE__ ) ) ) . '/languages' );

	}

	/**
	 * Redirect unauthorized users
	 */
	public static function cas_redirect_unauthorized_users() {

		if ( isset( $_GET['page'] ) && 'woocommerce_page_picklist' === $_GET['page'] ) {
			if ( ! self::cas_user_can_access_picklist() ) {
				wp_safe_redirect( admin_url('admin.php?page=wc-admin') );
				exit;
			}
		}

	}

	/**
	 * Save attributes data for the order items
	 */
	public static function cas_woo_picklist_save_all_attributes_to_order_item_meta( $item, $cart_item_key, $values, $order ) {
		
		$product = $values['data']; // Get the product object

		// Check if the product is a variation (not the parent variable product)
		if ( $product->is_type( 'variation' ) ) {
			// Get all attributes for the variation
			$variation_attributes = $product->get_attributes();

			foreach ( $variation_attributes as $attribute_name => $attribute_value ) {
				// Check if the attribute has a value
				if ( ! empty( $attribute_value ) ) {
					// Add the attribute name and value to the order item metadata
					$item->add_meta_data( wc_attribute_label( $attribute_name ), $attribute_value, true );
				}
			}
		}
	}


	/**
	 * Make custom meta keys to look nice
	 */
	public static function cas_woo_picklist_custom_order_item_meta_key( $display_key, $meta, $item ) {
		
		if ( '_cas_pick_item_comment' === $meta->key ) {
			$display_key = __( 'Pick Comment', 'woo-picklist' );
		}

		return $display_key;
	}

	/**
	 * Create order item column
	 */
	public static function cas_add_custom_order_item_column_header() {
		?>
		<th class="item_custom_field"><?php echo esc_attr__( 'Pick Status', 'woo-picklist' ); ?></th>
		<?php
	}

	/**
	 * Set status if pick completed on order items
	 */
	public static function cas_display_custom_order_item_meta_admin( $_product, $item, $item_id ) {
		
		// Check if it's not a product line item
		if ( ! $item || ! $_product instanceof \WC_Product ) {
			echo '<td class="cas_item_custom_field"></td>'; // Ensure empty cells for non-line items
			return;
		}
	
		$order           = wc_get_order( $item->get_order_id() );
		$pick_status     = $order->get_meta( '_cas_pick_order_status', true );
		$pick_status_arr = ( is_array( $pick_status ) ) ? $pick_status : array();
		$pick_item_id    = ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id();
		$picked_obj      = $order->get_meta( '_cas_pick_order_object_items', true ); // New from vs 2.4.16

		if ( ! empty( $picked_obj ) && is_array( $picked_obj ) ) {
			if ( array_key_exists( $item_id, $picked_obj ) && $picked_obj[$item_id] === $item->get_quantity() ) {
				echo '<td class="cas_item_custom_field" style="font-size:14px;">' . esc_html__( 'DONE', 'woo-picklist' ) . ' <i class="fas fa-check-circle cas-icon"></i> </td>';
			} else {
				echo '<td class="cas_item_custom_field"></td>';
			}
		} else {
			if ( in_array( $pick_item_id, $pick_status_arr, true ) ) {
				echo '<td class="cas_item_custom_field" style="font-size:14px;">' . esc_html__( 'DONE', 'woo-picklist' ) . ' <i class="fas fa-check-circle cas-icon"></i> </td>';
			} else {
				echo '<td class="cas_item_custom_field"></td>';
			}
		}
	
	}

	/**
	 * Add Pick info to my account (customer)
	 */
	public static function cas_display_custom_meta_on_order_view( $item_id, $item, $order, $plain_text ) {
		
		$options    = get_option( 'picklist_options' );
		$use_status = ( ! empty( $options['cas_pick_my_order_statuses'] ) && 'yes' === $options['cas_pick_my_order_statuses'] ) ? true : false;
		
		if ( $use_status ) {

			$pick_status     = $order->get_meta( '_cas_pick_order_status', true );
			$pick_status_arr = ( is_array( $pick_status ) ) ? $pick_status : array();
			$pick_item_id    = ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id();
			$picked_obj      = $order->get_meta( '_cas_pick_order_object_items', true ); // New from vs 2.4.16

			if ( ! empty( $picked_obj ) && is_array( $picked_obj ) ) {
				if ( array_key_exists( $item_id, $picked_obj ) && $picked_obj[$item_id] === $item->get_quantity() ) {
					echo '<br/>' . esc_html__( 'Pick Status: Item Selected', 'woo-picklist' ) . ' <i class="fas fa-check-circle cas-icon"></i>';
				} else {
					echo '<br/>' . esc_html__( 'Pick Status:', 'woo-picklist' );
				}
			} else {
				if ( in_array( $pick_item_id, $pick_status_arr, true ) ) {
					echo '<br/>' . esc_html__( 'Pick Status: Item Selected', 'woo-picklist' ) . ' <i class="fas fa-check-circle cas-icon"></i>';
				} else {
					echo '<br/>' . esc_html__( 'Pick Status:', 'woo-picklist' );
				}
			}
		}

	}

	/**
	 * Pick List meta box on the edit order page
	 */
	public static function cas_woo_picklist_order_meta_box_callback( $object ) {
		
		$order          = ( $object instanceof WP_Post ) ? wc_get_order( $object->ID ) : $object;
		$pick_status    = $order->get_meta( '_cas_pick_order_status', true );
		$pick_percent   = $order->get_meta( '_cas_pick_order_status_percent', true );
		$pick_priority  = $order->get_meta( '_cas_pick_order_priority', true );
		$pick_admin_usr = $order->get_meta( '_cas_pick_admin_user', true );
		$url            = admin_url( 'admin.php?page=picklist' );
		$shipped_count  = ! empty( $order->get_meta( '_cas_pick_order_shipped_count', true ) ) ? intval( $order->get_meta( '_cas_pick_order_shipped_count', true ) ) : 0;
		$shop_managers  = Cas_Pick_Functions::cas_get_pick_list_users();

		?>
		<style>
			progress {
				width: 98%;
				max-width: 400px;
				height : 10px;
				border: 0px solid gray;
				border-radius:50px;
				margin-top: 10px;
			}
			progress::-webkit-progress-bar {
				background-color: silver;
				border-radius:50px;
			}
			progress::-webkit-progress-value {
				background-color: lightgreen;
				padding:5px;
				border-radius:50px;
			}
		</style>
		<input type="hidden" id="woo_pick_nonce" name="woo_pick_nonce" value="<?php echo esc_attr( wp_create_nonce( 'woo_pick_nonce_id' ) ); ?>" />
		<?php echo esc_attr__( 'Pick Progress:', 'woo-picklist' ); ?> <?php echo esc_attr( $pick_percent ? $pick_percent : 0 ); ?>% <br/>
		<progress value="<?php echo esc_attr( $pick_percent ); ?>" max="100"></progress><br/>
		<p style="font-size:medium;">
		<?php if ( 'yes' === $order->get_meta( '_cas_pick_order_shipped', true ) ) : ?> 
			<?php if ( $shipped_count > 1 ) : ?>
				<span style="color:#b32d2e;">
				<?php 
					/* translators: %s: shipment information - x times a shipment is done */
					echo esc_attr( sprintf( __( 'Order is Shipped #%1$d times', 'woo-picklist' ), $shipped_count ) ); 
				?>
				</span>
			<?php else : ?>
				<span style="color:#b32d2e;"><?php echo esc_attr__( 'Order is Shipped', 'woo-picklist' ); ?></span>
			<?php endif; ?>
		<?php endif; ?>
		</p>
		<p style="border-top:1px solid #ddd;padding-top:2px;">
		<?php 
			woocommerce_wp_select( array(
				'id'      => '_cas_pick_order_priority',
				'label'   => __( 'Pick Priority:', 'woo-picklist' ),
				'value'   => $pick_priority,
				'options' => array(
					''        => __( 'Select', 'woo-picklist' ),
					'low'     => __( 'Low', 'woo-picklist' ),
					'medium'  => __( 'Medium', 'woo-picklist' ),
					'high'    => __( 'High', 'woo-picklist' ),
					'on_hold' => __( 'On Hold', 'woo-picklist' ),
				),
				'wrapper_class' => 'form-field-wide'
			) );
		?>
		</p>
		<p class="form-field form-field-wide" style="border-top:1px solid #ddd;padding-top:15px;padding-bottom:5px;">
			<label for="_cas_pick_admin_user"><?php echo esc_attr__( 'Assigned To:', 'woo-picklist' ); ?></label>
			<select name="_cas_pick_admin_user" id="_cas_pick_admin_user" class="wc-enhanced-select">
				<option value=""><?php echo esc_attr__( 'Select', 'woo-picklist' ); ?></option>
				<?php foreach ( $shop_managers as $pick_user ) : ?>
					<option value="<?php echo esc_attr( $pick_user->user_login ); ?>" <?php selected( $pick_admin_usr, $pick_user->user_login ); ?>><?php echo esc_html( $pick_user->display_name ); ?></option>
				<?php endforeach; ?>
			</select>
		</p>
		<p style="border-top:1px solid #ddd;padding-top:10px;">
			<button class="button" type="submit" name="save" value="update"><?php echo esc_attr__( 'Save', 'woo-picklist' ); ?></button>
			<a class="button" href="<?php echo esc_url( $url ); ?>"><?php echo esc_attr__( 'Orders →', 'woo-picklist' ); ?></a>
			<a class="button button-primary" href="<?php echo esc_url( $url . '&tab=single&orderid=' . $order->get_id() ); ?>"><?php echo esc_attr__( 'Pick Order →', 'woo-picklist' ); ?></a>
		</p>
		<?php
	}

	/**
	 * Add meta box in the edit order screen
	 */
	public static function cas_woo_picklist_order_meta_box() {
		
		$screen = wc_get_container()->get( CustomOrdersTableController::class )->custom_orders_table_usage_is_enabled()
		? wc_get_page_screen_id( 'shop-order' )
		: 'shop_order';

		add_meta_box(
			'cas-woo-picklist-order-meta-box',
			__( 'Pick List', 'woo-picklist' ),
			'Cas_Pick_Woo::cas_woo_picklist_order_meta_box_callback',
			$screen,
			'side',
			'default'
		);

	}

	/**
	 * Make sure the metabox for Pick List is visible
	 */
	public static function cas_default_on_meta_box( $hidden_meta_boxes, $screen ) {
		// Target the WooCommerce order edit screen
		if ( 'shop_order' === $screen->id ) {
			// Remove the 'cas-woo-picklist-order-meta-box' from the hidden meta boxes array
			$hidden_meta_boxes = array_diff( $hidden_meta_boxes, array( 'cas-woo-picklist-order-meta-box' ) );
		}
		
		return $hidden_meta_boxes;
	}

	/**
	 * Save custom order data
	 */
	public static function cas_pick_save_custom_order_metadata( $order_id ) {
		
		$nonce = isset( $_REQUEST['woocommerce_meta_nonce'] ) ? sanitize_text_field( $_REQUEST['woocommerce_meta_nonce'] ) : ''; // @codingStandardsIgnoreLine.
		if ( ! wp_verify_nonce( $nonce, 'woocommerce_save_data' ) ) { 
			return;
		}

		if ( isset( $_POST[ '_cas_pick_order_priority' ] ) || isset( $_POST[ '_cas_pick_admin_user' ] ) ) {
			$order = wc_get_order( absint( $order_id ) );
			$order->update_meta_data( '_cas_pick_order_priority', wc_clean( $_POST[ '_cas_pick_order_priority' ] ) );
			$order->update_meta_data( '_cas_pick_admin_user', wc_clean( $_POST[ '_cas_pick_admin_user' ] ) );
			$order->save();
		}
	}

	/**
	 * Add option to pick from order list
	 */
	public static function cas_pick_list_add_pick_order_link( $actions, $order ) {
		
		// Define the action for picking an order
		$actions[ 'cas_pick_order' ] = array(
			'url'    => wp_nonce_url( admin_url( 'admin.php?page=picklist&tab=single&orderid=' . $order->get_id() ), 'cas_pick_order_action' ),
			'name'   => __( 'Pick Order', 'woo-picklist' ),
			'action' => 'cas-pick-icon',
		);

		return $actions;
	}

	/**
	 * Create style for woo order list
	 */
	public static function cas_woo_order_pick_list_icon_styles() {
		echo '<style>.cas-pick-icon::after { font-family: dashicons; content: "\f314"; }</style>';
	}

	/**
	 * Add settings
	 */
	public static function cas_pick_settings( $settings ) {

		$settings[] = include_once dirname( __FILE__ ) . '/class-pick-settings.php';
		return $settings;

	}

	/**
	 * Create menu
	 */
	public static function cas_pick_create_admin_menu() {

		// Check if user has access
		if ( ! self::cas_user_can_access_picklist() ) {
			return;
		}

		add_submenu_page(
			'woocommerce',
			__( 'Pick List', 'woo-picklist' ),
			__( 'Pick List', 'woo-picklist' ),
			'read', //'manage_woocommerce',
			'picklist',
			'cas_pick_main_page',
			3
		);

	}

	/**
	 * Check if the current user has access to Pick List
	 */
	public static function cas_user_can_access_picklist() {
		$user       = wp_get_current_user();
		$user_roles = (array) $user->roles;
		$options    = get_option( 'picklist_options', [] );

		// Get allowed roles from settings
		$allowed_roles = isset( $options['cas_picklist_roles'] ) ? array_values( (array) $options['cas_picklist_roles'] ) : [];

		// Always allow shop managers and administrators
		$default_roles = [ 'administrator', 'shop_manager' ];
		$allowed_roles = array_unique( array_merge( $allowed_roles, $default_roles ) );

		// Check if the user has one of the allowed roles
		return (bool) array_intersect( $user_roles, $allowed_roles );
	}

	/**
	 * Add scripts and style
	 *
	 * @param var $hook object.
	 */
	public static function cas_pick_header_scripts( $hook ) {

		if ( 'woocommerce_page_picklist' !== $hook ) {
			return;
		}

		wp_register_script( 'cas_jquery-ui', 'https://code.jquery.com/ui/1.14.0/jquery-ui.js', array( 'jquery' ), '1.14.0', true );
		wp_enqueue_script( 'cas_jquery-ui' );

		wp_register_script( 'cas_multiselect', plugins_url( '../assets/js/multiselect/jquery.multiselect.js', __FILE__ ), array( 'jquery' ), CAS_WOO_PICKLIST_VS, true );
		wp_enqueue_script( 'cas_multiselect' );

		wp_register_script( 'cas_multiselect_filter', plugins_url( '../assets/js/multiselect/jquery.multiselect.filter.js', __FILE__ ), array( 'jquery' ), CAS_WOO_PICKLIST_VS, true );
		wp_enqueue_script( 'cas_multiselect_filter' );

		wp_register_script( 'cas_barcodejs', plugins_url( '../assets/js/barcode/jquery-barcode.min.js', __FILE__ ), array('jquery'), '2.2', false );
		wp_enqueue_script( 'cas_barcodejs' );

		wp_register_script( 'exportjs', plugins_url( '../assets/js/tabletoexcel/tabletoexcel.min.js', __FILE__ ), array(), '1.0.4', false );
		wp_enqueue_script( 'exportjs' );

		wp_register_script( 'qrcodejs', 'https://cdn.rawgit.com/davidshimjs/qrcodejs/gh-pages/qrcode.min.js', array(), '1', false );
		wp_enqueue_script( 'qrcodejs' );

		wp_register_script( 'html2canvas', 'https://cdnjs.cloudflare.com/ajax/libs/html2canvas/1.4.1/html2canvas.min.js', array(), '1.4.1', false );
		wp_enqueue_script( 'html2canvas' );

		wp_register_script( 'jspdf', 'https://cdnjs.cloudflare.com/ajax/libs/jspdf/2.5.1/jspdf.umd.min.js', array(), '2.5.1', false );
		wp_enqueue_script( 'jspdf' );

		
		wp_enqueue_script( 'select2' ); // WooCommerce includes this by default in the admin
		wp_enqueue_style( 'select2' );  // WooCommerce includes the styles

		if ( 'true' === CAS_WOO_PICKLIST_PROD ) {
			wp_register_script( 'cas_pick_scripts', plugins_url( '../assets/js/picklist.min.js', __FILE__ ), array( 'jquery' ), CAS_WOO_PICKLIST_VS, true );
		} else {
			wp_register_script( 'cas_pick_scripts', plugins_url( '../assets/js/picklist.js', __FILE__ ), array( 'jquery' ), CAS_WOO_PICKLIST_VS, true );
		}

		wp_register_style( 'woocommerce_admin', plugins_url( '../plugins/woocommerce/assets/css/admin.css' ), array(), '1.2.12', 'all' );
		wp_enqueue_style( 'woocommerce_admin' );

		wp_register_style( 'fonta', 'https://use.fontawesome.com/releases/v6.4.2/css/all.css', array(), '6.4.2' );
		wp_enqueue_style( 'fonta' );

		wp_register_style( 'jquery-ui-css', 'https://code.jquery.com/ui/1.14.0/themes/base/jquery-ui.css', array(), '1.14.0', 'all' );
		wp_enqueue_style( 'jquery-ui-css' );

		wp_register_style( 'cas_multiselect_css', plugins_url( '../assets/css/multiselect/jquery.multiselect.css', __FILE__ ), array(), CAS_WOO_PICKLIST_VS );
		wp_enqueue_style( 'cas_multiselect_css' );

		wp_register_style( 'cas_multiselect_filter_css', plugins_url( '../assets/css/multiselect/jquery.multiselect.filter.css', __FILE__ ), array(), CAS_WOO_PICKLIST_VS );
		wp_enqueue_style( 'cas_multiselect_filter_css' );

		wp_register_style( 'cas_picklist_css', plugins_url( '../assets/css/picklist.css', __FILE__ ), array(), CAS_WOO_PICKLIST_VS );
		wp_enqueue_style( 'cas_picklist_css' );

		wp_register_style( 'cas_picklist_dark_css', plugins_url( '../assets/css/picklist-dark.css', __FILE__ ), array(), CAS_WOO_PICKLIST_VS );
		wp_enqueue_style( 'cas_picklist_dark_css' );
		
		// Localize the script with new data
		$options = get_option( 'picklist_options' );

		$obj_data = array(
			'notify_orders'     => $options[ 'cas_pick_noti_orders' ],
			'notify_changes'    => $options[ 'cas_pick_noti_changes' ],
			'auto_refresh'      => $options[ 'cas_pick_auto_refresh' ],
			'select_status'     => __( 'Select Order Statuses', 'woo-picklist' ),
			'select_priority'   => __( 'Select Priority', 'woo-picklist' ),
			'select_progress'   => __( 'Select Progress', 'woo-picklist' ),
			'select_users'      => __( 'Select Assigned User', 'woo-picklist' ),
			'select_customer'   => __( 'Select a Customer', 'woo-picklist' ),
			'barcode_type'      => $options[ 'cas_pick_barcode_type' ],
			'barcode_error'     => __( 'Product not found for this barcode', 'woo-picklist' ),
			'scan_in_only'      => $options[ 'cas_pick_scan_only_in' ],
			'scan_sounds_off'   => $options[ 'cas_pick_scan_sound_on' ],
			'ship_message'      => __( 'Pick is not completed, are you sure you want to ship anyway?', 'woo-picklist' ),
			'select_default'    => __( 'Select', 'woo-picklist' ),
			'sort_default'      => __( 'Sort by...', 'woo-picklist' ),
			'use_dark_mode'     => isset( $options[ 'cas_use_dark_mode' ] ) ? $options[ 'cas_use_dark_mode' ] : '',
			'use_ship_order_qr' => isset( $options[ 'cas_shipping_include_pick_qr' ] ) ? $options[ 'cas_shipping_include_pick_qr' ] : '',
			'ship_label_size'   => isset( $options[ 'cas_shipping_label_paper_size' ] ) ? $options[ 'cas_shipping_label_paper_size' ] : '',
			'currency_symbol' => get_woocommerce_currency_symbol(),
			'currency_pos'    => get_option( 'woocommerce_currency_pos' ), // 'left', 'right', etc.
			'decimal_sep'     => wc_get_price_decimal_separator(),
			'thousand_sep'    => wc_get_price_thousand_separator(),
			'decimals'        => wc_get_price_decimals(),
		);

		wp_localize_script( 'cas_pick_scripts', 'cas_object', $obj_data );
		wp_enqueue_script( 'cas_pick_scripts' );

	}

	/**
	 * On activation notice
	 */
	public static function cas_pick_admin_activation_notice_success() {

		$allowed_tags = array(
			'a' => array(
				'class'  => array(),
				'href'   => array(),
				'target' => array(),
				'title'  => array(),
			),
		);

		/* translators: %s: url for documentation */
		$read_doc = __( 'Read the <a href="%1$s" target="_blank"> extension documentation </a> for more information.', 'woo-picklist' );
		$out_str  = sprintf( $read_doc, esc_url( 'https://docs.woocommerce.com/document/picklist/' ) );

		if ( get_transient( 'cas-pick-admin-notice-activated' ) ) {
			?>
			<div class="updated woocommerce-message">
				<a class="woocommerce-message-close notice-dismiss" href="<?php echo esc_url( wp_nonce_url( add_query_arg( 'wc-hide-notice', 'cas_pick_admin_activation_notice_success' ), 'woocommerce_hide_notices_nonce', '_wc_notice_nonce' ) ); ?>"><?php esc_html_e( 'Dismiss', 'woo-picklist' ); ?></a>
				<p>
					<?php esc_html_e( 'Thank you for installing Pick List for WooCommerce. You can now start working with Pick Lists on orders', 'woo-picklist' ); ?>
					<?php echo wp_kses( $out_str, $allowed_tags ); ?>
				</p>
				<p class="submit">
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=picklist' ) ); ?>" class="button-primary"><?php esc_html_e( 'Pick List', 'woo-picklist' ); ?></a>
					<a href="<?php echo esc_url( admin_url( 'admin.php?page=wc-settings&tab=picklist' ) ); ?>" class="button-secondary"><?php esc_html_e( 'Settings', 'woo-picklist' ); ?></a>
				</p>
			</div>

			<?php
			delete_transient( 'cas-pick-admin-notice-activated' );
		}
	}


	/**
	 * Set footer text.
	 */
	public static function cas_pick_set_footer_text( $text ) {

		$page = filter_input( 1, 'page', FILTER_SANITIZE_FULL_SPECIAL_CHARS );

		if ( 'picklist' === $page ) {
			$img_file = plugins_url( '../assets/images/consortia-100.png', __FILE__ );

			/* translators: %s: url to vendor and logo */ 
			printf( esc_html__( '- developed by %1$s%2$s%3$s', 'woo-picklist' ), '<a href="https://www.consortia.no/en/" target="_blank">', '<img src="' . esc_attr( $img_file ) . '" class="cas-logo">', '</a>' );
			?>
			<a href="https://woocommerce.com/products/picklist/#reviews" target="_blank" title="<?php esc_attr_e( 'Rate this plugin', 'woo-picklist' ); ?>" style="padding-left:4px; text-decoration: none;">
				<?php esc_html_e( 'Love it? Rate it ★★★★★', 'woo-picklist' ); ?>
			</a>
			<?php
		
		} else {

			return $text;

		}

	}

	/**
	 * Check if user can see prices 
	 */
	public static function cas_pick_hide_prices_for_user_callback() {
		
		$options      = get_option( 'picklist_options' );
		$users        = ( isset( $options['cas_pick_hide_price_users'] ) && ! empty( $options['cas_pick_hide_price_users'] ) ) ? $options['cas_pick_hide_price_users'] : array();
		$current_user = wp_get_current_user()->user_login;
		$hide         = false;

		if ( count( $users ) > 0 ) {
			foreach ( $users as $user ) {
				if ( $user === $current_user ) {
					$hide = true;
					break;
				}
			}
		}

		return $hide;

	}

	/**
	 * Generate Shipping label (from ajax)
	 */
	public static function cas_generate_shipping_label() {

		if ( ! isset( $_GET['order_id'] ) ) {
			wp_send_json_error(['message' => 'Order ID is missing.']);
		}
		
		$order_id     = absint( $_GET['order_id'] );
		$order        = wc_get_order( $order_id );
		$priority     = strtoupper( sanitize_text_field( $_GET['priority'] ?? $priority ) );
		$weight       = ! empty( sanitize_text_field( $_GET['weight'] ) ) ? sanitize_text_field( $_GET['weight'] ) : '';
		$barcode      = ! empty( sanitize_text_field( $_GET['barcode'] ) ) ? sanitize_text_field( $_GET['barcode'] ) : '';
		$barcode_type = ! empty( sanitize_text_field( $_GET['barcode_type'] ) ) ? sanitize_text_field( $_GET['barcode_type'] ) : '';
		$order_note   = ! empty( sanitize_text_field( $_GET['order_note'] ) ) ? sanitize_text_field( $_GET['order_note'] ) : '';
		$icons        = ! empty( $_GET['icons'] ) ? json_decode( stripslashes( sanitize_text_field( $_GET['icons'] ) ), true ) : [];
		$is_return    = ! empty( sanitize_text_field( $_GET['is_return'] ) ) ? sanitize_text_field( $_GET['is_return'] ) : '';
		$return       = 'yes' === $is_return;

		if ( ! $order ) {
			wp_send_json_error( ['message' => 'Invalid order.'] );
		}
	
		$label_html = self::cas_shipping_label_template( $order, $priority, $weight, $barcode, $icons, $barcode_type, $order_note, $return );
		
		wp_send_json_success( ['html' => $label_html] );
	}


	/**
	 * Generate Shipping Label (template)
	 */
	public static function cas_shipping_label_template( $order, $priority, $weight, $barcode, $icons, $barcode_type, $order_note, $return ) {
		if ( ! $order instanceof WC_Order ) {
			return;
		}
	
		// Get plugin settings
		$options           = get_option( 'picklist_options', [] );
		$use_order_qr      = isset( $options['cas_shipping_include_pick_qr'] ) && 'yes' === $options['cas_shipping_include_pick_qr'];
		$barcode_format    = ! empty( $barcode_type ) ? $barcode_type : ( isset( $options['cas_shipping_carrier_barcode_format'] ) ? $options['cas_shipping_carrier_barcode_format'] : 'code128' );
		$shipping_priority = $priority; // P = Priority, E = Economy, S = Standard

		// Get WooCommerce Store Address
		$store_address_1 = get_option( 'woocommerce_store_address' );
		$store_address_2 = get_option( 'woocommerce_store_address_2' );
		$store_city      = get_option( 'woocommerce_store_city' );
		$store_postcode  = get_option( 'woocommerce_store_postcode' );
		$store_country   = get_option( 'woocommerce_default_country' );
		//$store_state     = get_option( 'woocommerce_store_state' );
		$weight_unit     = get_option( 'woocommerce_weight_unit' );

		// Get full country & state names (instead of codes)
		$store_country_full = WC()->countries->countries[$store_country] ? WC()->countries->countries[$store_country] : $store_country;
		$store_state_full   = '';

		// Construct Full Store Address
		$store_address = trim( "{$store_address_1} {$store_address_2}, {$store_city}, {$store_state_full} {$store_postcode}, {$store_country_full}" );

		// Get order details
		$order_id         = $order->get_id();
		/* translators: %s: shipping method (e.g. 'UPS', 'DHL', etc.) */
		$shipping_method  = ! $return ? $order->get_shipping_method() : sprintf( __( '%s – Return', 'woo-picklist' ), $order->get_shipping_method() );
		$shipping_company = $order->get_shipping_company() ? $order->get_shipping_company() : '';
		$carrier_logo     = ''; // Placeholder for carrier logo logic
		$shipping_display = ! empty( $shipping_method ) ? $shipping_method : $shipping_company;
	
		// Get recipient details
		$recipient_name     = trim( $order->get_shipping_first_name() . ' ' . $order->get_shipping_last_name() );
		$recipient_address  = trim( $order->get_shipping_address_1() . ' ' . $order->get_shipping_address_2() );
		$recipient_city     = $order->get_shipping_city();
		$recipient_postcode = $order->get_shipping_postcode();
		$recipient_country  = WC()->countries->countries[$order->get_shipping_country()] ?? $order->get_shipping_country();
	
		// Dynamically set carrier logo (optional)
		if ( strpos( strtolower( $shipping_method ), 'ups' ) !== false ) {
			$carrier_logo = plugins_url( '../assets/images/ups-logo.svg', __FILE__ );
		} elseif ( strpos( strtolower( $shipping_method ), 'fedex' ) !== false ) {
			$carrier_logo = plugins_url( '../assets/images/fedex-logo.svg', __FILE__ );
		}

		ob_start();
		?>
		
		<div class="cas-shipping-label">
			<table class="shipping-label-table">
				<!-- Header Row: Carrier Logo + Priority -->
				<tr>
					<td colspan="2" class="carrier-logo">
						<?php if ( ! empty( $carrier_logo ) ) : ?>
							<img src="<?php echo esc_url( $carrier_logo ); ?>" alt="Carrier Logo">
						<?php else : ?>
							<span class="cas-shipping-company"><?php echo esc_attr( $shipping_display ); ?></span>
						<?php endif; ?>
					</td>
					<td class="priority-label" style="text-align:center;">
						<h1><?php echo esc_html( strtoupper( $shipping_priority ) ); ?></h1>
					</td>
				</tr>
	
				<!-- Sender Information -->
				<tr>
					<td colspan="2" class="sender-info">
						<strong><?php echo esc_attr__( 'From:', 'woo-picklist' ); ?></strong><br>
						<div style="text-align:left;">
							<?php if ( $return ) : ?>
								<?php echo esc_html( $recipient_name ); ?><br>
								<?php echo esc_html( $recipient_address ); ?><br>
								<?php echo esc_html( $recipient_city . ', ' . $recipient_postcode ); ?><br>
								<?php echo esc_html( $recipient_country ); ?>
							<?php else : ?>
								<?php echo esc_html( get_bloginfo('name') ); ?><br>
								<span class="sender-address"><?php echo esc_attr( $store_address ); ?></span>
							<?php endif; ?>
						</div>
					</td>
					<td style="text-align:center;">
						<?php if ( in_array( 'snow', $icons ) ) : ?>
							<i class="fa-solid fa-snowflake shipping-icon" title="Keep Cold" style="display:block;"></i>
							<?php echo esc_attr__( 'Keep Cold', 'woo-picklist' ); ?>
						<?php endif; ?>
					</td>
				</tr>
				
				<!-- Recipient Information -->
				<tr>
					<td colspan="2" class="recipient-info">
						<strong class="ship-to"><?php echo esc_attr__( 'Ship To:', 'woo-picklist' ); ?></strong><br>
						<div class="cas-to-address">
							<?php if ( $return ) : ?>
								<?php echo esc_html( get_bloginfo('name') ); ?><br>
								<span class="sender-address"><?php echo esc_attr( $store_address ); ?></span>
							<?php else : ?>
								<?php echo esc_html( $recipient_name ); ?><br>
								<?php echo esc_html( $recipient_address ); ?><br>
								<?php echo esc_html( $recipient_city . ', ' . $recipient_postcode ); ?><br>
								<?php echo esc_html( $recipient_country ); ?>
							<?php endif; ?>
						</div>
					</td>
					<td class="shipping-icons">
						<?php if ( in_array( 'fragile', $icons ) ) : ?>
							<i class="fa-solid fa-wine-glass-empty shipping-icon" title="Fragile"></i>
						<?php endif; ?>
						<?php if ( in_array( 'keep-dry', $icons ) ) : ?>
							<i class="fa-solid fa-umbrella shipping-icon" title="Keep Dry"></i>
						<?php endif; ?>
						<?php if ( in_array( 'this-side-up', $icons ) ) : ?>
							<i class="fa-solid fa-up-long shipping-icon" title="This Side Up"></i>
						<?php endif; ?>
					</td>
				</tr>
	
				<!-- QR Code / Order Number -->
				<tr>
					<td colspan="2">
						<strong><?php echo esc_attr__( 'Order No:', 'woo-picklist' ); ?></strong> <?php echo esc_html( $order->get_order_number() ); ?><br/>
						<strong><?php echo esc_attr__( 'Weight:', 'woo-picklist' ); ?></strong> <?php echo esc_html( $weight . ' ' . $weight_unit ); ?><br/>
						<strong><?php echo esc_attr__( 'Note:', 'woo-picklist' ); ?></strong> <?php echo esc_html( $order_note ); ?>
					</td>
					<td class="qr-code">
						<?php if ( $use_order_qr ) : ?>
							<div id="shipping-label-qr-code-<?php echo esc_attr( $order_id ); ?>" class="cas-shipping-label-qr"></div>
						<?php endif; ?>
					</td>
				</tr>
	
				<!-- Order Weight & Shipping Method -->
				<tr>
					<td colspan="3" class="shipping-method">
						<strong><?php echo esc_attr__( 'Shipping Method:', 'woo-picklist' ); ?></strong> <?php echo esc_html( $shipping_method ); ?>
						<?php if ( $shipping_company ) : ?>
							<br/>
							<strong><?php echo esc_attr__( 'Shipping Company:', 'woo-picklist' ); ?></strong> <?php echo esc_html( $shipping_company ); ?>
						<?php endif; ?>
					</td>
				</tr>

				<!-- Tracking ID and barcode / QR Code for carrier -->
				<tr>
					<td colspan="3" class="barcode" style="border-bottom:none;">
						<?php if ( ! empty( $barcode ) ) : ?>
							<div class="tracking"><?php echo esc_attr__( 'Tracking ID: #', 'woo-picklist' ); ?><?php echo esc_attr( $barcode ); ?></div>
							<div class="cas-shipping-label-barcode">
								<div id="track-<?php echo esc_attr( $order_id ); ?>" data-barcode-value="<?php echo esc_attr( $barcode ); ?>" data-barcode-format="<?php echo esc_attr( $barcode_format ); ?>"></div>
							</div>
						<?php endif; ?>
					</td>
				</tr>

			</table>
		</div>
		

	
		<style>
		.recipient-info {
			height:110px;
			vertical-align:middle;
			border-bottom:none;
		}
		.cas-to-address {
			font-weight: 600;
			font-size:medium;
			padding-top:4px;
		}

		.shipping-icons {
			display: flex;
			justify-content: center; /* Centers horizontally */
			align-items: center; /* Centers vertically */
			gap: 5px; /* Adds spacing between icons */
			padding: 5px;
			height: 110px;
			vertical-align:middle;
			border-bottom:none;
		}

		.shipping-icon {
			font-size: 30px; /* Adjust size as needed */
			margin: 5px;
			color: black; /* Ensure visibility */
		}

		.cas-shipping-company {
			display: flex;
			align-items: center;
			justify-content: center;
			width: 100%;
			text-align: center;
			white-space: normal;      /* allow wrapping for long text */
			overflow-wrap: break-word;
			font-weight: bold;
			line-height: 1.2;
			padding-bottom: 2px;
			/* Responsive font size: min 1rem, ideal 2.5vw, max 2rem */
			font-size: clamp(1rem, 2.5vw, 2rem);
		}

		.cas-shipping-label {
			border: 2px solid #000;
			padding: 10px;
			font-family: Arial, sans-serif;
			color: black;
			display: flex;
			justify-content: center;
			align-items: center;
			overflow: hidden;
		}

		.shipping-label-table {
			width: 100%;
			border-collapse: collapse;
			table-layout: fixed; /* Prevents table from expanding beyond parent */
		}

		.shipping-label-table td {
			border-bottom: 1px solid #000;
			padding: 6px;
			word-wrap: break-word; /* Ensures long text doesn't overflow */
			overflow: hidden;
			text-align: left;
		}

		.carrier-logo img {
			max-height: 80px;
			max-width: 80%;
			display: block;
		}

		.priority-label {
			text-align: center;
			font-size: 22px;
			font-weight: bold;
			vertical-align: middle;
			width: 30%; /* Keep it compact */
			padding-left:25px;
			border-left: 1px solid black;
		}

		.sender-info, .recipient-info, .order-weight, .shipping-method {
			font-size: 14px;
			text-align: left;
		}
		.order-weight, .shipping-method {
			border:none;
		}

		.sender-address {
			font-size: 13px;
			color: #333;
			word-wrap: break-word;
		}

		.qr-code {
			text-align: center;
			padding: 5px;
		}

		/* Center barcode and QR code */
		.cas-shipping-label-barcode {
			display: flex;
			justify-content: center;
			align-items: center;
			text-align: center;
			margin-top: 5px;
			width: 100%;
			border:none;
		}

		/* Tracking number text */
		.tracking {
			text-align: center;
			font-weight: bold;
			font-size: 12px;
			text-transform: uppercase;
			color: black;
			padding-bottom: 5px;
		}

		/* Improve QR Code layout */
		.cas-shipping-label-qr {
			display: flex;
			justify-content: center;
			align-items: center;
			margin-top: 5px;
			width: 100%;
		}

		.cas-shipping-label-qr canvas, 
		.cas-shipping-label-qr img {
			display: block;
			margin: auto;
			max-width: 80px;
			max-height: 80px;
		}

		</style>
	
		<?php
		return ob_get_clean();
	}
	

}
Cas_Pick_Woo::cas_pick_admin_hooks();
