<?php
/**
 * Pick List
 *
 * @package picklist/includes
 */

defined( 'ABSPATH' ) || exit;

const CAS_PICK_FUNCTIONS = 'Cas_Pick_Functions';

/**
 * Class for functions
 */
class Cas_Pick_Functions {

	/**
	 * Get orders
	 */
	public static function cas_get_orders( $orderstatusfilter = array() ) {

		$page_num       = ! empty( filter_input( INPUT_GET, 'paged', FILTER_VALIDATE_INT ) ) ? filter_input( INPUT_GET, 'paged', FILTER_VALIDATE_INT ) : 1;
		$search_str     = ! empty( filter_input( INPUT_GET, 's', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) ) ? filter_input( INPUT_GET, 's', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) : '';
		$order_statuses = wc_get_order_statuses();
		$options        = get_option( 'picklist_options' );
		$ordering       = ! empty( $options['cas_pick_ordering'] ) ? $options['cas_pick_ordering'] : 'DESC';

		//Get order statuses from settings (values)
		if ( ! empty( $options['cas_pick_order_statuses'] ) && is_array( $options['cas_pick_order_statuses'] ) && count( $options['cas_pick_order_statuses'] ) > 0 ) {
			$order_statuses = $options['cas_pick_order_statuses'];
		} else {
			//Use the keys from order statuses
			$order_statuses = array_keys( $order_statuses );
		}

		//Use order status filter if set (values)
		if ( count( $orderstatusfilter ) > 0 ) {
			$order_statuses = $orderstatusfilter;
		}

		/**
		 * Filter to add or remove order statuses
		 * 
		 * @since 2.3.1
		 */
		$order_statuses = apply_filters( 'cas_picklist_order_statuses', $order_statuses );
	
		$args = array(
			'orderby'     => 'date_created',
			'order'       => $ordering,
			'type'        => 'shop_order',
			'post_status' => $order_statuses,
			'paginate'    => true,
			'limit'       => ( ! empty( $options['cas_pick_per_page'] ) ) ? intval( $options['cas_pick_per_page'] ) : 30,
			'page'        => intval( $page_num ),
		);

		//Search in orders
		if ( strlen( $search_str ) > 0 ) {
			$result_arr       = wc_order_search( wc_clean( wp_unslash( $search_str ) ) );
			$args['post__in'] = array_merge( $result_arr, array( 0 ) );
		}

		$date_from = '';
		$date_to   = '';

		//Set date range
		if ( ! empty( $options['cas_pick_order_date_range'] ) ) {

			switch ( $options['cas_pick_order_date_range'] ) {
				case 'week':
					$date_from = strtotime( '-8 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'this_month':
					$date_from = gmdate('Y-m-01');
					$date_to   = strtotime( 'last day of this month' );
					break;
				case 'last_month':
					$date_from = strtotime( 'first day of previous month' );
					$date_to   = strtotime( 'last day of previous month' );
					break;
				case 'last_3_month':
					$date_from = strtotime( '-91 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'last_6_month':
					$date_from = strtotime( '-181 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'last_12_month':
					$date_from = strtotime( '-366 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'year':
					$date_from = strtotime( 'January 01' );
					$date_to   = strtotime( 'tomorrow' );
					break;	
			}

			if ( strlen( $date_from ) > 0 && strlen( $date_to ) > 0 ) {
				$d = array(
					'date_created' => $date_from . '...' . $date_to,
				);
	
				$args = array_merge( $args, $d );
			}
		}

		return wc_get_orders( $args );
		
	}

	/**
	 * Get unpicked orders
	 */
	public static function cas_get_unpicked_orders( $orderstatusfilter = array(), $limit_to = 0 ) {

		$page_num       = ! empty( filter_input( INPUT_GET, 'paged', FILTER_VALIDATE_INT ) ) ? filter_input( INPUT_GET, 'paged', FILTER_VALIDATE_INT ) : 1;
		$search_str     = ! empty( filter_input( INPUT_GET, 's', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) ) ? filter_input( INPUT_GET, 's', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) : '';
		$current_order  = ! empty( filter_input( INPUT_GET, 'orderid', FILTER_VALIDATE_INT ) ) ? filter_input( INPUT_GET, 'orderid', FILTER_VALIDATE_INT ) : '';
		$order_statuses = wc_get_order_statuses();
		$options        = get_option( 'picklist_options' );
		$ordering       = ! empty( $options['cas_pending_pick_ordering'] ) ? $options['cas_pending_pick_ordering'] : 'ASC';

		//Get order statuses from settings (values)
		if ( ! empty( $options['cas_pending_pick_order_statuses'] ) && is_array( $options['cas_pending_pick_order_statuses'] ) && count( $options['cas_pending_pick_order_statuses'] ) > 0 ) {
			$order_statuses = $options['cas_pending_pick_order_statuses'];
		} else {
			//Use the keys from order statuses
			$order_statuses = array_keys( $order_statuses );
		}

		//Use order status filter if set (values)
		if ( count( $orderstatusfilter ) > 0 ) {
			$order_statuses = $orderstatusfilter;
		}

		/**
		 * Filter to add or remove order statuses
		 * 
		 * @since 2.3.1
		 */
		$order_statuses = apply_filters( 'cas_picklist_order_statuses', $order_statuses );
	
		$args = array();

		if ( $limit_to > 0 ) {
			
			$args = array(
				'orderby'      => 'date_created',
				'order'        => $ordering,
				'type'         => 'shop_order',
				'post_status'  => $order_statuses,
				'limit'        => $limit_to,
				'post__not_in' => array( $current_order ),
				'meta_query'   => array(
					'relation' => 'OR',
					array(
						'key'     => '_cas_pick_order_status_percent',
						'value'   => array( '0', '' ),
						'compare' => 'IN',
					),
					array(
						'key'     => '_cas_pick_order_status_percent',
						'compare' => 'NOT EXISTS',
					)
				),
			);
			
			return wc_get_orders( $args );

		} else {

			$args = array(
				'orderby'     => 'date_created',
				'order'       => $ordering,
				'type'        => 'shop_order',
				'post_status' => $order_statuses,
				'paginate'    => true,
				'limit'       => ( ! empty( $options['cas_pending_pick_per_page'] ) ) ? intval( $options['cas_pending_pick_per_page'] ) : 30,
				'page'        => intval( $page_num ),
				'meta_query'  => array(
					'relation' => 'OR',
					array(
						'key'     => '_cas_pick_order_status_percent',
						'value'   => 100,
						'compare' => '!=',
						'type'    => 'NUMERIC',
					),
					array(
						'key'     => '_cas_pick_order_status_percent',
						'compare' => 'NOT EXISTS',
					),
				),
			);

		}

		//Search in orders
		if ( strlen( $search_str ) > 0 ) {
			$result_arr       = wc_order_search( wc_clean( wp_unslash( $search_str ) ) );
			$args['post__in'] = array_merge( $result_arr, array( 0 ) );
		}

		$date_from = '';
		$date_to   = '';

		//Set date range
		if ( ! empty( $options['cas_pending_pick_order_date_range'] ) ) {

			switch ( $options['cas_pending_pick_order_date_range'] ) {
				case 'week':
					$date_from = strtotime( '-8 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'this_month':
					$date_from = gmdate('Y-m-01');
					$date_to   = strtotime( 'last day of this month' );
					break;
				case 'last_month':
					$date_from = strtotime( 'first day of previous month' );
					$date_to   = strtotime( 'last day of previous month' );
					break;
				case 'last_3_month':
					$date_from = strtotime( '-91 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'last_6_month':
					$date_from = strtotime( '-181 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'last_12_month':
					$date_from = strtotime( '-366 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'year':
					$date_from = strtotime( 'January 01' );
					$date_to   = strtotime( 'tomorrow' );
					break;	
			}

			if ( strlen( $date_from ) > 0 && strlen( $date_to ) > 0 ) {
				$d = array(
					'date_created' => $date_from . '...' . $date_to,
				);
	
				$args = array_merge( $args, $d );
			}
		}
		
		return wc_get_orders( $args );
		

	}

	
	/**
	 * Get customer orders
	 *
	 * @param string|int $customer_id_or_email Customer ID or email.
	 * @param array $order_statuses Order statuses to filter by.
	 * @return array List of customer orders with pagination.
	 */
	public static function cas_get_customer_orders( $customer_id_or_email = '', $order_statuses = array() ) {
		$page_num        = filter_input( INPUT_GET, 'paged', FILTER_VALIDATE_INT ) ? filter_input( INPUT_GET, 'paged', FILTER_VALIDATE_INT ) : 1;
		$search_str      = trim( filter_input( INPUT_GET, 's', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) );
		$customer_select = trim( filter_input( INPUT_GET, 'customer_email', FILTER_SANITIZE_FULL_SPECIAL_CHARS ) );

		if ( ! empty( $customer_select ) ) {
			$customer_id_or_email = $customer_select;
		}

		if ( empty( $customer_id_or_email ) && empty( $search_str ) ) {
			return array(
				'orders'      => array(),
				'total_pages' => 0,
				'total'       => 0,
			);
		}

		$args = array(
			'type'        => 'shop_order',
			'post_status' => ! empty( $order_statuses ) ? $order_statuses : array_keys( wc_get_order_statuses() ),
			'paginate'    => true,
			'page'        => $page_num,
			'limit'       => 40,
		);

		
		// Use customer ID or email directly
		if ( is_numeric( $customer_id_or_email ) && $customer_id_or_email > 0 ) {
			$args['customer_id'] = intval( $customer_id_or_email );
		} elseif ( is_email( $customer_id_or_email ) ) {
			$args['billing_email'] = strtolower( sanitize_email( $customer_id_or_email ) );
		}

		// Search
		if ( ! empty( $search_str ) ) {
			$result_arr       = wc_order_search( wc_clean( wp_unslash( $search_str ) ) );
			$args['post__in'] = array_merge( $result_arr, array( 0 ) );
		}

		$orders_data = wc_get_orders( $args );

		return array(
			'orders'      => $orders_data->orders,
			'total_pages' => $orders_data->max_num_pages,
			'total'       => $orders_data->total,
		);
	}

	/**
	 * Get unique customers dropdown
	 *
	 * @return string HTML select element with unique customer emails and names.
	 */
	public static function cas_get_unique_customers_dropdown() {
		 $args = array(
			'role__in' => array( 'customer', 'subscriber' ),
			'orderby'  => 'display_name',
			'order'    => 'ASC',
			'fields'   => array( 'ID', 'display_name', 'user_email' ),
		);

		$users = get_users( $args );

		$no_customers_found = __( 'No customers found', 'woo-picklist' );

		 if ( empty( $users ) ) {
			 return wp_kses( '<select><option>' . $no_customers_found . '</option></select>', array(
				'select' => array(),
				'option' => array()
			 ) );
		 }

		$html  = '<select name="customer_email" id="cas_customer_email_select">';
		$html .= '<option value="">' . esc_html__( 'Select a customer', 'woo-picklist' ) . '</option>';

		 foreach ( $users as $user ) {
			 $email = esc_attr( $user->user_email );
			 $name  = esc_html( $user->display_name );
			 $html .= '<option value="' . $email . '">' . $name . ' (' . $email . ')</option>';
		 }

		$html .= '</select>';

		echo wp_kses( $html, array(
			'select' => array( 'name' => array(), 'id' => array() ),
			'option' => array( 'value' => array() ),
		) );

	}


	/**
	 * Get orders - for order totals
	 */
	private static function cas_get_orders_for_totals() {

		$order_statuses = wc_get_order_statuses();
		$options        = get_option( 'picklist_options' );
		$ordering       = ! empty( $options['cas_pick_ordering'] ) ? $options['cas_pick_ordering'] : 'DESC';

		//Get order statuses from settings (values)
		if ( ! empty( $options['cas_pick_order_statuses'] ) && is_array( $options['cas_pick_order_statuses'] ) && count( $options['cas_pick_order_statuses'] ) > 0 ) {
			$order_statuses = $options['cas_pick_order_statuses'];
		} else {
			//Use the keys from order statuses
			$order_statuses = array_keys( $order_statuses );
		}

		/**
		 * Filter to add or remove order statuses
		 * 
		 * @since 2.3.1
		 */
		$order_statuses = apply_filters( 'cas_picklist_order_statuses', $order_statuses );
	
		$args = array(
			'orderby'     => 'date_created',
			'order'       => $ordering,
			'type'        => 'shop_order',
			'post_status' => $order_statuses,
			'limit'       => -1,
		);

		$date_from = '';
		$date_to   = '';

		//Set date range
		if ( ! empty( $options['cas_pick_order_date_range'] ) ) {

			switch ( $options['cas_pick_order_date_range'] ) {
				case 'week':
					$date_from = strtotime( '-8 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'this_month':
					$date_from = gmdate('Y-m-01');
					$date_to   = strtotime( 'last day of this month' );
					break;
				case 'last_month':
					$date_from = strtotime( 'first day of previous month' );
					$date_to   = strtotime( 'last day of previous month' );
					break;
				case 'last_3_month':
					$date_from = strtotime( '-91 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'last_6_month':
					$date_from = strtotime( '-181 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'last_12_month':
					$date_from = strtotime( '-366 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'year':
					$date_from = strtotime( 'January 01' );
					$date_to   = strtotime( 'tomorrow' );
					break;	
			}

			if ( strlen( $date_from ) > 0 && strlen( $date_to ) > 0 ) {
				$d = array(
					'date_created' => $date_from . '...' . $date_to,
				);
	
				$args = array_merge( $args, $d );
			}
		}

		return wc_get_orders( $args );
		
	}

	/**
	 * Get unpicked orders for totals
	 */
	private static function cas_get_unpicked_orders_for_totals() {

		$order_statuses = wc_get_order_statuses();
		$options        = get_option( 'picklist_options' );
		$ordering       = ! empty( $options['cas_pending_pick_ordering'] ) ? $options['cas_pending_pick_ordering'] : 'ASC';

		//Get order statuses from settings (values)
		if ( ! empty( $options['cas_pending_pick_order_statuses'] ) && is_array( $options['cas_pending_pick_order_statuses'] ) && count( $options['cas_pending_pick_order_statuses'] ) > 0 ) {
			$order_statuses = $options['cas_pending_pick_order_statuses'];
		} else {
			//Use the keys from order statuses
			$order_statuses = array_keys( $order_statuses );
		}

		/**
		 * Filter to add or remove order statuses
		 * 
		 * @since 2.3.1
		 */
		$order_statuses = apply_filters( 'cas_picklist_order_statuses', $order_statuses );
	
		$args = array(
			'orderby'     => 'date_created',
			'order'       => $ordering,
			'type'        => 'shop_order',
			'post_status' => $order_statuses,
			'limit'       => -1,
			'meta_query'  => array(
				'relation' => 'OR',
				array(
					'key'     => '_cas_pick_order_status_percent',
					'value'   => '100',
					'compare' => '!=', 
					'type'    => 'NUMERIC',
				),
				array(
					'key'     => '_cas_pick_order_status_percent',
					'compare' => 'NOT EXISTS',
				),
				array(
					'key'     => '_cas_pick_order_status_percent',
					'value'   => '',
					'compare' => '=',
				)
			),
		);

		$date_from = '';
		$date_to   = '';

		//Set date range
		if ( ! empty( $options['cas_pending_pick_order_date_range'] ) ) {

			switch ( $options['cas_pending_pick_order_date_range'] ) {
				case 'week':
					$date_from = strtotime( '-8 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'this_month':
					$date_from = gmdate('Y-m-01');
					$date_to   = strtotime( 'last day of this month' );
					break;
				case 'last_month':
					$date_from = strtotime( 'first day of previous month' );
					$date_to   = strtotime( 'last day of previous month' );
					break;
				case 'last_3_month':
					$date_from = strtotime( '-91 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'last_6_month':
					$date_from = strtotime( '-181 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'last_12_month':
					$date_from = strtotime( '-366 days' );
					$date_to   = strtotime( 'tomorrow' );
					break;
				case 'year':
					$date_from = strtotime( 'January 01' );
					$date_to   = strtotime( 'tomorrow' );
					break;	
			}

			if ( strlen( $date_from ) > 0 && strlen( $date_to ) > 0 ) {
				$d = array(
					'date_created' => $date_from . '...' . $date_to,
				);
	
				$args = array_merge( $args, $d );
			}
		}
		
		return wc_get_orders( $args );
		

	}

	/**
	 * Get stats for the order or pending orders tab
	 */
	public static function pick_get_order_tabs_totals_callback( $data ) {

		if ( ! check_ajax_referer( 'cas_pick_id', 'nonce' ) ) {
			wp_die();
		}
		
		$data             = wp_unslash( $_POST );
		$total            = 0;
		$items            = 0;
		$average          = 0;
		$orders           = 0;
		$shipping         = 0;
		$average_shipping = 0;

		$orderlines = array();

		if ( 'orders' === $data['query'] ) {

			if ( get_transient( 'cas_pick_totals_tab_orders' ) ) {
				wp_send_json( get_transient( 'cas_pick_totals_tab_orders' ) );
				wp_die();
			}
			
			$orderlines = self::cas_get_orders_for_totals();

		} else {
			
			if ( get_transient( 'cas_pick_totals_tab_pending' ) ) {
				wp_send_json( get_transient( 'cas_pick_totals_tab_pending' ) );
				wp_die();
			}

			$orderlines = self::cas_get_unpicked_orders_for_totals();
			
		}

		foreach ( $orderlines as $order ) {
			$total    += $order->get_total();
			$items    += $order->get_item_count();
			$shipping += $order->get_shipping_total();
		}

		$orders = count( $orderlines );

		if ( $orders > 0 ) {
			$average          = $total / $orders;
			$average_shipping = $shipping / $orders;
		}

		$return_arr = array(
			'total'            => wc_price( $total ),
			'items'            => $items,
			'average'          => wc_price( $average ),
			'orders'           => $orders,
			'shipping'         => wc_price( $shipping ),
			'average_shipping' => wc_price( $average_shipping ),
		);

		if ( 'orders' === $data['query'] ) {
			set_transient( 'cas_pick_totals_tab_orders', $return_arr, 5 * MINUTE_IN_SECONDS );
		} else {
			set_transient( 'cas_pick_totals_tab_pending', $return_arr, 5 * MINUTE_IN_SECONDS );
		}

		wp_send_json( $return_arr );
		wp_die();
	}

	/**
	 * Update order priority
	 */
	public static function cas_set_pick_order_priority( $data ) {
		
		if ( ! check_ajax_referer( 'cas_pick_id', 'nonce' ) ) {
			wp_die();
		}
		
		$data  = wp_unslash( $_POST );
		$order = wc_get_order( intval( $data['orderid'] ) );
	
		$order->update_meta_data( '_cas_pick_order_priority', $data['priority'] );
		$order->save();

		wp_die();
	}

	/**
	 * Get select dropdown with users that can be assigned to a pick
	 */
	public static function cas_get_pick_list_shop_managers( $selected_user = '', $order_id = '' ) {

		$shop_managers = self::cas_get_pick_list_users();
		asort( $shop_managers ); // Sort ASC
		
		?>
		<select name="pick_assigned_user" class="cas-pick-user-select" data-order-id="<?php echo esc_attr( $order_id ); ?>">
		<option value=""><?php echo esc_attr__( 'Select', 'woo-picklist' ); ?></option>
		<?php foreach ( $shop_managers as $pick_user ) : ?>
			<option value="<?php echo esc_attr( $pick_user->user_login ); ?>" <?php echo esc_attr( $selected_user === $pick_user->user_login ? 'selected="selected"' : '' ); ?>><?php echo esc_html( $pick_user->display_name ); ?></option>
		<?php endforeach; ?>
		</select>

		<?php
	}

	/**
	 * Get users that can be assigned to a pick
	 */
	public static function cas_get_pick_list_users() {

		$shop_managers = ( false !== get_transient( 'cas_pick_list_shop_managers' ) && get_transient( 'cas_pick_list_shop_managers' ) ) ? get_transient( 'cas_pick_list_shop_managers' ) : '';
		
		if ( '' === $shop_managers ) {

			$options       = get_option( 'picklist_options', [] );
			$allowed_roles = isset( $options['cas_picklist_roles'] ) ? array_values( (array) $options['cas_picklist_roles'] ) : [];
			$default_roles = [ 'administrator', 'shop_manager' ];
			$allowed_roles = array_unique( array_merge( $allowed_roles, $default_roles ) );
			
			$shop_managers = get_users( array(
				'role__in' => (array) $allowed_roles,
			));
			set_transient( 'cas_pick_list_shop_managers', $shop_managers, 5 * MINUTE_IN_SECONDS );
		}

		return $shop_managers;
	}

	/**
	 * Assign user to pick order
	 */
	public static function cas_assign_user_to_pick_order( $data ) {
		
		if ( ! check_ajax_referer( 'cas_pick_id', 'nonce' ) ) {
			wp_die();
		}
		
		$data      = wp_unslash( $_POST );
		$order     = wc_get_order( intval( $data['orderid'] ) );
		$selected  = $data['user'];
		$usr_login = wp_get_current_user()->user_login;

		if ( '' !== $selected ) {
			$usr_login = $selected;
		}

		$order->update_meta_data( '_cas_pick_admin_user', $usr_login );
		$order->save();

		set_transient( 'cas_notice_pick_order_changed', array( 'changed' => true ) );

		$return_arr = array(
			'key' => $usr_login,
		);

		wp_send_json( $return_arr );
		wp_die();

	}

	/**
	 * Get status priority color
	 */
	public static function cas_get_priority_status_color( $priority ) {
		
		$priority_color = 'silver';

		switch ( $priority ) {
			case 'low':
				$priority_color = 'cornflowerblue';
				break;
			case 'medium':
				$priority_color = 'orange';
				break;
			case 'high':
				$priority_color = 'red';
				break;
			case 'on_hold':
				$priority_color = 'black';
				break;
		}

		return $priority_color;
	}

	/**
	 * Get customer role
	 */
	public static function cas_get_customer_role( $customer_id ) {
		
		$roles = wp_roles()->get_names();
		$user  = get_userdata( $customer_id );
		$role  = '';
		
		if ( '' !== $user ) {
			$role = translate_user_role( $roles[ implode( '', $user->roles ) ] );
		}

		return $role;

	}

	/**
	 * Get barcode on products with optimized transient caching.
	 */
	private static function cas_get_product_data_field_value( $item_id ) {
		
		$options_transient_key = 'cas_picklist_product_options';
		$cached_options        = get_transient( $options_transient_key );
		
		// If the options were not found in the cache, fetch and cache them.
		if ( false === $cached_options ) {

			$options        = get_option( 'picklist_options' );
			$cached_options = array(
				'custom_field'  => ( ! empty( $options['cas_pick_product_c1'] ) && strlen( $options['cas_pick_product_c1'] ) > 0 ) ? $options['cas_pick_product_c1'] : '',
				'product_field' => ( ! empty( $options['cas_pick_product_c1_alt'] ) && strlen( $options['cas_pick_product_c1_alt'] ) > 0 ) ? $options['cas_pick_product_c1_alt'] : '',
			);
			
			set_transient( $options_transient_key, $cached_options, 30000 );
		}

		// Use the cached options.
		$custom_field  = $cached_options['custom_field'];
		$product_field = $cached_options['product_field'];
		$code          = '';

		if ( ! empty( $custom_field ) ) {
			$code = wc_get_product( intval( $item_id ) )->get_meta( $custom_field );
		} else {
			switch ( $product_field ) {
				case 'id':
					$code = $item_id;
					break;
				case 'sku':
					$code = wc_get_product( intval( $item_id ) )->get_sku();
					break;
			}
		}

		return $code;
	}

	/**
	 * Check barcode on order
	 */
	public static function cas_pick_list_find_product_by_barcode( $data ) {
		
		check_ajax_referer( 'cas_pick_id', 'nonce' );

		$data           = wp_unslash( $_POST );
		$barcode        = $data['barcode'];
		$orderId        = $data['order'];
		$order          = wc_get_order( intval( $orderId ) );
		$items          = $order->get_meta( '_cas_pick_order_status', true );
		$items_qty_arr  = $order->get_meta( '_cas_pick_order_status_qty', true );
		$percent        = $order->get_meta( '_cas_pick_order_status_percent', true );
		$item_arr       = is_array( $items ) ? $items : array();
		$is_picked      = false;
		$qty            = 0;
		$barcode_item   = '';
		$success        = false;
		$already_picked = false;
		$picked_qty     = 0;
		
		foreach ( $order->get_items() as $item ) {
			$item_id = ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id();
			$code    = self::cas_get_product_data_field_value( $item_id );

			if ( empty( $code ) ) {
				break;
			}

			if ( $code === $barcode ) {

				$success      = true;
				$barcode_item = $item_id;
				$qty          = $item->get_quantity();

				//Only allow add when barcode scanning
				if ( ! in_array( $item_id, $item_arr ) ) {

					//Add item to pick order
					array_push( $item_arr, $item_id );
					
					if ( ! is_array( $items_qty_arr ) ) {
						$items_qty_arr = array();
					}

					$update_order_item = true;
					
					// Check if the item's quantity is greater than 1.
					if ( $qty > 1 ) {
						// Check if the item has already been scanned.
						if ( array_key_exists( $item_id, $items_qty_arr ) ) {
							// Increment the scan count for the item.
							$items_qty_arr[$item_id] += 1;
						} else {
							// If this is the first scan, initialize the scan count for the item.
							$items_qty_arr[$item_id] = 1;
						}

						// Set picked_qty to the current scan count for the item.
						$picked_qty = $items_qty_arr[$item_id];
						
						if ( $items_qty_arr[$item_id] >= $qty ) {
							$update_order_item = true; // All quantities have been scanned.
						} else {
							$update_order_item = false; // More scans needed.
						}
					} else {
						// For items with a quantity of 1, just consider them fully picked.
						$picked_qty        = 1;
						$update_order_item = true; // Item's quantity is 1 and considered fully picked.
					}

					$order->update_meta_data( '_cas_pick_order_status_qty', $items_qty_arr );

					//Update
					if ( $update_order_item ) {
						//Percent
						$percent = ( count( $item_arr ) > 0 ) ? number_format( count( $item_arr ) / intval( count( $order->get_items() ) ) * 100, 2 ) : 0;
						// Update order meta data
						$order->update_meta_data( '_cas_pick_order_status', $item_arr );
						// Update percent
						$order->update_meta_data( '_cas_pick_order_status_percent', strval( $percent ) );
					}

					//Add more info
					$order = self::cas_add_partial_pick_information( $order, $percent );

					//Save
					$order->save();

					$is_picked = true;

				} else {
					$already_picked = true;
				}

				break;
			}
			
		}

		$return_arr = array(
			'isPicked'  => $is_picked,
			'itemId'    => $barcode_item,
			'orderId'   => intval( $orderId ),
			'success'   => $success,
			'percent'   => $percent,
			'isDone'    => $already_picked,
			'item_qty'  => $picked_qty,
			'sum_qty'   => $qty,
		);

		wp_send_json( $return_arr );
		wp_die();
	}

	/**
	 * Save comment on order item
	 */
	public static function pick_update_pick_orderitem_comment_data_callback( $data ) {
		check_ajax_referer( 'cas_pick_id', 'nonce' );

		$data    = wp_unslash( $_POST );
		$comment = $data['comment'];
		$orderId = $data['orderId'];
		$itemId  = $data['itemId'];
		$order   = wc_get_order( intval( $orderId ) );
		$item    = $order->get_item( $itemId ); 

		$item->update_meta_data( '_cas_pick_item_comment', $comment );
		$item->save();

		wp_die();
	}

	/**
	 * Check barcode on order - NEW
	 */
	public static function cas_pick_list_find_product_by_barcode_id( $data ) {
		
		check_ajax_referer( 'cas_pick_id', 'nonce' );

		$data           = wp_unslash( $_POST );
		$barcode        = $data['barcode'];
		$orderId        = $data['order'];
		$order_item_id  = $data['itemId'];
		$scan_all       = $data['scanAll'];
		$order          = wc_get_order( intval( $orderId ) );
		$items          = $order->get_meta( '_cas_pick_order_status', true );
		$items_qty_arr  = $order->get_meta( '_cas_pick_order_status_qty', true );
		$percent        = $order->get_meta( '_cas_pick_order_status_percent', true );
		$picked_obj     = $order->get_meta( '_cas_pick_order_object_items', true ); // New from vs 2.4.16
		$item_arr       = is_array( $items ) ? $items : array();
		$is_picked      = false;
		$qty            = 0;
		$success        = false;
		$already_picked = false;
		$picked_qty     = 0;
		$input_qty      = ! empty( $data['qty'] ) ? intval( $data['qty'] ) : 1;
		$use_picked_obj = false; // Check if save to new object
				
		$item = $order->get_item( $order_item_id ); 

		if ( ! empty( $item ) ) {

			$item_id = ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id();
			$success = true;
			$qty     = $item->get_quantity();

			if ( ! is_array( $items_qty_arr ) ) {
				$items_qty_arr  = array();
				$item_id        = $order_item_id;
				$use_picked_obj = true;
			}

			if ( ! empty( $picked_obj ) ) { // Use new object for saving
				$items_qty_arr  = $picked_obj;
				$item_id        = $order_item_id;
				$use_picked_obj = true;
			}

			// Manual qty is added
			if ( 'yes' !== $scan_all && $input_qty < $qty ) {
				$current_picked_qty        = isset( $items_qty_arr[ $item_id ] ) ? intval( $items_qty_arr[ $item_id ] ) : 0;
				$to_add                    = min( $input_qty + $current_picked_qty, $qty );
				$items_qty_arr[ $item_id ] = $to_add;
				$already_picked            = $current_picked_qty >= $qty;
			} else {
				$items_qty_arr[ $item_id ] = $qty;
			}


			if ( ! $already_picked ) {
				
				//Add if fully picked
				if ( ( ! in_array( $item_id, $item_arr ) ) && ( intval( $items_qty_arr[$item_id] ) === $qty ) ) {
					array_push( $item_arr, $item_id );
				} 

				// Set picked_qty to the current scan count for the item.
				$picked_qty = $items_qty_arr[$item_id];
				
				if ( $use_picked_obj ) {
					$order->update_meta_data( '_cas_pick_order_object_items', $items_qty_arr ); // New object for saving
				} else {
					$order->update_meta_data( '_cas_pick_order_status_qty', $items_qty_arr );
				}

				$is_picked = true;
			}

		}

		$picked_count = 0;
		$picked_items = $items_qty_arr;
		
		if ( is_array( $picked_items ) ) {
			foreach ( array_values( $picked_items ) as $value ) {
				if ( ! empty( $value ) ) {
					$picked_count += intVal( $value );
				}	
			}
		}

		/**
		 * Get filtered order item count
		 * 
		 * @since 2.5.8
		 */
		$order_item_count = self::cas_get_filtered_item_count( $order );

		// Update
		$percent = ( ! empty( $order_item_count ) && $picked_count > 0 ) ? ( $picked_count / $order_item_count ) * 100 : 0;
		// Update order meta data
		$order->update_meta_data( '_cas_pick_order_status', $item_arr );
		// Update percent
		$order->update_meta_data( '_cas_pick_order_status_percent', strval( number_format( round( $percent ), 2 ) ) );
		// Add more info
		$order = self::cas_add_partial_pick_information( $order, $percent );

		// Save
		$order->save();

			
		$return_arr = array(
			'isPicked'  => $is_picked,
			'itemId'    => $order_item_id, //$barcode_item,
			'orderId'   => intval( $orderId ),
			'success'   => $success,
			'percent'   => $percent,
			'isDone'    => $already_picked,
			'item_qty'  => $picked_qty,
			'sum_qty'   => $qty,
			'count_tot' => $picked_count . '/' . $order_item_count,
		);

		wp_send_json( $return_arr );
		wp_die();
	}

	/**
	 * Add partial order information
	 */
	private static function cas_add_partial_pick_information( $order, $percent ) {

		$options     = get_option( 'picklist_options' );
		$user        = $order->get_meta( '_cas_pick_admin_user', true );
		$disable_msg = ( ! empty( $options['cas_pick_disable_customer_messages'] ) && 'yes' === $options['cas_pick_disable_customer_messages'] ) ? true : false;

		// Add user if not already set
		if ( empty( $user ) ) {
			$current_user = wp_get_current_user();
			$order->update_meta_data( '_cas_pick_admin_user', $current_user->user_login );
			$order->add_order_note( __( 'Pick is assigned to: ', 'woo-picklist' ) . $current_user->display_name, 0 );
			$user = $current_user->user_login;
		}

		// Change status on order (start pick)
		if ( ! empty( $options['cas_pick_order_processing'] ) && 'yes' === $options['cas_pick_order_processing'] ) {
			$start_status = !empty($options['cas_pick_start_status']) ? $options['cas_pick_start_status'] : false;
			if ( $start_status ) {
				$order->update_status( $start_status, '', true );
			}
		}

		// Set status to completed if 100% picked
		if ( 100.00 === floatval( $percent ) ) {
			$customer_note = ! empty( $options['cas_pick_message_done'] ) ? $options['cas_pick_message_done'] : __( 'Order Picked, ready for shipment.', 'woo-picklist' );
			$order->add_order_note( __( 'Order Picked (100% done)', 'woo-picklist' ), 0 );
			
			//If customer messages
			if ( ! $disable_msg ) {
				$order->add_order_note( $customer_note, 1 ); // Customer note
			}

			// Change status when completed
			if ( ! empty( $options['cas_pick_order_completed'] ) && 'yes' === $options['cas_pick_order_completed'] ) {
				$completed_status = ! empty( $options['cas_pick_completed_status'] ) ? $options['cas_pick_completed_status'] : false;
				if ( $completed_status ) {
					$order->update_status( $completed_status, '', true );
				}
			}
		}

		return $order;
	}

	/**
	 * Update pick on order item (add or remove)
	 */
	public static function cas_update_pick_orderitem_data( $data ) {

		check_ajax_referer( 'cas_pick_id', 'nonce' );
		
		$data       = wp_unslash( $_POST );
		$item_id    = intval( $data['itemid'] ); //product or variation id
		$item_qty   = intval( $data['quantity'] );
		$add_item   = boolval( $data['add'] ); //check if we add or remove pick item
		$order      = wc_get_order( intval( $data['orderid'] ) );
		$items      = $order->get_meta( '_cas_pick_order_status', true );
		$items_qty  = $order->get_meta( '_cas_pick_order_status_qty', true );
		$user       = $order->get_meta( '_cas_pick_admin_user', true );
		$picked_obj = $order->get_meta( '_cas_pick_order_object_items', true ); // New from vs 2.4.16
		$options    = get_option( 'picklist_options' );
		$count      = 0;
		$item_arr   = is_array( $items ) ? $items : array();
		$bulk_pick  = isset( $data['bulkPick'] ) ? $data['bulkPick'] : 'no';
		
		//Order item info
		$orderItemId    = intval( $data['orderItemId'] );
		$orderItem      = $order->get_item( $orderItemId ); 
		$orderItem_qty  = $orderItem->get_quantity();
		$total_picked   = 0;
		$use_picked_obj = false; // Check if save to new object

		if ( ! is_array( $items_qty ) ) {
			$items_qty      = array();
			$item_id        = $orderItemId;
			$use_picked_obj = true;
		}

		if ( is_array( $picked_obj ) ) { // Use new object for saving
			$items_qty      = $picked_obj;
			$item_id        = $orderItemId;
			$use_picked_obj = true;
		}

		
		if ( $add_item ) {
			
			//Manual qty is added
			if ( ( 'yes' !== $bulk_pick ) && ( $item_qty < $orderItem_qty ) ) {
				$current_picked_qty  = ! empty( $items_qty[$item_id] ) ? $items_qty[$item_id] : 0;
				$to_add              = ( ( $item_qty + $current_picked_qty ) >= $orderItem_qty ) ? $orderItem_qty : $current_picked_qty + $item_qty;
				$items_qty[$item_id] = $to_add;
			} else {
				$items_qty[$item_id] = $item_qty;
			}

			//Add if fully picked
			if ( ( ! in_array( $item_id, $item_arr ) ) && ( $orderItem_qty === $items_qty[$item_id] ) ) {
				array_push( $item_arr, $item_id );
			}
			
			$total_picked = $items_qty[$item_id];

			//Update qty info
			if ( $use_picked_obj ) {
				$order->update_meta_data( '_cas_pick_order_object_items', $items_qty ); // New object for saving
			} else {
				$order->update_meta_data( '_cas_pick_order_status_qty', $items_qty );
			}
			
		} else {
			if ( in_array( $item_id, $item_arr ) ) {
				$key = array_search( $item_id, $item_arr, true );
				unset( $item_arr[ $key ] );
				$item_arr = array_values( $item_arr );
			}
			//Update info for barcode scanning
			if ( array_key_exists( $item_id, $items_qty ) ) {
				$items_qty[$item_id] = '';
				if ( $use_picked_obj ) {
					$order->update_meta_data( '_cas_pick_order_object_items', $items_qty ); // New object for saving
				} else {
					$order->update_meta_data( '_cas_pick_order_status_qty', $items_qty );
				}
			}
		}

		$picked_count = 0;
		$picked_items = $items_qty;
		
		if ( is_array( $picked_items ) ) {
			foreach ( array_values( $picked_items ) as $value ) {
				if ( ! empty( $value ) ) {
					$picked_count += intVal( $value );
				}	
			}
		}
	
		// Update order meta data
		$order->update_meta_data( '_cas_pick_order_status', $item_arr );

		/**
		 * Get filtered order item count
		 * 
		 * @since 2.5.8
		 */
		$order_item_count = self::cas_get_filtered_item_count( $order );
		
		//Percent
		$percent = ( ! empty( $order_item_count ) && $picked_count > 0 ) ? ( $picked_count / $order_item_count ) * 100 : 0;
		$order->update_meta_data( '_cas_pick_order_status_percent', strval( number_format( round( $percent, 2 ), 2 ) ) );
		
		//Add more info
		$order = self::cas_add_partial_pick_information( $order, $percent, $options );
		$order->save();

		set_transient( 'cas_notice_pick_order_changed', array( 'changed' => true ) );

		$return_arr = array(
			'item-pick' => $orderItemId, //$item_id,
			'items'     => $item_arr,
			'percent'   => $percent,
			'orderid'   => intval( $data['orderid'] ),
			'user'      => $user,
			'status'    => wc_get_order_status_name( $order->get_status() ),
			'add'       => $add_item,
			'sum_qty'   => $orderItem_qty,
			'tot_pick'  => $total_picked,
			'count_tot' => $picked_count . '/' . $order_item_count,
		);

		wp_send_json( $return_arr );
		wp_die();

	}

	/**
	 * Get notices
	 */
	public static function cas_pick_list_check_changes() {

		check_ajax_referer( 'cas_pick_id', 'nonce' );

		$pick_changed = 'false';
		$new_order    = 'false';
		$wc_logger    = wc_get_logger();

		if ( false !== get_transient( 'cas_notice_pick_order_changed' ) && get_transient( 'cas_notice_pick_order_changed' )['changed'] ) {
			$pick_changed = 'true';
			if ( false === delete_transient( 'cas_notice_pick_order_changed' ) ) {
				$wc_logger->log( 'error', 'Error deleting transient: cas_notice_pick_order_changed', 'Pick List' );
			}
		}

		if ( false !== get_transient( 'cas_notice_pick_new_order' ) && get_transient( 'cas_notice_pick_new_order' )['changed'] ) {
			$new_order = 'true';
			if ( false === delete_transient( 'cas_notice_pick_new_order' ) ) {
				$wc_logger->log( 'error', 'Error deleting transient: cas_notice_pick_new_order', 'Pick List' );
			}
		}

		$return_arr = array(
			'pickChanged' => $pick_changed,
			'newOrder'    => $new_order,
		);

		wp_send_json( $return_arr );
		wp_die();
	}

	/**
	 * Check if there is new orders
	 */
	public static function cas_new_order_in_store( $order_id, $order ) {
		set_transient( 'cas_notice_pick_new_order', array( 'changed' => true ) );

		//Delete transients for stats
		delete_transient( 'cas_pick_totals_tab_orders' );
		delete_transient( 'cas_pick_totals_tab_pending' );
	}

	/**
	 * Bulk update order - set items to picked
	 */
	public static function cas_pick_bulk_update_items( $data ) {
		
		check_ajax_referer( 'cas_pick_id', 'nonce' );
		
		$data        = wp_unslash( $_POST );
		$order       = wc_get_order( intval( $data['orderid'] ) );
		$user        = $order->get_meta( '_cas_pick_admin_user', true );
		$item_arr    = array();
		$options     = get_option( 'picklist_options' );
		$pick_arr    = array();
		$disable_msg = ( ! empty( $options['cas_pick_disable_customer_messages'] ) && 'yes' === $options['cas_pick_disable_customer_messages'] ) ? true : false;
		
		foreach ( $order->get_items() as $item ) {
			//array_push( $item_arr, ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id() );
			array_push( $item_arr, $item->get_id() ); // Use item id
			$pick_arr[$item->get_id()] = $item->get_quantity();
		}

		$customer_note = ( strlen( $options['cas_pick_message_done'] ) > 0 ) ? $options['cas_pick_message_done'] : __( 'Order Picked, ready for shipping', 'woo-picklist' );

		$order->update_meta_data( '_cas_pick_order_status', $item_arr );
		//$order->update_meta_data( '_cas_pick_order_status_qty', $order->get_item_count() );
		$order->update_meta_data( '_cas_pick_order_status_qty', '' ); // Clear and put data in new object
		$order->update_meta_data( '_cas_pick_order_object_items', $pick_arr ); // New object for saving
		
		$order->update_meta_data( '_cas_pick_order_status_percent', '100' );
		$order->add_order_note( __( 'Order Picked ( 100% done )', 'woo-picklist' ), 0 );
		
		//Add customer note
		if ( ! $disable_msg ) {
			$order->add_order_note( $customer_note, 1 ); //Customer note
		}

		if ( '' === $user ) {
			$order->update_meta_data( '_cas_pick_admin_user', wp_get_current_user()->user_login );
			$user = wp_get_current_user()->user_login;
		}

		//Change status to completed
		if ( 'yes' === $options['cas_pick_order_completed'] ) {
			$order->update_status( 'completed', '', true );
		}

		$order->save();

		set_transient( 'cas_notice_pick_order_changed', array( 'changed' => true ) );

		wp_die();
	}

	/**
	 * Reset pick data on order
	 */
	public static function cas_reset_pick_order_data( $data ) {
	
		check_ajax_referer( 'cas_pick_id', 'nonce' );
		
		$data  = wp_unslash( $_POST );
		$order = wc_get_order( intval( $data['orderid'] ) );
		
		$order->update_meta_data( '_cas_pick_order_status', '' );
		$order->update_meta_data( '_cas_pick_order_status_qty', '' );
		$order->update_meta_data( '_cas_pick_order_status_percent', '' );
		$order->update_meta_data( '_cas_pick_admin_user', '' );
		$order->update_meta_data( '_cas_pick_order_shipped', '' );
		$order->update_meta_data( '_cas_pick_order_shipped_count', 0 );
		$order->add_order_note( __( 'Pick is reset ( 0% )', 'woo-picklist' ), 0 );
		$order->update_meta_data( '_cas_pick_order_object_items', '' ); // New object for saving picked items

		//Change status to processing
		$order->update_status( 'processing', '', true );

		$order->save();

		set_transient( 'cas_notice_pick_order_changed', array( 'changed' => true ) );

		wp_die();
	
	}

	/**
	 * Add order note
	 */
	public static function cas_pick_add_order_note( $data ) {
		
		check_ajax_referer( 'cas_pick_id', 'nonce' );
		
		$data  = wp_unslash( $_POST );
		$order = wc_get_order( intval( $data['orderid'] ) );
		
		$order->add_order_note( $data['message'], 0 );
		$order->save();

		wp_die();

	}

	/**
	 * Set order status to shipped
	 */
	public static function cas_pick_set_order_status_shipped( $data ) {

		check_ajax_referer( 'cas_pick_id', 'nonce' );
		
		$data          = wp_unslash( $_POST );
		$order         = wc_get_order( intval( $data['orderid'] ) );
		$options       = get_option( 'picklist_options' );
		$message       = ( strlen( $options['cas_pick_message_shipped'] ) > 0 ) ? $options['cas_pick_message_shipped'] : __( 'Your order is shipped', 'woo-picklist' );
		$shipped_count = ! empty( $order->get_meta( '_cas_pick_order_shipped_count', true ) ) ? intval( $order->get_meta( '_cas_pick_order_shipped_count', true ) ) : 0;
		$disable_msg   = ( ! empty( $options['cas_pick_disable_customer_messages'] ) && 'yes' === $options['cas_pick_disable_customer_messages'] ) ? true : false;
	
		//Message to customer
		if ( $shipped_count > 0 ) {
			/* translators: %s: message to customer for x times of shipment */ 
			$message = sprintf( __( '%1$s. This is shipment #%2$d of your order.', 'woo-picklist' ), $message, $shipped_count + 1 );
		}

		//Add customer message
		if ( ! $disable_msg ) {
			$order->add_order_note( $message, 1 );
		}
		$order->update_meta_data( '_cas_pick_order_shipped', 'yes' );
		$order->update_meta_data( '_cas_pick_order_shipped_count', $shipped_count + 1 );
		$order->save();

		set_transient( 'cas_notice_pick_order_changed', array( 'changed' => true ) );

		wp_die();

	}

	/**
	 * Change order status
	 */
	public static function cas_pick_change_order_status_callback( $data ) {
		
		// Check nonce for security
		check_ajax_referer( 'cas_pick_id', 'nonce' );

		$data     = wp_unslash( $_POST );
		$order_id = isset( $data['orderId'] ) ? absint( $data['orderId'] ) : 0;
		$status   = isset( $data['status'] ) ? sanitize_text_field( $data['status'] ) : '';
		$order    = wc_get_order( $order_id );

		// Ensure order exists
		if ( ! $order ) {
			wp_send_json_error( array( 'message' => __( 'Invalid order ID.', 'woo-picklist' ) ), 400 );
			wp_die();
		}

		// Support custom statuses
		$valid_statuses = array_keys( wc_get_order_statuses() );

		// Normalize status
		if ( strpos( $status, 'wc-' ) !== 0 ) {
			$status = 'wc-' . $status;
		}

		// Validate statuses
		if ( ! in_array( $status, $valid_statuses, true ) ) {
			wp_send_json_error( [ 'message' => __( 'Invalid order status.', 'woo-picklist' ) ], 400 );
			wp_die();
		}

		$order->update_status( $status, '', true );
		$order->save();

		$return_arr = array(
			'status'      => $order->get_status(),
			'id'          => $order_id,
			'status_name' => wc_get_order_status_name( $order->get_status() ),
		);

		wp_send_json( $return_arr );
		wp_die();

	}

	/**
	 * Get paging (span)
	 */
	public static function cas_get_paging( $page_num, $max_num_pages ) {
		?>
		<span class="cas-pagination">
			<?php
			$args = array( // WPCS: XSS ok.
				'base'      => '%_%',
				'format'    => '?paged=%#%',
				'current'   => $page_num,
				'total'     => $max_num_pages,
				'prev_text' => is_rtl() ? '&rarr;' : '&larr;',
				'next_text' => is_rtl() ? '&larr;' : '&rarr;',
				'type'      => 'plain',
				'end_size'  => 1,
				'mid_size'  => 1,
			);
			echo wp_kses( paginate_links( $args ), 'post' );
			?>
		</span>
		<?php
	}

	/**
	 * Get allowed html tags
	 */
	public static function cas_allowed_html_tags() {
		$allowed_html = array(
			'br'   => array(), // Self-closing tag, no attributes
			'span' => array(
				'class' => true, // Allows class attribute
				'style' => true, // Allows inline styles
			),
			'p'    => array(
				'class' => true,
				'style' => true,
			),
			'strong' => array(), // Bold text
			'em'     => array(), // Italic text
			'a'      => array(
				'href'   => true,
				'title'  => true,
				'target' => true,
			),
			'ul'     => array(),
			'ol'     => array(),
			'li'     => array(),
			's'      => array( 
				'title' => true, 
			), // Strikethrough text
		);
		
		return $allowed_html;
	}

	/**
	 * Get the total quantity of order items after applying picklist filters.
	 *
	 * This function loops through all items in an order and counts only those
	 * that are allowed by the `cas_picklist_include_order_item` filter.
	 *
	 * @param WC_Order $order The WooCommerce order object.
	 * @return int The total count of included items (filtered).
	 */
	public static function cas_get_filtered_item_count( $order ) {
		$count = 0;
	
		foreach ( $order->get_items() as $item ) {
			$product_id = ( $item->get_variation_id() > 0 ) ? $item->get_variation_id() : $item->get_product_id();
			$product    = wc_get_product( $product_id );
	
			/**
			 * Filter to include or exclude order items from the picklist.
			 * 
			 * @since 2.5.8
			 */
			$include = apply_filters( 'cas_picklist_include_order_item', true, $item, $product, $order );
	
			if ( $include ) {
				$count += $item->get_quantity();
			}
		}
	
		/**
		 * Filter to modify the total count of order items.
		 * 
		 * @since 2.5.8
		 */
		return apply_filters( 'cas_picklist_order_item_count', $count, $order );
	}
	

	/**
	 * Update stock quantity for a product via AJAX.
	 */
	public static function cas_pick_update_stock_quantity() {
		
		check_ajax_referer( 'cas_pick_id', 'nonce' );

		$data       = wp_unslash( $_POST );
		$product_id = isset( $data['product_id'] ) ? absint( $data['product_id'] ) : '0';
		$qty        = isset( $data['quantity'] ) ? wc_stock_amount( $data['quantity'] ) : 0;

		if ( ! current_user_can( 'manage_woocommerce' ) ) {
			wp_send_json_error( 'Unauthorized' );
		}

		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			wp_send_json_error( 'Product not found' );
		}

		$product->set_stock_quantity( $qty );
		$product->save();

		$return_arr = array(
			'id'       => $product_id,
			'quantity' => $qty,
			'success'  => true,
			'message'  => __( 'Stock quantity updated successfully.', 'woo-picklist' ),
		);

		wp_send_json( $return_arr );
		wp_die();

	}

}

