<?php
/**
 * Pick List
 *
 * @package picklist/includes
 */

defined( 'ABSPATH' ) || exit;

/**
 * Settings for API.
 */
if ( class_exists( 'Cas_Pick_Settings', false ) ) {
	return new Cas_Pick_Settings();
}

/**
 * Class for settings
 */
class Cas_Pick_Settings extends WC_Settings_Page {

	/**
	 * Constructor.
	 */
	public function __construct() {
		$this->id    = 'picklist';
		$this->label = __( 'Pick List', 'woo-picklist' );

		parent::__construct();
		
		register_setting( 'picklist_options_group', 'picklist_options_settings' );

	}

	/**
	 * Get sections
	 *
	 * @return array
	 */
	public function get_sections() {
	
		$sections = array(
			''            => __( 'General', 'woo-picklist' ),
			'unpicked'    => __( 'Pending Pickup Orders', 'woo-picklist' ),
			'pick'        => __( 'Pick Order', 'woo-picklist' ),
			'integration' => __( 'Custom Order fields', 'woo-picklist' ),
			'product'     => __( 'Add Product fields', 'woo-picklist' ),
			'barcode'     => __( 'Barcode Integration', 'woo-picklist' ),
			'shipping'    => __( 'Shipping Labels', 'woo-picklist' ),
		);

		/**
		 * Filter to add id and sections
		 * 
		 * @since 2.3.1
		 */
		return apply_filters( 'woocommerce_get_sections_' . $this->id, $sections ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		
	}

	/**
	 * Settings array
	 *
	 * @return array
	 */
	public function get_settings( $section = '' ) {

		global $current_section;
		$section  = $current_section;
		$settings = array();

		switch ( $section ) {

			case '':
				$settings = array( 

					array(
						'name' => __( 'Pick List', 'woo-picklist' ),
						'type' => 'title',
						'desc' => __( 'The following options are used to configure Pick List - mainly the Orders tab', 'woo-picklist' ),
						'id'   => 'cas_pick_id_label',
					),

					array(
						'name'     => __( 'Orders per page', 'woo-picklist' ),
						'desc_tip' => __( 'Used on Pick List orders', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_per_page]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Default Order statuses', 'woo-picklist' ),
						'desc_tip' => __( 'Select the default order statuses for Pick List order table', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_statuses]',
						'type'     => 'multiselect',
						'options'  => wc_get_order_statuses(),
						'class'    => 'cas-multiselect',
					),	

					array(
						'name'     => __( 'Show Orders in date range', 'woo-picklist' ),
						'desc_tip' => __( 'Select date range (none selected gets all orders)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_date_range]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array(
							''              => __( 'Select', 'woo-picklist' ),
							'today'         => __( 'Today', 'woo-picklist' ),
							'week'          => __( 'Last 7 days', 'woo-picklist' ),
							'this_month'    => __( 'This month', 'woo-picklist' ),
							'last_3_month'  => __( 'Last 3 month', 'woo-picklist' ),
							'last_6_month'  => __( 'Last 6 month', 'woo-picklist' ),
							'last_12_month' => __( 'Last 12 month', 'woo-picklist' ),
							'year'          => __( 'Year ( January - today )', 'woo-picklist' ),
						),
					),

					array(
						'name'     => __( 'Order', 'woo-picklist' ),
						'desc_tip' => __( 'Default newest to oldest', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_ordering]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array(
							''     => __( 'Select', 'woo-picklist' ),
							'asc'  => __( 'ASC', 'woo-picklist' ),
							'desc' => __( 'DESC (default)', 'woo-picklist' ),
						),
					),

					array(
						'name'     => __( 'Show Company', 'woo-picklist' ),
						'desc_tip' => __( 'Show Company in order table and pick order', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_show_company]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Show Role', 'woo-picklist' ),
						'desc_tip' => __( 'Show Role (e.g. VIP / B2B) in order table and pick order', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_show_role]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Show Shipping', 'woo-picklist' ),
						'desc_tip' => __( 'Show Shipping information in orders and pending pickup orders tabs', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_show_shipping_information]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Notification: New Orders', 'woo-picklist' ),
						'desc_tip' => __( 'Show notification when new orders arrive', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_noti_orders]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Notification: Order Pick', 'woo-picklist' ),
						'desc_tip' => __( 'Show notification when changes is done on the order pick', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_noti_changes]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Automatically Update', 'woo-picklist' ),
						'desc_tip' => __( 'Update Pick List order table when new orders arrive or changes in pick orders. If on, no notifications are shown!', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_auto_refresh]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Pick Status on My Orders', 'woo-picklist' ),
						'desc_tip' => __( 'If enabled every order item gets a status line in this format - Pick Status: Item Selected (if picked)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_my_order_statuses]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Hide Price Information for user(s)', 'woo-picklist' ),
						'desc_tip' => __( 'Select users you want to hide prices and totals for. Hides in orders, pending pickup orders, pick order and prints.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_hide_price_users]',
						'type'     => 'multiselect',
						'options'  => array_merge( array( '' => __( 'Select Users', 'woo-picklist' ) ), $this->cas_get_picklist_users() ),
						'class'    => 'cas-multiselect',
					),
					
					array(
						'name'     => __( 'User Roles Access', 'woo-picklist' ),
						'desc_tip' => __( 'Enable user roles that have access to Pick List. Default roles is: Administrators and Shop Managers.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_picklist_roles]',
						'type'     => 'multiselect',
						'options'  => array_merge( array( '' => __( 'Select Roles', 'woo-picklist' ) ), $this->cas_get_picklist_roles_options() ),
						'class'    => 'cas-multiselect',
					),

					array(
						'name'     => __( 'Use Dark Mode (style)', 'woo-picklist' ),
						'desc_tip' => __( 'Check to enable dark mode style for Pick List', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_use_dark_mode]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),
					
					array(
						'type' => 'sectionend',
						'id'   => 'cas_pick_general',
					),
				
				);

				break;
			case 'unpicked':
				$settings = array(

					array(
						'name' => __( 'Pick List', 'woo-picklist' ),
						'type' => 'title',
						'desc' => __( 'The following options are used to configure Pending Pickup Orders', 'woo-picklist' ),
						'id'   => 'cas_pending_pick_id_label',
					),

					array(
						'name'     => __( 'Orders per page', 'woo-picklist' ),
						'desc_tip' => __( 'Used on Pick List orders', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pending_pick_per_page]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Default Order statuses', 'woo-picklist' ),
						'desc_tip' => __( 'Select the default order statuses for Pick List order table', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pending_pick_order_statuses]',
						'type'     => 'multiselect',
						'css'      => 'min-width:200px;min-height:200px;',
						'options'  => wc_get_order_statuses(),
						'class'    => 'cas-multiselect',
					),	

					array(
						'name'     => __( 'Show Orders in date range', 'woo-picklist' ),
						'desc_tip' => __( 'Select date range (none selected gets all orders)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pending_pick_order_date_range]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array(
							''              => __( 'Select', 'woo-picklist' ),
							'today'         => __( 'Today', 'woo-picklist' ),
							'week'          => __( 'Last 7 days', 'woo-picklist' ),
							'this_month'    => __( 'This month', 'woo-picklist' ),
							'last_3_month'  => __( 'Last 3 month', 'woo-picklist' ),
							'last_6_month'  => __( 'Last 6 month', 'woo-picklist' ),
							'last_12_month' => __( 'Last 12 month', 'woo-picklist' ),
							'year'          => __( 'Year ( January - today )', 'woo-picklist' ),
						),
					),

					array(
						'name'     => __( 'Order', 'woo-picklist' ),
						'desc_tip' => __( 'Default old to new', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pending_pick_ordering]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array(
							''     => __( 'Select', 'woo-picklist' ),
							'asc'  => __( 'ASC (default)', 'woo-picklist' ),
							'desc' => __( 'DESC', 'woo-picklist' ),
						),
					),

					array(
						'type' => 'sectionend',
						'id'   => 'cas_pick_unpicked',
					),
				);
				break;
			case 'pick':
				$settings = array( 

					array(
						'name' => __( 'Pick List', 'woo-picklist' ),
						'type' => 'title',
						'desc' => __( 'The following options are used to configure Pick List for WooCommerce', 'woo-picklist' ),
						'id'   => 'cas_pick_id_label',
					),

					array(
						'name'     => __( 'Company Logo URL', 'woo-picklist' ),
						'desc_tip' => __( 'Add your company logo URL, used in print pick list / packing slips', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_site_logo]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
						'default'  => '', 
					),
					

					array(
						'name'     => __( 'Change Status when Pick starts', 'woo-picklist' ),
						'desc_tip' => __( 'Automatically changes the order status to e.g. processing (select below) when pick starts (when e.g. 10% progress)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_processing]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Status Pick Started', 'woo-picklist' ),
						'desc_tip' => __( 'Select a status to e.g. Processing (only works if change status on start is checked)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_start_status]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array_merge( array( '' => __( '— Select —', 'woo-picklist' ) ), wc_get_order_statuses() ),
						'default'  => '',
					),	

					array(
						'name'     => __( 'Change Status when Pick Completed', 'woo-picklist' ),
						'desc_tip' => __( 'Atomatically changes the order status to e.g. Completed (select below) when a pick is 100% done', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_completed]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Status Pick Completed', 'woo-picklist' ),
						'desc_tip' => __( 'Select a status (only works if change status on complted is checked)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_completed_status]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array_merge( array( '' => __( '— Select —', 'woo-picklist' ) ), wc_get_order_statuses() ),
						'default'  => '',
					),	

					array(
						'name'     => __( 'Customer message - shipped', 'woo-picklist' ),
						'desc_tip' => __( 'Message to customer when order is shipped, default: Order shipped', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_message_shipped]',
						'type'     => 'textarea',
						'css'      => 'min-width:200px;height:150px;',
					),

					array(
						'name'     => __( 'Customer message - pick completed', 'woo-picklist' ),
						'desc_tip' => __( 'Message to customer when order is picked, default: Order Picked, ready for shipment', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_message_done]',
						'type'     => 'textarea',
						'css'      => 'min-width:200px;height:150px;',
					),

					array(
						'name'     => __( 'Disable Customer Messages', 'woo-picklist' ),
						'desc_tip' => __( 'Do not add customer messages in the extension', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_disable_customer_messages]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Default Packing Slip note', 'woo-picklist' ),
						'desc_tip' => __( 'A default (thank you) text to use on packing slips (below order items in print)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_message_slip]',
						'type'     => 'textarea',
						'css'      => 'min-width:200px;height:200px;',
					),
					
					array(
						'name'     => __( 'Include Packing Slip note by default', 'woo-picklist' ),
						'desc_tip' => __( 'Atomatically adds packing slip note to every order (print)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_message_slip_on]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Hide Product Categories', 'woo-picklist' ),
						'desc_tip' => __( 'Hides product categories in pick table, grid and prints', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_hide_category]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Show Billing Address in Prints', 'woo-picklist' ),
						'desc_tip' => __( 'Shows billing addresses in prints (pick list and packing slip)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_show_billing_prints]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Include Grid View', 'woo-picklist' ),
						'desc_tip' => __( 'Check to enable grid view and toggle. Note: this can slow down picking performance if you e.g. use barcode scanning.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_include_grid_pick_view]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Show Dimensions Column', 'woo-picklist' ),
						'desc_tip' => __( 'Shows dimensions column (length, width and height) in pick order table and calculates volume++', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_show_th_dimensions]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),
					
					array(
						'name'     => __( 'Hide Next Orders button', 'woo-picklist' ),
						'desc_tip' => __( 'Hides next orders button and reduces queries to database (faster e.g. when barcode scanning)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_hide_next_unpicked_btn]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Hide Stock Column', 'woo-picklist' ),
						'desc_tip' => __( 'Hides stock column in pick order table', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_hide_th_stock]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Hide Price Column', 'woo-picklist' ),
						'desc_tip' => __( 'Hides price column in pick order table', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_hide_th_price]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Hide Weight Column', 'woo-picklist' ),
						'desc_tip' => __( 'Hides weigth column in pick order table', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_hide_th_weight]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'type' => 'sectionend',
						'id'   => 'cas_pick_item',
					),
				);
				break;
			case 'integration':
				$settings = array( 

					array(
						'name' => __( 'Pick List', 'woo-picklist' ),
						'type' => 'title',
						'desc' => __( 'The following options are used to configure Pick List for WooCommerce', 'woo-picklist' ),
						'id'   => 'cas_pick_id_label',
					),

					array(
						'name'     => __( 'Custom field 1', 'woo-picklist' ),
						'desc_tip' => __( 'Add a custom order field to Pick List', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_custom1]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Custom field 1 label', 'woo-picklist' ),
						'desc_tip' => __( 'Add label for custom field 1', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_custom1_lbl]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Use Custom 1 -> Order table', 'woo-picklist' ),
						'desc_tip' => __( 'Show custom field 1 in order table', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_custom1_pick]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),
					
					array(
						'name'     => __( 'Use Custom 1 -> Pick Order', 'woo-picklist' ),
						'desc_tip' => __( 'Show custom field 1 on pick order tab', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_custom1_order]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Custom field 2', 'woo-picklist' ),
						'desc_tip' => __( 'Add a custom order field to Pick List', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_custom2]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Custom field 2 label', 'woo-picklist' ),
						'desc_tip' => __( 'Add label for custom field 1', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_custom2_lbl]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Use Custom 2 -> Order table', 'woo-picklist' ),
						'desc_tip' => __( 'Show custom field 2 in order table', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_custom2_pick]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Use Custom 2 -> Pick Order', 'woo-picklist' ),
						'desc_tip' => __( 'Show custom field 2 on pick order tab', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_order_custom2_order]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'type' => 'sectionend',
						'id'   => 'cas_pick_integration',
					),
				);
				break;
			case 'product':
				$settings = array( 

					array(
						'name' => __( 'Pick List', 'woo-picklist' ),
						'type' => 'title',
						'desc' => __( 'Add custom product fields to the Pick Order list. If more than this is needed, there is option to add as many (Custom Fields) as you like using a hook - read how to in documentation.', 'woo-picklist' ),
						'id'   => 'cas_pick_id_label_custom_product',
					),

					array(
						'name'     => __( 'Regular Product Fields', 'woo-picklist' ),
						'desc_tip' => __( 'Select regular product fields (will be shown below Name and SKU)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_regular_product_fields]',
						'type'     => 'multiselect',
						'css'      => 'min-width:200px;',
						'class'    => 'cas-multiselect',
						'options'  => self::cas_get_picklist_standard_field_options(),
					),
				
					array(
						'name'     => __( 'Custom Field 1', 'woo-picklist' ),
						'desc_tip' => __( 'Select custom product field to use', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_1]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'class'    => 'cas_interval',
						'options'  => array_merge( array( '' => __( 'Select Custom Field', 'woo-picklist' ) ), $this->cas_get_all_custom_product_fields() ),
					),

					array(
						'name'     => __( 'Custom field 1 label', 'woo-picklist' ),
						'desc_tip' => __( 'Add label for custom field 1', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_1_lbl]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Use Custom Field 1', 'woo-picklist' ),
						'desc_tip' => __( 'Check this to use custom product field 1.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_1_use]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Custom Field 2', 'woo-picklist' ),
						'desc_tip' => __( 'Select custom product field to use', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_2]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'class'    => 'cas_interval',
						'options'  => array_merge( array( '' => __( 'Select Custom Field', 'woo-picklist' ) ), $this->cas_get_all_custom_product_fields() ),
					),

					array(
						'name'     => __( 'Custom field 2 label', 'woo-picklist' ),
						'desc_tip' => __( 'Add label for custom field 2', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_2_lbl]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Use Custom Field 2', 'woo-picklist' ),
						'desc_tip' => __( 'Check this to use custom product field 2.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_2_use]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Custom Field 3', 'woo-picklist' ),
						'desc_tip' => __( 'Select custom product field to use', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_3]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'class'    => 'cas_interval',
						'options'  => array_merge( array( '' => __( 'Select Custom Field', 'woo-picklist' ) ), $this->cas_get_all_custom_product_fields() ),
					),

					array(
						'name'     => __( 'Custom field 3 label', 'woo-picklist' ),
						'desc_tip' => __( 'Add label for custom field 3', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_3_lbl]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Use Custom Field 3', 'woo-picklist' ),
						'desc_tip' => __( 'Check this to use custom product field 3.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_3_use]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Custom Field 4', 'woo-picklist' ),
						'desc_tip' => __( 'Select custom product field to use', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_4]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'class'    => 'cas_interval',
						'options'  => array_merge( array( '' => __( 'Select Custom Field', 'woo-picklist' ) ), $this->cas_get_all_custom_product_fields() ),
					),

					array(
						'name'     => __( 'Custom field 4 label', 'woo-picklist' ),
						'desc_tip' => __( 'Add label for custom field 4', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_4_lbl]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Use Custom Field 4', 'woo-picklist' ),
						'desc_tip' => __( 'Check this to use custom product field 4.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_custom_product_field_4_use]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'type' => 'sectionend',
						'id'   => 'cas_pick_custom_product_fields',
					),
				);
				break;
			case 'barcode':
				$settings = array(
					array(
						'name' => __( 'Pick List', 'woo-picklist' ),
						'type' => 'title',
						'desc' => __( 'The following options are used to configure integration of barcode on product items', 'woo-picklist' ),
						'id'   => 'cas_pick_id_label',
					),
					array(
						'name'     => __( 'Barcode field (product field)', 'woo-picklist' ),
						'desc_tip' => __( 'Add a custom product field for the barcode (integration)', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_product_c1]',
						'type'     => 'text',
						'css'      => 'min-width:200px;',
					),

					array(
						'desc_tip' => __( 'Adds the selected field to the above input', 'woo-picklist' ),
						'id'       => 'picklist-field-select',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array_merge( array( '' => __( 'Select custom field (for barcodes)', 'woo-picklist' ) ), $this->cas_get_all_custom_product_fields() ),
					),

					array(
						'name'     => __( 'Use product field', 'woo-picklist' ),
						'desc_tip' => __( 'Select a common product field in the list - only select this if you use this for barcode.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_product_c1_alt]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array( 
							''                  => __( 'Select', 'woo-picklist' ), 
							'id'                => __( 'Id', 'woo-picklist' ),
							'sku'               => __( 'SKU', 'woo-picklist' ),
							'_global_unique_id' => __( 'Global Unique ID', 'woo-picklist' ),
							),
						'default'  => '',
					),

					array(
						'name'     => __( 'Use Barcode?', 'woo-picklist' ),
						'desc_tip' => __( 'Check this to add barcode on products, uses barcode field (pri 1) or product field if set.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_product_c1_is_bar]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Barcode type', 'woo-picklist' ),
						'desc_tip' => __( 'Select the type of barcode you use, check with integration vendor.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_barcode_type]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array( 
							''           => __( 'Select', 'woo-picklist' ), 
							'ean8'       => __( 'EAN 8 (GTIN-8) (only numeric) ', 'woo-picklist' ),
							'ean13'      => __( 'EAN 13 (GTIN-13) (only numeric)', 'woo-picklist' ),
							'upc'        => __( 'UPC-A (GTIN-12) (only numeric)', 'woo-picklist' ),
							'code11'     => __( 'CODE 11 (only numeric)', 'woo-picklist' ),
							'code39'     => __( 'CODE 39', 'woo-picklist' ),
							'code93'     => __( 'CODE 93', 'woo-picklist' ),
							'code128'    => __( 'CODE 128', 'woo-picklist' ),
							'codabar'    => __( 'Codabar (only numeric)', 'woo-picklist' ),
							'std25'      => __( 'Standard 2 of 5 (industrial)', 'woo-picklist' ),
							'msi'        => __( 'MSI (Modified Plessey)', 'woo-picklist' ),
							'int25'      => __( 'Interleaved 2 of 5', 'woo-picklist' ),
							'datamatrix' => __( 'Data Matrix (datamatrix)', 'woo-picklist' ),
							),
						'default'  => '',
					),
					
					array(
						'name'     => __( 'Turn of scan sounds?', 'woo-picklist' ),
						'desc_tip' => __( 'Check this to turn off confirmation and error sounds (sound works on most browsers).', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_scan_sound_on]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Bulk scan items', 'woo-picklist' ),
						'desc_tip' => __( 'Enable (checked) if you want to scan e.g. 20 items on a order item in one scan (default 1 scan = 1 item qty).', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_pick_scan_all_qty]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'type' => 'sectionend',
						'id'   => 'cas_pick_product',
					),
				);
				break;
			case 'shipping':
				$settings = array(
					array(
						'name' => __( 'Pick List', 'woo-picklist' ),
						'type' => 'title',
						'desc' => __( 'The following options are used to configure shipping labels', 'woo-picklist' ),
						'id'   => 'cas_pick_id_shipping_label',
					),

					array(
						'name'     => __( 'Disable Shipping Labels', 'woo-picklist' ),
						'desc_tip' => __( 'Check to disable (hide) the print shipping label button (e.g. if you do this in another plugin).', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_hide_shipping_label_button]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'name'     => __( 'Label Size', 'woo-picklist' ),
						'desc_tip' => __( 'Select shipping label (paper) size. Default 4in x 6in if none is set.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_shipping_label_paper_size]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array_merge( array( '' => __( 'Select Label Size', 'woo-picklist' ) ), self::cas_get_label_sizes() ),
					),

					array(
						'name'     => __( 'Barcode format', 'woo-picklist' ),
						'desc_tip' => __( 'Carrier barcode format - can be overwritten in shipping label dialog, default Code128 if none is set.', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_shipping_carrier_barcode_format]',
						'type'     => 'select',
						'css'      => 'min-width:200px;',
						'options'  => array_merge( array( '' => __( 'Select default format', 'woo-picklist' ) ), self::cas_get_shipping_barcode_format() ),
					),

					array(
						'name'     => __( 'Include Order QR', 'woo-picklist' ),
						'desc_tip' => __( 'Check to include QR Code with URL direct to Pick Order (tab).', 'woo-picklist' ),
						'id'       => 'picklist_options[cas_shipping_include_pick_qr]',
						'type'     => 'checkbox',
						'css'      => 'min-width:200px;',
					),

					array(
						'type' => 'sectionend',
						'id'   => 'cas_pick_shipping',
					),
				);
				break;
		}

		/**
		 * Filter to add or settings
		 * 
		 * @since 2.3.1
		 */
		return apply_filters( 'wc_' . $this->id . '_settings', $settings ); // phpcs:ignore WordPress.NamingConventions.ValidHookName.UseUnderscores
		
	}

	/**
	 * Get regular shipping label sizes
	 */
	public static function cas_get_label_sizes() {
		return array(
			'4x6' => __( '4" × 6" (100mm × 150mm)', 'woo-picklist' ),
			'4x8' => __( '4" × 8" (100mm × 200mm)', 'woo-picklist' ),
			'6x8' => __( '6" × 8" (150mm × 200mm)', 'woo-picklist' ),
			'A6'  => __( 'A6 (105mm × 148mm)', 'woo-picklist' ),
			'A5'  => __( 'A5 (148mm × 210mm)', 'woo-picklist' ),
		);
	}

	/**
	 * Get shipping barcode format
	 */
	public static function cas_get_shipping_barcode_format() {
		return array(
			'code128' => __( 'Code 128 (Default for UPS, FedEx, DHL)', 'woo-picklist' ),
			'code39'  => __( 'Code 39 (Older FedEx, some carriers)', 'woo-picklist' ),
			'qr'      => __( 'QR Code (Amazon, some carriers)', 'woo-picklist' ),
		);
	}

	/**
	 * Get picklist field options
	 * 
	 * @return array
	 */
	public static function cas_get_picklist_standard_field_options() {
		
		$standard_fields = array(
			'shipping_class'    => __( 'Shipping Class', 'woo-picklist' ),
			'weight'            => __( 'Weight', 'woo-picklist' ),
			'dimensions'        => __( 'Dimensions (L×W×H)', 'woo-picklist' ),
			'price'             => __( 'Price', 'woo-picklist' ),
			'sale_price'        => __( 'Sale Price', 'woo-picklist' ),
			'description'       => __( 'Description', 'woo-picklist' ),
			'short_description' => __( 'Short Description', 'woo-picklist' ),
		);
		
		return array_merge(
			array( '' => __( 'Select Field', 'woo-picklist' ) ),
			$standard_fields
		);
	}

	/**
	 * Get order statuses
	 */
	public function cas_pick_get_order_statuses() {
		return wc_get_order_statuses();
	}

	/**
	 * Add output functions
	 */
	public function output() {

		global $current_section;

		$settings = $this->get_settings( $current_section );
		WC_Admin_Settings::output_fields( $settings );

		?>
		<script>
		jQuery(document).ready(function($) {
			$('#picklist-field-select').on('change', function() {
				// Get the selected value from the dropdown
				var selectedField = $(this).val();

				// Update the barcode text input field with the selected value
				$('#picklist_options\\[cas_pick_product_c1\\]').val(selectedField);
			});

			// Initialize Select2 on the specific multiselect field
			$('.cas-multiselect').select2({
				allowClear: true,
				width: '300px'
			});

		});
		</script>
		<?php
	}

	/**
	 * Save settings and trigger the 'woocommerce_update_options_'.id action.
	 */
	public function save() {
		$this->save_settings_for_current_section();
		$this->do_update_options_action();

		delete_transient( 'cas_picklist_product_options' );
		delete_transient( 'cas_pick_totals_tab_orders' );
		delete_transient( 'cas_pick_totals_tab_pending' );
	}

	/**
	 * Get custom fields options
	 */
	private function cas_get_all_custom_product_fields() {
		global $wpdb;

		$meta_keys = $wpdb->get_col(
			$wpdb->prepare(
				"SELECT DISTINCT pm.meta_key
				FROM $wpdb->postmeta pm
				JOIN $wpdb->posts p ON pm.post_id = p.ID
				WHERE pm.meta_key NOT BETWEEN '_' AND '_z'
				AND pm.meta_key NOT LIKE %s
				ORDER BY p.post_modified DESC, pm.meta_key
				LIMIT %d",
				$wpdb->esc_like('_') . '%',
				60
			)
		);

		$custom_fields = array();
	
		foreach ( $meta_keys as $key ) {
			$custom_fields[$key] = $key; // Use meta key as both the key and the label
		}
	
		return $custom_fields;
	}

	/**
	 * Get users
	 */
	private static function cas_get_picklist_users() {
		
		//Make sure all users is added
		delete_transient( 'cas_pick_list_shop_managers' );

		$shop_managers = Cas_Pick_Functions::cas_get_pick_list_users();

		$users = array();

		foreach ( $shop_managers as $pick_user ) {
			$users[$pick_user->user_login] = $pick_user->display_name;
		}

		return $users;

	}

	/**
	 * Get roles for Pick List settings (excluding administrator & shop manager)
	 */
	private static function cas_get_picklist_roles_options() {
		// Get all roles in WordPress
		$all_roles = wp_roles()->roles;

		// Remove roles that should always have access
		unset( $all_roles['administrator'] );
		unset( $all_roles['shop_manager'] );

		// Filter out roles that do NOT have 'read' capability
		$valid_roles = [];
		foreach ( $all_roles as $role_key => $role_data ) {
			if ( ! empty( $role_data['capabilities']['read'] ) ) {
				$valid_roles[$role_key] = $role_data['name'];
			}
		}
	
		return $valid_roles;
	}
	
}
return new Cas_Pick_Settings();
