<?php
/**
 * WC_BIS_Compatibility class
 *
 * @package  WooCommerce Back In Stock Notifications
 * @since    1.0.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

/**
 * Handles compatibility with other WC extensions.
 *
 * @class    WC_BIS_Compatibility
 * @version  2.0.5
 */
class WC_BIS_Compatibility {

	/**
	 * Min required plugin versions to check.
	 *
	 * @var array
	 */
	private static $required = array(
		'wc_bundles' => '6.5.1',
	);

	/**
	 * Setup compatibility class.
	 */
	public static function init() {
		// Initialize.
		self::load_modules();
	}

	/**
	 * Initialize.
	 *
	 * @return void
	 */
	protected static function load_modules() {

		if ( is_admin() ) {
			// Check plugin min versions.
			add_action( 'admin_init', array( __CLASS__, 'add_compatibility_notices' ) );
		}

		// Include core compatibility class.
		self::core_includes();

		// Declare HPOS compatibility.
		add_action( 'before_woocommerce_init', array( __CLASS__, 'declare_hpos_compatibility' ) );

		// Declare Blocks compatibility.
		add_action( 'before_woocommerce_init', array( __CLASS__, 'declare_blocks_compatibility' ) );

		// Load modules.
		add_action( 'plugins_loaded', array( __CLASS__, 'module_includes' ), 100 );
		add_action( 'before_woocommerce_init', array( __CLASS__, 'deferred_module_includes' ) );

		// Prevent initialization of deprecated mini-extensions usually loaded on 'plugins_loaded' -- 10.
		self::unload_modules();
	}

	/**
	 * Core compatibility functions.
	 *
	 * @return void
	 */
	public static function core_includes() {
		require_once WC_BIS_ABSPATH . 'includes/compatibility/core/class-wc-bis-core-compatibility.php';
	}

	/**
	 * Declare HPOS( Custom Order tables) compatibility.
	 *
	 * @since 1.4.1
	 */
	public static function declare_hpos_compatibility() {

		if ( ! class_exists( 'Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
			return;
		}

		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', WC_BIS()->get_plugin_basename(), true );
	}

	/**
	 * Declare cart/checkout Blocks compatibility.
	 *
	 * @since 1.6.5
	 */
	public static function declare_blocks_compatibility() {

		if ( ! class_exists( 'Automattic\WooCommerce\Utilities\FeaturesUtil' ) ) {
			return;
		}

		\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'cart_checkout_blocks', WC_BIS()->get_plugin_basename(), true );
	}

	/**
	 * Prevent deprecated mini-extensions from initializing.
	 *
	 * @return void
	 */
	protected static function unload_modules() {
		// Silence.
	}

	/**
	 * Load compatibility classes.
	 *
	 * @return void
	 */
	public static function module_includes() {

		$module_paths = array();

		// WC Product Bundles support.
		if ( class_exists( 'WC_Bundles' ) && defined( 'WC_PB_VERSION' ) && version_compare( WC_PB_VERSION, self::$required['wc_bundles'] ) >= 0 ) {
			$module_paths['wc_bundles'] = 'modules/class-wc-bis-bundles-compatibility.php';
		}

		// WC Pre-Orders support.
		if ( class_exists( 'WC_Pre_Orders' ) && defined( 'WC_VERSION' ) && version_compare( WC_VERSION, '3.6' ) >= 0 ) {
			$module_paths['wc_pre_orders'] = 'modules/class-wc-bis-pre-orders-compatibility.php';
		}

		// WCS support.
		if ( class_exists( 'WC_Subscriptions' ) || class_exists( 'WC_Subscriptions_Core_Plugin' ) ) {
			$module_paths['wc_subscriptions'] = 'modules/class-wc-bis-wcs-compatibility.php';
		}

		/**
		 * 'woocommerce_bis_compatibility_modules' filter.
		 *
		 * Use this to filter the required compatibility modules.
		 *
		 * @since  1.0.0
		 * @param  array $module_paths
		 */
		$module_paths = apply_filters( 'woocommerce_bis_compatibility_modules', $module_paths );

		foreach ( $module_paths as $path ) {
			require_once $path;
		}
	}

	/**
	 * Load time-sensitive compatibility classes.
	 *
	 * Runs on 'before_woocommerce_init' which runs on init with priority 0. This is to ensure that i10n needs of the FeaturesUtil class are met.
	 *
	 * @since 2.0.5
	 *
	 * @return void
	 */
	public static function deferred_module_includes() {
		$module_paths = array();

		// Product Editor support.
		if ( class_exists( 'Automattic\WooCommerce\Utilities\FeaturesUtil' ) && Automattic\WooCommerce\Utilities\FeaturesUtil::feature_is_enabled( 'product_block_editor' ) ) {
			$module_paths['wc_product_editor'] = 'modules/class-wc-bis-product-editor-compatibility.php';
		}

		foreach ( $module_paths as $path ) {
			require_once $path;
		}
	}

	/**
	 * Checks versions of compatible/integrated/deprecated extensions.
	 *
	 * @return void
	 */
	public static function add_compatibility_notices() {

		// WC Product Bundles version check.
		if ( class_exists( 'WC_Bundles' ) ) {

			$required_version = self::$required['wc_bundles'];

			if ( ! defined( 'WC_PB_VERSION' ) || version_compare( WC_PB_VERSION, $required_version ) < 0 ) {

				$extension      = __( 'Product Bundles', 'woocommerce-back-in-stock-notifications' );
				$extension_full = __( 'WooCommerce Product Bundles', 'woocommerce-back-in-stock-notifications' );
				$extension_url  = 'https://woocommerce.com/products/product-bundles/';
				/* translators: %1$s extension name, %2$s extension url, %3$s extension full name */
				$notice = sprintf( __( 'The installed version of <strong>%1$s</strong> is not supported by <strong>WooCommerce Back In Stock Notifications</strong>. Please update <a href="%2$s" target="_blank">%3$s</a> to version <strong>%4$s</strong> or higher.', 'woocommerce-back-in-stock-notifications' ), $extension, $extension_url, $extension_full, $required_version );

				WC_BIS_Admin_Notices::add_dismissible_notice(
					$notice,
					array(
						'dismiss_class' => 'wc_bundles_lt_' . $required_version,
						'type'          => 'warning',
					)
				);
			}
		}
	}
}

WC_BIS_Compatibility::init();
