<?php
/**
 * WC_BIS_Product_Editor_Compatibility class
 *
 * @package  WooCommerce Back In Stock Notifications
 * @since    1.7.0
 */

// Exit if accessed directly.
if ( ! defined( 'ABSPATH' ) ) {
	exit;
}

use Automattic\WooCommerce\Admin\BlockTemplates\BlockInterface;

/**
 * WooCommerce Product Editor compatibility.
 *
 * @version  1.7.0
 */
class WC_BIS_Product_Editor_Compatibility {

	/**
	 * Initialize integration.
	 */
	public static function init() {

		add_action( 'init', array( __CLASS__, 'register_blocks' ) );
		add_action( 'woocommerce_block_template_area_product-form_after_add_block_inventory', array( __CLASS__, 'add_blocks' ) );

		add_filter( 'woocommerce_rest_prepare_product_object', array( __CLASS__, 'rest_handle_product_response' ), 90, 3 );
		add_filter( 'woocommerce_rest_prepare_product_variation_object', array( __CLASS__, 'rest_handle_product_response' ), 90, 3 );
		add_filter( 'woocommerce_rest_pre_insert_product_object', array( __CLASS__, 'rest_handle_product_save' ), 10, 2 );
	}

	/**
	 * Register product editor blocks.
	 */
	public static function register_blocks() {
		if ( ! class_exists( 'Automattic\WooCommerce\Admin\PageController' ) || ! Automattic\WooCommerce\Admin\PageController::is_admin_page() ) {
			return;
		}

		if ( ! class_exists( 'Automattic\WooCommerce\Admin\Features\ProductBlockEditor\BlockRegistry' ) ) {
			return;
		}

		// Add register calls here.
		Automattic\WooCommerce\Admin\Features\ProductBlockEditor\BlockRegistry::get_instance()->register_block_type_from_metadata( WC_BIS_ABSPATH . 'assets/dist/admin/product-editor/flash-notices/block.json' );
		Automattic\WooCommerce\Admin\Features\ProductBlockEditor\BlockRegistry::get_instance()->register_block_type_from_metadata( WC_BIS_ABSPATH . 'assets/dist/admin/product-editor/notices/block.json' );
	}

	/**
	 * Add blocks to the Product Editor.
	 *
	 * @param BlockInterface $group The template interface.
	 */
	public static function add_blocks( $group ) {

		// Sanity.
		if ( ! method_exists( $group, 'add_block' ) || ! method_exists( $group, 'add_section' ) ) {
			return;
		}

		$hide_conditions = array();
		$supported_types = wc_bis_get_supported_types();

		foreach ( $supported_types as $type ) {

			// Skip variations.
			if ( 'variation' === $type ) {
				continue;
			}

			$hide_conditions[] = sprintf( "editedProduct.type !== '%s'", $type );
		}

		// Add the flash notices area (include variations.)
		$group->add_block(
			array(
				'id'             => 'wc-bis-flash-notices',
				'blockName'      => 'woocommerce-back-in-stock-notifications/flash-notices',
				'hideConditions' => array(
					array(
						// Include variations here.
						'expression' => implode( ' && ', array_merge( $hide_conditions, array( "postType !== 'product_variation'" ) ) ),
					),
				),
			)
		);

		// These will show everywhere that is supported by this wc_bis_get_supported_types (except on variations.)
		$group->add_block(
			array(
				'id'             => 'wc-bis-notices',
				'blockName'      => 'woocommerce-back-in-stock-notifications/notices',
				'attributes'     => array(
					'type' => 'info',
				),
				'hideConditions' => array(
					array(
						// Include variations here.
						'expression' => implode( ' && ', $hide_conditions ),
					),
				),
			)
		);

		// Create a new section.
		$section = $group->add_section(
			array(
				'id'             => 'wc-bis-section',
				'attributes'     => array(
					'title' => __( 'Stock Notifications', 'woocommerce-back-in-stock-notifications' ),
				),
				'hideConditions' => array(
					array(
						'expression' => implode( ' && ', $hide_conditions ),
					),
				),
			)
		);

		if ( 'yes' === get_option( 'wc_bis_allow_signups', 'yes' ) ) {
			// Add the checkbox.
			$section->add_block(
				array(
					'id'         => 'wc-bis-product-enabled',
					'blockName'  => 'woocommerce/product-toggle-field',
					'attributes' => array(
						'label'          => __(
							'Let customers sign up to be notified when this product is restocked',
							'woocommerce-back-in-stock-notifications'
						),
						'tooltip'        => __(
							'When enabled, customers can sign up to be notified when this product is restocked.',
							'woocommerce-back-in-stock-notifications'
						),
						'property'       => '_wc_bis_enabled',
						'checkedValue'   => 'yes',
						'uncheckedValue' => 'no',
					),
				)
			);
			return;
		}
		$section->add_block(
			array(
				'id'         => 'wc-bis-disabled-notice',
				'blockName'  => 'woocommerce/product-notice-field',
				'attributes' => array(
					'message' => sprintf( __( 'Sign-ups for stock notifications are disabled for all products in the store. To control sign-ups for this product, first enable the global <a href="%s">"Allow sign-ups"</a> option.', 'woocommerce-back-in-stock-notifications' ), '/wp-admin/admin.php?page=wc-settings&tab=bis_settings' ),
				),
			)
		);
	}

	/**
	 * Add the meta data to the rest response.
	 *
	 * @param WP_REST_Response $response The response.
	 * @param WC_Product       $product The product.
	 * @param WP_REST_Request  $request The request.
	 *
	 * @return array
	 */
	public static function rest_handle_product_response( $response, $product, $request ) {

		if ( 'edit' !== $request->get_param( 'context' ) ) {
			return $response;
		}

		if ( ! $product->is_type( wc_bis_get_supported_types() ) ) {
			return $response;
		}

		// Revert the negative logic on the checkbox into the '_wc_bis_enabled' runtime meta.
		// Hint: Variations will be handled by the parent product.
		$has_feature_enabled               = ! $product->is_type( 'variation' ) ? 'yes' !== $product->get_meta( '_wc_bis_disabled', true ) : 'yes' !== get_post_meta( $product->get_parent_id(), '_wc_bis_disabled', true );
		$response->data['_wc_bis_enabled'] = $has_feature_enabled ? 'yes' : 'no';

		// Note: We only need to check the response for the stock_status key, as that is the only key that triggers a sync action.
		// However, it might be possible from the sync action to trigger from other 3PD-related conditions (Stock Management Third-Parties etc.) If issues occur, we might need to adjust this logic.
		if ( $request->has_param( 'stock_status' ) ) {

			if ( ! class_exists( 'WC_BIS_Admin_Notices' ) ) {
				require_once WC_BIS_ABSPATH . 'includes/admin/class-wc-bis-admin-notices.php';
			}

			// Force a sync action.
			// Hint: Shutdown sync will be skipped.
			WC_BIS()->sync->sync();

			// Grab flash notices happened during sync.
			$notices = self::rest_prepare_flash_notices( (array) WC_BIS_Admin_Notices::$meta_box_notices );
			if ( ! empty( $notices ) ) {
				$response->data['_wc_bis_notices'] = $notices;
			}

			// Prevent notices from saving in the database during shutdown.
			WC_BIS_Admin_Notices::$meta_box_notices = array();
		}

		if ( ! $has_feature_enabled || 'publish' !== $product->get_status() ) {

			$product_ids = array( $product->get_id() );
			if ( $product->is_type( 'variable' ) ) {
				$product_ids = array_merge( $product_ids, $product->get_children() );
			}

			// Count existing and active notifications.
			$response->data['_wc_bis_notifications_count'] = (int) wc_bis_get_notifications_count( $product_ids, true );
		}

		return $response;
	}

	/**
	 * Save the meta data to the rest response.
	 *
	 * @param WC_Product      $product The product.
	 * @param WP_REST_Request $request The request.
	 *
	 * @return WC_Product
	 */
	public static function rest_handle_product_save( $product, $request ) {

		// Prevent from saving.
		if ( $request->has_param( '_wc_bis_enabled' ) ) {

			$params     = $request->get_params();
			$is_enabled = 'yes' === $params['_wc_bis_enabled'];

			if ( $is_enabled ) {
				$product->delete_meta_data( '_wc_bis_disabled' );
			} else {
				$product->add_meta_data( '_wc_bis_disabled', 'yes', true );
			}
		}

		return $product;
	}

	/**
	 * Prepare flash notices for the REST response.
	 *
	 * @param array $notices_data The notices from WC_BIS_Admin_Notices.
	 *
	 * @return array
	 */
	public static function rest_prepare_flash_notices( $notices_data ) {

		if ( empty( $notices_data ) ) {
			return array();
		}

		if ( ! is_array( $notices_data ) ) {
			$notices_data = array( $notices_data );
		}

		$allowed_types = array( 'error', 'warning', 'success', 'info' );
		$notices       = array_map(
			function ( $notice ) use ( $allowed_types ) {
				return array(
					'type'    => in_array( $notice['type'], $allowed_types, true ) ? $notice['type'] : 'info',
					'message' => wp_strip_all_tags( $notice['content'] ),
					'actions' => ! empty( $notice['actions'] ) ? array_filter(
						array_map(
							function ( $action ) {
								if ( empty( $action['text'] ) || empty( $action['name'] ) ) {
										return false;
								}

								return array(
									'name' => wp_strip_all_tags( sanitize_text_field( $action['name'] ) ),
									'text' => wp_strip_all_tags( sanitize_text_field( $action['text'] ) ),
									'url'  => isset( $action['url'] ) ? esc_url_raw( $action['url'] ) : '',
									'data' => isset( $action['data'] ) ? wc_clean( $action['data'] ) : array(),
								);
							},
							$notice['actions']
						)
					) : array(),
				);
			},
			$notices_data
		);

		return $notices;
	}
}

WC_BIS_Product_Editor_Compatibility::init();
