<?php

/**
 * Copyright (c) 2015-2021 Jamez Picard
 * Simple and approximate geo location.
 * Using avg and near lat/long for zip code prefixes and postal code prefixes.
 * Also using state approx lat/long
 * Data updated 2016/11
 * THIS SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 */
class cpwebservice_location {

    private $postal_loc = null;
    private $zip_loc    = null;
    private $state_loc  = null;
    
    public function distance($lat, $lng, $country, $state, $zip){
        // Get GeoLoc.
        $loc = null;
        if ($country == 'CA'){
            $loc = $this->lookup_postal_location($this->postal_prefix($zip));
        } elseif(in_array($country, array('US','PR','VI','GU','UM','MP'))){
            $loc = $this->lookup_zip_location($this->postal_prefix($zip));
            // Backup plan if not found: use state.
            if ($loc == null){
                $loc = $this->lookup_state_location($state);
            }
        } else {
            return null; // international
        }
    
        if (!empty($loc) && !empty($lat) && !empty($lng)){
            $distance = $this->geo_distance($loc[0],$loc[1],$lat,$lng);
    
            return $distance;
        }
        return null;
    }
    
    public function distance_state($lat, $lng, $country, $state){
        // Get GeoLoc.
        $loc = $this->lookup_state_location($state);

        if (!empty($loc) && !empty($lat) && !empty($lng)){
            $distance = $this->geo_distance($loc[0],$loc[1],$lat,$lng);
    
            return $distance;
        }
        return 0;
    }
    
    // Not used.
    private function distance_postal($country1, $zip1, $country2, $zip2){
        // Get GeoLoc.
        $loc1 = null;
        if ($country1 == 'CA'){
            $loc1 = $this->lookup_postal_location($zip1);
        } elseif(in_array($country1, array('US','PR','VI','GU','UM','MP'))){
            $loc1 = $this->lookup_zip_location($zip1);
        }
        
        // Get GeoLoc.
        $loc2 = null;
        if ($country2 == 'CA'){
            $loc2 = $this->lookup_postal_location($zip2);
        } elseif(in_array($country2, array('US','PR','VI','GU','UM','MP'))){
            $loc2 = $this->lookup_zip_location($zip2);
        }
        
        if (!empty($loc1) && !empty($loc2)){
            $distance = $this->geo_distance($loc1[0],$loc1[1],$loc2[0],$loc2[1]);
            
            return $distance;
        }
        return 0;
    }
    
    /*
     * Gets 3-digit prefix of Postal or Zip code.
     */
    public function postal_prefix($postalcode){
        return strlen($postalcode) >= 3 ? substr($postalcode, 0, 3) : $postalcode;
    }
    
    /*
     * Calculates distance between points, given latitude/longitude
     */
    public function geo_distance($lat1, $lon1, $lat2, $lon2){
         
        $theta = $lon1 - $lon2;
        $dist = sin(deg2rad($lat1)) * sin(deg2rad($lat2)) +  cos(deg2rad($lat1)) * cos(deg2rad($lat2)) * cos(deg2rad($theta));
        $dist = acos($dist);
        $dist = rad2deg($dist);
        $km = ($dist * 60 * 1.8531596);
         
        return $km;
    }
    
    /*
     * Provide approx postal code latitude/longitude
     */
    public function lookup_postal_location($postal_prefix){
        if ($this->postal_loc==null) { $this->postal_loc = json_decode('{"A0A":[47.4852371,-53.1029892],"A0B":[47.6097107,-53.5578461],"A0C":[48.4616356,-53.5887413],"A0E":[47.3981895,-54.9111137],"A0G":[49.1859894,-54.3039551],"A0H":[48.7377319,-55.5968857],"A0J":[49.513073,-56.0406837],"A0K":[50.7441597,-56.5364609],"A0L":[49.0103951,-57.6445351],"A0M":[47.621788,-58.7197533],"A0N":[48.2324486,-58.7486115],"A0P":[54.4347229,-60.1786575],"A0R":[53.2072258,-65.428009],"A1A":[47.3529015,-52.3932991],"A1B":[46.3927116,-51.435276],"A1C":[47.3884697,-52.5209198],"A1E":[47.3967209,-52.5930786],"A1G":[47.5101166,-52.7566643],"A1H":[41.1212769,-45.7393646],"A1K":[46.8692398,-51.8581123],"A1L":[45.9988899,-51.1605797],"A1M":[47.3198776,-52.5356216],"A1N":[46.7575836,-51.9665451],"A1S":[46.641964,-51.8647232],"A1V":[48.8211823,-54.4560623],"A1W":[46.8009071,-52.138401],"A1X":[46.2348633,-51.6145592],"A1Y":[47.7343941,-53.2252464],"A2A":[48.2660789,-54.8859406],"A2B":[48.9523125,-55.6661453],"A2H":[48.6999474,-57.6541634],"A2N":[47.6960068,-57.5570984],"A2V":[51.7318039,-65.3780823],"A5A":[46.9860039,-52.6474762],"A8A":[48.978199,-57.1663857],"B0C":[46.5514488,-60.5095329],"B0E":[45.865612,-61.010479],"B0H":[45.5583496,-61.6921043],"B0J":[44.6132126,-63.923069],"B0K":[45.6339378,-62.8532143],"B0L":[45.7976418,-64.1177292],"B0M":[45.4975624,-63.7925949],"B0N":[45.0298042,-63.6775551],"B0P":[45.0516472,-64.710762],"B0R":[44.5004501,-64.6538315],"B0S":[44.8191071,-65.3358307],"B0T":[43.9783745,-65.1342926],"B0V":[44.5827675,-65.8636017],"B0W":[43.8545227,-65.870018],"B1A":[45.9726524,-59.6707191],"B1B":[46.1010284,-59.924408],"B1C":[45.9526024,-59.9551735],"B1E":[46.1839638,-60.015667],"B1G":[46.2101974,-60.0182724],"B1H":[46.1324959,-59.9457016],"B1J":[45.5201149,-59.9205894],"B1K":[45.874897,-60.0380249],"B1L":[46.073349,-60.2616463],"B1M":[46.1114273,-60.1263695],"B1N":[46.0650711,-60.0520096],"B1P":[46.0067825,-60.0101891],"B1R":[45.9538155,-59.9957237],"B1S":[45.9348717,-59.9703865],"B1T":[45.3657646,-59.4067726],"B1V":[46.2384987,-60.2274971],"B1W":[45.9380379,-60.6424675],"B1X":[46.219902,-60.4407501],"B1Y":[46.1521187,-60.2033501],"B2A":[45.9166794,-59.9227524],"B2C":[45.7702789,-61.5108948],"B2E":[45.7496643,-60.6119156],"B2G":[45.4691505,-61.7840843],"B2H":[45.5417747,-62.579731],"B2J":[45.4645691,-62.4451637],"B2N":[45.2348289,-63.0983543],"B2R":[44.5172386,-63.2264099],"B2S":[44.7059441,-63.1002007],"B2T":[44.1195183,-62.5356522],"B2V":[44.6611252,-63.4841118],"B2W":[44.5463409,-63.3243637],"B2X":[44.6983414,-63.541172],"B2Y":[44.597847,-63.45298],"B2Z":[44.6955566,-63.4322815],"B3A":[44.6159248,-63.4891701],"B3B":[44.3498878,-63.0920334],"B3E":[44.5569305,-63.1043777],"B3G":[44.451313,-63.233551],"B3H":[44.4285469,-63.289978],"B3J":[44.1704369,-62.9109726],"B3K":[44.5574226,-63.4527168],"B3L":[44.5113983,-63.4160805],"B3M":[44.1758537,-62.949276],"B3N":[44.5468903,-63.492733],"B3P":[44.489727,-63.4182472],"B3R":[44.2816277,-63.1525917],"B3S":[43.1087723,-61.4604454],"B3T":[43.3469543,-61.9628754],"B3V":[44.2066154,-63.1234474],"B3Z":[44.3106613,-63.3652382],"B4A":[44.2744064,-63.0263023],"B4B":[43.9077072,-62.5743103],"B4C":[44.5504341,-63.3615417],"B4E":[42.7418938,-60.7902718],"B4G":[44.8429413,-63.6861534],"B4H":[45.7779808,-64.1103058],"B4N":[44.777523,-64.0643845],"B4P":[44.970211,-64.2063751],"B4R":[45.0639839,-64.5826645],"B4V":[44.2650375,-64.4028244],"B5A":[43.8442116,-66.1097336],"B6L":[44.9280777,-62.5912781],"B9A":[45.6355438,-61.3671608],"C0A":[37.5891647,-51.0134926],"C0B":[46.5313454,-63.884964],"C1A":[46.0447769,-62.8514709],"C1B":[28.2665672,-38.588047],"C1C":[27.6155872,-37.6605568],"C1E":[27.4737873,-37.5017281],"C1N":[46.1117897,-63.385376],"E1A":[44.8694916,-63.012722],"E1B":[44.923748,-63.2002869],"E1C":[45.9434814,-64.5834503],"E1E":[45.5508041,-64.076889],"E1G":[44.7605629,-62.9627075],"E1H":[45.7106361,-64.1209717],"E1J":[45.6743889,-64.3374176],"E1N":[46.8243179,-65.1808014],"E1V":[46.8593559,-65.333725],"E1W":[46.2946281,-63.0375748],"E1X":[46.8695259,-64.0922318],"E2A":[47.6139908,-65.6716385],"E2E":[45.2251282,-65.6952438],"E2G":[44.0107002,-63.8511925],"E2H":[45.2146606,-65.82724],"E2J":[45.1769562,-65.8395309],"E2K":[45.1739235,-65.9157639],"E2L":[44.8342934,-65.4196472],"E2M":[45.1458549,-65.9517136],"E2N":[44.6327744,-64.9318695],"E2P":[45.2492447,-66.0021896],"E2R":[45.2751007,-66.0091629],"E2S":[45.021183,-65.3595734],"E2V":[45.4331627,-65.8844604],"E3A":[45.8733292,-66.4685745],"E3B":[45.6147156,-66.196907],"E3C":[44.9307251,-65.2956924],"E3E":[45.6364822,-66.4347458],"E3G":[44.8783875,-65.0661392],"E3L":[45.0523834,-67.0545654],"E3N":[47.9280396,-66.6250916],"E3V":[47.3022079,-68.2317734],"E3Y":[46.9892731,-67.6082764],"E3Z":[46.9047394,-67.5831528],"E4A":[46.1261787,-65.8803635],"E4B":[45.9657478,-65.9727478],"E4C":[45.7244644,-65.7280197],"E4E":[45.3660316,-64.9859161],"E4G":[45.8009224,-65.463501],"E4H":[45.6308594,-64.4916458],"E4J":[45.6369553,-64.51548],"E4K":[45.7971802,-64.3092957],"E4L":[45.7769585,-64.1641922],"E4M":[45.8521538,-63.6289635],"E4N":[45.8629951,-63.8441658],"E4P":[45.463459,-63.4320984],"E4R":[46.0677452,-64.2343292],"E4S":[46.0666962,-64.225441],"E4T":[46.2006645,-64.7778091],"E4V":[46.2191238,-64.4844284],"E4W":[46.3483391,-64.4742889],"E4X":[46.5824165,-64.8171234],"E4Y":[46.7276039,-65.3756332],"E4Z":[45.8887711,-65.1374435],"E5A":[45.31427,-67.158493],"E5B":[45.0337715,-66.8980713],"E5C":[44.3713913,-65.7247086],"E5E":[44.6369247,-66.5500717],"E5G":[44.7036095,-66.7738953],"E5H":[44.7689018,-66.2456055],"E5J":[44.8689041,-66.0173264],"E5K":[44.9863434,-65.6841431],"E5L":[45.6798058,-66.4821472],"E5M":[45.6412086,-66.1109924],"E5N":[45.1420097,-65.2722015],"E5P":[45.4686241,-65.2986221],"E5R":[44.7524834,-64.7619629],"E5S":[45.3998718,-66.0989304],"E5T":[45.5016632,-65.6705017],"E5V":[44.9264488,-66.4453354],"E6A":[46.3943939,-66.5467224],"E6B":[46.0226631,-66.4079742],"E6C":[46.1825104,-66.5846939],"E6E":[45.6211548,-66.518837],"E6G":[45.9510918,-67.3483582],"E6H":[45.6668167,-67.2758942],"E6J":[45.5954742,-67.3080444],"E6K":[45.581871,-66.7601852],"E6L":[45.7775459,-66.5469513],"E7A":[47.1028328,-68.3644485],"E7B":[47.1651039,-67.9734192],"E7C":[47.1279831,-67.8718414],"E7E":[47.1921921,-67.9090881],"E7G":[46.9246254,-67.4847336],"E7H":[46.4785652,-67.2973785],"E7J":[46.5253105,-67.621788],"E7K":[46.4410553,-67.6020737],"E7L":[46.4470444,-67.5229034],"E7M":[46.0061836,-67.3826904],"E7N":[46.0919685,-67.5847092],"E7P":[46.3105125,-67.497139],"E8A":[47.5318146,-67.3934479],"E8B":[47.6583557,-67.3264236],"E8C":[47.9341431,-66.2107773],"E8E":[47.8452911,-66.1736298],"E8G":[47.5013618,-65.4578171],"E8J":[47.6826096,-65.5922623],"E8K":[47.5318832,-65.5082245],"E8L":[47.3343086,-64.7557678],"E8M":[47.3584862,-64.5708237],"E8N":[47.7819176,-65.0328598],"E8P":[47.6711464,-64.9726791],"E8R":[47.2687302,-64.2697983],"E8S":[47.6365738,-64.5468292],"E8T":[47.7402229,-64.4897766],"E9A":[47.0327988,-64.9600296],"E9B":[46.1926346,-65.0836563],"E9C":[46.552372,-66.1665802],"E9E":[46.6299248,-65.4133377],"E9G":[47.1516914,-64.9633179],"E9H":[47.3703079,-64.9922256],"G0A":[46.93396,-71.3558502],"G0C":[46.7813148,-63.4725266],"G0E":[49.1859016,-65.5705795],"G0G":[51.0869522,-64.9481354],"G0H":[49.1767311,-68.1420517],"G0J":[48.5650597,-67.5118103],"G0K":[48.3368607,-68.3661728],"G0L":[47.6700706,-69.0121078],"G0M":[46.0277939,-70.7277374],"G0N":[45.6009941,-70.4581757],"G0P":[44.9535332,-70.0724792],"G0R":[46.7966957,-70.5175705],"G0S":[46.491394,-71.3498611],"G0T":[48.2159081,-69.6718063],"G0V":[48.4598274,-70.9010468],"G0W":[48.6343536,-71.9825058],"G0X":[46.6225014,-72.6047821],"G0Y":[44.3680801,-69.0404129],"G0Z":[46.1770134,-72.1716385],"G1A":[44.93964,-68.367897],"G1B":[46.0098152,-69.8523788],"G1C":[46.6511307,-70.8386002],"G1E":[46.2314148,-70.2318497],"G1G":[46.1159096,-70.1292343],"G1H":[46.7319794,-71.0803146],"G1J":[46.0806313,-70.0765839],"G1K":[46.6478348,-70.9748383],"G1L":[46.6966019,-71.036499],"G1M":[46.8184967,-71.2675858],"G1N":[46.7336349,-71.1402359],"G1P":[46.7403259,-71.2036896],"G1R":[46.750103,-71.1286621],"G1S":[46.3862038,-70.6242752],"G1T":[46.6223717,-71.0248108],"G1V":[46.7112656,-71.1782303],"G1W":[46.7641296,-71.2889481],"G1X":[46.5461464,-70.9741898],"G1Y":[46.7587967,-71.3505325],"G2A":[46.7020035,-71.1256104],"G2B":[46.7929726,-71.2707291],"G2C":[46.389267,-70.6579285],"G2E":[46.468071,-70.8413315],"G2G":[46.2182617,-70.5054779],"G2J":[46.268425,-70.4085922],"G2K":[45.4439354,-69.1689606],"G2L":[46.6879425,-70.9645309],"G2M":[46.424511,-70.5680237],"G2N":[46.6108818,-70.8859787],"G3A":[46.5873833,-71.1930161],"G3B":[46.3258286,-70.3396378],"G3C":[46.7404251,-70.9687347],"G3E":[46.5443954,-70.8343353],"G3G":[46.8938484,-71.3653641],"G3H":[46.3318291,-71.0239563],"G3J":[45.4522552,-69.2883224],"G3K":[45.7044563,-69.6732788],"G3L":[46.5872726,-71.3544083],"G3M":[45.2349014,-69.5095596],"G3N":[45.2608337,-69.1745834],"G3Z":[46.3829041,-68.9441376],"G4A":[47.0037613,-69.2079391],"G4R":[49.0481262,-64.8218002],"G4S":[49.9883156,-66.0541916],"G4T":[47.0135078,-61.3812027],"G4V":[49.1259575,-66.4628143],"G4W":[47.8042068,-66.0894165],"G4X":[48.3282471,-63.7252846],"G4Z":[49.21772,-68.1712036],"G5A":[47.5753899,-69.9910736],"G5B":[49.5592766,-66.2534027],"G5C":[49.1959114,-68.27034],"G5H":[48.33004,-67.8281174],"G5J":[47.7167931,-66.3894272],"G5L":[48.3772621,-68.4395142],"G5M":[42.8082199,-60.4555054],"G5N":[48.0162888,-67.9591141],"G5R":[47.254715,-68.7006683],"G5T":[46.8068314,-67.574234],"G5V":[45.77005,-68.7422562],"G5X":[45.8498039,-70.2212524],"G5Y":[45.8320198,-70.230545],"G5Z":[45.2090149,-69.2686462],"G6A":[46.1344833,-70.6721954],"G6B":[45.3853683,-70.5819092],"G6C":[46.3855286,-70.5575256],"G6E":[45.1539192,-69.0486679],"G6G":[46.067379,-71.245903],"G6H":[45.8298645,-70.9358902],"G6J":[46.0237045,-70.3395386],"G6K":[45.3541756,-69.236824],"G6L":[45.7454567,-71.1029892],"G6P":[45.9383507,-71.7725067],"G6R":[46.0316086,-71.9747391],"G6S":[43.7984314,-68.4042282],"G6T":[44.0379105,-68.7961731],"G6V":[45.9547615,-69.86409],"G6W":[46.5170898,-70.8479538],"G6X":[46.7170715,-71.267189],"G6Z":[46.6369553,-71.0721817],"G7A":[45.7366562,-69.8538513],"G7B":[48.208252,-70.7070618],"G7G":[47.9783058,-70.3819122],"G7H":[47.983345,-70.4167175],"G7J":[47.4547615,-69.671936],"G7K":[44.3571701,-65.1905594],"G7N":[46.7912025,-68.8665924],"G7P":[44.4299088,-65.2854691],"G7S":[48.1178017,-70.733551],"G7T":[34.9598236,-51.4259987],"G7X":[48.1075935,-70.8114166],"G7Y":[44.6630745,-65.7853622],"G7Z":[48.4294815,-71.2662582],"G8A":[47.0404129,-69.2465286],"G8B":[48.4029655,-71.4272919],"G8C":[47.0654564,-69.4836121],"G8E":[48.4453201,-71.4168472],"G8G":[48.2345428,-71.5753555],"G8H":[46.8738136,-69.7898407],"G8J":[47.3668785,-70.5577545],"G8K":[48.658165,-72.4417343],"G8L":[48.7802582,-72.0731583],"G8M":[48.8574219,-72.4760132],"G8N":[47.8372307,-70.8599548],"G8P":[49.7653542,-74.1476059],"G8T":[46.2561798,-72.3409576],"G8V":[44.9733505,-70.2946777],"G8W":[43.6364174,-68.2671509],"G8Y":[46.2467308,-72.4266968],"G8Z":[46.2723465,-72.4518356],"G9A":[46.2024727,-72.3414993],"G9B":[43.0562859,-67.5264206],"G9C":[38.7477112,-60.6879425],"G9H":[45.2057915,-70.6517563],"G9N":[46.5634689,-72.7413635],"G9P":[46.00037,-71.935112],"G9R":[46.3174973,-72.3621826],"G9T":[45.8078766,-71.4317322],"G9X":[47.3017693,-72.5769501],"H0M":[45.0016823,-74.6439362],"H1A":[45.4568977,-73.1519318],"H1B":[45.5867729,-73.4177475],"H1C":[45.5061989,-73.3184204],"H1E":[45.556797,-73.4574814],"H1G":[45.6091499,-73.624855],"H1H":[45.5894852,-73.6393127],"H1J":[45.0509071,-72.6730728],"H1K":[45.1884651,-72.870491],"H1L":[45.4655571,-73.3034363],"H1M":[45.5871277,-73.5561829],"H1N":[45.3864632,-73.2339401],"H1P":[45.5954742,-73.5933456],"H1R":[45.5848961,-73.6071167],"H1S":[45.4671707,-73.4031677],"H1T":[45.5729637,-73.5715103],"H1V":[45.5539284,-73.5391159],"H1W":[45.5408745,-73.5502625],"H1X":[45.5555725,-73.5729904],"H1Y":[45.5494194,-73.580452],"H1Z":[45.5714111,-73.6191559],"H2A":[45.5608521,-73.5991364],"H2B":[45.5730057,-73.6494827],"H2C":[45.5595741,-73.6584167],"H2E":[45.5512619,-73.6120987],"H2G":[45.544281,-73.592865],"H2H":[45.378212,-73.3183365],"H2J":[45.529274,-73.5837021],"H2K":[45.1275024,-72.9088821],"H2L":[45.2992172,-73.213623],"H2M":[45.4100189,-73.4113541],"H2N":[45.5396957,-73.650589],"H2P":[44.2976494,-71.6197205],"H2R":[45.4161682,-73.4188538],"H2S":[45.0311775,-72.7941208],"H2T":[45.3923416,-73.3798981],"H2V":[45.4462318,-73.4912338],"H2W":[45.3447876,-73.2992401],"H2X":[45.2223816,-73.0910873],"H2Y":[45.5056458,-73.5592117],"H2Z":[44.3302002,-71.6635895],"H3A":[41.7870522,-67.5638733],"H3B":[45.5029182,-73.5725861],"H3C":[44.7410889,-72.3715134],"H3E":[43.943676,-71.0826645],"H3G":[45.4989357,-73.5872803],"H3H":[45.4949303,-73.5815353],"H3J":[45.3191032,-73.2942429],"H3K":[45.2332344,-73.1614151],"H3L":[45.5476494,-73.6694336],"H3M":[45.5392609,-73.6919556],"H3N":[45.5303841,-73.6313171],"H3P":[45.5216064,-73.6375656],"H3R":[45.5114136,-73.6476212],"H3S":[45.5064316,-73.6291275],"H3T":[45.5005569,-73.6198578],"H3V":[44.667881,-72.2758636],"H3W":[45.4898529,-73.6331482],"H3X":[45.4818115,-73.6405945],"H3Y":[45.487339,-73.6043167],"H3Z":[45.0919724,-72.9659653],"H4A":[45.4756241,-73.6154404],"H4B":[45.4662552,-73.6306],"H4C":[44.831604,-72.5339355],"H4E":[45.4571075,-73.5946732],"H4G":[45.4630737,-73.5704346],"H4H":[45.4466782,-73.5809555],"H4J":[45.3728447,-73.4489975],"H4K":[45.5245895,-73.7278748],"H4L":[45.3991585,-73.4920044],"H4M":[45.0991592,-73.0342026],"H4N":[45.0465393,-72.8899002],"H4P":[44.6536827,-72.2873383],"H4R":[44.1093597,-71.4486465],"H4S":[44.7308846,-72.5073547],"H4T":[45.1403961,-73.1369553],"H4V":[45.4674683,-73.648407],"H4W":[45.3615685,-73.4934158],"H4X":[45.4548988,-73.6516571],"H4Y":[45.4765663,-73.7174683],"H4Z":[45.5161705,-73.5414352],"H5A":[45.5048332,-73.5622482],"H5B":[44.6591873,-72.1886978],"H7A":[45.274025,-72.9542694],"H7B":[44.5144043,-71.7439346],"H7C":[45.3319702,-73.1988525],"H7E":[45.2816849,-73.1630096],"H7G":[45.5750885,-73.6833649],"H7H":[45.4251518,-73.4081192],"H7J":[45.1807404,-72.9116135],"H7K":[45.3117867,-73.2381287],"H7L":[45.3473473,-73.3647156],"H7M":[45.549942,-73.6367645],"H7N":[45.5146332,-73.6301727],"H7P":[44.9105148,-72.7214966],"H7R":[45.3246384,-73.486351],"H7S":[45.5742722,-73.7303314],"H7T":[44.1328278,-71.4642792],"H7V":[45.4666443,-73.6046753],"H7W":[44.8826218,-72.7004852],"H7X":[45.1773415,-73.2386017],"H7Y":[45.1758804,-73.2835388],"H8N":[44.2416115,-71.6846085],"H8P":[45.4267235,-73.6052933],"H8R":[45.2636185,-73.3725052],"H8S":[45.0410385,-73.0287781],"H8T":[45.2482071,-73.3871765],"H8Y":[45.5067329,-73.8000641],"H8Z":[45.5064316,-73.8412857],"H9A":[45.2981911,-73.5096359],"H9B":[45.3846054,-73.6276016],"H9C":[45.3822899,-73.6821747],"H9E":[45.4872665,-73.9002075],"H9G":[45.4758339,-73.8367081],"H9H":[45.471386,-73.8582611],"H9J":[45.3777046,-73.7544708],"H9K":[45.4578896,-73.898407],"H9P":[45.4569702,-73.7622528],"H9R":[45.3530197,-73.6400146],"H9S":[45.3900185,-73.6829147],"H9W":[45.4314346,-73.8659134],"H9X":[45.4160576,-73.9270172],"J0A":[45.862545,-71.9966812],"J0B":[45.3843193,-71.8562851],"J0C":[45.9179077,-72.4439011],"J0E":[45.1001282,-72.2417374],"J0G":[45.2597198,-71.6313553],"J0H":[45.5859566,-72.7042007],"J0J":[45.1432228,-73.1970978],"J0K":[46.1639442,-73.589592],"J0L":[45.4932899,-73.3613815],"J0M":[57.3486595,-74.3008804],"J0N":[45.5712738,-73.9374619],"J0P":[45.3757591,-74.2446213],"J0R":[45.8062744,-74.0275421],"J0S":[45.1414146,-74.0657806],"J0T":[46.0952492,-74.3813553],"J0V":[45.7465668,-74.7885437],"J0W":[46.5144882,-75.4401627],"J0X":[45.832859,-76.095459],"J0Y":[48.550518,-77.8375015],"J0Z":[47.7972565,-78.6811676],"J1A":[44.8733978,-71.4093933],"J1C":[45.14748,-71.4425201],"J1E":[44.5562553,-70.5128479],"J1G":[45.121151,-71.4120865],"J1H":[45.2073479,-71.6039734],"J1J":[45.0467529,-71.3487244],"J1K":[45.3857117,-71.9351883],"J1L":[44.9723167,-71.274086],"J1M":[45.0339432,-71.333168],"J1N":[44.2883835,-70.275177],"J1R":[45.3922844,-72.0245361],"J1S":[45.4718666,-71.8408051],"J1T":[45.7048836,-71.8182144],"J1X":[45.1342468,-71.9248047],"J1Z":[45.7721443,-72.2035522],"J2A":[44.6512871,-70.5484848],"J2B":[45.251194,-71.5062027],"J2C":[45.1490784,-71.337532],"J2E":[41.2585831,-65.1902618],"J2G":[45.3740616,-72.6851349],"J2H":[43.8881226,-70.2675018],"J2J":[45.3123665,-72.6239395],"J2K":[44.4743271,-71.5601196],"J2L":[44.329258,-71.1224594],"J2M":[42.312191,-67.7202072],"J2N":[44.6193771,-71.9088898],"J2R":[45.0783005,-72.0325775],"J2S":[45.4382668,-72.6637115],"J2T":[44.9982834,-71.9399033],"J2W":[45.2412872,-73.1058884],"J2X":[45.1043472,-72.9144821],"J2Y":[44.5647049,-72.1018982],"J3A":[45.0879288,-72.8744736],"J3B":[45.2378578,-73.1640244],"J3E":[45.4072952,-73.049408],"J3G":[45.0439529,-72.3772507],"J3H":[45.4283066,-72.9925079],"J3L":[44.2483521,-71.3510513],"J3M":[44.2374649,-71.2380295],"J3N":[45.1284332,-72.6515427],"J3P":[44.152462,-70.1194],"J3R":[42.736515,-67.9109726],"J3T":[45.3265877,-71.1822739],"J3V":[45.0192642,-72.5222397],"J3X":[45.6763649,-73.4235916],"J3Y":[44.7423248,-72.2118454],"J3Z":[43.0178795,-69.4076157],"J4B":[45.5062103,-73.2917252],"J4G":[45.5631523,-73.4770432],"J4H":[45.2833824,-73.0965424],"J4J":[45.476162,-73.390892],"J4K":[45.5178032,-73.4964218],"J4L":[45.4538002,-73.354599],"J4M":[45.4086761,-73.2419128],"J4N":[44.1490364,-71.1977997],"J4P":[45.5067101,-73.5079346],"J4R":[45.4937325,-73.502037],"J4S":[45.4809074,-73.493187],"J4T":[45.4971008,-73.4683685],"J4V":[45.3919067,-73.3126373],"J4W":[45.4680252,-73.482399],"J4X":[45.0873299,-72.9056396],"J4Y":[43.3856812,-70.127121],"J4Z":[44.6808014,-72.1780167],"J5A":[44.7441711,-72.5496826],"J5B":[45.2009392,-73.2552032],"J5C":[45.3995209,-73.5656357],"J5J":[44.5258026,-71.8340302],"J5K":[44.2565002,-71.7006912],"J5L":[44.9897385,-72.7661972],"J5M":[44.9334526,-72.2775726],"J5R":[44.96521,-72.7830582],"J5T":[45.2058258,-72.1987457],"J5V":[46.2543411,-72.9451675],"J5W":[45.7078972,-73.2037048],"J5X":[45.3307381,-72.6643143],"J5Y":[45.5910568,-73.1299362],"J5Z":[45.0000992,-72.2669449],"J6A":[45.2019501,-72.5894775],"J6E":[45.7327995,-72.9670944],"J6J":[45.2269707,-73.5095596],"J6K":[44.7491493,-72.7451172],"J6N":[44.7898598,-73.0029678],"J6R":[45.0315552,-73.2776718],"J6S":[45.1888695,-73.987854],"J6T":[45.0613632,-73.8250732],"J6V":[45.2197418,-72.7394409],"J6W":[45.7011604,-73.631897],"J6X":[45.3731613,-73.1198425],"J6Y":[45.4095154,-73.2960281],"J6Z":[45.6691055,-73.7721481],"J7A":[45.6389465,-73.7961426],"J7B":[45.6577263,-73.8169098],"J7C":[45.0948715,-72.9399643],"J7E":[45.5430374,-73.6791611],"J7G":[45.6143112,-73.8334351],"J7H":[44.236702,-71.6146774],"J7J":[42.8314667,-69.2924347],"J7K":[44.9073486,-72.2505493],"J7L":[45.6675606,-73.5367966],"J7M":[45.1335258,-72.698143],"J7N":[44.1018066,-71.521904],"J7P":[45.5736732,-73.8790054],"J7R":[45.3376961,-73.5582809],"J7T":[44.9034004,-73.3342056],"J7V":[44.6439667,-72.7863159],"J7W":[45.3703766,-73.9774857],"J7X":[42.1986389,-69.1868973],"J7Y":[45.4637794,-73.4829254],"J7Z":[45.5023575,-73.5513763],"J8A":[45.1933975,-72.8421173],"J8B":[45.7630234,-73.826416],"J8C":[43.0595398,-69.4632645],"J8E":[45.1041222,-72.9082794],"J8G":[45.2700806,-73.7331314],"J8H":[44.8703651,-73.0424423],"J8L":[45.253952,-74.8681259],"J8M":[45.4232788,-75.2693253],"J8N":[45.579586,-75.5225754],"J8P":[44.9756546,-74.7742844],"J8R":[44.0539742,-73.2139206],"J8T":[45.4773598,-75.7056122],"J8V":[45.3190384,-75.4018326],"J8X":[45.2527542,-75.4050903],"J8Y":[45.4013481,-75.6646118],"J8Z":[45.470417,-75.7596283],"J9A":[43.9655762,-73.3117981],"J9B":[45.0682678,-75.0562744],"J9E":[46.1326752,-75.5543594],"J9H":[44.7950706,-74.8142319],"J9J":[42.166832,-70.3864365],"J9L":[46.1165161,-74.8035202],"J9P":[47.8883553,-77.4396362],"J9T":[48.3629494,-77.7777405],"J9V":[46.7187119,-78.4154968],"J9X":[47.7887764,-78.278511],"J9Y":[29.0575848,-47.7058411],"J9Z":[48.8006401,-79.2020798],"K0A":[45.2919312,-75.7842636],"K0B":[45.5180931,-74.750885],"K0C":[45.1533241,-74.8729248],"K0E":[44.6735916,-75.7727966],"K0G":[44.8583946,-76.0813599],"K0H":[44.5292931,-76.6366577],"K0J":[45.5755272,-77.1902084],"K0K":[44.2020607,-77.4809875],"K0L":[44.5764542,-78.0834961],"K0M":[44.7191086,-78.7032242],"K1A":[45.4124107,-75.69767],"K1B":[45.4230423,-75.5935669],"K1C":[45.1773758,-75.0552139],"K1E":[45.4730148,-75.5050278],"K1G":[45.3522263,-75.5655136],"K1H":[45.387928,-75.6584167],"K1J":[45.2510223,-75.2861252],"K1K":[45.4380493,-75.646904],"K1L":[45.3639603,-75.5401917],"K1M":[45.4454575,-75.6798019],"K1N":[44.997406,-74.9726105],"K1P":[45.1570435,-75.3500137],"K1R":[45.2343559,-75.4110184],"K1S":[45.3544769,-75.6130447],"K1T":[44.6390877,-74.4462738],"K1V":[45.1634216,-75.3694458],"K1W":[41.3051262,-68.6613007],"K1X":[45.2784157,-75.6261597],"K1Y":[45.3983421,-75.7311478],"K1Z":[45.3187981,-75.6276016],"K2A":[45.3170853,-75.6616974],"K2B":[45.3621178,-75.7897797],"K2C":[45.302597,-75.645401],"K2E":[45.2699585,-75.5869446],"K2G":[45.325264,-75.7431412],"K2H":[45.2269897,-75.639534],"K2J":[43.5415573,-72.8536301],"K2K":[45.0613899,-75.4526978],"K2L":[45.3010597,-75.8879318],"K2M":[44.4021149,-74.3737793],"K2P":[44.9967117,-74.9916077],"K2R":[45.2800865,-75.803627],"K2S":[44.7596703,-75.0791245],"K2T":[41.563858,-69.6398849],"K2V":[45.2991142,-75.9008331],"K2W":[45.3575401,-75.9518814],"K4A":[44.4615631,-73.8036423],"K4B":[45.4097633,-75.4058914],"K4C":[45.5023422,-75.398819],"K4K":[45.5476913,-75.2792587],"K4M":[41.3788948,-69.2390671],"K4P":[44.5947609,-74.4765015],"K4R":[43.0384636,-71.6683197],"K6A":[45.6048279,-74.6036377],"K6H":[44.9226837,-74.5322647],"K6J":[45.0235405,-74.7491531],"K6K":[43.315815,-71.8752365],"K6T":[44.6099014,-75.7652817],"K6V":[44.5295448,-75.5822678],"K7A":[44.8606453,-75.9415512],"K7C":[44.931076,-75.8135452],"K7G":[44.3288193,-76.1750641],"K7H":[44.9009171,-76.2557144],"K7K":[43.9535446,-75.9740829],"K7L":[44.1937141,-76.4051437],"K7M":[44.2104912,-76.509552],"K7N":[43.1498337,-74.7868347],"K7P":[44.030735,-76.1859741],"K7R":[43.9160805,-76.3866043],"K7S":[44.0251846,-73.9997253],"K7V":[44.661335,-75.321434],"K8A":[45.758316,-77.0229187],"K8B":[45.8069382,-77.0871048],"K8H":[45.722374,-76.9818192],"K8N":[44.000351,-77.0799408],"K8P":[43.9063377,-76.9349823],"K8R":[44.1398849,-77.4577713],"K8V":[43.3401794,-76.2286606],"K9A":[43.6960793,-77.6821136],"K9H":[44.2821617,-78.2589951],"K9J":[44.23069,-78.200531],"K9K":[44.2865944,-78.366394],"K9L":[44.3310776,-78.3037872],"K9V":[44.0275459,-78.1462631],"L0A":[44.1151657,-78.4879684],"L0B":[44.0634499,-78.7653351],"L0C":[44.1790314,-79.0657349],"L0E":[44.3117104,-79.2788696],"L0G":[44.0400124,-79.6360321],"L0H":[43.9439774,-79.2538605],"L0J":[43.8441544,-79.6350174],"L0K":[43.8950577,-78.1032028],"L0L":[42.6817322,-76.529541],"L0M":[44.3286476,-79.969101],"L0N":[44.0120163,-80.1703491],"L0P":[43.5423584,-79.9795837],"L0R":[43.1934166,-79.6971893],"L0S":[42.4898643,-78.2320786],"L1A":[43.7865105,-78.0028],"L1B":[38.4486389,-68.8152771],"L1C":[43.1403961,-77.2962494],"L1E":[42.9760323,-77.1209564],"L1G":[43.8824425,-78.8019104],"L1H":[43.8636894,-78.7874374],"L1J":[43.8899956,-78.8774338],"L1K":[43.2301178,-77.5848923],"L1L":[43.9511871,-78.8890991],"L1M":[43.5483932,-78.2226257],"L1N":[43.4962273,-78.2411957],"L1P":[42.8037148,-77.0099411],"L1R":[42.7811012,-76.9044724],"L1S":[43.7696571,-78.8920364],"L1T":[43.6723289,-78.6876678],"L1V":[43.7934456,-79.0434494],"L1W":[43.6546555,-78.7981415],"L1X":[41.2333374,-74.3680573],"L1Y":[43.9550705,-79.1026077],"L1Z":[40.1266365,-72.2650757],"L2A":[42.9079628,-78.9386673],"L2E":[43.027916,-78.9497757],"L2G":[42.8990211,-78.7615433],"L2H":[41.4686737,-76.1421661],"L2J":[43.1262894,-79.1056213],"L2M":[43.1604843,-79.1554947],"L2N":[43.1939888,-79.2519989],"L2P":[42.8404388,-78.6465759],"L2R":[43.1643791,-79.2465515],"L2S":[43.0128098,-79.0156937],"L2T":[43.0589218,-79.0863037],"L2V":[42.1824265,-77.4905624],"L2W":[43.1656799,-79.2770081],"L3B":[42.897747,-79.0806427],"L3C":[43.0004883,-79.264061],"L3K":[42.8913193,-79.2521133],"L3M":[42.8697853,-78.9625397],"L3P":[42.726963,-77.1824036],"L3R":[43.8505974,-79.3243713],"L3S":[43.8013153,-79.1906815],"L3T":[43.4445839,-78.7132187],"L3V":[44.5056076,-79.2248306],"L3X":[41.7969666,-75.4250031],"L3Y":[43.9996872,-79.3515015],"L3Z":[40.0628624,-72.2726135],"L4A":[42.5228767,-76.6285553],"L4B":[43.6754608,-79.0765228],"L4C":[43.7964401,-79.3048706],"L4E":[42.9346466,-77.6407471],"L4G":[43.9146309,-79.3152542],"L4H":[42.0770035,-76.4070511],"L4J":[43.7031288,-79.254921],"L4K":[43.7150688,-79.3286819],"L4L":[43.6703186,-79.3524628],"L4M":[44.2419128,-79.3866196],"L4N":[44.3660545,-79.6943588],"L4P":[44.1747437,-79.3646545],"L4R":[44.1627007,-78.8397827],"L4S":[43.1353073,-78.0481796],"L4T":[43.7158432,-79.6434174],"L4V":[43.6983643,-79.6201782],"L4W":[43.6351814,-79.6179199],"L4X":[43.6170197,-79.5819092],"L4Y":[43.5289536,-79.4579086],"L4Z":[43.61409,-79.6468887],"L5A":[43.5872307,-79.6073303],"L5B":[43.5941694,-79.6527507],"L5C":[43.5639992,-79.6506195],"L5E":[43.5828629,-79.5641251],"L5G":[43.4970551,-79.461525],"L5H":[43.5413475,-79.613739],"L5J":[43.4524307,-79.5138855],"L5K":[43.2456665,-79.1413803],"L5L":[43.5363426,-79.6930313],"L5M":[43.5423164,-79.6810684],"L5N":[43.5330811,-79.6613235],"L5P":[43.6903915,-79.6237717],"L5R":[43.6031532,-79.6687698],"L5S":[43.6797142,-79.6746368],"L5T":[43.6559486,-79.6696777],"L5V":[42.9955215,-78.594902],"L5W":[43.6313133,-79.720108],"L6A":[43.3253174,-78.5407639],"L6B":[38.8139915,-70.0818634],"L6C":[41.672039,-75.325119],"L6E":[40.0156898,-72.2559128],"L6G":[43.8496399,-79.3326263],"L6H":[43.3703957,-79.5076675],"L6J":[43.3677063,-79.4715652],"L6K":[40.5382614,-74.3624496],"L6L":[43.2472916,-79.4202271],"L6M":[41.7032661,-76.5541229],"L6P":[39.927887,-72.6847458],"L6R":[42.7845154,-77.9849396],"L6S":[43.6924973,-79.6576385],"L6T":[43.2453499,-78.8393478],"L6V":[43.7051468,-79.761795],"L6W":[43.545742,-79.492424],"L6X":[43.3664055,-79.2148819],"L6Y":[42.1308556,-76.9659576],"L6Z":[43.7261429,-79.7935867],"L7A":[43.4402275,-79.3436966],"L7B":[41.4349556,-75.004631],"L7C":[39.3985214,-71.849762],"L7E":[43.5684357,-79.1667099],"L7G":[43.3673859,-79.3981171],"L7J":[43.6307716,-80.0358353],"L7K":[43.8726273,-80.004097],"L7L":[43.3473854,-79.7068939],"L7M":[43.3067398,-79.6760483],"L7N":[43.3472595,-79.7784042],"L7P":[43.2035866,-79.5624084],"L7R":[43.2138824,-79.5759964],"L7S":[43.3246117,-79.8078766],"L7T":[43.3099899,-79.8430862],"L8E":[42.6973076,-78.7359085],"L8G":[43.2174225,-79.7465363],"L8H":[43.1344528,-79.5864487],"L8J":[42.7784233,-79.0147247],"L8K":[43.2273521,-79.799408],"L8L":[43.0080147,-79.3815613],"L8M":[43.2462845,-79.8364944],"L8N":[43.2566681,-79.8684845],"L8P":[43.0585251,-79.5224152],"L8R":[43.2624397,-79.8750992],"L8S":[43.2573242,-79.9138031],"L8T":[43.2205429,-79.8303223],"L8V":[43.2278748,-79.8488464],"L8W":[42.6682167,-78.8667145],"L9A":[43.2297974,-79.8695908],"L9B":[40.9059029,-75.6457596],"L9C":[43.0599213,-79.5867538],"L9G":[43.1661491,-79.8871765],"L9H":[43.2710152,-79.959053],"L9J":[35.9053078,-66.3009033],"L9K":[35.8094292,-66.2347717],"L9L":[44.1002693,-78.9581375],"L9M":[25.3484306,-45.2465096],"L9N":[44.1072235,-79.4926147],"L9P":[44.1077347,-79.1284027],"L9R":[41.3857803,-74.850914],"L9S":[44.2870522,-79.4977264],"L9T":[40.3034897,-73.9711456],"L9V":[43.9478149,-80.1048508],"L9W":[43.6635475,-79.6344757],"L9Y":[43.2403793,-77.9701309],"L9Z":[44.3988953,-79.8757553],"M1B":[43.7649727,-79.1295929],"M1C":[43.6318932,-78.8740463],"M1E":[43.7657585,-79.1906509],"M1G":[43.7682571,-79.2172623],"M1H":[43.7663078,-79.239006],"M1J":[43.7461128,-79.2360306],"M1K":[43.7294884,-79.2639694],"M1L":[43.7088699,-79.2847137],"M1M":[43.6457939,-79.0856018],"M1N":[43.6180534,-79.1232681],"M1P":[43.7596092,-79.2708969],"M1R":[43.7476845,-79.3047028],"M1S":[43.5353127,-78.8028641],"M1T":[43.7814865,-79.304245],"M1V":[43.7104836,-79.0892258],"M1W":[43.799614,-79.3209915],"M1X":[43.4403152,-78.5239029],"M2H":[43.8011246,-79.3593369],"M2J":[43.5949707,-79.0145721],"M2K":[43.2965851,-78.5120163],"M2L":[43.7530174,-79.3801575],"M2M":[43.7912102,-79.4109802],"M2N":[43.6007309,-79.1042175],"M2P":[43.3061028,-78.5973206],"M2R":[43.6942062,-79.2886963],"M3A":[43.7534256,-79.3277054],"M3B":[43.7446098,-79.3589783],"M3C":[43.3742638,-78.7021637],"M3H":[43.6057243,-79.1816559],"M3J":[43.1235733,-78.330307],"M3K":[43.7335434,-79.4688644],"M3L":[41.4847412,-75.4201584],"M3M":[43.5076523,-79.0901642],"M3N":[43.6276932,-79.2848434],"M4A":[43.7275848,-79.3121338],"M4B":[43.7066879,-79.3080139],"M4C":[43.6387863,-79.218132],"M4E":[43.6779861,-79.296402],"M4G":[43.6274033,-79.2231216],"M4H":[43.7047997,-79.3462753],"M4J":[43.6863632,-79.336174],"M4K":[43.6803474,-79.3512955],"M4L":[43.5914307,-79.1755142],"M4M":[43.3583717,-78.7904434],"M4N":[43.7268105,-79.3957443],"M4P":[43.7099495,-79.390625],"M4R":[43.7138329,-79.4050751],"M4S":[43.4507332,-78.9304886],"M4T":[43.6895142,-79.388298],"M4V":[43.5544357,-79.1632843],"M4W":[43.6774254,-79.3817291],"M4X":[43.6684074,-79.3694839],"M4Y":[43.3751907,-78.8403473],"M5A":[43.2066727,-78.5487671],"M5B":[43.4137802,-78.9332199],"M5C":[43.653038,-79.3768082],"M5E":[42.2879448,-76.8958664],"M5G":[42.5227394,-77.3246689],"M5H":[43.3335648,-78.8082504],"M5J":[42.974762,-78.1663208],"M5K":[43.6487541,-79.3963242],"M5L":[43.6597252,-79.3998032],"M5M":[43.7311172,-79.4179382],"M5N":[43.5876465,-79.1944427],"M5P":[43.4983215,-79.0515747],"M5R":[43.5706329,-79.2152252],"M5S":[42.7752647,-77.7832947],"M5T":[43.6537285,-79.3976212],"M5V":[42.6413307,-77.5740738],"M5W":[42.8956146,-78.0055923],"M5X":[43.6487236,-79.3813934],"M6A":[43.7228851,-79.4430771],"M6B":[43.707592,-79.4444885],"M6C":[43.6915665,-79.4298172],"M6E":[43.688591,-79.4491196],"M6G":[43.6686516,-79.4199982],"M6H":[43.5185738,-79.1683044],"M6J":[43.3069878,-78.7887115],"M6K":[43.4530754,-79.0833435],"M6L":[43.7139206,-79.4845886],"M6M":[43.6353989,-79.3758698],"M6N":[43.6755295,-79.4788818],"M6P":[43.6614876,-79.4607468],"M6R":[43.6476898,-79.4461594],"M6S":[43.6537247,-79.4838257],"M7A":[42.4739685,-77.2239685],"M7Y":[43.664917,-79.3134003],"M8V":[43.3695526,-79.0677261],"M8W":[43.6008987,-79.5373535],"M8X":[43.6524467,-79.5106354],"M8Y":[43.5593224,-79.3610535],"M8Z":[43.6297798,-79.5172424],"M9A":[43.6092224,-79.4234161],"M9B":[43.5615883,-79.3861542],"M9C":[43.6451492,-79.5742264],"M9L":[43.7580261,-79.5601654],"M9M":[43.2532501,-78.6611099],"M9N":[43.7058983,-79.5158234],"M9P":[43.6927834,-79.5306473],"M9R":[43.6884575,-79.5561829],"M9V":[43.7412834,-79.5842667],"M9W":[43.6358986,-79.4298172],"N0A":[42.8835335,-79.9862213],"N0B":[42.6911507,-78.8818283],"N0C":[44.2609444,-80.5036087],"N0E":[42.9108772,-80.3956985],"N0G":[44.0173225,-81.0651093],"N0H":[44.6221886,-81.1153183],"N0J":[42.9982491,-80.6743774],"N0K":[43.5025177,-81.1476059],"N0L":[42.8340492,-81.2868729],"N0M":[43.3066292,-81.5611038],"N0N":[42.943779,-82.1399689],"N0P":[42.3903275,-82.2196655],"N0R":[42.1640968,-82.8017578],"N1A":[42.7302818,-79.2926712],"N1C":[43.4992523,-80.2362366],"N1E":[43.3372955,-79.8279266],"N1G":[43.4217453,-80.048378],"N1H":[43.5112801,-80.2093658],"N1K":[42.3090439,-78.040329],"N1L":[41.4636002,-76.409317],"N1M":[43.7063179,-80.3763199],"N1P":[42.4364624,-78.6177216],"N1R":[43.3343582,-80.2490158],"N1S":[43.1202469,-79.8969803],"N1T":[43.1676025,-79.8736572],"N2A":[42.4228058,-78.5456161],"N2B":[43.4599648,-80.4505081],"N2C":[43.1610947,-79.9655685],"N2E":[43.3017349,-80.2943497],"N2G":[43.3685379,-80.3454895],"N2H":[43.4113998,-80.4025269],"N2J":[43.3540382,-80.2965927],"N2K":[42.9358215,-79.4619522],"N2L":[43.3806,-80.3734283],"N2M":[43.4378548,-80.5097198],"N2N":[43.1270027,-79.9869614],"N2P":[41.7649002,-77.4103088],"N2R":[42.0848236,-78.0488892],"N2T":[42.493885,-78.7857437],"N2V":[43.1877975,-80.0094376],"N2Z":[44.1745262,-81.6336517],"N3A":[43.2074585,-80.3637085],"N3B":[43.4367065,-80.2700043],"N3C":[42.9574356,-79.4318848],"N3E":[43.4229279,-80.3596954],"N3H":[43.347847,-80.2643509],"N3L":[43.0636253,-80.1367874],"N3P":[43.1845322,-80.2354965],"N3R":[43.1009903,-80.1444168],"N3S":[42.8651009,-79.7349625],"N3T":[42.8301468,-79.7031555],"N3V":[43.1715012,-80.3012238],"N3W":[43.0652275,-79.94384],"N3Y":[42.5722504,-79.8156052],"N4B":[42.8455124,-80.4968643],"N4G":[42.8223953,-80.648468],"N4K":[44.4951057,-80.8037567],"N4L":[43.6454163,-78.8620605],"N4N":[44.0751305,-80.8778534],"N4S":[42.9329605,-80.3796616],"N4T":[42.8306274,-80.1346512],"N4V":[42.5870857,-79.7589951],"N4W":[43.7336349,-80.9547729],"N4X":[42.6229553,-79.9133759],"N4Z":[43.3649483,-80.9904327],"N5A":[43.2741241,-80.799469],"N5C":[42.8462563,-80.5235672],"N5H":[42.7705231,-80.9831543],"N5L":[42.6710434,-81.2200241],"N5P":[42.6744003,-80.9734802],"N5R":[42.6482239,-80.9505081],"N5V":[43.0093079,-81.1764221],"N5W":[42.8942871,-81.0131989],"N5X":[43.0357208,-81.2569962],"N5Y":[43.0137024,-81.2278976],"N5Z":[42.9703217,-81.2083588],"N6A":[42.9937172,-81.2536163],"N6B":[42.9845543,-81.2392502],"N6C":[42.9606209,-81.2436295],"N6E":[42.8556519,-81.0790939],"N6G":[41.8754234,-79.1600189],"N6H":[42.9812508,-81.3008652],"N6J":[42.8971214,-81.159317],"N6K":[42.952198,-81.322731],"N6L":[40.900692,-77.4609985],"N6M":[42.3999901,-80.0994568],"N6N":[42.5307693,-80.4692459],"N6P":[41.0731659,-77.8237915],"N7A":[43.5863724,-81.4074097],"N7G":[42.8013229,-81.2987213],"N7L":[41.9796753,-81.3582764],"N7M":[42.1682587,-81.7314682],"N7S":[42.7782326,-81.9637451],"N7T":[42.9048195,-82.2567444],"N7V":[43.0007401,-82.3902512],"N7W":[42.2561607,-80.9592133],"N7X":[43.0222511,-82.3153152],"N8A":[42.5948753,-82.3886642],"N8H":[42.0465164,-82.5991898],"N8M":[42.1694679,-82.8264542],"N8N":[42.2626839,-82.7778778],"N8P":[41.6052628,-81.4972076],"N8R":[42.1511803,-82.6107864],"N8S":[42.3300819,-82.948761],"N8T":[42.1244545,-82.5987549],"N8V":[42.2684708,-82.9633636],"N8W":[42.2899323,-82.9872284],"N8X":[42.2962456,-83.0179367],"N8Y":[42.1949272,-82.7513428],"N9A":[42.2670212,-82.9420776],"N9B":[42.1351585,-82.7304077],"N9C":[41.6230507,-81.7564011],"N9E":[42.2655602,-83.0250015],"N9G":[42.2490692,-82.9942627],"N9H":[41.7681694,-82.0982666],"N9J":[42.0235176,-82.6735077],"N9K":[42.2987633,-82.8696442],"N9V":[42.1082306,-83.0970688],"N9Y":[42.0343819,-82.7350845],"P0A":[45.6073685,-79.4332047],"P0B":[45.162159,-79.3775635],"P0C":[45.1816597,-79.6395645],"P0E":[44.8263474,-79.3526306],"P0G":[45.8273964,-79.8944473],"P0H":[46.2385101,-79.4263382],"P0J":[47.6137543,-79.7812424],"P0K":[48.3221626,-80.2639465],"P0L":[49.6942368,-82.6471024],"P0M":[45.9825058,-79.9959412],"P0N":[48.5096436,-81.1509552],"P0P":[45.9019623,-82.1692505],"P0R":[46.2867508,-83.713913],"P0S":[47.313324,-84.2984467],"P0T":[48.6917725,-89.5901718],"P0V":[51.2592735,-92.4061356],"P0W":[48.7562943,-94.1780396],"P0X":[49.660965,-94.4755249],"P0Y":[49.7857246,-95.1167755],"P1A":[43.7409821,-75.0592041],"P1B":[46.2106972,-79.2692184],"P1C":[43.0814095,-73.8534622],"P1H":[45.3311996,-79.2134323],"P1L":[44.9054718,-79.0590515],"P1P":[44.9274788,-79.368248],"P2A":[32.5409966,-57.3998451],"P2B":[46.1198082,-79.5016708],"P2N":[48.1483917,-80.0331268],"P3A":[46.469265,-80.85746],"P3B":[46.2641716,-80.5558243],"P3C":[46.3792114,-80.806366],"P3E":[46.1884804,-80.5099411],"P3G":[43.7091103,-76.2611389],"P3L":[45.8857384,-79.7034302],"P3N":[46.6100082,-81.0089645],"P3P":[46.6523933,-80.9651794],"P3Y":[45.8232346,-80.0859375],"P4N":[48.369873,-81.1575928],"P4P":[48.497303,-81.3481522],"P4R":[48.0177116,-80.6104736],"P5A":[46.2796822,-82.4713211],"P5E":[46.2529793,-81.7711792],"P5N":[49.4152679,-82.416008],"P6A":[46.3688622,-84.0327377],"P6B":[46.4309006,-84.1437149],"P6C":[46.4371681,-84.1881561],"P7A":[42.6793823,-78.5867538],"P7B":[48.384903,-89.1608963],"P7C":[48.3349304,-89.1630096],"P7E":[48.3741226,-89.2779922],"P7G":[31.3105011,-57.7005501],"P7J":[33.0072403,-60.9982567],"P7K":[25.6644421,-47.4141045],"P7L":[48.1962242,-89.5508957],"P8N":[49.5515633,-92.4064941],"P8T":[49.6701393,-91.1438065],"P9A":[48.6116638,-93.3975143],"P9N":[49.593956,-94.1180954],"R0A":[49.567337,-96.8833008],"R0B":[55.095047,-97.7626266],"R0C":[50.8245316,-97.5928268],"R0E":[50.0597038,-96.6110764],"R0G":[41.0734978,-81.1454086],"R0H":[50.0685959,-98.5051956],"R0J":[50.5022926,-100.1646881],"R0K":[49.5971298,-99.6311646],"R0L":[51.7008743,-100.4605179],"R0M":[49.7354012,-100.8919983],"R1A":[50.0660934,-96.8333435],"R1B":[50.0817566,-96.9336624],"R1N":[49.9056206,-98.1587753],"R2C":[49.6100693,-96.4475479],"R2E":[49.9744492,-97.0197601],"R2G":[49.8789482,-96.9450226],"R2H":[49.7022476,-96.768158],"R2J":[49.8649025,-97.0746841],"R2K":[49.9210434,-97.0851669],"R2L":[49.9083366,-97.1008224],"R2M":[49.7920876,-97.0133514],"R2N":[49.2481346,-96.0021286],"R2P":[49.9563904,-97.170845],"R2R":[49.8410454,-97.0197525],"R2V":[49.796093,-96.8273621],"R2W":[49.8655472,-97.0304794],"R2X":[49.9309578,-97.1666183],"R2Y":[49.8912468,-97.2995758],"R3A":[49.9024925,-97.1472931],"R3B":[49.790802,-96.9321442],"R3C":[49.8010788,-96.9616928],"R3E":[49.9078369,-97.175415],"R3G":[49.8863525,-97.1756439],"R3H":[49.9023666,-97.2075043],"R3J":[49.8093376,-97.1012573],"R3K":[49.8754845,-97.2935104],"R3L":[49.8691826,-97.1369705],"R3M":[49.8659401,-97.1628799],"R3N":[49.7287292,-96.9284515],"R3P":[49.8460121,-97.2031479],"R3R":[49.7854385,-97.1429214],"R3S":[49.8342361,-97.2681885],"R3T":[49.8138008,-97.1549301],"R3V":[49.487915,-96.6137161],"R3W":[46.0734291,-89.573761],"R3X":[47.0507011,-91.6436386],"R3Y":[43.2520065,-84.3934555],"R4A":[49.6376152,-96.3719254],"R4H":[49.8721046,-97.3715591],"R4J":[49.8539124,-97.4026337],"R4K":[49.8729286,-97.5009918],"R4L":[49.9429779,-97.609169],"R5A":[49.7255058,-97.072052],"R5G":[48.215332,-94.1298065],"R5H":[48.6537743,-94.6619263],"R6M":[48.3727226,-96.4615097],"R6W":[48.8684731,-97.3214264],"R7A":[49.4309082,-99.118721],"R7B":[49.5241966,-99.3452454],"R7C":[49.2898941,-98.779129],"R7N":[51.149868,-100.0493622],"R8A":[54.5956383,-101.5319366],"R8N":[55.7429733,-97.8633118],"R9A":[53.8132782,-101.2337723],"S0A":[51.3711243,-102.8641205],"S0C":[49.3981781,-103.0498962],"S0E":[52.9587097,-103.670723],"S0G":[50.5010376,-104.0448456],"S0H":[50.0794983,-106.1983109],"S0J":[55.3698273,-106.2781906],"S0K":[48.1308327,-97.9242783],"S0L":[51.4931335,-108.7290955],"S0M":[53.6704979,-108.6881714],"S0N":[50.0927467,-108.4756851],"S0P":[55.2272911,-102.4420319],"S2V":[50.7503624,-104.9301224],"S3N":[51.2105217,-102.4657745],"S4A":[48.9600067,-102.608551],"S4H":[49.5820694,-103.6781082],"S4L":[45.3971825,-93.995163],"S4N":[50.4156723,-104.4914246],"S4P":[50.3625336,-104.4289856],"S4R":[50.4793091,-104.6243134],"S4S":[50.3775063,-104.5408478],"S4T":[50.4582901,-104.6448288],"S4V":[50.230957,-104.1334534],"S4W":[46.1111946,-95.6750641],"S4X":[50.3010521,-104.2748566],"S4Y":[46.7218018,-96.8993149],"S4Z":[37.839901,-78.3991547],"S6H":[50.2840881,-105.318161],"S6J":[44.695446,-93.5649796],"S6K":[50.3785248,-105.5826797],"S6V":[53.1205788,-105.5956726],"S6W":[47.8601151,-95.1766129],"S6X":[53.1916542,-105.7037125],"S7H":[52.1143112,-106.6201477],"S7J":[52.0981789,-106.6240158],"S7K":[52.1202469,-106.5656128],"S7L":[52.1415634,-106.7032318],"S7M":[52.0185814,-106.4971771],"S7N":[52.1345482,-106.6071701],"S7P":[49.2169075,-100.5391846],"S7R":[45.7543602,-93.563736],"S7S":[52.1583405,-106.590065],"S7T":[41.4556198,-84.9157867],"S7V":[45.7911072,-93.6472626],"S7W":[43.9007034,-89.7152328],"S9A":[52.5345573,-107.7927628],"S9H":[50.1339645,-107.4708252],"S9V":[53.2738075,-110.000618],"S9X":[54.1265755,-108.4339981],"T0A":[40.6849442,-83.8334656],"T0B":[37.0377083,-77.9318924],"T0C":[52.5461807,-113.0787201],"T0E":[53.6445961,-115.0791855],"T0G":[55.105999,-114.6795578],"T0H":[56.4314156,-118.097641],"T0J":[51.0055923,-112.2701416],"T0K":[49.5465355,-112.6981735],"T0L":[49.8077469,-112.0545883],"T0M":[50.5305901,-111.1956635],"T0P":[56.2870255,-111.0819473],"T0V":[59.8685379,-111.6329422],"T1A":[50.0026054,-110.608963],"T1B":[49.7747459,-110.1382294],"T1C":[50.0612488,-110.6772919],"T1G":[49.6764412,-111.8888931],"T1H":[49.1625557,-111.5626526],"T1J":[48.5850182,-110.3189163],"T1K":[49.6760712,-112.8363876],"T1L":[51.1790962,-115.5697403],"T1M":[49.101757,-111.2064285],"T1P":[50.1527481,-111.4067993],"T1R":[50.57341,-111.8962402],"T1S":[49.8359108,-111.9720993],"T1V":[50.1693535,-112.9429474],"T1W":[50.6259232,-114.3163376],"T1X":[50.7028542,-113.1487045],"T1Y":[51.0781326,-113.963768],"T1Z":[44.2675514,-98.8009186],"T2A":[51.0507965,-113.9711609],"T2B":[51.0272827,-113.9961014],"T2C":[49.9398079,-111.6578293],"T2E":[51.0353889,-113.9686966],"T2G":[51.0346832,-114.0484543],"T2H":[50.9929962,-114.0585175],"T2J":[50.942543,-114.0393829],"T2K":[51.0557899,-113.97155],"T2L":[51.0889359,-114.1198502],"T2M":[50.8882103,-113.6806183],"T2N":[51.058197,-114.1049194],"T2P":[50.8021278,-113.5155106],"T2R":[51.0414925,-114.0757065],"T2S":[51.0281792,-114.0714874],"T2T":[50.9888725,-113.9923706],"T2V":[50.9274139,-113.9770737],"T2W":[50.948616,-114.1068115],"T2X":[49.4865265,-110.8822327],"T2Y":[50.8242264,-113.9022141],"T2Z":[50.5163841,-113.0633621],"T3A":[51.0804405,-114.0530853],"T3B":[50.6049271,-113.1049576],"T3C":[51.0457993,-114.1180496],"T3E":[50.6211586,-113.2216949],"T3G":[50.6854935,-113.1995773],"T3H":[49.3133316,-110.3225021],"T3J":[50.3114471,-112.1586151],"T3K":[50.3848648,-112.3740082],"T3L":[51.1219635,-114.2370529],"T3M":[47.0742912,-105.4297485],"T3N":[28.5175304,-63.5646553],"T3P":[32.7079391,-72.9275818],"T3R":[47.4604225,-105.9168854],"T3S":[50.9408798,-113.8920975],"T3Z":[50.9710655,-114.1127167],"T4A":[43.6925621,-97.1268234],"T4B":[49.3515816,-109.7242813],"T4C":[50.6768074,-113.3138351],"T4E":[50.6541595,-110.282196],"T4G":[52.0225868,-113.9524536],"T4H":[50.7294693,-111.7699738],"T4J":[52.6786194,-113.5797272],"T4L":[52.1853218,-113.1105804],"T4N":[52.1000481,-113.4488373],"T4P":[51.174408,-111.381546],"T4R":[51.9681587,-113.161438],"T4S":[51.4744759,-112.2614441],"T4T":[52.3786545,-114.9212265],"T4V":[52.8112259,-112.4031448],"T4X":[44.3227615,-94.2148056],"T5A":[53.4725571,-113.1924438],"T5B":[53.5615234,-113.4767303],"T5C":[53.5870247,-113.470192],"T5E":[52.8038254,-111.839447],"T5G":[53.5653687,-113.4996262],"T5H":[53.4779816,-113.3366547],"T5J":[53.4156036,-113.2253571],"T5K":[53.5364037,-113.5129776],"T5L":[53.5256233,-113.422226],"T5M":[53.5632744,-113.5524139],"T5N":[53.553051,-113.5477905],"T5P":[53.427887,-113.3131943],"T5R":[53.5264053,-113.5749435],"T5S":[51.2314339,-108.7161942],"T5T":[53.4776726,-113.5598679],"T5V":[52.4861031,-111.2808685],"T5W":[53.5502243,-113.441452],"T5X":[51.8667107,-109.8500214],"T5Y":[51.3247566,-108.6058502],"T5Z":[52.7366028,-111.7542267],"T6A":[53.4656944,-113.2822495],"T6B":[52.7051811,-111.6990433],"T6C":[53.5226517,-113.4591064],"T6E":[53.5063744,-113.4833679],"T6G":[53.5213547,-113.5180511],"T6H":[53.4995651,-113.5238266],"T6J":[53.4717293,-113.5067978],"T6K":[53.4639816,-113.4494553],"T6L":[53.4602585,-113.4150238],"T6M":[52.7072639,-111.9927826],"T6N":[53.4564171,-113.4812088],"T6P":[53.2441826,-112.8215561],"T6R":[52.7848358,-112.1410599],"T6S":[53.566761,-113.3393173],"T6T":[51.0929604,-108.3407211],"T6V":[48.2795525,-102.256691],"T6W":[41.7388649,-88.7048035],"T6X":[47.9821434,-101.8879471],"T7A":[53.2242432,-114.9704666],"T7E":[26.0792732,-56.6647873],"T7N":[53.8605309,-113.8541031],"T7P":[54.1546555,-113.8561935],"T7S":[53.4762497,-114.2576218],"T7V":[53.2681274,-117.2701187],"T7X":[52.0821915,-110.768898],"T7Y":[53.4399109,-113.8274384],"T7Z":[52.8202438,-112.5031586],"T8A":[53.5270653,-113.2896042],"T8B":[53.4646912,-113.2815399],"T8C":[53.4533653,-113.1978531],"T8E":[53.5248222,-113.1219635],"T8G":[53.5066643,-113.0262451],"T8H":[51.0863876,-108.0616531],"T8L":[52.5752029,-110.8045349],"T8N":[53.6063385,-113.5489349],"T8R":[52.572525,-111.0710449],"T8S":[56.0741501,-116.9634094],"T8T":[45.6849136,-96.4659805],"T8V":[54.9919128,-118.4045792],"T8W":[44.9177208,-96.7705307],"T8X":[53.9036255,-116.0241776],"T9A":[52.8858261,-113.1946945],"T9C":[53.108242,-111.2402267],"T9E":[52.0820427,-111.0161133],"T9G":[41.8156052,-89.1269226],"T9H":[56.5547981,-111.0757904],"T9J":[56.70895,-111.3719559],"T9K":[56.4366913,-110.8467178],"T9M":[53.7890434,-108.8781815],"T9N":[53.7566795,-109.6939774],"T9S":[54.1164513,-112.0554504],"T9V":[52.2827339,-107.9737091],"T9W":[52.8409843,-110.8539047],"T9X":[52.9434662,-109.9893494],"V0A":[50.8410873,-116.421875],"V0B":[47.5515862,-111.3444366],"V0C":[56.7627563,-122.1331863],"V0E":[47.6410866,-111.8822327],"V0G":[40.2749214,-95.2335892],"V0H":[49.2636909,-119.083313],"V0J":[47.212513,-108.9685669],"V0K":[44.5717087,-104.6872787],"V0L":[52.0716667,-123.3018646],"V0M":[49.2849236,-121.851799],"V0N":[43.2129631,-107.8255157],"V0P":[50.2977371,-125.7717896],"V0R":[48.9929237,-123.9269714],"V0S":[48.5529671,-124.4223022],"V0T":[52.9067078,-129.0223389],"V0V":[54.2265816,-130.0942383],"V0W":[59.5547714,-133.686203],"V0X":[49.2758636,-120.8715363],"V1A":[49.2904472,-115.0737762],"V1B":[50.1259727,-118.920723],"V1C":[49.274044,-115.2178802],"V1E":[50.6924934,-119.269722],"V1G":[54.8696404,-118.3462677],"V1H":[50.1400375,-119.1100006],"V1J":[55.8981476,-120.0899048],"V1K":[49.7169151,-119.839325],"V1L":[49.3454514,-116.8943863],"V1M":[48.9738159,-122.1433411],"V1N":[49.2538223,-117.504158],"V1P":[48.8475609,-116.8557129],"V1R":[49.0971985,-117.705452],"V1S":[50.0533257,-118.9738464],"V1T":[50.2648888,-119.2702332],"V1V":[49.3731575,-118.1158447],"V1W":[49.4561157,-118.5701294],"V1X":[49.7061157,-118.9384995],"V1Y":[49.8473244,-119.3904343],"V1Z":[49.3898201,-118.3649979],"V2A":[49.3232536,-119.1879196],"V2B":[50.5720901,-120.0283279],"V2C":[50.2665977,-119.3243942],"V2E":[50.2236099,-119.3020172],"V2G":[52.1361542,-122.1334457],"V2H":[50.7375565,-120.2527008],"V2J":[52.9733658,-122.4916687],"V2K":[53.9387703,-122.683815],"V2L":[53.7003822,-122.2842941],"V2M":[53.9210243,-122.7958984],"V2N":[53.6280518,-122.2478256],"V2P":[49.10149,-121.7610855],"V2R":[48.5802841,-120.6736832],"V2S":[48.978096,-122.1092911],"V2T":[48.931633,-122.042366],"V2V":[49.0769768,-122.1250381],"V2W":[48.1602249,-119.9315109],"V2X":[49.1915932,-122.548645],"V2Y":[47.9942169,-119.8094635],"V2Z":[49.0651588,-122.599823],"V3A":[48.8460541,-122.0519943],"V3B":[48.8438492,-121.6868286],"V3C":[49.1589127,-122.5506058],"V3E":[47.0990334,-117.3074799],"V3G":[48.2752037,-120.2547607],"V3H":[49.2880821,-122.8541412],"V3J":[49.0992126,-122.4644699],"V3K":[49.1950722,-122.7398911],"V3L":[48.7389183,-121.7067261],"V3M":[49.0059738,-122.4497528],"V3N":[48.9525223,-122.2555084],"V3R":[49.1543846,-122.7262039],"V3S":[49.1108017,-122.764061],"V3T":[48.2607117,-120.529541],"V3V":[49.0242271,-122.4842529],"V3W":[49.1374588,-122.8562164],"V3X":[48.7058716,-121.8479156],"V3Y":[48.8651161,-121.7965012],"V4A":[49.0041199,-122.7487946],"V4B":[48.8954811,-122.4775314],"V4C":[49.0053902,-122.5347672],"V4E":[49.1237411,-122.9039001],"V4G":[48.631073,-121.7357788],"V4K":[48.8488159,-122.4646301],"V4L":[49.021389,-123.0592194],"V4M":[48.9363365,-122.8748398],"V4N":[48.7080383,-121.6036072],"V4P":[49.0557404,-122.8261642],"V4R":[47.8629379,-119.1605988],"V4S":[47.9000893,-119.1684952],"V4T":[49.5146103,-118.8389511],"V4V":[49.2952805,-117.5934143],"V4W":[49.0672836,-122.494812],"V4X":[48.3429451,-120.5699463],"V4Z":[49.1146011,-121.8456802],"V5A":[48.9753723,-122.2215805],"V5B":[49.2741737,-122.9776001],"V5C":[48.9035149,-122.0762177],"V5E":[49.2266998,-122.9573441],"V5G":[48.9381027,-122.2285767],"V5H":[48.4834862,-121.1314087],"V5J":[49.0880547,-122.6820908],"V5K":[49.0102959,-122.3697433],"V5L":[49.2048874,-122.8833237],"V5M":[48.9665489,-122.3367691],"V5N":[48.9153595,-122.2243423],"V5P":[49.1547203,-122.9015808],"V5R":[48.9703255,-122.3793716],"V5S":[48.7107086,-121.7726288],"V5T":[49.0949173,-122.6762009],"V5V":[48.9881096,-122.4384995],"V5W":[48.7672462,-121.9293747],"V5X":[48.9113388,-122.3307114],"V5Y":[48.3316307,-120.8132248],"V5Z":[49.008358,-122.5107193],"V6A":[49.1731606,-122.826973],"V6B":[48.9482117,-122.2859039],"V6C":[49.0102348,-122.4299393],"V6E":[48.7087097,-121.6918335],"V6G":[48.8726883,-122.0955353],"V6H":[49.0730057,-122.6712112],"V6J":[48.6941414,-121.7253418],"V6K":[48.8204117,-122.0368881],"V6L":[49.0075684,-122.547287],"V6M":[49.0666962,-122.7239304],"V6N":[48.9882507,-122.5575638],"V6P":[49.0143929,-122.6290359],"V6R":[49.1885681,-123.0018616],"V6S":[48.5734024,-121.491188],"V6T":[49.2672768,-123.2399826],"V6V":[49.1845932,-123.048378],"V6W":[49.1520119,-123.0508423],"V6X":[48.8706779,-122.3351364],"V6Y":[48.9033928,-122.4908218],"V6Z":[48.7343674,-121.7684402],"V7A":[49.1390533,-123.118187],"V7B":[48.7037582,-121.9388123],"V7C":[49.0468445,-122.8866043],"V7E":[48.9912033,-122.8134232],"V7G":[49.3185921,-122.9555283],"V7H":[49.0558243,-122.344574],"V7J":[49.3249092,-123.0346451],"V7K":[49.3436813,-123.0390625],"V7L":[49.1723785,-122.7004776],"V7M":[48.8768616,-121.9711304],"V7N":[49.3403893,-123.0737534],"V7P":[49.1171722,-122.5944138],"V7R":[49.348381,-123.0995255],"V7S":[49.2401886,-122.8819046],"V7T":[49.3317642,-123.1446838],"V7V":[49.2631798,-122.9978027],"V7W":[49.3578491,-123.2633362],"V7X":[49.2675209,-123.1127548],"V7Y":[49.2818413,-123.1239548],"V8A":[37.3513374,-93.3069839],"V8B":[47.1263847,-116.7226028],"V8C":[54.0607643,-128.6464233],"V8G":[54.4628258,-128.447525],"V8J":[54.3134613,-130.3150787],"V8K":[48.7662735,-123.294693],"V8L":[48.5613747,-123.1957779],"V8M":[48.3876038,-122.9380341],"V8N":[48.4274368,-123.1919098],"V8P":[48.4011154,-123.1893311],"V8R":[48.4364243,-123.3233414],"V8S":[48.4182167,-123.3264008],"V8T":[48.3742828,-123.1933594],"V8V":[48.2754364,-123.0012894],"V8W":[48.3385696,-123.1138763],"V8X":[48.2817497,-122.8864212],"V8Y":[48.4051857,-123.0906372],"V8Z":[48.3951607,-123.1816788],"V9A":[48.3561211,-123.1929092],"V9B":[47.7317085,-121.6486511],"V9C":[47.5293274,-121.2492981],"V9E":[48.5151291,-123.4370346],"V9G":[49.0038834,-123.8115387],"V9H":[49.516552,-124.1174164],"V9J":[49.7876892,-125.0727158],"V9K":[49.2805023,-124.3100967],"V9L":[48.7142067,-123.5237427],"V9M":[49.6861343,-124.9219513],"V9N":[49.6111717,-124.7997894],"V9P":[49.2141228,-124.0405502],"V9R":[48.9630508,-123.462204],"V9S":[49.0181503,-123.5420456],"V9T":[48.893013,-123.1959],"V9V":[49.2397766,-124.0239716],"V9W":[49.964222,-125.1472473],"V9X":[49.1013374,-123.8818817],"V9Y":[49.1689224,-124.595665],"V9Z":[47.4504242,-121.3532791],"X0A":[68.1559372,-76.260025],"X0B":[68.5665512,-101.2621765],"X0C":[63.4932022,-90.7266541],"X0E":[61.4932594,-117.0381927],"X0G":[60.2358932,-123.4629364],"X1A":[61.863678,-113.3363419],"Y0A":[60.1191788,-130.7037201],"Y0B":[62.1338272,-136.3330536],"Y1A":[60.2792511,-134.0881042]}', true); }
        return isset($this->postal_loc[$postal_prefix]) ? $this->postal_loc[$postal_prefix] : null;
    }
    
    /*
     * Provide approx zip code latitude/longitude
    */
    public function lookup_zip_location($zip_prefix){
        if ($this->zip_loc == null) { $this->zip_loc = json_decode('{"100":[40.7131996,-73.9869995],"101":[40.7099991,-73.9899979],"102":[40.7118988,-69.3659973],"103":[40.5800018,-74.1399994],"104":[40.8400002,-73.8700027],"105":[41.1855011,-73.7720032],"106":[41.0237999,-73.7480011],"107":[40.9449997,-73.8470001],"108":[40.9220009,-73.7779999],"109":[41.2591019,-74.1610031],"110":[40.7691002,-73.7030029],"111":[40.7599983,-73.9179993],"112":[40.6399994,-73.9400024],"113":[40.7503014,-73.8349991],"114":[40.6866989,-73.7819977],"115":[40.7019005,-73.6330032],"116":[40.582901,-73.8270035],"117":[40.7970009,-73.2399979],"118":[40.7636986,-73.5070038],"119":[40.9224014,-72.5350037],"120":[42.6745987,-73.9420013],"121":[42.6783981,-73.9240036],"122":[42.6599998,-73.7900009],"123":[42.7999992,-73.9199982],"124":[42.0732994,-74.197998],"125":[41.7580986,-73.8539963],"126":[41.6899986,-73.9199982],"127":[41.6806984,-74.7639999],"128":[43.4903984,-73.7099991],"129":[44.5569992,-73.9150009],"130":[42.9651985,-76.2129974],"131":[43.0649986,-76.2639999],"132":[43.0400009,-76.1389999],"133":[43.1495018,-75.2210007],"134":[43.1081009,-75.2720032],"135":[43.0999985,-75.2300034],"136":[44.2858009,-75.5179977],"137":[42.2402,-75.5289993],"138":[42.3302002,-75.5940018],"139":[42.0999985,-75.9100037],"140":[42.7955017,-78.625],"141":[42.8120003,-78.7180023],"142":[42.8800011,-78.8499985],"143":[43.0900002,-79.0100021],"144":[42.9684982,-77.6110001],"145":[42.9418983,-77.4869995],"146":[43.1599998,-77.6100006],"147":[42.1974983,-78.8610001],"148":[42.3022003,-77.1029968],"149":[42.0800018,-76.8000031],"150":[40.4616013,-80.0550003],"151":[40.4140015,-79.9189987],"152":[40.4300995,-79.9680023],"153":[40.0395012,-80.1669998],"154":[39.9520988,-79.6849976],"155":[39.9770012,-78.8769989],"156":[40.3044014,-79.5309982],"157":[40.7195015,-79.0220032],"158":[41.2486992,-78.7030029],"159":[40.3548012,-78.8519974],"160":[40.9273987,-79.8740005],"161":[41.1459999,-80.3430023],"162":[41.0130005,-79.3730011],"163":[41.5727005,-79.5490036],"164":[41.8862991,-80.0270004],"165":[42.1199989,-80.0800018],"166":[40.4766006,-78.3440018],"167":[41.8170013,-78.5],"168":[40.9370995,-77.9950027],"169":[41.8297005,-77.2699966],"170":[40.3959999,-77.0289993],"171":[40.2700005,-76.8799973],"172":[39.9948006,-77.7610016],"173":[39.8973999,-76.8720016],"174":[39.9589005,-76.737999],"175":[40.0513,-76.2429962],"176":[40.0373001,-76.3069992],"177":[41.2751007,-77.1159973],"178":[40.8795013,-76.7949982],"179":[40.6857986,-76.2699966],"180":[40.6330986,-75.4459991],"181":[40.5900002,-75.4700012],"182":[40.9281998,-75.9550018],"183":[41.0704994,-75.223999],"184":[41.5563011,-75.4020004],"185":[41.3987999,-75.6679993],"186":[41.3017006,-76.052002],"187":[41.2428017,-75.8769989],"188":[41.8185997,-76.0699997],"189":[40.3456001,-75.1689987],"190":[40.0023994,-75.2170029],"191":[39.9506989,-75.1600037],"192":[39.9500008,-75.1600037],"193":[39.9255981,-75.6989975],"194":[40.1651001,-75.4209976],"195":[40.4026985,-75.8789978],"196":[40.3300018,-75.9199982],"197":[39.6473999,-75.6729965],"198":[39.7299995,-75.5299988],"199":[38.8676987,-75.4339981],"200":[38.8904991,-77.0279999],"201":[38.9163017,-77.5719986],"202":[38.889801,-77.0289993],"203":[38.8897018,-77.0289993],"204":[38.8902016,-77.0289993],"205":[38.8888016,-77.0270004],"206":[38.411499,-76.7259979],"207":[38.9236984,-76.8300018],"208":[39.0898018,-77.1559982],"209":[39.0078011,-77.0149994],"210":[39.3688011,-76.560997],"211":[39.3806,-76.6409988],"212":[39.3017998,-76.6029968],"214":[38.9762993,-76.4929962],"215":[39.5896988,-78.9649963],"216":[38.8628998,-76.0410004],"217":[39.4906006,-77.4759979],"218":[38.2673988,-75.5979996],"219":[39.5653,-75.9380035],"220":[38.8386993,-77.2529984],"221":[38.7869987,-77.2190018],"222":[38.8699989,-77.098999],"223":[39.2719002,-73.5100021],"224":[38.1216011,-77.0650024],"225":[38.0363998,-77.0070038],"226":[39.0191002,-78.2170029],"227":[38.4705009,-78.0479965],"228":[38.5489998,-78.810997],"229":[38.0168991,-78.5579987],"230":[37.5825996,-77.1179962],"231":[37.5332985,-77.0199966],"232":[37.5499992,-77.4589996],"233":[37.3445015,-75.9560013],"234":[37.2034988,-75.9970016],"235":[36.8400002,-76.2799988],"236":[37.0906982,-76.4160004],"237":[36.8300018,-76.2900009],"238":[36.9640999,-77.4150009],"239":[37.0149994,-78.413002],"240":[37.1874008,-80.0400009],"241":[37.0819016,-80.1439972],"242":[36.8300018,-82.4550018],"243":[36.8325005,-81.1230011],"244":[38.0694008,-79.3560028],"245":[37.1455994,-79.1279984],"246":[37.2044983,-81.7959976],"247":[37.3879013,-81.1969986],"248":[37.4650993,-81.6220016],"249":[37.8452988,-80.4160004],"250":[38.1903992,-81.4919968],"251":[38.2275009,-81.5220032],"252":[38.6134987,-81.5559998],"253":[38.3499985,-81.6299973],"254":[39.3992996,-78.0289993],"255":[38.2853012,-82.1669998],"256":[37.7521019,-82.0410004],"257":[38.4099998,-82.4300003],"258":[37.7857018,-81.2060013],"259":[37.7792015,-81.0660019],"260":[40.1683006,-80.6220016],"261":[39.2207985,-81.2910004],"262":[38.7652016,-80.0559998],"263":[39.1939011,-80.5130005],"264":[39.2542992,-80.3960037],"265":[39.571701,-80.0970001],"266":[38.4544983,-80.8150024],"267":[39.3526001,-78.9599991],"268":[38.9504013,-79.0019989],"270":[36.2434998,-80.3919983],"271":[36.0999985,-80.2399979],"272":[35.8825989,-79.6809998],"273":[35.9370995,-79.6309967],"274":[36.0699997,-79.8239975],"275":[35.8736,-78.5619965],"276":[35.8213997,-78.6520004],"277":[35.9799995,-78.9100037],"278":[35.9087982,-77.4250031],"279":[36.101799,-76.1910019],"280":[35.364399,-81.0179977],"281":[35.2896004,-80.7639999],"282":[35.1907005,-80.8280029],"283":[34.9758987,-78.9840012],"284":[34.2994003,-78.197998],"285":[35.0085983,-77.1100006],"286":[36.0635986,-81.3659973],"287":[35.4407997,-82.6819992],"288":[35.5699997,-82.5400009],"289":[35.0685997,-83.9339981],"290":[33.9669991,-80.8069992],"291":[33.8886986,-80.8690033],"292":[34,-81.0299988],"293":[34.8646011,-81.8990021],"294":[32.9516983,-80.1039963],"295":[34.0573997,-79.4029999],"296":[34.6720009,-82.5479965],"297":[34.8759003,-80.9309998],"298":[33.5167999,-81.7659988],"299":[32.5373993,-80.9039993],"300":[33.8739014,-84.1930008],"301":[34.0091019,-84.8590012],"302":[33.3244019,-84.4749985],"303":[33.7750015,-84.387001],"304":[32.4149017,-82.1200027],"305":[34.5133018,-83.7200012],"306":[33.885601,-83.2639999],"307":[34.7494011,-85.086998],"308":[33.3303986,-82.3580017],"309":[33.4500008,-81.9899979],"310":[32.5946007,-83.4560013],"311":[33.7510986,-84.4599991],"312":[32.8300018,-83.6500015],"313":[31.9062996,-81.4189987],"314":[32.0800018,-81.0999985],"315":[31.3404007,-82.1350021],"316":[30.9741993,-83.2020035],"317":[31.4564991,-83.8659973],"318":[32.5200005,-84.7160034],"319":[32.4949989,-84.8700027],"320":[30.1629009,-82.1050034],"321":[29.3045006,-81.4250031],"322":[30.3232994,-81.6289978],"323":[30.3332005,-84.2610016],"324":[30.4860001,-85.5559998],"325":[30.5349998,-86.9560013],"326":[29.5881996,-82.439003],"327":[28.7658997,-81.3249969],"328":[28.4953995,-81.3700027],"329":[28.0149994,-80.5759964],"330":[25.7621002,-80.3830032],"331":[25.7723007,-80.211998],"332":[25.7749996,-80.2040024],"333":[26.1387005,-80.1399994],"334":[26.6123009,-80.163002],"335":[28.0942001,-82.25],"336":[27.9500008,-82.4789963],"337":[27.8330002,-82.7030029],"338":[27.8938007,-81.7080002],"339":[26.6937008,-81.8919983],"341":[26.1347008,-81.6969986],"342":[27.3167992,-82.4530029],"344":[29.0111008,-82.3050003],"346":[28.3008995,-82.6100006],"347":[28.4988003,-81.6029968],"349":[27.2947006,-80.3550034],"350":[33.5625,-86.6969986],"351":[33.4945984,-86.6279984],"352":[33.5200005,-86.7900009],"354":[33.139801,-87.7279968],"355":[33.9630013,-87.6350021],"356":[34.6845016,-87.2910004],"357":[34.7271004,-86.3550034],"358":[34.7000008,-86.6200027],"359":[34.2778015,-85.9509964],"360":[32.1464005,-86.1340027],"361":[32.3499985,-86.2799988],"362":[33.5634003,-85.7020035],"363":[31.2915001,-85.5410004],"364":[31.4074001,-87.0680008],"365":[30.8661995,-87.9339981],"366":[30.6821995,-88.0419998],"367":[32.2983017,-87.3860016],"368":[32.5791016,-85.3219986],"369":[32.1338005,-88.2529984],"370":[36.1181984,-86.8430023],"371":[36.0601997,-86.7020035],"372":[36.1699982,-86.7799988],"373":[35.2633018,-85.3389969],"374":[35.0600014,-85.25],"375":[35.125,-90.0250015],"376":[36.3786011,-82.3460007],"377":[36.153801,-83.8690033],"378":[36.0970001,-83.7639999],"379":[35.9668007,-83.9469986],"380":[35.4866982,-89.4690018],"381":[35.098999,-89.9899979],"382":[36.2818985,-88.7249985],"383":[35.6036987,-88.5569992],"384":[35.2900009,-87.2929993],"385":[36.1777992,-85.4179993],"386":[34.5901985,-89.8270035],"387":[33.5616989,-90.8089981],"388":[34.3082008,-88.6520004],"389":[33.767601,-89.9410019],"390":[32.4402008,-90.1500015],"391":[32.3535004,-90.2470016],"392":[32.3199997,-90.1999969],"393":[32.3606987,-88.8150024],"394":[31.3351002,-89.3030014],"395":[30.4120007,-88.9919968],"396":[31.3479996,-90.5400009],"397":[33.5265999,-88.8270035],"398":[31.2884998,-84.6320038],"399":[33.7599983,-84.4199982],"400":[38.1178017,-85.3280029],"401":[37.8619995,-86.1419983],"402":[38.2200012,-85.7399979],"403":[38.0569992,-84.1790009],"404":[37.5112,-84.4260025],"405":[38.0402985,-84.4509964],"406":[38.1899986,-84.8600006],"407":[36.9827995,-84.0749969],"408":[36.8927002,-83.2699966],"409":[36.9289017,-83.7040024],"410":[38.7854996,-84.387001],"411":[38.3722,-82.9339981],"412":[37.8554993,-82.7050018],"413":[37.5713005,-83.5039978],"414":[37.8627014,-83.163002],"415":[37.4472008,-82.3669968],"416":[37.4900017,-82.7350006],"417":[37.2360992,-83.2190018],"418":[37.2148018,-82.8529968],"420":[36.9082985,-88.5709991],"421":[36.8820992,-86.0630035],"422":[36.9449997,-86.9860001],"423":[37.4866982,-87.0270004],"424":[37.5041008,-87.6500015],"425":[37.1405983,-84.7109985],"426":[36.7724991,-84.6949997],"427":[37.3362007,-85.7440033],"430":[40.1551018,-82.9219971],"431":[39.6940994,-82.8889999],"432":[39.9799995,-82.9800034],"433":[40.523201,-83.3389969],"434":[41.4564018,-83.2929993],"435":[41.4561996,-84.1269989],"436":[41.6585007,-83.5709991],"437":[39.8488007,-81.6910019],"438":[40.2537003,-81.8379974],"439":[40.2154007,-80.8410034],"440":[41.5446014,-81.3850021],"441":[41.453701,-81.6760025],"442":[41.1130981,-81.5820007],"443":[41.0800018,-81.5199966],"444":[41.0924988,-80.7450027],"445":[41.0900002,-80.6399994],"446":[40.6655006,-81.4649963],"447":[40.8100014,-81.3700027],"448":[41.0280991,-82.762001],"449":[40.7599983,-82.5199966],"450":[39.3857002,-84.487999],"451":[39.1097984,-83.9179993],"452":[39.0909004,-84.5090027],"453":[39.9863014,-84.2730026],"454":[39.7700005,-84.1900024],"455":[39.9199982,-83.7900009],"456":[38.9267998,-82.810997],"457":[39.3391991,-81.8010025],"458":[40.8207016,-84.1800003],"459":[39.0900002,-84.5100021],"460":[40.1358986,-86.0299988],"461":[39.6483002,-86.1520004],"462":[39.7697983,-86.1399994],"463":[41.4430008,-87.1539993],"464":[41.5663986,-87.3290024],"465":[41.4473,-86.0469971],"466":[41.6699982,-86.2600021],"467":[41.167099,-85.1930008],"468":[41.0699997,-85.1299973],"469":[40.7192001,-86.0950012],"470":[39.1525993,-85.060997],"471":[38.3686981,-85.9670029],"472":[39.0625992,-85.7429962],"473":[40.0497017,-85.2060013],"474":[39.0404015,-86.6660004],"475":[38.4118996,-86.9950027],"476":[38.1372986,-87.4599991],"477":[37.9799995,-87.5400009],"478":[39.4564018,-87.3150024],"479":[40.3922005,-87.0429993],"480":[42.6619987,-82.9120026],"481":[42.2205009,-83.4970016],"482":[42.3800011,-83.0979996],"483":[42.6206017,-83.3099976],"484":[43.2397003,-83.2829971],"485":[43.0154991,-83.6780014],"486":[43.8012009,-84.2549973],"487":[43.9235001,-83.5670013],"488":[43.0256996,-84.6829987],"489":[42.700901,-84.5500031],"490":[42.2131996,-85.5849991],"491":[41.8779984,-86.4280014],"492":[42.0373993,-84.3399963],"493":[43.1987991,-85.5390015],"494":[43.2192993,-86.1259995],"495":[42.9556999,-85.6569977],"496":[44.5344009,-85.6350021],"497":[45.5583,-84.5429993],"498":[46.0676003,-87.1940002],"499":[46.7336006,-88.814003],"500":[41.6955986,-93.7730026],"501":[41.5997009,-93.4560013],"502":[41.6749001,-93.5530014],"503":[41.5735016,-93.6190033],"504":[43.1567001,-93.3420029],"505":[42.7400017,-94.4369965],"506":[42.6255989,-92.4509964],"507":[42.4840012,-92.3339996],"508":[40.9550018,-94.5449982],"509":[41.5712013,-93.6179962],"510":[42.592701,-95.9029999],"511":[42.5,-96.3899994],"512":[43.2584991,-96.0279999],"513":[43.2482986,-95.1360016],"514":[42.0872002,-95.0630035],"515":[41.4487991,-95.5540009],"516":[40.7200012,-95.3590012],"520":[42.4738998,-90.9440002],"521":[43.1535988,-91.7149963],"522":[41.8838005,-91.7389984],"523":[41.9015007,-91.7279968],"524":[41.9700012,-91.6600037],"525":[40.9606018,-92.4250031],"526":[40.8683014,-91.4449997],"527":[41.6551018,-90.7399979],"528":[41.5499992,-90.5999985],"530":[43.4492989,-88.1989975],"531":[42.7327003,-88.2160034],"532":[43.0578995,-87.9570007],"534":[42.7200012,-87.8000031],"535":[42.9382019,-89.5960007],"537":[43.0699997,-89.3799973],"538":[42.8135986,-90.7450027],"539":[43.6148987,-89.5159988],"540":[45.0587997,-92.4889984],"541":[44.8563004,-88.2259979],"542":[44.5419998,-87.5589981],"543":[44.5200005,-87.9800034],"544":[44.8679008,-89.788002],"545":[45.9003983,-89.875],"546":[43.844101,-90.9150009],"547":[44.8260002,-91.5920029],"548":[46.0180016,-91.6399994],"549":[44.1971016,-88.8150024],"550":[45.0455017,-93.0139999],"551":[44.9393005,-93.0960007],"553":[44.9744987,-93.8600006],"554":[44.9600983,-93.2600021],"555":[45.0001984,-93.7910004],"556":[47.4785004,-91],"557":[47.167099,-92.7659988],"558":[46.7799988,-92.1100006],"559":[43.8984985,-92.2839966],"560":[43.9954987,-93.961998],"561":[43.8893013,-95.663002],"562":[45.0164986,-95.6520004],"563":[45.7436981,-94.5979996],"564":[46.5713997,-94.5169983],"565":[46.8283997,-96.1259995],"566":[47.8307991,-94.4309998],"567":[48.4243011,-96.3359985],"569":[38.8863983,-77.0169983],"570":[43.4788017,-96.9929962],"571":[43.540699,-96.7279968],"572":[45.0247993,-97.1080017],"573":[43.7943993,-98.3899994],"574":[45.347599,-98.7389984],"575":[43.7448997,-100.4199982],"576":[45.5441017,-101.3700027],"577":[44.0848007,-103.0800018],"580":[46.6772003,-97.2910004],"581":[46.8699989,-96.8099976],"582":[48.2061005,-97.6029968],"583":[48.3576012,-99.211998],"584":[46.9441986,-98.9980011],"585":[46.7795982,-100.8899994],"586":[46.7825012,-102.9000015],"587":[48.3566017,-101.5500031],"588":[48.2644005,-103.5],"590":[45.8152008,-108.6600037],"591":[45.7457008,-108.4899979],"592":[48.3289986,-105.3099976],"593":[46.3152008,-105.3899994],"594":[47.7318993,-111.25],"595":[48.4786987,-109.4400024],"596":[46.5457993,-111.9199982],"597":[45.6693993,-112.1299973],"598":[46.9990997,-114.2600021],"599":[48.3263016,-114.4199982],"600":[42.2144012,-88.0029984],"601":[41.9658012,-88.211998],"602":[42.0129013,-87.6750031],"603":[41.8950005,-87.810997],"604":[41.5187988,-87.9140015],"605":[41.7327995,-88.2860031],"606":[41.8344994,-87.6809998],"607":[41.9459991,-87.7639999],"608":[41.7174988,-87.7020035],"609":[40.8189011,-87.9280014],"610":[42.2019997,-89.5169983],"611":[42.2873993,-89.0469971],"612":[41.4845009,-90.3509979],"613":[41.3394012,-89.1999969],"614":[40.837101,-90.4110031],"615":[40.6853981,-89.6900024],"616":[40.7347984,-89.5970001],"617":[40.4972992,-88.9199982],"618":[40.1004982,-88.1169968],"619":[39.6736984,-88.2109985],"620":[39.0531006,-89.9929962],"622":[38.4356995,-89.7959976],"623":[40.0278015,-91.0550003],"624":[39.0481987,-88.2050018],"625":[39.7025986,-89.2040024],"626":[39.8191986,-89.9889984],"627":[39.7789001,-89.6399994],"628":[38.3031998,-88.6689987],"629":[37.5471992,-88.9899979],"630":[38.4464989,-90.6549988],"631":[38.6300011,-90.2399979],"633":[38.9497986,-90.9950027],"634":[39.9612007,-91.7699966],"635":[40.1926994,-92.5780029],"636":[37.6562004,-90.6080017],"637":[37.3182983,-89.7689972],"638":[36.4939995,-89.7770004],"639":[36.8512001,-90.512001],"640":[39.0775986,-94.211998],"641":[39.0900002,-94.5800018],"644":[40.1043015,-94.6849976],"645":[39.7599983,-94.8499985],"646":[39.861599,-93.5670013],"647":[38.085701,-94.2080002],"648":[36.9533997,-94.3600006],"649":[39.0900002,-94.5800018],"650":[38.4502983,-92.2699966],"651":[38.5699997,-92.1900024],"652":[39.1764984,-92.3799973],"653":[38.7524986,-93.2450027],"654":[37.6762009,-91.802002],"655":[37.6363983,-91.9069977],"656":[37.0508003,-93.1230011],"657":[37.0456009,-92.9899979],"658":[37.1899986,-93.2789993],"660":[38.9107018,-95.0940018],"661":[39.0980988,-94.6309967],"662":[38.9000015,-94.6800003],"664":[39.3783989,-96.0479965],"665":[39.3180008,-96.0950012],"666":[39.0299988,-95.6900024],"667":[37.5886993,-95.0889969],"668":[38.3506012,-96.3649979],"669":[39.7747002,-97.7080002],"670":[37.5493011,-97.5930023],"671":[37.5386009,-97.6660004],"672":[37.6781998,-97.336998],"673":[37.1859016,-95.7539978],"674":[38.9328003,-97.7369995],"675":[38.2290993,-98.8270035],"676":[39.3163986,-99.3949966],"677":[39.3521004,-101.0699997],"678":[37.7925987,-100.6500015],"679":[37.1343002,-101.3799973],"680":[41.4669991,-96.4100037],"681":[41.2494011,-96.0039978],"683":[40.4595985,-96.8249969],"684":[40.531601,-96.6070023],"685":[40.8100014,-96.6800003],"686":[41.4715996,-97.5029984],"687":[42.3891983,-97.711998],"688":[41.1301994,-98.9079971],"689":[40.3409004,-98.8119965],"690":[40.3235016,-100.8600006],"691":[41.2639999,-101.5899963],"692":[42.7900009,-100.5800018],"693":[42.1553001,-103.0299988],"700":[29.8833008,-90.1549988],"701":[30.0599995,-89.9300003],"703":[29.6676006,-90.7789993],"704":[30.5580997,-90.197998],"705":[30.2203007,-92.1200027],"706":[30.3738995,-93.1650009],"707":[30.5053997,-91.1740036],"708":[30.4405003,-91.1220016],"710":[32.4888992,-93.4820023],"711":[32.4771004,-93.7679977],"712":[32.5497017,-91.9970016],"713":[31.3183002,-92.0130005],"714":[31.5475998,-92.8410034],"716":[33.6941986,-91.7570038],"717":[33.5027008,-92.7979965],"718":[33.6544991,-93.8219986],"719":[34.3863983,-93.5250015],"720":[34.948101,-91.987999],"721":[34.8944016,-92.0950012],"722":[34.7212982,-92.3470001],"723":[35.107399,-90.5179977],"724":[36.0359993,-90.7300034],"725":[35.9975014,-91.7829971],"726":[36.1568985,-92.8659973],"727":[36.1665001,-94.1589966],"728":[35.2612991,-93.3889999],"729":[35.3305016,-94.2249985],"730":[35.3008995,-97.7369995],"731":[35.4592018,-97.5110016],"733":[30.2600002,-97.7399979],"734":[34.2305984,-97.1569977],"735":[34.5175018,-98.7249985],"736":[35.5032005,-99.2839966],"737":[36.3790016,-98.1060028],"738":[36.4459991,-99.487999],"739":[36.7426987,-101.5],"740":[36.2528,-96.1800003],"741":[36.1199989,-95.9100037],"743":[36.5447006,-94.9779968],"744":[35.6442986,-95.4300003],"745":[34.6641998,-95.6869965],"746":[36.6911011,-97.1429977],"747":[34.0135994,-95.5970001],"748":[35.1702003,-96.6790009],"749":[35.2090988,-94.685997],"750":[33.0359993,-96.7829971],"751":[32.5555992,-96.4660034],"752":[32.7900009,-96.7600021],"753":[32.7938004,-96.762001],"754":[33.3555984,-95.8059998],"755":[33.2736015,-94.3280029],"756":[32.4553986,-94.6289978],"757":[32.2384987,-95.3320007],"758":[31.5060997,-95.7070007],"759":[31.2978001,-94.3929977],"760":[32.6575012,-97.3280029],"761":[32.7546005,-97.3259964],"762":[33.4003983,-97.2710037],"763":[33.7958984,-98.7440033],"764":[32.5228004,-98.447998],"765":[31.0874996,-97.5179977],"766":[31.7542992,-97.0370026],"767":[31.5559006,-97.1819992],"768":[31.3227997,-99.3150024],"769":[31.4025993,-100.5599976],"770":[29.7600002,-95.3799973],"772":[29.7600002,-95.3799973],"773":[30.3335991,-95.3290024],"774":[29.4810009,-95.9319992],"775":[29.5799007,-95.0400009],"776":[30.0863991,-94.1070023],"777":[30.0842991,-94.1399994],"778":[30.6569996,-96.3300018],"779":[28.8896999,-96.9160004],"780":[28.9253006,-98.8830032],"781":[29.1956997,-97.9800034],"782":[29.4500008,-98.5],"783":[27.7196007,-97.8259964],"784":[27.7999992,-97.3899994],"785":[26.2679005,-98.0319977],"786":[30.2691002,-97.9000015],"787":[30.2600994,-97.7399979],"788":[29.2516994,-100.0999985],"789":[29.9246006,-96.8030014],"790":[35.3368988,-101.5199966],"791":[35.2000008,-101.8099976],"792":[34.2675018,-100.6600037],"793":[33.5005989,-102.1600037],"794":[33.5699997,-101.8700027],"795":[32.6544991,-100.1800003],"796":[32.4473,-99.7409973],"797":[31.7348003,-102.4599991],"798":[30.7873001,-104.75],"799":[31.8369999,-106.4199982],"800":[39.7937012,-104.9499969],"801":[39.5242996,-104.8300018],"802":[39.7589989,-104.8700027],"803":[40.0200005,-105.25],"804":[39.8382988,-105.9000015],"805":[40.3959999,-105.1100006],"806":[40.3297005,-104.6399994],"807":[40.4687996,-103.0500031],"808":[39.077301,-103.8799973],"809":[38.8610992,-104.75],"810":[37.9249001,-103.8000031],"811":[37.4399986,-106.1900024],"812":[38.5047989,-106.1200027],"813":[37.4235001,-108.4199982],"814":[38.4342003,-108],"815":[39.0579987,-108.6200027],"816":[39.6941986,-107.3799973],"820":[41.2681999,-105.0500031],"821":[44.5699997,-110.5100021],"822":[42.2910004,-104.5299988],"823":[41.6393013,-107.1299973],"824":[44.4431992,-108.3499985],"825":[43.0909996,-108.7399979],"826":[43.0157013,-106.5599976],"827":[44.3372002,-104.8300018],"828":[44.709301,-106.8300018],"829":[41.733799,-109.8700027],"830":[43.585701,-110.6699982],"831":[42.5194016,-110.7200012],"832":[42.9174995,-112.3899994],"833":[42.8105011,-114.2600021],"834":[44.0124016,-112.1100006],"835":[46.1543007,-116.2799988],"836":[43.8869019,-116.3000031],"837":[43.6062012,-116.1999969],"838":[47.6780014,-116.5400009],"840":[40.6376991,-111.4100037],"841":[40.7224007,-111.8799973],"842":[41.25,-111.9700012],"843":[41.6982002,-112.0899963],"844":[41.2226982,-111.9700012],"845":[38.6763,-110.2799988],"846":[39.547699,-111.8399963],"847":[37.7989998,-112.6900024],"850":[33.5102997,-112.0699997],"851":[33.0517006,-111.5199966],"852":[33.4350014,-111.8000031],"853":[33.4748001,-112.8799973],"855":[33.3237991,-110.3099976],"856":[31.8295994,-110.4400024],"857":[32.2230988,-110.9400024],"859":[34.2145996,-109.8899994],"860":[35.7228012,-111.1800003],"863":[34.6678009,-112.25],"864":[35.2550011,-114.1500015],"865":[36.0354996,-109.4400024],"870":[35.2033997,-106.75],"871":[35.1195984,-106.6200027],"873":[35.4912987,-108.5999985],"874":[36.7120018,-108.2300034],"875":[36.1282997,-105.9000015],"877":[36.0312996,-104.9499969],"878":[34.0649986,-107.7300034],"879":[32.9550018,-107.2699966],"880":[32.4231987,-107.5299988],"881":[34.2052002,-103.4800034],"882":[32.7975006,-103.8899994],"883":[33.2476006,-105.6299973],"884":[35.6637993,-103.7699966],"885":[31.8400002,-106.4300003],"889":[36.1399994,-115.2799988],"890":[36.4933014,-115.2799988],"891":[36.1334991,-115.1800003],"893":[39.2821999,-115.3099976],"894":[39.6557007,-119.0699997],"895":[39.5325012,-119.8000031],"897":[39.1335983,-119.75],"898":[41.0950012,-115.5899963],"900":[34.0368004,-118.2900009],"901":[34.0133018,-118.1900024],"902":[33.9458008,-118.3499985],"903":[33.9491997,-118.3399963],"904":[34.0144997,-118.4700012],"905":[33.8359985,-118.3199997],"906":[33.9199982,-118.0199966],"907":[33.7944984,-118.1800003],"908":[33.7871017,-118.1600037],"910":[34.1791992,-118.1100006],"911":[34.142601,-118.1200027],"912":[34.1631012,-118.2399979],"913":[34.2616997,-118.5599976],"914":[34.1767998,-118.4599991],"915":[34.1722984,-118.3199997],"916":[34.1612015,-118.3799973],"917":[34.0637016,-117.8399963],"918":[34.0849991,-118.1200027],"919":[32.6932983,-116.8499985],"920":[33.0984001,-117.1100006],"921":[32.7691994,-117.1200027],"922":[33.5985985,-115.9199982],"923":[34.4700012,-116.9800034],"924":[34.1260986,-117.2900009],"925":[33.7803001,-117.1999969],"926":[33.6240005,-117.7799988],"927":[33.7341003,-117.8499985],"928":[33.8367004,-117.8300018],"930":[34.2813988,-119.0899963],"931":[34.4147987,-119.7699966],"932":[35.8869019,-119.2099991],"933":[35.3636017,-119.0100021],"934":[35.1856995,-120.5800018],"935":[35.5271988,-118.0999985],"936":[36.8405991,-119.6800003],"937":[36.7546997,-119.7799988],"938":[36.75,-119.6699982],"939":[36.4986992,-121.5999985],"940":[37.4692993,-122.25],"941":[37.7657013,-122.4100037],"942":[38.5665016,-121.4800034],"943":[37.4185982,-122.1399994],"944":[37.5480003,-122.3099976],"945":[37.9367981,-122.0599976],"946":[37.787899,-122.2200012],"947":[37.8699989,-122.2600021],"948":[37.9440002,-122.3300018],"949":[38.0755005,-122.6399994],"950":[37.1021996,-121.8600006],"951":[37.3089981,-121.8600006],"952":[38.0922012,-120.9800034],"953":[37.6324997,-120.6800003],"954":[38.8452988,-123.0299988],"955":[40.7345009,-123.9100037],"956":[38.6306,-121.2699966],"957":[38.8376007,-120.9599991],"958":[38.5778999,-121.4400024],"959":[39.4948997,-121.5100021],"960":[40.8586006,-122.2699966],"961":[39.8908997,-120.25],"967":[20.4083996,-157.1900024],"968":[21.3157997,-157.8600006],"969":[11.6146002,147.3970032],"970":[45.4295006,-122.2300034],"971":[45.530201,-123.4899979],"972":[45.5093002,-122.6500015],"973":[44.7494011,-123.1699982],"974":[43.5511017,-123.4300003],"975":[42.3800011,-123.0899963],"976":[42.4325981,-121.1999969],"977":[43.8400002,-120.5199966],"978":[45.2013016,-118.5999985],"979":[43.8737984,-117.5599976],"980":[47.544899,-122.1399994],"981":[47.5876999,-122.3199997],"982":[48.3555984,-122.2799988],"983":[47.4272995,-122.6500015],"984":[47.2014008,-122.4599991],"985":[46.9266014,-123.2200012],"986":[45.9278984,-122.4899979],"988":[47.8572006,-119.8899994],"989":[46.6609001,-120.5299988],"990":[47.6156006,-117.4400024],"991":[47.8194008,-117.9800034],"992":[47.674099,-117.3700027],"993":[46.3577995,-118.9599991],"994":[46.2033005,-117.2099991],"995":[60.1226006,-154.7899933],"996":[60.2083015,-156.6199951],"997":[65.7145996,-153.25],"998":[58.0144997,-134.8699951],"999":[55.6292,-132.1499939],"064":[41.4343987,-72.7669983],"005":[40.8100014,-73.0400009],"029":[41.8143005,-71.4209976],"019":[42.5932007,-70.8960037],"038":[43.3703003,-71.0319977],"060":[41.8544998,-72.7890015],"077":[40.3031006,-74.0970001],"016":[42.2611008,-71.810997],"051":[43.2378006,-72.6090012],"026":[41.7293015,-70.1669998],"063":[41.4846992,-72.0329971],"058":[44.661499,-72.1279984],"083":[39.4229012,-75.0230026],"059":[44.7770996,-71.6439972],"013":[42.5960999,-72.5630035],"041":[43.6593018,-70.2419968],"025":[41.5769997,-70.5589981],"039":[43.1800003,-70.7070007],"010":[42.2596016,-72.5650024],"056":[44.3277016,-72.5630035],"015":[42.2160988,-71.810997],"085":[40.2734985,-74.6569977],"023":[42.0096016,-70.8789978],"080":[39.8176003,-74.9789963],"036":[43.1674995,-72.3570023],"062":[41.8219986,-72.0780029],"008":[18.0006008,-64.8190002],"024":[42.3497009,-71.2070007],"047":[46.7075005,-68.2450027],"055":[42.6500015,-71.1399994],"082":[39.2664986,-74.6800003],"084":[39.3432999,-74.4599991],"009":[18.3913994,-66.0749969],"020":[42.1497002,-71.0410004],"067":[41.6290016,-73.1529999],"006":[18.2886009,-66.8509979],"031":[42.9954987,-71.4540024],"054":[44.5987015,-73.0530014],"071":[40.7270012,-74.1709976],"011":[42.1054993,-72.5319977],"050":[43.8171997,-72.3700027],"087":[39.9743004,-74.1419983],"018":[42.6194,-71.1900024],"022":[42.3464012,-71.0589981],"053":[42.9179993,-72.7300034],"032":[43.5125008,-71.6449966],"043":[44.293499,-69.7639999],"078":[40.9292984,-74.7529984],"076":[40.9213982,-74.0159988],"040":[43.7271004,-70.5059967],"045":[43.9325981,-69.572998],"027":[41.7608986,-71.064003],"052":[43.0255013,-73.1409988],"079":[40.736599,-74.5100021],"037":[43.6907005,-72.1449966],"044":[45.1130981,-68.7200012],"042":[44.2955017,-70.3619995],"057":[43.6539993,-73.052002],"061":[41.7531013,-72.6839981],"068":[41.1866989,-73.4309998],"073":[40.7108994,-74.060997],"030":[42.8758011,-71.4759979],"049":[44.7472992,-69.6449966],"093":[-0.885,0.6706],"033":[43.2216988,-71.5400009],"081":[39.9239998,-75.0820007],"046":[44.537899,-68.0039978],"069":[41.0900002,-73.5500031],"048":[44.1038017,-69.0950012],"014":[42.5840988,-71.776001],"075":[40.9150009,-74.1669998],"066":[41.1912994,-73.1900024],"070":[40.7541008,-74.223999],"007":[18.1515999,-66.1110001],"035":[44.4880981,-71.4899979],"065":[41.3179016,-72.9199982],"017":[42.3358002,-71.4430008],"086":[40.2200012,-74.7600021],"088":[40.5354996,-74.6210022],"012":[42.3446007,-73.2249985],"072":[40.6624985,-74.211998],"074":[41.039299,-74.2949982],"021":[42.3353004,-71.0619965],"028":[41.6659012,-71.5230026],"089":[40.4786987,-74.4420013],"034":[42.9099998,-72.2030029]}', true); }
        return isset($this->zip_loc[$zip_prefix]) ? $this->zip_loc[$zip_prefix] : null;
    }
    
    /*
     * Provide approx state latitude/longitude 
     */
    public function lookup_state_location($state){
        if ($this->state_loc == null) { $this->state_loc = json_decode('{"AK":[61.385,-152.2683],"AL":[32.799,-86.8073],"AR":[34.9513,-92.3809],"AS":[14.2417,-170.7197],"AZ":[33.7712,-111.3877],"CA":[36.17,-119.7462],"CO":[39.0646,-105.3272],"CT":[41.5834,-72.7622],"DC":[38.8964,-77.0262],"DE":[39.3498,-75.5148],"FL":[27.8333,-81.717],"GA":[32.9866,-83.6487],"HI":[21.1098,-157.5311],"IA":[42.0046,-93.214],"ID":[44.2394,-114.5103],"IL":[40.3363,-89.0022],"IN":[39.8647,-86.2604],"KS":[38.5111,-96.8005],"KY":[37.669,-84.6514],"LA":[31.1801,-91.8749],"MA":[42.2373,-71.5314],"MD":[39.0724,-76.7902],"ME":[44.6074,-69.3977],"MI":[43.3504,-84.5603],"MN":[45.7326,-93.9196],"MO":[38.4623,-92.302],"MP":[14.8058,145.55051],"MS":[32.7673,-89.6812],"MT":[46.9048,-110.3261],"NC":[35.6411,-79.8431],"ND":[47.5362,-99.793],"NE":[41.1289,-98.2883],"NH":[43.4108,-71.5653],"NJ":[40.314,-74.5089],"NM":[34.8375,-106.2371],"NV":[38.4199,-117.1219],"NY":[42.1497,-74.9384],"OH":[40.3736,-82.7755],"OK":[35.5376,-96.9247],"OR":[44.5672,-122.1269],"PA":[40.5773,-77.264],"PR":[18.2766,-66.335],"RI":[41.6772,-71.5101],"SC":[33.8191,-80.9066],"SD":[44.2853,-99.4632],"TN":[35.7449,-86.7489],"TX":[31.106,-97.6475],"UT":[40.1135,-111.8535],"VA":[37.768,-78.2057],"VI":[18.0001,-64.8199],"VT":[44.0407,-72.7093],"WA":[47.3917,-121.5708],"WI":[44.2563,-89.6385],"WV":[38.468,-80.9696],"WY":[42.7475,-107.2085]}', true); }
        return isset($this->state_loc[$state]) ? $this->state_loc[$state] : null;
    }

    
}