<?php
/**
 * WooCommerce Global Payments HPP
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Global Payments HPP to newer
 * versions in the future. If you wish to customize WooCommerce Global Payments HPP for your
 * needs please refer to https://docs.woocommerce.com/document/woocommerce-global-payments/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2012-2025, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Global_Payments\API\APM;

defined( 'ABSPATH' ) or exit;

use SkyVerge\WooCommerce\PluginFramework\v5_15_9 as Framework;

/**
 * The Local Payments Refund API request class.
 *
 * @since 2.4.0
 */
class Payment_Credit_Request extends \WC_Realex_Redirect_API_Request {


	/** @var \WC_Order order object */
	protected $order;


	/**
	 * Constructs the class.
	 *
	 * @since 2.4.0
	 *
	 * @param \WC_Order $order order object
	 */
	public function __construct( \WC_Order $order ) {

		$this->order = $order;
	}


	/**
	 * Sets the data for a Local Payments refund (payment credit).
	 *
	 * The API expects the fields in the request data to be always included in particular order.
	 *
	 * @since 2.4.0
	 */
	public function set_refund_data() {

		$this->request_type = 'payment-credit';

		$this->request_data = [
			'amount'        => [
				'@attributes' => [
					'currency' => $this->get_order()->get_currency(),
				],
				'@value'      => $this->get_order()->refund->amount * 100,
			],
			'orderid'       => $this->get_order()->refund->realex_id,
			'pasref'        => $this->get_order()->refund->trans_id,
			'paymentmethod' => $this->get_order()->refund->payment_method,
			'comments' => [
				'comment' => [
					'@attributes' => [
						'id' => 1,
					],
					'@value' => Framework\SV_WC_Helper::str_truncate( preg_replace( '/[^A-Za-z0-9-_+ \'".,]*/', '', $this->get_order()->refund->reason ), 255 ),
				],
			],
			'sha1hash'      => '', // the value will be provided in WC_Realex_Redirect_API_Request::get_data() but the field needs to be defined before refundhash
			'refundhash'    => sha1( $this->get_order()->refund->password ),
		];
	}


	/**
	 * Gets the parameters used to generate the SHA-1 hash.
	 *
	 * @since 2.4.0
	 *
	 * @return array
	 */
	protected function get_hash_values() {

		$values = [
			$this->request_data['orderid'],
		];

		if ( ! empty( $this->request_data['amount'] ) ) {

			if ( is_array( $this->request_data['amount'] ) ) {
				$values[] = $this->request_data['amount']['@value'];
				$values[] = $this->request_data['amount']['@attributes']['currency'];
			} else {
				$values[] = $this->request_data['amount'];
				$values[] = '';
			}

		} else {

			$values[] = ''; // intentionally blank
			$values[] = ''; // intentionally blank
		}

		$values[] = $this->request_data['paymentmethod'];

		return $values;
	}


	/**
	 * Gets the order object associated with this request.
	 *
	 * @since 2.4.0
	 *
	 * @return \WC_Order
	 */
	protected function get_order() {

		return $this->order;
	}


	/**
	 * Gets the string representation of this request with any and all sensitive elements masked or removed.
	 *
	 * @since 2.0.0
	 *
	 * @return string
	 */
	public function to_string_safe() {

		$string = parent::to_string_safe();

		// mask the refund SHA-1 hash
		if ( isset( $this->request_data[ $this->get_root_element() ]['refundhash'] ) && $hash = $this->request_data[ $this->get_root_element() ]['refundhash'] ) {
			$string = str_replace( $hash, str_repeat( '*', strlen( $hash ) ), $string );
		}

		return $string;
	}


}
