<?php
/**
 * WooCommerce Global Payments HPP
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Global Payments HPP to newer
 * versions in the future. If you wish to customize WooCommerce Global Payments HPP for your
 * needs please refer to https://docs.woocommerce.com/document/woocommerce-global-payments/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2012-2025, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Global_Payments\HPP\APM;

defined( 'ABSPATH' ) or exit;

/**
 * An APM Redirect Response
 *
 * @since 2.4.0
 */
class Redirect_Response extends \WC_Realex_Redirect_API_HPP_Response {


	/**
	 * Determines if the transaction was held.
	 *
	 * @since 2.4.0
	 *
	 * @return bool
	 */
	public function transaction_held() {

		return '01' === $this->get_status_code();
	}


	/**
	 * Gets the response payment type.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_payment_type() {

		return $this->get_value( 'PAYMENTMETHOD' );;
	}


	/**
	 * Gets the customer's account number.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_account_number() {

		return $this->get_value( 'ACCOUNTNUMBER' );
	}


	/**
	 * Gets the payment descriptor.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_payment_descriptor() {

		return $this->get_value( 'HPP_APM_DESCRIPTOR' );
	}


	/**
	 * Gets the customer's bank identifier code.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_bic_code() {

		return $this->get_value( 'HPP_CUSTOMER_BIC' );
	}


	/**
	 * Determines whether a Status Response is expected to arrive in the next few seconds.
	 *
	 * WAITFORNOTIFICATION = 1 means the transaction will most likely remain pending for hours or days.
	 *
	 * @since 2.4.0
	 *
	 * @return boolean
	 */
	public function should_wait_for_notification() {

		return '1' !== $this->get_value( 'WAITFORNOTIFICATION' );
	}


	/**
	 * Gets the string representation of this request with any and all sensitive elements masked or removed.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function to_string_safe() {

		$string = parent::to_string_safe();

		// mask the account number
		$string = str_replace( $this->get_account_number(), str_repeat( '*', strlen( $this->get_account_number() ) ), $string );

		// mask the BIC code
		$string = str_replace( $this->get_bic_code(), str_repeat( '*', strlen( $this->get_bic_code() ) ), $string );

		return $string;
	}


}
