<?php
/**
 * WooCommerce Global Payments HPP
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Global Payments HPP to newer
 * versions in the future. If you wish to customize WooCommerce Global Payments HPP for your
 * needs please refer to https://docs.woocommerce.com/document/woocommerce-global-payments/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2012-2023, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Global_Payments\HPP\APM;

defined( 'ABSPATH' ) or exit;

/**
 * A single Alternative Payment Method.
 *
 * @since 2.4.0
 */
class Alternative_Payment_Method {


	/** @var string payment method ID */
	private $id;

	/** @var string payment method name */
	private $name;

	/** @var string[] list of country codes for the countries where the payment method is supported */
	private $countries = [];

	/** @var string[] list of currency codes for the currencies that the payment method supports */
	private $currencies = [];

	/** @var bool whether the method should be hidden from admin settings and not offered as a payment option */
	private $hidden = false;


	/**
	 * Constructor.
	 *
	 * @since 2.4.0
	 *
	 * @param array $args payment method properties
	 */
	public function __construct( $args ) {

		$args = wp_parse_args( $args, [
			'countries'  => [],
			'currencies' => [],
			'hidden'     => false,
		] );

		if ( ! isset( $args['id'] ) ) {
			throw new \InvalidArgumentException( 'The ID of the payment method must be defined. No value given.' );
		}

		if ( ! isset( $args['name'] ) ) {
			throw new \InvalidArgumentException( 'The name of the payment method must be defined. No value given.' );
		}

		$this->id         = $args['id'];
		$this->name       = $args['name'];
		$this->countries  = (array) $args['countries'];
		$this->currencies = (array) $args['currencies'];
		$this->hidden     = (bool) $args['hidden'];
	}


	/**
	 * Gets the ID of the payment method.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_id() {

		return $this->id;
	}


	/**
	 * Gets the name of the payment method.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_name() {

		return $this->name;
	}


	/**
	 * Gets the list of country codes for the countries where the payment method is supported.
	 *
	 * An empty list means the payment method is supported in all countries.
	 *
	 * @since 2.4.0
	 *
	 * @return string[]
	 */
	public function get_countries() {

		return $this->countries;
	}


	/**
	 * Gets the list of currency codes for the supported currencies.
	 *
	 * An empty list means the payment method supports all currencies.
	 *
	 * @since 2.4.0
	 *
	 * @return string[]
	 */
	public function get_currencies() {

		return $this->currencies;
	}


	/**
	 * Determines whether the given currency is supported by this payment method.
	 *
	 * @since 2.4.0
	 *
	 * @param string $currency currency code
	 * @return bool
	 */
	public function supports_currency( $currency ) {

		$supported = true;

		if ( $currencies = $this->get_currencies() ) {
			$supported = in_array( $currency, $currencies, true );
		}

		return $supported;
	}


	/**
	 * Determines whether the payment method can be displayed as a payment option for new transactions.
	 *
	 * Hidden payment methods are not shown on the gateway settings as available payment methods either.
	 *
	 * @since 2.4.0
	 *
	 * @return bool
	 */
	public function is_hidden() {

		return $this->hidden;
	}


	/**
	 * Gets APM specific transaction data from the given response.
	 *
	 * @since 2.4.0
	 *
	 * @param Redirect_Response $response response object
	 *
	 * @return array
	 */
	public function get_transaction_data( Redirect_Response $response ) {

		return [
			'apm_payment_descriptor' => $response->get_payment_descriptor(),
		];
	}


}
