<?php
/**
 * WooCommerce Global Payments HPP
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade WooCommerce Global Payments HPP to newer
 * versions in the future. If you wish to customize WooCommerce Global Payments HPP for your
 * needs please refer to https://docs.woocommerce.com/document/woocommerce-global-payments/ for more information.
 *
 * @author      SkyVerge
 * @copyright   Copyright (c) 2012-2023, SkyVerge, Inc.
 * @license     http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Global_Payments\HPP\APM;

defined( 'ABSPATH' ) or exit;

/**
 * An APM Status Response
 *
 * @since 2.4.0
 */
class Status_Response extends Redirect_Response {


	/**
	 * Constructs the class.
	 *
	 * @since 2.4.0
	 *
	 * @param array $data payment response data
	 */
	public function __construct( $data ) {

		unset( $data['wc-api'] );
		unset( $data['status_update'] );

		// make sure all data keys are uppercase
		parent::__construct( array_combine( array_map( 'strtoupper', array_keys( $data ) ), $data ) );
	}


	/**
	 * Gets the merchant ID that generated this response.
	 *
	 * The Status Response uses MERCHANTID instead of MERCHANT_ID
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_merchant_id() {

		return $this->get_value( 'MERCHANTID' );
	}


	/**
	 * Gets the order number associated with this response.
	 *
	 * The Status Response uses ORDERID instead of ORDER_ID
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_order_number() {

		return $this->get_value( 'ORDERID' );
	}


	/**
	 * Gets the payment descriptor.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_payment_descriptor() {

		return $this->get_value( 'PAYMENTPURPOSE' );
	}


	/**
	 * Gets the customer's bank identifier code.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_bic_code() {

		return $this->get_value( 'BIC' );
	}


	/**
	 * Gets the customer's international bank account number.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_iban_code() {

		return $this->get_value( 'IBAN' );
	}


	/**
	 * Gets the customer's bank name.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_bank_name() {

		return $this->get_value( 'BANKNAME' );
	}


	/**
	 * Gets the bank code for the customer's account.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function get_bank_code() {

		return $this->get_value( 'BANKCODE' );
	}


	/**
	 * Determines if this is an IPN response.
	 *
	 * @since 2.4.0
	 *
	 * @return true
	 */
	public function is_ipn() {

		return true;
	}


	/**
	 * Gets the parameters used to generate the sha1 hash.
	 *
	 * @since 2.4.0
	 *
	 * @return array
	 */
	public function get_hash_parameters() {

		return [
			$this->get_timestamp(),
			$this->get_merchant_id(),
			$this->get_order_number(),
			$this->get_status_code(),
			$this->get_status_message(),
			$this->get_transaction_id(),
			$this->get_payment_type(),
		];
	}


	/**
	 * Determines whether the response data includes the payment amount.
	 *
	 * @since 2.4.0
	 *
	 * @return bool
	 */
	public function has_amount() {

		return false;
	}


	/**
	 * Gets the string representation of this request with any and all sensitive elements masked or removed.
	 *
	 * @since 2.4.0
	 *
	 * @return string
	 */
	public function to_string_safe() {

		$string = parent::to_string_safe();

		// mask the IBAN code
		$string = str_replace( $this->get_iban_code(), str_repeat( '*', strlen( $this->get_iban_code() ) ), $string );

		// mask the bank code
		$string = str_replace( $this->get_bank_code(), str_repeat( '*', strlen( $this->get_bank_code() ) ), $string );

		// mask the bank name
		$string = str_replace( $this->get_bank_name(), str_repeat( '*', strlen( $this->get_bank_name() ) ), $string );

		return $string;
	}


}
