<?php
/**
 * WooCommerce Memberships Role Handler
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future. If you wish to customize this plugin for your
 * needs please refer to https://www.skyverge.com/product/woocommerce-memberships-role-handler/
 * for more information.
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2017-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Memberships\RoleHandler;

defined( 'ABSPATH' ) or exit;

use SkyVerge\WooCommerce\PluginUpdater as Updater;

/**
 * Adds the ability to change role based on member active or inactive status, ensuring compatibility for plugins that leverage role
 *  for discounts or other functionality.
 *
 * Adds a settings UI to tie roles to specific membership plans
 *
 * @since 1.0.0
 */
class Plugin {


	const VERSION = '1.2.4';

	/** @var \SkyVerge\WooCommerce\Memberships\RoleHandler\Plugin single instance of this plugin */
	protected static $instance;

	/** @var Updater\License $license license class instance */
	protected $license;

	/** @var \SkyVerge\WooCommerce\Memberships\RoleHandler\Admin admin instance */
	protected $admin;


	/**
	 * Plugin constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// load translations
		add_action( 'init', array( $this, 'load_translation' ) );

		// add plugin links
		add_filter( 'plugin_action_links_' . plugin_basename( $this->get_plugin_file() ), array( $this, 'add_plugin_links' ) );

		// trigger roles changes with membership status
		add_action( 'wc_memberships_user_membership_status_changed', array( $this, 'maybe_trigger_status_role_change' ), 10, 3 );
		add_action( 'wc_memberships_user_membership_saved',          array( $this, 'maybe_trigger_creation_role_change' ), 10, 2 );

		// set member role from import data
		add_filter( 'wc_memberships_csv_import_user_memberships_data', array( $this, 'csv_import_parse_member_role' ), 10, 4 );
		add_action( 'wc_memberships_csv_import_user_membership',       array( $this, 'csv_import_set_member_role' ), 10, 3 );

		// export member role in CSV output
		add_filter( 'wc_memberships_csv_export_user_memberships_headers',            array( $this, 'csv_export_member_role_header' ) );
		add_filter( 'wc_memberships_csv_export_user_memberships_member_role_column', array( $this, 'csv_export_member_role_column' ), 10, 3 );

		// change roles when membership is deleted
		add_action( 'delete_post', array( $this, 'maybe_trigger_deleted_role_change' ), 15 );

		$this->includes();

		if ( is_admin() && ! is_ajax() ) {

			$this->admin_includes();

			// run every time
			$this->install();
		}
	}


	/** Plugin methods ***************************************/


	/**
	 * Include the required files.
	 *
	 * @since 1.0.0
	 */
	public function includes() {

		// FIX: SkyVerge updater breaks WordPress updates screen
//		if ( ! class_exists( 'Updater\\License' ) ) {
//			require_once( $this->get_plugin_path() . '/lib/skyverge/updater/class-skyverge-plugin-license.php' );
//		}
//
//		// item ID is from skyverge.com download WP_Post ID
//		$this->license = new Updater\License( $this->get_plugin_file(), $this->get_plugin_path(), $this->get_plugin_url(), $this->get_plugin_name(), $this->get_version(), 9087 );
	}


	/**
	 * Include admin required files.
	 *
	 * @since 1.0.0
	 */
	public function admin_includes() {

		require_once( $this->get_plugin_path() . '/includes/admin/class-wc-memberships-role-handler-admin.php' );
		$this->admin = new Admin();
	}


	/**
	 * Trigger a role change when membership status changes
	 *
	 * @since 1.0.0
	 *
	 * @param \WC_Memberships_User_Membership $user_membership the user's membership
	 * @param string $old_status the previous membership status
	 * @param string $new_status the new membership status
	 */
	public function maybe_trigger_status_role_change( $user_membership, $old_status, $new_status ) {

		$user_id       = $user_membership->get_user_id();
		$active_role   = get_option( 'wc_memberships_role_handler_member_role', 'customer' );
		$inactive_role = get_option( 'wc_memberships_role_handler_inactive_role', 'customer' );

		// is this membership active or inactive?
		if ( in_array( $new_status, wc_memberships()->get_user_memberships_instance()->get_active_access_membership_statuses(), true ) ) {

			$from_role = $inactive_role;
			$to_role   = $active_role;

		} else {

			// be sure there are no other active memberships before changing roles
			foreach ( wc_memberships_get_user_memberships( $user_id ) as $membership ) {
				if ( $membership->is_active() ) {
					return;
				}
			}

			// if we're here, there are no other active memberships, you may fire when ready
			$from_role = $active_role;
			$to_role   = $inactive_role;
		}

		wc_memberships_role_handler_change_user_role( $user_id, $from_role, $to_role );
	}


	/**
	 * Maybe trigger a role change when a new membership has been created (if active)
	 *
	 * @since 1.0.0
	 *
	 * @param \WC_Memberships_Membership_Plan $plan The plan that user was granted access to
	 * @param array $args {
	 *  @type int|string $user_id user ID for the membership
	 *  @type int|string $user_membership_id post ID for the new user membership
	 *  @type bool $is_update true if the membership is being updated, false if new
	 * }
	 */
	public function maybe_trigger_creation_role_change( $plan, $args ) {

		// bail if plan is not yet be set or the user ID is undetermined
		if ( ! $plan instanceof \WC_Memberships_Membership_Plan || empty( $args['user_id'] ) ) {
			return;
		}

		// check if the user is an active member of the plan, skipping cache
		if ( wc_memberships_is_user_active_member( $args['user_id'], $plan->get_id(), false ) ) {

			$active_role   = get_option( 'wc_memberships_role_handler_member_role', 'customer' );
			$inactive_role = get_option( 'wc_memberships_role_handler_inactive_role', 'customer' );

			wc_memberships_role_handler_change_user_role( $args['user_id'], $inactive_role, $active_role );
		}
	}


	/**
	 * Ensures role changes when required while memberships are deleted.
	 *
	 * @since 1.0.0
	 *
	 * @param int $post_id the post ID being deleted
	 */
	public function maybe_trigger_deleted_role_change( $post_id ) {

		// bail out if the post being deleted is not a user membership
		if ( 'wc_user_membership' !== get_post_type( $post_id ) ) {
			return;
		}

		// change role if there are no other active memberships
		if ( $user_membership = wc_memberships_get_user_membership( $post_id ) ) {

			// be sure there are no other active memberships before changing roles
			foreach ( wc_memberships_get_user_memberships( $user_membership->get_user_id() ) as $membership ) {

				// don't check the one we're deleting :)
				if ( $post_id === $membership->get_id() ) {
					continue;
				}

				if ( $membership->is_active() ) {
					return;
				}
			}

			// if there are no other active memberships, you may fire when ready
			wc_memberships_role_handler_change_user_role(
				$user_membership->get_user_id(),
				get_option( 'wc_memberships_role_handler_member_role', 'customer' ),
				get_option( 'wc_memberships_role_handler_inactive_role', 'customer' )
			);
		}
	}


	/**
	 * Parses member role data from a CSV imported row for setting the member role later.
	 *
	 * @internal
	 *
	 * @since 1.1.0
	 *
	 * @param array $import_data data to be imported
	 * @param string $action import action (create or update records)
	 * @param array $columns column data
	 * @param array $row row data
	 * @return array associative array data
	 */
	public function csv_import_parse_member_role( $import_data, $action, $columns, $row ) {

		if ( is_array( $import_data ) ) {
			$import_data['member_role'] = ! empty( $row['member_role'] ) && is_string( $row['member_role'] ) ? $row['member_role'] : null;
		}

		return $import_data;
	}


	/**
	 * Sets a user role based on CSV import data.
	 *
	 * @internal
	 *
	 * @since 1.1.0
	 *
	 * @param \WC_Memberships_User_Membership $user_membership
	 * @param string $action import action (create or update)
	 * @param array $import_data CSV import data
	 */
	public function csv_import_set_member_role( $user_membership, $action, $import_data ) {

		if (      is_array( $import_data )
		     &&   $user_membership instanceof \WC_Memberships_User_Membership
		     && ! empty( $import_data['member_role'] )
		     &&   ( $user = $user_membership->get_user() ) ) {

			wc_memberships_role_handler_change_user_role( $user->ID, array_shift( $user->roles ), $import_data['member_role'] );
		}
	}


	/**
	 * Sets the CSV header for exporting the member role.
	 *
	 * @internal
	 *
	 * @since 1.1.0
	 *
	 * @param array $headers array of CSV headers
	 * @return array
	 */
	public function csv_export_member_role_header( $headers ) {

		$new_headers = array();

		if ( is_array( $headers ) ) {

			foreach ( $headers as $k => $v ) {

				$new_headers[ $k ] = $v;

				if ( 'member_email' === $k ) {
					$new_headers['member_role'] = 'member_role';
				}
			}
		}

		if ( ! isset( $new_headers['member_role'] ) ) {
			$new_headers['member_role'] = 'member_role';
		}

		return $new_headers;
	}


	/**
	 * Sets the member role for the corresponding exported CSV column.
	 *
	 * @internal
	 *
	 * @since 1.1.0
	 *
	 * @param string $value value (the member role)
	 * @param string $column_name CSV column name
	 * @param \WC_Memberships_User_Membership $user_membership membership object being exported
	 * @return string
	 */
	public function csv_export_member_role_column( $value, $column_name, $user_membership ) {

		if (    'member_role' === $column_name
		     && $user_membership instanceof \WC_Memberships_User_Membership
		     && ( $user = $user_membership->get_user() ) ) {

			$role  = array_shift( $user->roles );
			$value = is_string( $role ) ? $role : '';
		}

		return $value;
	}


	/** Helper methods ***************************************/


	/**
	 * Gets the admin class instance.
	 *
	 * @since 1.0.0
	 * @return \SkyVerge\WooCommerce\Memberships\RoleHandler\Admin admin instance
	 */
	public function get_admin_instance() {
		return $this->admin;
	}


	/**
	 * Gets the updater class instance.
	 *
	 * @since 1.0.0
	 *
	 * @return Updater\License
	 */
	public function get_license_instance() {
		return $this->license;
	}


	/**
	 * Adds plugin page links
	 *
	 * @since 1.0.0
	 *
	 * @param array $links all plugin links
	 * @return array $links all plugin links + our custom links (i.e., "Settings")
	 */
	public function add_plugin_links( $links ) {

		$plugin_links = array(
			'<a href="' . admin_url( 'admin.php?page=wc-settings&tab=memberships' ) . '">' . __( 'Configure', 'woocommerce-memberships-role-handler' ) . '</a>',
			'<a href="http://skyverge.com/product/woocommerce-memberships-role-handler/#usage">' . __( 'Docs', 'woocommerce-memberships-role-handler' ) . '</a>',
		);

		$text = $this->get_license_instance()->is_license_valid() ? __( 'License', 'woocommerce-memberships-role-handler' ) : __( 'Get updates', 'woocommerce-memberships-role-handler' );

		$plugin_links[] = '<a href="' . $this->get_license_instance()->get_license_settings_url() . '">' . $text . '</a>';

		return array_merge( $plugin_links, $links );
	}


	/**
	 * Main Plugin Instance, ensures only one instance is/can be loaded
	 *
	 * @since 1.0.0
	 *
	 * @see wc_memberships_role_handler()
	 * @return \SkyVerge\WooCommerce\Memberships\RoleHandler\Plugin
	 */
	public static function instance() {
		if ( is_null( self::$instance ) ) {
			self::$instance = new self();
		}
		return self::$instance;
	}


	/**
	 * Cloning instances is forbidden due to singleton pattern.
	 *
	 * @since 1.0.0
	 */
	public function __clone() {
		/* translators: Placeholders: %s - plugin name */
		_doing_it_wrong( __FUNCTION__, sprintf( esc_html__( 'You cannot clone instances of %s.', 'woocommerce-memberships-role-handler' ), $this->get_plugin_name() ), '1.0.0' );
	}


	/**
	 * Unserializing instances is forbidden due to singleton pattern.
	 *
	 * @since 1.0.0
	 */
	public function __wakeup() {
		/* translators: Placeholders: %s - plugin name */
		_doing_it_wrong( __FUNCTION__, sprintf( esc_html__( 'You cannot unserialize instances of %s.', 'woocommerce-memberships-role-handler' ), $this->get_plugin_name() ), '1.0.0' );
	}


	/**
	 * Load Translations
	 *
	 * @since 1.0.0
	 */
	public function load_translation() {
		// localization
		load_plugin_textdomain( 'woocommerce-memberships-role-handler', false, dirname( plugin_basename( __FILE__ ) ) . '/i18n/languages' );
	}


	/**
	 * Helper to return the plugin name.
	 *
	 * @since 1.0.0
	 *
	 * @return string plugin name
	 */
	public function get_plugin_name() {
		return __( 'WooCommerce Memberships Role Handler', 'woocommerce-memberships-role-handler' );
	}


	/**
	 * Helper to get the plugin path.
	 *
	 * @since 1.0.0
	 *
	 * @return string the plugin path
	 */
	public function get_plugin_path() {
		return untrailingslashit( plugin_dir_path( $this->get_file() ) );
	}


	/**
	 * Gets the main plugin file.
	 *
	 * @since 1.0.0
	 *
	 * @return string
	 */
	public function get_plugin_file() {

		$slug = dirname( plugin_basename( $this->get_file() ) );
		return trailingslashit( $slug ) . $slug . '.php';
	}


	/**
	 * Helper to get the plugin file.
	 *
	 * @since 1.0.0
	 *
	 * @return string the plugin version
	 */
	public function get_file() {
		return __FILE__;
	}


	/**
	 * Helper to get the plugin URL.
	 *
	 * @since 1.0.0
	 *
	 * @return string the plugin URL
	 */
	public function get_plugin_url() {
		return untrailingslashit( plugins_url( '/', $this->get_file() ) );
	}


	/**
	 * Helper to get the plugin version.
	 *
	 * @since 1.0.0
	 *
	 * @return string the plugin version
	 */
	public function get_version() {
		return self::VERSION;
	}


	/**
	 * Checks if a plugin is active.
	 *
	 * @since 1.0.0
	 *
	 * @param string $plugin_name plugin name, as the plugin-filename.php
	 * @return boolean true if the named plugin is installed and active
	 */
	public static function is_plugin_active( $plugin_name ) {

		$active_plugins = (array) get_option( 'active_plugins', array() );

		if ( is_multisite() ) {
			$active_plugins = array_merge( $active_plugins, array_keys( get_site_option( 'active_sitewide_plugins', array() ) ) );
		}

		$plugin_filenames = array();

		foreach ( $active_plugins as $plugin ) {

			if ( false !== strpos( $plugin, '/' ) ) {

				// normal plugin name (plugin-dir/plugin-filename.php)
				list( , $filename ) = explode( '/', $plugin );

			} else {

				// no directory, just plugin file
				$filename = $plugin;
			}

			$plugin_filenames[] = $filename;
		}

		return in_array( $plugin_name, $plugin_filenames );
	}


	/** Lifecycle methods ***************************************/


	/**
	 * Run every time. Used since the activation hook is not executed when updating a plugin.
	 *
	 * @since 1.0.0
	 */
	private function install() {

		// get current version to check for upgrade
		$installed_version = get_option( 'wc_memberships_role_handler_version' );

		// force upgrade to 1.0.0
		if ( ! $installed_version ) {
			$this->upgrade( '1.0.0' );
		}

		// upgrade if installed version lower than plugin version
		if ( -1 === version_compare( $installed_version, self::VERSION ) ) {
			$this->upgrade( self::VERSION );
		}

	}


	/**
	 * Perform any version-related changes.
	 *
	 * @since 1.0.0
	 *
	 * @param int $version the currently installed version of the plugin
	 */
	private function upgrade( $version ) {

		// update the installed version option
		update_option( 'wc_memberships_role_handler_version', $version );
	}


}

