<?php
/**
 * WooCommerce Memberships Role Handler
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future. If you wish to customize this plugin for your
 * needs please refer to https://www.skyverge.com/product/woocommerce-memberships-role-handler/
 * for more information.
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2017-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

namespace SkyVerge\WooCommerce\Memberships\RoleHandler;

defined( 'ABSPATH' ) or exit;

/**
 * Class \SkyVerge\WooCommerce\Memberships\RoleHandler\Admin
 *
 * @since 1.0.0
 */
class Admin {


	/**
	 * Admin constructor.
	 *
	 * @since 1.0.0
	 */
	public function __construct() {

		// add settings
		add_filter( 'wc_memberships_general_settings', array( $this, 'add_settings' ) );
	}


	/**
	 * Adds new plugin settings after the general Memberships settings.
	 *
	 * @since 1.0.0
	 *
	 * @param array $settings original Memberships settings
	 * @return array updated settings
	 */
	public function add_settings( $settings ) {

		return array_merge( $settings, $this->get_settings() );
	}


	/**
	 * Gets plugin settings
	 *
	 * @since 1.0.0
	 */
	protected function get_settings() {

		if ( ! function_exists( 'get_editable_roles' ) ) {
			require_once( ABSPATH . 'wp-admin/includes/user.php' );
		}

		$roles         = get_editable_roles();
		$roles_options = array();

		foreach ( $roles as $role => $details ) {
			$roles_options[ $role ] = translate_user_role( $details['name'] );
		}

		$settings = array(

			array(
				'title' => __( 'Member Roles', 'wcm-role-handler' ),
				'type'  => 'title',
				'id'    => 'wc_memberships_role_handler_options',
				/* translators: %1$s - <em>, %2$s - </em> */
				'desc'  => sprintf( __( 'Choose the default roles to assign to active and inactive members. Users with the %1$sadministrator%2$s or %1$sshop manager%2$s role will never be allocated these roles to prevent locking out admin users.', 'wcm-role-handler' ), '<em>', '</em>' ),
			),

			array(
				'name'     => __( 'Member Default Role', 'wcm-role-handler' ),
				'desc'     => __( 'When a membership is activated, new members will be assigned this role.', 'wcm-role-handler' ),
				'id'       => 'wc_memberships_role_handler_member_role',
				'css'      => 'min-width:150px;',
				'default'  => 'customer',
				'type'     => 'select',
				'options'  => $roles_options,
				'desc_tip' => true,
			),

			array(
				'name'     => __( 'Inactive Member Role', 'wcm-role-handler' ),
				'desc'     => __( 'If a member becomes inactive (e.g., the membership is cancelled or expired), the member will be assigned this role.', 'wcm-role-handler' ),
				'id'       => 'wc_memberships_role_handler_inactive_role',
				'css'      => 'min-width:150px;',
				'default'  => 'customer',
				'type'     => 'select',
				'options'  => $roles_options,
				'desc_tip' => true,
			),

			array( 'type' => 'sectionend', 'id' => 'wc_memberships_role_handler_options' ),
		);

		/**
		 * Filters the plugin settings.
		 *
		 * @since 1.0.0
		 *
		 * @param string[] $settings the plugin settings array
		 */
		return apply_filters( 'wc_memberships_role_handler_settings', $settings );
	}


}
