<?php
/**
 * WooCommerce Memberships Role Handler
 *
 * This source file is subject to the GNU General Public License v3.0
 * that is bundled with this package in the file license.txt.
 * It is also available through the world-wide-web at this URL:
 * http://www.gnu.org/licenses/gpl-3.0.html
 * If you did not receive a copy of the license and are unable to
 * obtain it through the world-wide-web, please send an email
 * to license@skyverge.com so we can send you a copy immediately.
 *
 * DISCLAIMER
 *
 * Do not edit or add to this file if you wish to upgrade to newer
 * versions in the future. If you wish to customize this plugin for your
 * needs please refer to https://www.skyverge.com/product/woocommerce-memberships-role-handler/
 * for more information.
 *
 * @author    SkyVerge
 * @copyright Copyright (c) 2017-2020, SkyVerge, Inc. (info@skyverge.com)
 * @license   http://www.gnu.org/licenses/gpl-3.0.html GNU General Public License v3.0
 */

defined( 'ABSPATH' ) or exit;

/**
 * Returns the One True Instance of \SkyVerge\WooCommerce\Memberships\RoleHandler\Plugin
 *
 * @since 1.0.0
 *
 * @return \SkyVerge\WooCommerce\Memberships\RoleHandler\Plugin
 */
function wc_memberships_role_handler() {
	return \SkyVerge\WooCommerce\Memberships\RoleHandler\Plugin::instance();
}


/**
 * Updates user role for non-shop employees and removes any previous role.
 *
 * @since 1.0.0
 *
 * @param int $user_id user ID for the updated user membership
 * @param string $from_role the role to move away from (if set)
 * @param string $to_role the new role to potentially move to
 */
function wc_memberships_role_handler_change_user_role( $user_id, $from_role, $to_role ) {

	$user = get_userdata( $user_id );

	// if there are no roles for this person, there's probably a reason
	if ( empty( $user->roles ) ) {
		return;
	}

	// never change a shop admin's role to avoid locking out admins
	if ( in_array( array( 'shop_manager', 'administrator' ), $user->roles ) ) {
		return;
	}

	/**
	 * Lets actors prevent a user role change if desired.
	 *
	 * @param bool $change_role whether role should change or not, return false to prevent change
	 * @param \WP_User $user user object being changed
	 * @param string $from_role the role that will be removed
	 * @param string $to_role the role that will be added
	 */
	if ( ! apply_filters( 'wc_memberships_role_handler_update_role', true, $user, $from_role, $to_role ) ) {
		return;
	}

	// ensure the user has the current role before moving away
	if ( in_array( $from_role, $user->roles, true ) ) {
		$user->remove_role( $from_role );
	}

	$user->add_role( $to_role );

	/**
	 * Fires when a user has had a role changed via membership status change.
	 *
	 * @param \WP_User $user user object being changed
	 * @param string $from_role the role that will be removed
	 * @param string $to_role the role that will be added
	 */
	do_action( 'wc_memberships_role_handler_updated_role', $user, $to_role, $from_role );
}
