<?php

namespace Ademti\WoocommerceMsrp\Admin;

use Ademti\WoocommerceMsrp\Dependencies\League\Container\Container;
use WC_Product;
use WP_Post;
use function defined;
use function dirname;
use function get_woocommerce_currency_symbol;
use function plugins_url;
use function sanitize_key;
use function sanitize_text_field;
use function update_post_meta;
use function wc_format_decimal;
use function wc_get_product;
use function wp_enqueue_script;
use function wp_unslash;
use const WC_PRODUCT_ADDONS_VERSION;
use const WOOCOMMERCE_MSRP_VERSION;

class Admin {

	// Dependencies.
	protected QuickEdit         $quick_edit;
	protected ProductListColumn $product_list_column;
	protected ImportExport      $import_export;

	/**
	 * @var string
	 */
	private string $plugin_url;

	/**
	 * @param  ImportExport  $import_export
	 * @param  QuickEdit  $quick_edit
	 * @param  ProductListColumn  $product_list_column
	 */
	public function __construct(
		ImportExport $import_export,
		QuickEdit $quick_edit,
		ProductListColumn $product_list_column
	) {
		$this->import_export       = $import_export;
		$this->quick_edit          = $quick_edit;
		$this->product_list_column = $product_list_column;
	}

	/**
	 * Run the class features.
	 *
	 * @return void
	 */
	public function run() {

		$this->plugin_url = dirname( plugins_url( '', __FILE__ ), 2 );

		// Register settings to the WooCommerce settings page
		add_filter( 'woocommerce_general_settings', [ $this, 'settings_array' ] );

		// Enqueue JS.
		add_action( 'admin_init', [ $this, 'enqueue_js' ] );

		// Add meta box to the product page.
		add_action( 'woocommerce_product_options_pricing', [ $this, 'product_meta_field' ] );

		// Show the fields in the variation data.
		add_action( 'woocommerce_variation_options_pricing', [ $this, 'variation_show_fields' ], 10, 3 );

		// Save the variation data.
		add_action( 'woocommerce_save_product_variation', [ $this, 'variation_save_fields' ], 10, 2 );

		// Save the main MSRP price information.
		add_action( 'save_post_product', [ $this, 'save_product' ], 10, 1 );
		add_action( 'save_post_product_variation', [ $this, 'save_product' ], 10, 1 );

		// Support composite products extension.
		add_action( 'woocommerce_composite_product_options_pricing', [ $this, 'product_meta_field' ] );

		// Support for bulk modifying MSRP price on variations
		add_action(
			'woocommerce_variable_product_bulk_edit_actions',
			[ $this, 'variable_product_bulk_edit_actions' ]
		);
		add_action(
			'woocommerce_bulk_edit_variations',
			[ $this, 'variable_product_bulk_edit_actions_cb' ],
			10,
			4
		);

		// Trigger other admin classes.
		$this->import_export->run();
		$this->quick_edit->run();
		$this->product_list_column->run();
	}

	/**
	 * Add the settings to the WooCommerce settings page
	 */
	public function settings_array( $settings ) {
		// Heading
		$settings[] = [
			'title' => __( 'MSRP pricing options', 'woocommerce_msrp' ),
			'type'  => 'title',
			'id'    => 'woocommerce_msrp',
			'desc'  => __( 'Options controlling when, and how to display MSRP pricing', 'woocommerce_msrp' ),
		];
		// Show always / only if different / never
		$settings[] = [
			'name'     => __( 'Show MSRP Pricing?', 'woocommerce_msrp' ),
			'desc'     => __( 'Choose when to display MSRP prices.', 'woocommerce_msrp' ),
			'id'       => 'woocommerce_msrp_status',
			'std'      => 'always',
			'type'     => 'select',
			'options'  => [
				'always'    => __( 'Always show MSRP', 'woocommerce_msrp' ),
				'different' => __( 'Show if different from your price', 'woocommerce_msrp' ),
				'higher'    => __( 'Show if your price is lower than the MSRP', 'woocommerce_msrp' ),
				'never'     => __( 'Never show MSRP', 'woocommerce_msrp' ),
			],
			'desc_tip' => true,
		];
		// Description - text field
		$settings[] = [
			'name'     => __( 'MSRP Labelling', 'woocommerce_msrp' ),
			'desc'     => __( 'MSRP prices will be labelled with this description', 'woocommerce_msrp' ),
			'id'       => 'woocommerce_msrp_description',
			'std'      => __( 'MSRP', 'woocommerce_msrp' ),
			'type'     => 'text',
			'desc_tip' => true,
		];
		// 'Show savings' options
		$settings[] = [
			'name'     => __( 'Show savings?', 'woocommerce_msrp' ),
			'desc'     => __( 'Whether to show the saving against MSRP', 'woocommerce_msrp' ),
			'id'       => 'woocommerce_msrp_show_savings',
			'std'      => 'no',
			'type'     => 'select',
			'desc_tip' => true,
			'options'  => [
				'no'         => __( 'No', 'woocommerce_msrp' ),
				'amount'     => __( 'As a monetary amount', 'woocommerce_msrp' ),
				'percentage' => __( 'As a percentage', 'woocommerce_msrp' ),
			],
		];
		$settings[] = [
			'type' => 'sectionend',
			'id'   => 'woocommerce_msrp',
		];

		return $settings;
	}

	/**
	 * Display the meta field for MSRP prices on the product page
	 */
	public function product_meta_field(): void {
		woocommerce_wp_text_input(
			[
				'id'          => '_msrp_price',
				'class'       => 'wc_input_price short',
				'label'       => __( 'MSRP Price', 'woocommerce_msrp' ) . ' (' . get_woocommerce_currency_symbol() . ')',
				'description' => '',
				'data_type'   => 'price',
			]
		);
	}

	/**
	 * Show the fields for editing the MSRP on the variations panel on the post edit screen
	 *
	 * @param int     $loop           Position in the loop.
	 * @param array   $variation_data Variation data.
	 * @param WP_Post $variation      Post data.
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 */
	public function variation_show_fields( $loop, $variation_data, $variation ) {
		$variation_product = wc_get_product( $variation->ID );
		$msrp              = $variation_product->get_meta( '_msrp' );
		$msrp              = ! empty( $msrp ) ? $msrp : '';
		require $this->get_template_path( 'variation_show_fields' );
	}

	/**
	 * Save MSRP values for variable products
	 *
	 * @param int $product_id The parent product ID (Unused)
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 */
	public function variation_save_fields( $product_id, $idx ) {
		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if ( ! isset( $_POST['variable_post_id'][ $idx ] ) ) {
			return;
		}
		$variation_id = (int) sanitize_key( $_POST['variable_post_id'][ $idx ] );
		if ( ! empty( $_POST['variable_msrp'][ $idx ] ) ) {
			$msrp = (float) $_POST['variable_msrp'][ $idx ];
			$msrp = wc_format_decimal( $msrp );
		} else {
			$msrp = '';
		}
		// phpcs:enable WordPress.Security.NonceVerification.Missing
		$variation = wc_get_product( $variation_id );
		$variation->update_meta_data( '_msrp', $msrp );
		$variation->save();
		// phpcs:enable WordPress.Security.NonceVerification.Missing
	}

	/**
	 * Save the product meta information
	 *
	 * @param int $product_id The product ID
	 */
	public function save_product( $product_id ) {
		// phpcs:disable WordPress.Security.NonceVerification.Missing
		// Verify if this is an auto save routine.
		// If it is our form has not been submitted, so we dont want to do anything
		if ( defined( 'DOING_AUTOSAVE' ) && DOING_AUTOSAVE ) {
			return;
		}

		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if ( ! isset( $_POST['_msrp_price'] ) ) {
			return;
		}
		$msrp    = wc_format_decimal( sanitize_text_field( wp_unslash( $_POST['_msrp_price'] ) ) );
		$product = wc_get_product( $product_id );
		$product->update_meta_data( '_msrp_price', $msrp );
		$product->save();
		// phpcs:enable WordPress.Security.NonceVerification.Missing
	}

	/**
	 * Render the MSRP bulk-action options in the dropdown.
	 */
	public function variable_product_bulk_edit_actions() {
		require $this->get_template_path( 'variable_product_bulk_edit_actions' );
	}

	/**
	 * Handler a request to perform a bulk action.
	 *
	 * Calls the relevant function depending on the action being requested.
	 */
	public function variable_product_bulk_edit_actions_cb( $bulk_action, $data, $product_id, $variations ): void {
		switch ( $bulk_action ) {
			case 'msrp_set_prices':
				$this->bulk_action_set_prices( $data, $product_id, $variations );
				break;
			case 'msrp_clear_prices':
				$this->bulk_action_clear_prices( $data, $product_id, $variations );
				break;
			default:
				break;
		}
	}

	/**
	 * Update a set of variations with a given MSRP price.
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 */
	private function bulk_action_set_prices( $data, $product_id, $variations ) {
		if ( ! isset( $data['value'] ) ) {
			return;
		}
		foreach ( $variations as $variation_id ) {
			$variation = wc_get_product( $variation_id );
			$variation->update_meta_data( '_msrp', $data['value'] );
			$variation->save();
		}
	}

	/**
	 * Clear the MSRP prices off a set of variations.
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 */
	private function bulk_action_clear_prices( $data, $product_id, $variations ) {
		foreach ( $variations as $variation_id ) {
			$variation = wc_get_product( $variation_id );
			$variation->delete_meta_data( '_msrp' );
			$variation->save();
		}
	}

	/**
	 * Load JS when needed.
	 */
	public function enqueue_js() {
		wp_enqueue_script(
			'woocommerce_msrp_admin',
			$this->plugin_url . '/js/admin.js',
			[ 'jquery' ],
			WOOCOMMERCE_MSRP_VERSION,
			true
		);
	}

	/**
	 * @param $template
	 *
	 * @return string
	 */
	private function get_template_path( $template ) {
		return dirname( __DIR__, 2 ) . '/templates/' . $template . '.php';
	}
}
