<?php

namespace Ademti\WoocommerceMsrp;

use Exception;
use WC_Product;
use function _x;
use function apply_filters;
use function defined;
use function esc_html;
use function floor;
use function get_option;
use function get_post_meta;
use function plugins_url;
use function round;
use function sprintf;
use function wc_get_price_to_display;
use function wc_get_product;
use function wc_price;
use function wp_enqueue_script;
use function wp_localize_script;
use function wp_register_script;

/**
 * @SuppressWarnings(PHPMD.ExcessiveClassComplexity)
 */
class Frontend {
	/**
	 * @var string
	 */
	private string $plugin_url;

	/**
	 * Script arguments to be added to JS via wp_localize_script()
	 */
	private array $script_args = [];

	/**
	 * Run the classes features.
	 */
	public function run(): void {

		$this->plugin_url = dirname( plugins_url( '', __FILE__ ) );

		if ( ! is_admin() ) {
			// Add hooks for JS and CSS
			add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_styles' ] );
			add_action( 'wp_enqueue_scripts', [ $this, 'enqueue_scripts' ] );
			add_action( 'wp_footer', [ $this, 'footer' ] );
		}

		// Make sure the information is available to JS.
		add_action( 'woocommerce_available_variation', [ $this, 'add_msrp_to_js' ], 10, 3 );

		// Single Product Page.
		add_action( 'woocommerce_single_product_summary', [ $this, 'show_msrp' ], 7, 0 );

		// Loop
		add_action( 'woocommerce_after_shop_loop_item_title', [ $this, 'show_msrp' ], 9, 0 );

		// Composite products extension.
		add_action( 'woocommerce_composite_before_price', [ $this, 'show_msrp' ], 7, 0 );

		// Grouped product table
		add_action( 'woocommerce_grouped_product_list_before_price', [ $this, 'show_grouped_msrp' ], 9 );
	}

	/**
	 * Adds MSRP information to the JS used for variations.
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 * @param  array  $variation_data
	 * @param  WC_Product  $product
	 * @param  WC_Product  $variation
	 *
	 * @return array
	 * @throws Exception
	 */
	public function add_msrp_to_js(
		array $variation_data,
		WC_Product $product,
		WC_Product $variation
	): array {
		$msrp = apply_filters(
			'woocommerce_msrp_product_msrp_frontend',
			$variation->get_meta( '_msrp' ),
			$variation->get_id()
		);
		if ( empty( $msrp ) ) {
			return $variation_data;
		}
		// Adjust for tax input/display settings.
		$msrp = wc_get_price_to_display( $variation, [ 'price' => $msrp ] );
		// Get the display settings we need.
		$msrp_show_savings = get_option( 'woocommerce_msrp_show_savings', 'no' );
		// Add the data to the JS data array.
		$variation_data['msrp']           = $msrp;
		$variation_data['msrp_html']      = wc_price( $msrp );
		$variation_data['non_msrp_price'] = wc_get_price_to_display(
			$variation,
			[ 'price' => $variation->get_price() ]
		);
		$variation_data['msrp_saving']    = $this->msrp_saving_html(
			$variation_data['msrp'],
			$variation_data['non_msrp_price'],
			$msrp_show_savings
		);

		return $variation_data;
	}

	/**
	 * Generate the markup to show the single price / msrp value pair.
	 *
	 * @param  float  $msrp  The MSRP price.
	 * @param  float  $price  The actual price.
	 * @param  string  $type  'no', 'amount', 'percentage'
	 *
	 * @return string        The saving as markup.
	 * @throws Exception
	 */
	private function msrp_saving_html( float $msrp, float $price, string $type ): string {
		if ( 'no' === $type ) {
			return '';
		}

		$saving = $this->get_msrp_saving( $msrp, $price, $type );
		if ( $saving === null ) {
			return '';
		}
		if ( 'percentage' === $type ) {
			$saving .= '%';
		} else {
			$saving = apply_filters(
				'woocommerce_msrp_saving_amount_html',
				wc_price( $saving ),
				$saving,
				$msrp,
				$price
			);
		}

		return sprintf(
			// translators: %s is the saving amount which may be a financial amount, or a percentage.
			__( 'You save %s', 'woocommerce_msrp' ),
			$saving
		);
	}

	/**
	 * Generate the markup to show the saving range for a variable product.
	 *
	 * @param  WC_Product  $current_product  The variable product.
	 * @param  string  $type
	 *
	 * @return string                       The saving as markup.
	 * @throws Exception
	 */
	private function msrp_saving_html_variable( WC_Product $current_product, string $type ): string {
		if ( 'no' === $type ) {
			return '';
		}
		$savings = $this->get_savings_for_variable_product( $current_product, $type );
		if ( empty( $savings ) ) {
			return '';
		}
		if ( 'percentage' === $type ) {
			if ( $savings[0] === $savings[1] ) {
				return sprintf(
				// translators: %1$d%% is the saving amount which is a percentage.
					__( 'You save %1$d%%', 'woocommerce_msrp' ),
					$savings[0]
				);
			}

			return sprintf(
			// translators: %1$d%% is the saving amount is a percentage.
				__( 'You save up to %1$d%%', 'woocommerce_msrp' ),
				$savings[1]
			);
		}
		if ( 'amount' === $type ) {
			if ( $savings[0] === $savings[1] ) {
				return sprintf(
				// translators: %s is the saving amount which may be a financial amount, or a percentage.
					__( 'You save %s', 'woocommerce_msrp' ),
					wc_price( $savings[0] )
				);
			}

			return sprintf(
			// translators: %1$s is the saving amount which may be a financial amount, or a percentage.
				__( 'You save up to %1$s', 'woocommerce_msrp' ),
				wc_price( $savings[1] )
			);
		}

		return '';
	}

	/**
	 * Enqueue javascript required to show MSRPs on variable products.
	 */
	public function enqueue_scripts(): void {
		$suffix = ( defined( 'WP_DEBUG' ) && WP_DEBUG ) ? '' : '.min';
		wp_register_script(
			'woocommerce_msrp',
			$this->plugin_url . "/js/frontend{$suffix}.js",
			[ 'jquery' ],
			WOOCOMMERCE_MSRP_VERSION,
			true
		);
		$this->script_args = [
			'msrp_status'      => get_option( 'woocommerce_msrp_status' ),
			'msrp_description' => get_option( 'woocommerce_msrp_description' ),
		];
	}

	/**
	 * Output our Javascript.
	 */
	public function footer(): void {
		wp_localize_script( 'woocommerce_msrp', 'woocommerce_msrp', $this->script_args );
		wp_enqueue_script( 'woocommerce_msrp' );
	}


	/**
	 * Enqueue frontend stylesheets
	 */
	public function enqueue_styles(): void {
		wp_enqueue_style(
			'woocommerce_msrp',
			$this->plugin_url . '/css/frontend.css',
			[], // FIXME - was dependent on wc-components, but that broke things??
			WOOCOMMERCE_MSRP_VERSION,
		);
	}


	/**
	 * Wrapper function to add markup required when showing the MSRP in a
	 * grouped product table list
	 *
	 * @param  WC_Product  $current_product
	 */
	public function show_grouped_msrp( WC_Product $current_product ): void {
		echo '<td class="woocommerce_msrp_price">';
		$this->show_msrp( $current_product );
		echo '</td>';
	}

	/**
	 * Get the MSRP for a non-variable product
	 *
	 * @param  WC_Product  $current_product  The product the MSRP is required for
	 *
	 * @return string                  The MSRP, or empty string
	 */
	public function get_msrp_for_single_product( WC_Product $current_product ): string {
		return apply_filters(
			'woocommerce_msrp_product_msrp_frontend',
			$current_product->get_meta( '_msrp_price' ),
			$current_product->get_id()
		);
	}

	/**
	 * Get the MSRP for a variable product. This will be the highest and
	 * lowest MSRP across all variations.
	 *
	 * @param  WC_Product  $current_product  The product the MSRP is required for
	 *
	 * @return array                  The MSRP range info, or empty array.
	 *
	 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
	 * @SuppressWarnings(PHPMD.NPathComplexity)
	 */
	public function get_msrp_for_variable_product( WC_Product $current_product ): array {
		$children = $current_product->get_children();
		if ( ! $children ) {
			$msrp = $this->get_msrp_for_single_product( $current_product );

			return [ $msrp, $msrp ];
		}
		$lowest_msrp  = '';
		$highest_msrp = '';
		foreach ( $children as $child_id ) {
			// FIXME - "CRUD" way of doing this is ....?
			$child_msrp = apply_filters(
				'woocommerce_msrp_product_msrp_frontend',
				get_post_meta( $child_id, '_msrp', true ),
				$child_id
			);
			if ( false === $child_msrp || '' === $child_msrp ) {
				continue;
			}
			$child_msrp = wc_get_price_to_display( $current_product, [ 'price' => $child_msrp ] );
			if ( empty( $lowest_msrp ) || $child_msrp < $lowest_msrp ) {
				$lowest_msrp = $child_msrp;
			}
			if ( empty( $highest_msrp ) || $child_msrp > $highest_msrp ) {
				$highest_msrp = $child_msrp;
			}
		}
		if ( '' === $lowest_msrp ) {
			return [];
		}

		return [ $lowest_msrp, $highest_msrp ];
	}

	/**
	 * Get the higher / lower savings bound for a variable product.
	 *
	 * Compares the actual price and MSRP price for each variation and
	 * returns the highest / lowest savings found.
	 *
	 * @param  WC_Product  $current_product  The variable product.
	 * @param  string  $type  Whether to compare actual
	 *                                       savings, or percentage savings.
	 *
	 * @return array                         Array containing the lowest and
	 *                                       highest saving amounts.
	 *
	 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
	 * @SuppressWarnings(PHPMD.NPathComplexity)
	 */
	public function get_savings_for_variable_product( WC_Product $current_product, string $type = 'amount' ): array {
		$children       = $current_product->get_children();
		$lowest_saving  = '';
		$highest_saving = '';
		foreach ( $children as $child_id ) {
			// Grab the child's MSRP price.
			// FIXME - "CRUD" way of doing this is ....?
			$msrp = apply_filters(
				'woocommerce_msrp_product_msrp_frontend',
				get_post_meta( $child_id, '_msrp', true ),
				$child_id
			);
			// Skip it if it doesn't have one
			if ( false === $msrp || '' === $msrp ) {
				continue;
			}
			// ... or if the product does not have a price.
			$child_product       = wc_get_product( $child_id );
			$child_product_price = $child_product->get_price();
			if ( false === $child_product_price || '' === $child_product_price ) {
				continue;
			}

			// Adjust MSRP for input/display preferences
			$msrp = wc_get_price_to_display( $child_product, [ 'price' => $msrp ] );
			// Grab the child's price.
			$selling_price = wc_get_price_to_display( $child_product, [ 'price' => $child_product_price ] );

			// Calculate the saving.
			if ( 'amount' === $type ) {
				$saving = round( $msrp - $selling_price, 2, PHP_ROUND_HALF_UP );
				if ( $saving <= 0 ) {
					continue;
				}
			} else {
				$saving = $msrp - $selling_price;
				if ( $saving <= 0 ) {
					continue;
				}
				$saving = floor( $saving / $msrp * 100 );
			}
			if ( empty( $lowest_saving ) || $saving < $lowest_saving ) {
				$lowest_saving = $saving;
			}
			if ( empty( $highest_saving ) || $saving > $highest_saving ) {
				$highest_saving = $saving;
			}
		}
		if ( '' === $lowest_saving ) {
			return [];
		}

		return [ $lowest_saving, $highest_saving ];
	}

	/**
	 * @param  WC_Product  $product
	 * @param  string  $show_logic
	 * @param  string  $label
	 * @param  string  $show_saving
	 *
	 * @return void
	 * @throws Exception
	 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
	 * @SuppressWarnings(PHPMD.NPathComplexity)
	 * @SuppressWarnings(PHPMD.ExcessiveMethodLength)
	 */
	public function show_msrp_with_options(
		WC_Product $product,
		string $show_logic,
		string $label,
		string $show_saving
	): void {
		$product_type = $product->get_type();

		if ( 'variable' === $product_type ) {
			$msrp = $this->get_msrp_for_variable_product( $product );
			if ( empty( $msrp ) ) {
				return;
			}
			$current_product_price = $product->get_price();

			$show_msrp = false;
			if ( 'always' === $show_logic ) {
				$show_msrp = true;
			}
			if ( 'different' === $show_logic &&
				( $current_product_price !== $msrp[0] ||
					$current_product_price !== $msrp[1] )
			) {
				$show_msrp = true;
			}
			if ( 'higher' === $show_logic &&
				$current_product_price < $msrp[0]
			) {
				$show_msrp = true;
			}
			if ( apply_filters( 'woocommerce_msrp_hide_variation_price_ranges', false ) ) {
				$show_msrp = false;
			}

			if ( ! apply_filters(
				'woocommerce_msrp_show_msrp_pricing',
				$show_msrp,
				$current_product_price,
				$msrp,
				$product
			) ) {
				return;
			}

			if ( $msrp[0] === $msrp[1] ) {
				// phpcs:disable WordPress.WP.I18n.NoEmptyStrings
				$price_string = sprintf(
					// translators: %1$s is the MSRP price string.
					_x( '%1$s', 'The MSRP price string', 'woocommerce_msrp' ),
					wc_price( $msrp[0] )
				);
				// phpcs:enable Generic.WhiteSpace.DisallowSpaceIndent.SpacesUsed
				$this->script_args['msrp_ranged'] = false;
			} else {
				// translators: %1$s and %2$s are the lower and higher MSRP prices respectively.
				$template                         = _x(
					'%1$s - %2$s',
					'First arg is lowest MSRP price for variables, second is the highest',
					'woocommerce_msrp'
				);
				$price_string                     = sprintf( $template, wc_price( $msrp[0] ), wc_price( $msrp[1] ) );
				$this->script_args['msrp_ranged'] = true;
			}
			echo '<div class="woocommerce_msrp">';
			$msrp_info = '<span class="woocommerce_msrp_price">' . $price_string . '</span>';
			if ( $show_saving !== 'no' ) {
				$msrp_info .= ' <div class="woocommerce_msrp_saving">' .
								$this->msrp_saving_html_variable( $product, $show_saving ) .
								'</div>';
			}

			// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped
			printf(
			// translators: %1$s is the MSRP label, %2$s is the MSRP price with savings
				esc_html( _x( '%1$s: %2$s', 'MSRP output for variable products', 'woocommerce_msrp' ) ),
				esc_html( $label ),
				$msrp_info
			);
			// phpcs:enable WordPress.Security.EscapeOutput.OutputNotEscaped
			echo '</div>';
		} else {
			$msrp = $this->get_msrp_for_single_product( $product );
			if ( empty( $msrp ) ) {
				return;
			}
			// Adjust MSRP for input/display preferences
			$msrp = wc_get_price_to_display( $product, [ 'price' => $msrp ] );

			$selling_price = wc_get_price_to_display(
				$product,
				[ ' price' => $product->get_price() ]
			);

			$show_msrp = false;
			if ( 'always' === $show_logic ) {
				$show_msrp = true;
			}
			if ( $selling_price !== $msrp && 'different' === $show_logic ) {
				$show_msrp = true;
			}
			if ( $selling_price < $msrp && 'higher' === $show_logic ) {
				$show_msrp = true;
			}

			if ( ! apply_filters(
				'woocommerce_msrp_show_msrp_pricing',
				$show_msrp,
				$selling_price,
				[ $msrp, $msrp ],
				$product
			) ) {
				return;
			}

			echo '<div class="woocommerce_msrp">';
			$msrp_info = '<span class="woocommerce_msrp_price">' . wc_price( $msrp ) . '</span>';
			if ( $show_saving !== 'no' ) {
				$msrp_info .= '<div class="woocommerce_msrp_saving">' .
								$this->msrp_saving_html( $msrp, $selling_price, $show_saving ) .
								'</div>';
			}

			// phpcs:disable WordPress.Security.EscapeOutput.OutputNotEscaped
			printf(
			// translators: %1$s is the MSRP label, %2$s is the MSRP price with savings
				esc_html( _x( '%1$s: %2$s', 'MSRP output for non-variable products', 'woocommerce_msrp' ) ),
				esc_html( $label ),
				$msrp_info
			);
			// phpcs:enable WordPress.Security.EscapeOutput.OutputNotEscaped
			echo '</div>';
		}
	}

	/**
	 * @param $product_id
	 * @param $show_logic
	 * @param $label
	 * @param $show_saving
	 *
	 * @return void
	 * @throws Exception
	 */
	public function show_msrp_in_block( int $product_id, string $show_logic, string $label, string $show_saving ) {
		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return;
		}
		$this->show_msrp_with_options( $product, $show_logic, $label, $show_saving );
	}

	/**
	 * @param  int  $product_id
	 *
	 * @return void
	 * @throws Exception
	 */
	public function show_default_msrp_in_block( int $product_id ) {
		$product = wc_get_product( $product_id );
		if ( ! $product ) {
			return;
		}
		$msrp_status       = get_option( 'woocommerce_msrp_status' );
		$msrp_description  = get_option( 'woocommerce_msrp_description' );
		$msrp_show_savings = get_option( 'woocommerce_msrp_show_savings', 'no' );
		$this->show_msrp_with_options( $product, $msrp_status, $msrp_description, $msrp_show_savings );
	}

	/**
	 * Show the MSRP on the frontend
	 *
	 * @param  WC_Product|null  $current_product
	 *
	 * @throws Exception
	 */
	public function show_msrp( $current_product = null ) {

		global $product;

		if ( ! $current_product ) {
			$current_product = $product;
		}

		$msrp_status       = get_option( 'woocommerce_msrp_status' );
		$msrp_description  = get_option( 'woocommerce_msrp_description' );
		$msrp_show_savings = get_option( 'woocommerce_msrp_show_savings', 'no' );

		$this->show_msrp_with_options( $current_product, $msrp_status, $msrp_description, $msrp_show_savings );
	}

	/**
	 * @param  float  $msrp
	 * @param  float  $price
	 * @param  string  $type
	 *
	 * @return float|null
	 */
	public function get_msrp_saving( float $msrp, float $price, string $type = 'amount' ): ?float {
		if ( $price > $msrp ||
			empty( $price ) ||
			empty( $msrp ) ||
			$msrp === $price ) {
			return null;
		}

		if ( $type === 'percentage' ) {
			$percentage = ( $msrp - $price ) / $msrp * 100;

			return apply_filters(
				'woocommerce_msrp_saving_percentage_html',
				floor( $percentage ),
				$percentage,
				$msrp,
				$price
			);
		}

		return $msrp - $price;
	}
}
