<?php

namespace Ademti\WoocommerceMsrp\Integrations;

use WC_Product_Addons_Helper;
use function add_filter;
use function esc_html;
use function get_option;
use function sprintf;

class ProductAddons {

	/**
	 * @var string
	 */
	private string $plugin_url;

	public function run() {
		$this->plugin_url = dirname( plugins_url( '', __FILE__ ), 2 );

		// Enqueue needed CSS.
		add_action( 'admin_init', [ $this, 'admin_init' ] );

		// Admin side features.
		add_action(
			'woocommerce_product_addons_panel_option_heading',
			[ $this, 'product_addon_option_heading' ],
			10,
			0
		);
		add_action(
			'woocommerce_product_addons_panel_option_row',
			[ $this, 'product_addon_option_row' ],
			10,
			4
		);
		add_action(
			'woocommerce_product_addons_after_adjust_price',
			[ $this, 'product_addon_price_field' ],
			10,
			2
		);
		add_filter( 'woocommerce_product_addons_save_data', [ $this, 'product_addon_save' ], 10, 2 );

		// Frontend features.
		// Add support for product add-ons extension (add-ons with options)
		add_filter( 'woocommerce_product_addons_option_price', [ $this, 'product_addons_show_options_msrp' ], 10, 4 );
		// Add support for product add-ons extension (add-ons without options)
		add_filter( 'woocommerce_product_addons_price', [ $this, 'product_addons_show_msrp' ], 10, 2 );
	}

	/**
	 * Load CSS when needed.
	 */
	public function admin_init() {
		wp_enqueue_style(
			'woocommerce_msrp_admin',
			$this->plugin_url . '/css/admin.css',
			[],
			WOOCOMMERCE_MSRP_VERSION
		);
	}

	/**
	 * Render the MSRP for add-ons WITHOUT options.
	 *
	 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
	 * @SuppressWarnings(PHPMD.NPathComplexity)
	 */
	public function product_addons_show_msrp( string $html, array $addon ): string {

		// User has chosen not to show MSRP, don't show it
		$msrp_status = get_option( 'woocommerce_msrp_status' );
		if ( empty( $msrp_status ) || 'never' === $msrp_status ) {
			return $html;
		}

		// Get the info we need.
		$raw_price = $addon['price'] ?? ( $addon['options'][0]['price'] ?? null );
		$raw_msrp  = $addon['msrp'] ?? ( $addon['options'][0]['msrp'] ?? null );

		if ( empty( $raw_msrp ) ) {
			return $html;
		}

		$raw_price = WC_Product_Addons_Helper::get_product_addon_price_for_display( $raw_price );
		$raw_msrp  = WC_Product_Addons_Helper::get_product_addon_price_for_display( $raw_msrp );

		$msrp_description = get_option( 'woocommerce_msrp_description' );
		$msrp_price_html  = ! empty( $raw_msrp ) ? wc_price( $raw_msrp ) : '';

		// Return the original markup if the settings preclude showing the MSRP.
		if ( 'different' === $msrp_status && $raw_price === $raw_msrp ) {
			return $html;
		}
		if ( 'higher' === $msrp_status && $raw_price >= $raw_msrp ) {
			return $html;
		}

		$html .= '<span class="woocommerce_msrp">';
		$html .= sprintf(
		// translators: %1$s is the MSRP label, %2$s is the MSRP price value.
			esc_html( _x( '(%1$s: %2$s)', 'MSRP for product add-ons without options', 'woocommerce_msrp' ) ),
			$msrp_description,
			'<span class="woocommerce_msrp_price">' . $msrp_price_html . '</span>'
		);
		$html .= '</span>';

		// Return the modified markup including MSRP.
		return $html;
	}

	/**
	 * Render the MSRP for add-ons WITH options.
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 * @SuppressWarnings(PHPMD.CyclomaticComplexity)
	 * @SuppressWarnings(PHPMD.NPathComplexity)
	 */
	public function product_addons_show_options_msrp( string $html, array $option, $idx, string $type ): string {
		// User has chosen not to show MSRP, don't show it
		$msrp_status = get_option( 'woocommerce_msrp_status' );
		if ( empty( $msrp_status ) || 'never' === $msrp_status ) {
			return $html;
		}

		// Get the info we need.
		$price_template = '<span class="woocommerce_msrp_price">%1$s</span>';

		$template = '<span class="woocommerce_msrp">' .
					esc_html(
					// translators: %1$s is the MSRP label, and %2$s is the MSRP value.
						_x(
							'%1$s: %2$s',
							'MSRP price for product add-ons with options',
							'woocommerce_msrp'
						)
					) .
					'</span>';

		if ( 'select' === $type ) {
			$price_template = '%1$s';
			$template       = esc_html(
			// translators: %1$s is the MSRP label, and %2$s is the MSRP value.
				_x(
					'%1$s: %2$s',
					'MSRP price for drop-down product add-ons',
					'woocommerce_msrp'
				)
			);
		}
		$raw_price = isset( $option['price'] ) ? $option['price'] : null;
		$raw_msrp  = isset( $option['msrp'] ) ? $option['msrp'] : null;
		if ( empty( $raw_msrp ) ) {
			return $html;
		}

		$raw_price = WC_Product_Addons_Helper::get_product_addon_price_for_display( $raw_price );
		$raw_msrp  = WC_Product_Addons_Helper::get_product_addon_price_for_display( $raw_msrp );

		$msrp_description = get_option( 'woocommerce_msrp_description' );
		$msrp_price_html  = ! empty( $option['msrp'] ) ? wc_price( $raw_msrp ) : '';

		// Check whether we should show the MSRP, and either return the original markup
		// if not, or the MSRP included markup.
		if ( 'different' === $msrp_status && $raw_price === $raw_msrp ) {
			return $html;
		}
		if ( 'higher' === $msrp_status && $raw_price >= $raw_msrp ) {
			return $html;
		}

		return $html . ' ' .
				sprintf(
					$template,
					$msrp_description,
					sprintf( $price_template, $msrp_price_html )
				);
	}

	/**
	 * Output a heading for the product add-ons table.
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 */
	public function product_addon_option_heading() {
		require $this->get_template_path( 'product_addon_option_heading_v3' );
	}

	/**
	 * Output the markup for an option in the product add-ons table.
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 * @SuppressWarnings(PHPMD.UnusedLocalVariable)
	 */
	public function product_addon_option_row( $post, $addon, $loop, $option ) {
		$msrp = $option['msrp'] ?? '';
		require $this->get_template_path( 'product_addon_option_row_v3' );
	}

	/**
	 * Output the MSRP field for add-ons which do not have options.
	 *
	 * @param $addon
	 * @param $loop
	 *
	 * @SuppressWarnings(PHPMD.UnusedLocalVariable)
	 */
	public function product_addon_price_field( $addon, $loop ) {
		if ( is_null( $loop ) ) {
			$loop = 0;
		}

		$msrp = isset( $addon['msrp'] ) ? $addon['msrp'] : '';
		require $this->get_template_path( 'product_addon_price_field' );
	}

	/**
	 * Save the MSRP for product addons if they've been passed in.
	 */
	public function product_addon_save( $data, $idx ) {
		// phpcs:disable WordPress.Security.NonceVerification.Missing
		if (
			defined( 'WC_PRODUCT_ADDONS_VERSION' ) &&
			version_compare( WC_PRODUCT_ADDONS_VERSION, '3.0.0', '>=' ) &&
			in_array(
				$data['type'],
				[
					'custom_text',
					'custom_textarea',
					'file_upload',
					'input_multiplier',
				],
				true
			)
		) {
			if ( isset( $_POST['product_addon_msrp'][ $idx ] ) ) {
				$data['msrp'] = (float) $_POST['product_addon_msrp'][ $idx ];
			}
		} elseif ( isset( $_POST['product_addon_option_msrp'][ $idx ] ) ) {
			// phpcs:ignore WordPress.Security.ValidatedSanitizedInput
			foreach ( $_POST['product_addon_option_msrp'][ $idx ] as $option_idx => $value ) {
				$data['options'][ (int) $option_idx ]['msrp'] = (float) $value;
			}
		}

		return $data;
		// phpcs:enable WordPress.Security.NonceVerification.Missing
	}

	/**
	 * @param $template
	 *
	 * @return string
	 */
	private function get_template_path( $template ) {
		return dirname( __DIR__, 2 ) . '/templates/' . $template . '.php';
	}
}
