<?php

namespace Ademti\WoocommerceMsrp\RestHandlers;

use WP_Post;
use WP_REST_Request;
use WP_REST_Response;

/**
 * Handles the MSRP price on products in the Legacy WooCommerce REST API.
 */
class LegacyRestHandler extends AbstractRestHandler {

	/**
	 * Run the REST handler.
	 */
	public function run(): void {
		// General REST support.
		parent::run();
		// REST API support for WooCommerce pre-v2.6.
		add_filter( 'woocommerce_rest_prepare_product', [ $this, 'rest_api_price_output' ], 10, 2 );
		add_action( 'woocommerce_rest_insert_product', [ $this, 'rest_api_maybe_update_msrp' ], 10, 2 );
		add_action(
			'woocommerce_rest_save_product_variation',
			[ $this, 'rest_api_maybe_update_msrp_variation' ],
			10,
			3
		);
		add_filter( 'woocommerce_rest_product_schema', [ $this, 'rest_api_product_schema' ], 99 );
	}

	/**
	 * Include MSRP Prices in REST API GET responses on products.
	 */
	public function rest_api_price_output( WP_REST_Response $response, WP_Post $post ): WP_REST_Response {
		$product = wc_get_product( $post );
		if ( 'variable' === $product->get_type() ) {
			if ( ! count( $response->data['variations'] ) ) {
				$response->data['msrp_price'] = $this->get_msrp_for_single_product( $product );

				return $response;
			}
			foreach ( $response->data['variations'] as $idx => $variation ) {
				if ( isset( $response->data['variations'][ $idx ]['msrp_price'] ) ) {
					continue;
				}
				$product = wc_get_product( $variation['id'] );
				$response->data['variations'][ $idx ]['msrp_price'] = $this->get_msrp_for_variation( $product );
			}

			return $response;
		}
		// Do nothing if we already have the data.
		if ( isset( $response->data['msrp_price'] ) ) {
			return $response;
		}
		if ( 'variation' === $product->get_type() ) {
			$response->data['msrp_price'] = $this->get_msrp_for_variation( $product );
		} else {
			$response->data['msrp_price'] = $this->get_msrp_for_single_product( $product );
		}

		return $response;
	}

	/**
	 * Allow MSRP prices to be updated via REST API.
	 */
	public function rest_api_maybe_update_msrp( WP_Post $post, WP_REST_Request $request ): void {
		if ( ! isset( $request['msrp_price'] ) ) {
			return;
		}
		$product = wc_get_product( $post );

		$key = '_msrp_price';
		if ( 'variation' === $product->get_type() ) {
			$key = '_msrp';
		}

		$product = wc_get_product( $post->ID );

		if ( empty( $request['msrp_price'] ) ) {
			$product->delete_meta_data( $key );
		} else {
			$product->update_meta_data( $key, $request['msrp_price'] );
		}

		$product->save();
	}

	/**
	 * @param $variation_id
	 * @param $menu_order
	 * @param $data
	 *
	 * @SuppressWarnings(PHPMD.UnusedFormalParameter)
	 */
	public function rest_api_maybe_update_msrp_variation( $variation_id, $menu_order, $data ) {
		$this->rest_api_maybe_update_msrp( get_post( $variation_id ), $data );
	}
}
