<?php

/**
 * Class WCPP_Print_Template
 *
 * @package WooCommerce_PickingPal
 *          This class assists with the rendering of print templates
 */
class WCPP_Print_Template {

	private $templates_config = array(
		'default'                      => array(
			'heading'        => 'PickingPal Document',
			'visibility_key' => 'PICK_WINDOW_VIEW',
		),
		'template_print_pick_list'     => array(
			'heading'        => 'Pick List',
			'visibility_key' => 'PICK_LIST_VIEW',
		),
		'template_print_packing_slip'  => array(
			'heading'        => 'Packing Slip',
			'visibility_key' => 'PACKING_SLIP_VIEW',
		),
		'template_print_pick_ticket'   => array(
			'heading'        => 'Pick Ticket',
			'visibility_key' => 'PICK_TICKET_VIEW',
		),
		'template_print_unpicked_slip' => array(
			'heading'        => 'Unpicked Slip',
			'visibility_key' => 'PACKING_SLIP_VIEW',
		),
	);
	protected $template_id    = null;
	protected $template_data  = null;

	/**
	 * @param string     $template_id The template ID
	 * @param WC_Order[] $orders Optional Array of WC Order objects
	 */
	public function __construct( $template_id, $orders = null ) {
		$this->template_id = $template_id;
		$this->load_template_data();
		if ( $orders ) {
			$this->load_orders_data( $orders );
		}
	}

	/**
	 * Renders the template
	 *
	 * @return void
	 * @throws Exception
	 */
	public function render_template() {
		WCPP_Admin_Helper::wcpp_render_template(
			$this->template_id,
			array(
				'data' => $this->get_template_data(),
			)
		);
		$this->render_print_js();
	}

	/**
	 * Returns the data object used by the template
	 *
	 * @return WCPP_Print_Template_Data
	 * @throws Exception
	 */
	public function get_template_data() {
		return new WCPP_Print_Template_Data( $this->template_data );
	}

	/**
	 * Add data to pass to the template
	 *
	 * @param string $key
	 * @param mixed  $value
	 *
	 * @return void
	 * @throws Exception Will throw an exception if the key already exists.
	 */
	public function add_template_data( $key, $value ) {
		if ( isset( $this->template_data[ $key ] ) ) {
			throw new Exception( 'Template data key already exists: ' . $key );
		}
		$this->template_data[ $key ] = $value;
	}

	private function load_template_data() {
		$template_heading    = $this->get_templates_config_value( 'heading' );
		$v_control           = WCPP_Visibility_Controller::init();
		$v_key               = constant( 'WCPP_Visibility_Controller::' . $this->get_templates_config_value( 'visibility_key' ) );
		$warehouse_locations = array_filter(
			WCPP()->settings_handler->get_warehouse_locations( 'only_enabled' ),
			function ( $data, $key ) use ( $v_control, $v_key ) {
				return $v_control->is_visible( $key, $v_key );
			},
			ARRAY_FILTER_USE_BOTH
		);
		$this->template_data = array(
			'date_generated'           => date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ), current_time( 'timestamp' ) ),
			'url_template_css'         => $this->get_templates_css_url(),
			'tax_display'              => WCPP()->settings_handler->get_option( 'show_taxes' ) ? 'incl' : '',
			'warehouse_locations'      => $warehouse_locations,
			'additional_skus'          => array_filter(
				WCPP()->settings_handler->get_skus( 'only_enabled' ),
				function ( $data, $key ) use ( $v_control, $v_key ) {
					return $v_control->is_visible( $key, $v_key );
				},
				ARRAY_FILTER_USE_BOTH
			),
			'logo'                     => WCPP()->settings_handler->get_option( 'url_file' ),
			'show_product_images'      => $v_control->is_visible( $v_control::PRODUCT_IMAGE_FEATURE, $v_key ),
			'show_warehouse_locations' => $warehouse_locations && count( $warehouse_locations ),
			'show_customer_notes'      => $v_control->is_visible( $v_control::CUSTOMER_ORDER_NOTES_FEATURE, $v_key ),
			'contact_info'             => $contact_info = WCPP()->settings_handler->get_option( 'contact' ),
			'contact_info_formatted'   => nl2br( $contact_info ),
			'heading'                  => $template_heading,
		);
	}

	private function load_orders_data( $orders ) {
		$data = array();
		foreach ( $orders as $order ) {
			$items_data             = array();
			$backordered_items_data = array();
			$total_items_value      = 0.00;
			$total_items_tax        = 0.00;
			$refunded_items_data    = array();
			$picker_helper          = new WCPP_Picker_Data( $order->get_id() );
			$picker_data            = $picker_helper->get_data();
			$product_data           = $picker_data['products'];
			foreach ( $product_data as $item_data ) {
				$item_data = $this->apply_visibility_to_order_item_data( $item_data );
				if ( $item_data['backordered'] ) {
					$backordered_items_data[] = new WCPP_Print_Template_Data( $item_data );
				}
				if ( $item_data['is_digital'] || ! $item_data['qty_required'] ) {
					continue;
				}
				if ( $item_data['qty_required'] <= $item_data['backordered'] ) {
					continue;
				}
				$items_data[]       = new WCPP_Print_Template_Data( $item_data );
				$total_items_value += (float) $item_data['value'];
				$total_items_tax   += (float) $item_data['tax'];
			}
			$order_date_created = $order->get_date_created();
			$order_vars         = array(
				'id'                             => $order->get_id(),
				'order_number'                   => $order->get_order_number(),
				'barcode_id'                     => '(' . $order->get_order_number() . ')',
				'billing_phone'                  => $order->get_billing_phone(),
				'shipping_phone'                 => $order->get_shipping_phone(),
				'billing_address_formatted'      => $order->get_formatted_billing_address(),
				'shipping_address_formatted'     => $order->get_formatted_shipping_address(),
				'shipping_method'                => $order->get_shipping_method(),
				'order_total'                    => $order->get_total(),
				'order_total_formatted'          => $order->get_formatted_order_total( '', false ),
				'order_total_refunded'           => $order->get_total_refunded(),
				'order_total_refunded_formatted' => wc_price( $order->get_total_refunded() ),
				'order_date'                     => $order_date_created ?: '',
				'order_date_formatted'           => $order_date_created ? $order_date_created->date_i18n( get_option( 'date_format' ) . ' ' . get_option( 'time_format' ) ) : '',
				'customer_notes'                 => $order->get_customer_note() ? $order->get_customer_note() : '---',
				'items_data'                     => $items_data,
				'backordered_items_data'         => $backordered_items_data,
				'items_value'                    => $total_items_value,
				'items_value_formatted'          => wc_price( $total_items_value ),
				'items_tax'                      => $total_items_tax,
				'items_tax_formatted'            => wc_price( $total_items_tax ),
				'shipping_value'                 => $order->get_shipping_total(),
				'shipping_value_formatted'       => wc_price( $order->get_shipping_total() ),
				'shipping_tax'                   => $order->get_shipping_tax(),
				'shipping_tax_formatted'         => wc_price( $order->get_shipping_tax() ),
				'order_obj'                      => $order,
			);

			$data[ $order_vars['id'] ] = new WCPP_Print_Template_Data( $order_vars );
		}
		$this->template_data['orders_data'] = $data;
		$this->load_combined_items_data( $data );
	}

	/**
	 * Takes all items found in the templates order_data and groups them by product ID.
	 * The results will be assigned to `items_data` key.
	 *
	 * @param array $orders_data
	 *
	 * @return void
	 */
	private function load_combined_items_data( $orders_data ) {
		$combined_items_data           = array();
		$combined_items_qty            = 0;
		$combined_items_qty_required   = 0;
		$combined_items_qty_picked     = 0;
		$combined_items_qty_overpicked = 0;
		$combined_items_bundle_qty     = 0;
		$bin_number                    = 0;
		foreach ( $orders_data as $order_id => $order_data ) {
			++$bin_number;
			foreach ( $order_data->get_var( 'items_data' ) as $item_data_obj ) {
				$product_id                     = $item_data_obj->get_var( 'product_id' );
				$item_data                      = $item_data_obj->toArray();
				$combined_items_qty            += $item_data['qty'];
				$combined_items_qty_required   += $item_data['qty_required'];
				$combined_items_qty_picked     += $item_data['qty_picked'];
				$combined_items_qty_overpicked += $item_data['qty_overpicked'];
				$combined_items_bundle_qty     += $item_data['bundle_qty'];
				if ( ! isset( $combined_items_data[ $product_id ] ) ) {
					$combined_items_data[ $product_id ] = $item_data;
				} else {
					$combined_items_data[ $product_id ]['qty']            += $item_data['qty'];
					$combined_items_data[ $product_id ]['qty_required']   += $item_data['qty_required'];
					$combined_items_data[ $product_id ]['qty_picked']     += $item_data['qty_picked'];
					$combined_items_data[ $product_id ]['qty_overpicked'] += $item_data['qty_overpicked'];
					$combined_items_data[ $product_id ]['bundle_qty']     += $item_data['bundle_qty'];
				}
				// Add bin splits
				$combined_items_data[ $product_id ]['bins'][ $bin_number ] = new WCPP_Print_Template_Data(
					array(
						'id'           => $order_id,
						'qty'          => $item_data['qty'],
						'qty_required' => $item_data['qty_required'],
					)
				);
			}
		}
		if ( ( count( $orders_data ) > 1 ) && WCPP()->settings_handler->get_option( 'sort_by_warehouse_location' ) ) {
			$combined_items_data = WCPP()->helper->sort_array_empty_last( $combined_items_data, 'warehouse' );
		}

		foreach ( $combined_items_data as $key => $items_data ) {
			$combined_items_data[ $key ] = new WCPP_Print_Template_Data( $items_data );
		}

		$this->template_data['combined_items_data']           = $combined_items_data;
		$this->template_data['combined_items_qty']            = $combined_items_qty;
		$this->template_data['combined_items_qty_required']   = $combined_items_qty_required;
		$this->template_data['combined_items_qty_picked']     = $combined_items_qty_picked;
		$this->template_data['combined_items_qty_overpicked'] = $combined_items_qty_overpicked;
		$this->template_data['combined_items_bundle_qty']     = $combined_items_bundle_qty;
	}

	private function render_print_js() {
		echo '
		<script>
			if(document.images.length){ //wait for images to load
				Promise.all(Array.from(document.images).filter(img => !img.complete).map(img => new Promise(resolve => { img.onload = img.onerror = resolve; }))).then(() => {
					window.print();
				});
			} else {
				window.print();
			}
		</script>
		';
	}

	private function get_templates_css_url() {
		if ( file_exists( get_theme_file_path( 'woocommerce/woocommerce-pickingpal-template/templates.css' ) ) ) {
			return get_theme_file_uri( 'woocommerce/woocommerce-pickingpal-template/templates.css' );
		}
		return WCPP()->plugin_templates_url() . '/templates.css';
	}

	private function get_templates_config_value( $key ) {
		$config_id = isset( $this->templates_config[ $this->template_id ] ) ? $this->template_id : 'default';
		return isset( $this->templates_config[ $config_id ][ $key ] ) ? $this->templates_config[ $config_id ][ $key ] : null;
	}

	private function apply_visibility_to_order_item_data( $item_data ) {

		if ( isset( $this->template_data['warehouse_locations'] ) && isset( $item_data['warehouses'] ) ) {
			foreach ( $item_data['warehouses'] as $wh_key => $wh_value ) {
				if ( ! isset( $this->template_data['warehouse_locations'][ $wh_key ] ) ) {
					unset( $item_data['warehouses'][ $wh_key ] );
				}
			}
			$item_data['warehouses_formatted'] = implode(
				';',
				array_map(
					function ( $item ) {
						return $item['label'] . ':' . $item['value'];
					},
					$item_data['warehouses']
				)
			);
			// legacy key
			$item_data['warehouse'] = $item_data['warehouses_formatted'];
		}

		if ( isset( $this->template_data['additional_skus'] ) && isset( $item_data['additional_skus'] ) ) {
			foreach ( $item_data['additional_skus'] as $as_key => $as_data ) {
				if ( ! isset( $this->template_data['additional_skus'][ $as_key ] ) ) {
					unset( $item_data['additional_skus'][ $as_key ] );
				}
			}
			$item_data['additional_skus_formatted'] = implode(
				';',
				array_map(
					function ( $item ) {
						return $item['label'] . ':' . $item['value'];
					},
					$item_data['additional_skus']
				)
			);
		}

		return $item_data;
	}
}
