<?php

class WCPP_Settings {
	protected static $_instance = null;
	private $meta_key           = 'picking-pal-options';
	private $options;
	private $path_to_views;

	public function __construct() {
		if(WCPP()->is_lite()){
			$this->meta_key = $this->meta_key.'-lite';
		}
		$this->options       = $this->get_options();
		$this->path_to_views = untrailingslashit( plugin_dir_path( WCPP_PLUGIN_FILE ) . '/view/settings' );
	}

	public static function getInstance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}

		return self::$_instance;
	}

	public function get_options() {
		if ( $this->options ) {
			return $this->options;
		}

		return array_merge(
			$this->get_options_info( 'default' ),
			$this->load_options()
		);
	}

	public function get_option( $key ) {
		return isset( $this->options[ $key ] ) ? $this->options[ $key ] : null;
	}

	public function save_option( $key, $value ) {
		if ( ! isset( $this->options[ $key ] ) ) {
			return false;
		}

		$validated_value = $this->validate_value( $key, $value );

		if ( $validated_value ) {
			$this->options[ $key ] = $validated_value;

			return update_option( $this->meta_key, $this->options );
		}

		return false;
	}

	public function clear_option( $key ) {
		if ( ! isset( $this->options[ $key ] ) ) {
			return false;
		}

		unset( $this->options[ $key ] );

		return update_option( $this->meta_key, $this->options );
	}

	public function save_options( $input ) {
		$saved_options   = $this->get_options();
		$options_to_save = array_merge( $this->get_options_info( 'default' ), $this->validate( $input ) );

		$this->options = array_merge( $saved_options, $options_to_save );

		return update_option( $this->meta_key, array_merge( $saved_options, $options_to_save ) );
	}

	public function get_render_option_info() {
		$result = array();
		foreach ( $this->get_options_info() as $option ) {
			if ( ! empty( $option['key'] ) && ! empty( $option['section'] ) && ! empty( $option['view_path'] ) ) {
				$result[ $option['section'] ][] = array(
					'view_path' => $option['view_path'],
					'key'       => $option['key'],
				);
			}
		}

		return $result;
	}

	private function get_path_to_view( $view_name ) {
		return $this->path_to_views . '/' . $view_name . '.php';
	}

	private function get_options_info( $array_key = false ) {
		$options = array();

		$options[] = array(
			'key'                 => 'url_file',
			'name'                => 'url_file',
			'default'             => '',
			'deprecated_meta_key' => 'pickingpal-logo',
			'validate'            => FILTER_VALIDATE_URL,
			'view_path'           => $this->get_path_to_view( 'url_file' ),
			'section'             => 'company_info',
		);

		$options[] = array(
			'key'                 => 'contact',
			'name'                => 'contact',
			'default'             => '',
			'deprecated_meta_key' => 'pickingpal-contact',
			'view_path'           => $this->get_path_to_view( 'contact' ),
			'section'             => 'company_info',
		);

		$options[] = array(
			'key'                 => 'show_taxes',
			'name'                => 'pickingpal-show_taxes',
			'default'             => false,
			'deprecated_meta_key' => 'pickingpal-show_taxes',
			'validate'            => FILTER_VALIDATE_BOOLEAN,
			'view_path'           => $this->get_path_to_view( 'show_taxes' ),
			'section'             => 'company_info',
		);

		$options[] = array(
			'key'       => 'qty_override_sku',
			'view_path' => $this->get_path_to_view( 'qty_override_sku' ),
			'section'   => 'advanced',
		);

		$options[] = array(
			'key'                 => 'metakey_skus',
			'name'                => 'sku',
			'default'             => array(),
			'deprecated_meta_key' => '_wc_pickingpal_skus',
			'validate'            => null,
			'view_path'           => $this->get_path_to_view( 'metakey_skus' ),
			'section'             => 'advanced',
		);

		$options[] = array(
			'key'                 => 'warehouse_locations',
			'name'                => 'location',
			'default'             => array(),
			'deprecated_meta_key' => '_wc_pickingpal_warehouse_locations',
			'validate'            => null,
			'view_path'           => $this->get_path_to_view( 'warehouse_locations' ),
			'section'             => 'advanced',
		);

		$options[] = array(
			'key'                 => 'sort_by_warehouse_location',
			'name'                => 'pickingpal-sort_by_warehouse_location',
			'default'             => false,
			'deprecated_meta_key' => 'pickingpal-sort_by_warehouse_location',
			'validate'            => FILTER_VALIDATE_BOOLEAN,
			'view_path'           => $this->get_path_to_view( 'warehouse_locations' ),
			'section'             => 'advanced',
		);

		$options[] = array(
			'key'                 => 'active_fields',
			'name'                => 'active_fields',
			'default'             => array(),
			'deprecated_meta_key' => 'wc-picking-pal-active-fields',
			'validate'            => null,
			'view_path'           => $this->get_path_to_view( 'fields' ),
			'section'             => 'export',
		);

		$options[] = array(
			'key'                 => 'fields',
			'name'                => 'fields',
			'default'             => array(),
			'deprecated_meta_key' => 'wc-picking-pal-fields',
			'validate'            => null,
			'view_path'           => $this->get_path_to_view( 'fields' ),
			'section'             => 'export',
		);

		$options[] = array(
			'key'       => 'visibility',
			'name'      => 'visibility',
			'default'   => array(),
			'view_path' => $this->get_path_to_view( 'visibility' ),
			'section'   => 'advanced',
		);

		$options = apply_filters( 'wcpp_options_info', $options );

		if ( $array_key ) {
			$new_options = array();
			foreach ( $options as $option ) {
				if ( ! empty( $option['key'] ) && isset( $option[ $array_key ] ) ) {
					$new_options[ $option['key'] ] = $option[ $array_key ];
				}
			}

			return $new_options;
		}

		return $options;
	}

	private function load_options() {
		$options = get_option( $this->meta_key );

		// if starts first time, load old options
		if ( ! $options ) {
			$options = array();
			foreach ( $this->get_options_info() as $option_info ) {
				if ( isset( $option_info['deprecated_meta_key'] ) ) {
					if ( $deprecated_option = get_option( $option_info['deprecated_meta_key'] ) ) {
						$options[ $option_info['key'] ] = $deprecated_option;
					}
				}
			}
		}

		return $options;
	}

	private function validate_value( $key, $value ) {
		$names      = $this->get_options_info( 'name' );
		$validators = $this->get_options_info( 'validate' );

		if ( isset( $names[ $key ] ) ) {
			if ( isset( $validators[ $key ] ) ) {
				return filter_var( $value, $validators[ $key ] );
			} else {
				return $value;
			}
		}

		return $value;
	}

	private function validate( $input ) {
		$result     = array();
		$names      = $this->get_options_info( 'name' );
		$validators = $this->get_options_info( 'validate' );

		foreach ( $names as $key => $input_key ) {
			if ( isset( $input[ $input_key ] ) ) {
				if ( is_array( $input[ $input_key ] ) ) {
					$input[ $input_key ] = array_filter( $input[ $input_key ] );
				}

				if ( isset( $validators[ $key ] ) ) {
					if ( filter_var( $input[ $input_key ], $validators[ $key ] ) ) {
						$result[ $key ] = $input[ $input_key ];
					}
				} else {
					$result[ $key ] = $input[ $input_key ];
				}
			}
		}

		return $result;
	}

	public function get_warehouse_locations( $filter = 'available' ) {
		$allowed_filters = array( 'all', 'only_enabled', 'available' );
		if ( ! in_array( $filter, $allowed_filters ) ) {
			$filter = 'available';
		}

		$warehouse_locations = $this->get_option( 'warehouse_locations', array() );
		$warehouse_locations = array_filter(
			$warehouse_locations,
			function ( $e ) use ( $filter ) {
				if ( empty( $e['enabled'] ) ) {
					return false;
				}

				if ( 'all' == $filter ) {
					return true;
				} elseif ( 'only_enabled' == $filter || 'available' == $filter ) {
					return $e['enabled'] == 'on';
				}

				return false;
			}
		);

		return $warehouse_locations;
	}

	public function get_skus( $filter = 'available' ) {
		$allowed_filters = array( 'all', 'only_enabled', 'available' );
		if ( ! in_array( $filter, $allowed_filters ) ) {
			$filter = 'available';
		}

		$metakey_skus = $this->get_option( 'metakey_skus', array() );
		$metakey_skus = array_filter(
			$metakey_skus,
			function ( $e ) use ( $filter ) {
				if ( empty( $e['enabled'] ) ) {
					return false;
				}

				if ( 'all' == $filter ) {
					return true;
				} elseif ( 'only_enabled' == $filter || 'available' == $filter ) {
					return $e['enabled'] == 'on';
				}

				return false;
			}
		);

		return $metakey_skus;
	}
}
