<?php
/**
 * Installation related functions and actions.
 */

defined( 'ABSPATH' ) || exit;

/**
 * WCPP_Install Class.
 */
class WCPP_Install {

	const KNOWN_VARIANTS = array(
		'woocommerce-pickingpal-lite',
		'woocommerce-pickingpal-pro',
		'woocommerce-pickingpal',
	);

	/**
	 * DB updates and callbacks that need to be run per version.
	 *
	 * @var array
	 */
	private static $db_updates = array(
		'2.1.0' => array(
			'wc_pp_update_210_db_version',
		),
		'2.2.0' => array(
			'wc_pp_update_220_migrate_visibility_options',
			'wc_pp_update_220_migrate_log_table',
			'wc_pp_update_220_migrate_switch_order_status',
		),
	);

	/**
	 * Background update class.
	 *
	 * @var object
	 */
	private static $background_updater;

	/**
	 * Hook in tabs.
	 */
	public static function init() {
		add_action( 'init', array( __CLASS__, 'check_version' ), 5 );
		add_action( 'init', array( __CLASS__, 'init_background_updater' ), 5 );
	}

	/**
	 * Init background updates
	 */
	public static function init_background_updater() {
		include_once __DIR__ . '/class-wcpp-background-updater.php';
		self::$background_updater = new WCPP_Background_Updater();
	}

	/**
	 * Check version and run the updater is required.
	 *
	 * This check is done on all requests and runs if the versions do not match.
	 */
	public static function check_version() {
		if ( ! defined( 'IFRAME_REQUEST' ) && version_compare( get_option( 'pickingpal_version' ), WCPP_VERSION, '!=' ) ) {
			self::install();
			do_action( 'pickingpal_updated' );
		}
	}

	/**
	 * Install WCPP.
	 */
	public static function install() {
		if ( ! is_blog_installed() ) {
			return;
		}

		// Check if we are not already running this routine.
		if ( 'yes' === get_transient( 'wcpp_installing' ) ) {
			return;
		}

		// Before we activate this plugin, deactivate all other pickingpal plugins
		self::deactivate_pp_plugins();

		// If we made it till here nothing is running yet, lets set the transient now.
		set_transient( 'wcpp_installing', 'yes', MINUTE_IN_SECONDS * 10 );
		wc_maybe_define_constant( 'WCPP_INSTALLING', true );

		self::create_tables();
		self::update_wcpp_version();
		self::maybe_update_db_version();

		delete_transient( 'wcpp_installing' );
	}

	/**
	 * Update version to current.
	 */
	private static function update_wcpp_version() {
		delete_option( 'pickingpal_version' );
		add_option( 'pickingpal_version', WCPP_VERSION );
	}

	/**
	 * Is a DB update needed?
	 *
	 * @since  3.2.0
	 * @return boolean
	 */
	private static function needs_db_update() {
		$current_db_version = get_option( 'pickingpal_db_version' );
		$updates            = self::get_db_update_callbacks();

		return version_compare( $current_db_version, max( array_keys( $updates ) ), '<' );
	}


	/**
	 * See if we need to show or run database updates during install.
	 *
	 * @since 3.2.0
	 */
	private static function maybe_update_db_version() {
		if ( self::needs_db_update() ) {
			self::init_background_updater();
			self::update();
		} else {
			self::update_db_version();
		}
	}

	/**
	 * Get list of DB update callbacks.
	 *
	 * @since  3.0.0
	 * @return array
	 */
	public static function get_db_update_callbacks() {
		return self::$db_updates;
	}

	/**
	 * Push all needed DB updates to the queue for processing.
	 */
	private static function update() {
		$current_db_version = get_option( 'pickingpal_db_version' );
		$update_queued      = false;

		foreach ( self::get_db_update_callbacks() as $version => $update_callbacks ) {
			if ( version_compare( $current_db_version, $version, '<' ) ) {
				foreach ( $update_callbacks as $update_callback ) {
					self::$background_updater->push_to_queue( $update_callback );
					$update_queued = true;
				}
			}
		}

		if ( $update_queued ) {
			self::$background_updater->save()->dispatch();
		}
	}

	/**
	 * Update DB version to current.
	 *
	 * @param string|null $version New DB version or null.
	 */
	public static function update_db_version( $version = null ) {
		delete_option( 'pickingpal_db_version' );
		add_option( 'pickingpal_db_version', is_null( $version ) ? WCPP_VERSION : $version );
	}


	/**
	 * Set up the database tables which the plugin needs to function.
	 *
	 * Tables:
	 *      pp_pick_order_status
	 */
	private static function create_tables() {
		global $wpdb;

		$wpdb->hide_errors();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$log_table_name = $wpdb->prefix . 'pp_pick_order_status';
		if ( ! $wpdb->get_var( "SHOW TABLES LIKE '$log_table_name';" ) ) {
			dbDelta( self::get_log_table_schema( $log_table_name ) );
		}

		WCPP_Order_Actions_Log_Store::create_tables();
	}

	/**
	 * Get Table schema.
	 *
	 * @param string $log_table_name
	 *
	 * @return string
	 */
	private static function get_log_table_schema( $log_table_name ) {
		global $wpdb;

		$collate = '';

		if ( $wpdb->has_cap( 'collation' ) ) {
			$collate = $wpdb->get_charset_collate();
		}

		$log_table = "
				CREATE TABLE {$log_table_name} (
					id BIGINT UNSIGNED NOT NULL auto_increment,
					order_id BIGINT UNSIGNED NOT NULL,
					last_picked_by BIGINT UNSIGNED NOT NULL,
					date_scanned longtext,
					date_last_picked longtext,
					processing tinyint(1) NOT NULL DEFAULT '1',
					PRIMARY KEY (id),
					KEY order_id (order_id),
					KEY last_picked_by (last_picked_by)
				) $collate;
		";

		return $log_table;
	}

	/**
	 * Return a list of Picking Pal tables.
	 *
	 * @return array tables.
	 */
	public static function get_tables() {
		global $wpdb;

		$tables = array(
			"{$wpdb->prefix}pp_pick_order_status",
		);

		return $tables;
	}

	/**
	 * Drop tables.
	 *
	 * @return void
	 */
	public static function drop_tables() {
		global $wpdb;

		$tables = self::get_tables();

		foreach ( $tables as $table ) {
			$wpdb->query( "DROP TABLE IF EXISTS {$table}" ); // phpcs:ignore WordPress.WP.PreparedSQL.NotPrepared
		}

		WCPP_Order_Actions_Log_Store::drop_tables();
	}

	/**
	 * This method is used to disable all installed pickingpal plugins before
	 * activating this plugin.
	 *
	 * @return void
	 */
	private static function deactivate_pp_plugins() {
		$variants = self::KNOWN_VARIANTS;
		foreach ( $variants as $variant ) {
			deactivate_plugins( $variant . '/woocommerce-pickingpal.php' );
		}
	}
}

WCPP_Install::init();
