<?php
if ( ! defined( 'ABSPATH' ) ) {
	exit;
} // Exit if accessed directly

class WCPP_Order_Actions_Log_Store {
	/**
	 * @var int|null
	 */
	protected $id;

	/**
	 * @var int|null
	 */
	protected $order_id;

	/**
	 * @var int|null
	 */
	protected $user_id;

	/**
	 * @var int|null
	 */
	protected $action;
	const ACTION_LOAD_ORDER  = 1;
	const ACTION_PICK_ITEM   = 2;
	const ACTION_RESET_ORDER = 3;
	const ACTION_PICK_ORDER  = 4;

	/**
	 * GMT time in MySQl format
	 *
	 * @var string|null
	 */
	protected $action_date_gmt;

	/**
	 * @var array
	 */
	protected $data;

	/**
	 * @param int|null $id
	 */
	public function __construct( $id = null ) {
		$this->data = array();

		if ( is_int( $id ) ) {
			$this->load_using_id( $id );
		}
	}

	public static function get_table_name() {
		global $wpdb;

		return $wpdb->prefix . 'pp_order_actions_log';
	}

	public static function create_tables() {
		global $wpdb;

		$wpdb->hide_errors();

		require_once ABSPATH . 'wp-admin/includes/upgrade.php';

		$log_table_name = self::get_table_name();
		if ( $wpdb->get_var( "SHOW TABLES LIKE '$log_table_name';" ) ) {
			return;
		}

		$collate = '';

		if ( $wpdb->has_cap( 'collation' ) ) {
			$collate = $wpdb->get_charset_collate();
		}

		$r = dbDelta(
			"
				CREATE TABLE {$log_table_name} (
					id BIGINT UNSIGNED NOT NULL auto_increment,
					order_id BIGINT UNSIGNED NOT NULL,
					user_id BIGINT UNSIGNED NOT NULL,
					action_date_gmt longtext,
					action tinyint(1) NOT NULL,
					data longtext,
					PRIMARY KEY (id),
					KEY order_id (order_id),
					KEY user_id (user_id)
				) $collate;
		"
		);
	}

	public static function drop_tables() {
		global $wpdb;

		$table_name = self::get_table_name();
		$wpdb->query( "DROP TABLE IF EXISTS {$table_name}" ); // phpcs:ignore WordPress.WP.PreparedSQL.NotPrepared
	}

	/**
	 * @param int $action
	 *
	 * @return string
	 */
	public static function get_action_label( $action ) {
		if ( ! is_int( $action ) ) {
			return '';
		}

		$all_actions = array(
			self::ACTION_LOAD_ORDER,
			self::ACTION_PICK_ITEM,
			self::ACTION_RESET_ORDER,
			self::ACTION_PICK_ORDER,
		);

		if ( ! in_array( $action, $all_actions, true ) ) {
			return '';
		}

		if ( $action === self::ACTION_LOAD_ORDER ) {
			return __( 'Load order', 'woocommerce-pickingpal' );
		} elseif ( $action === self::ACTION_PICK_ITEM ) {
			return __( 'Pick item', 'woocommerce-pickingpal' );
		} elseif ( $action === self::ACTION_RESET_ORDER ) {
			return __( 'Reset', 'woocommerce-pickingpal' );
		} elseif ( $action === self::ACTION_PICK_ORDER ) {
			return __( 'Pick complete', 'woocommerce-pickingpal' );
		}

		return '';
	}

	/**
	 * @param int $order_id
	 */
	public function set_order_id( $order_id ) {
		if ( is_numeric( $order_id ) && ( $order_id = (int) $order_id ) ) {
			$this->order_id = $order_id;
		}
	}

	/**
	 * @return int|null
	 */
	public function get_order_id() {
		return $this->order_id;
	}

	/**
	 * @param WC_Order $order
	 */
	public function set_order( $order ) {
		if ( $order instanceof WC_Order ) {
			$this->set_order_id( $order->get_id() );
		}
	}

	/**
	 * @param int $user_id
	 */
	public function set_user_id( $user_id ) {
		if ( is_numeric( $user_id ) && ( $user_id = (int) $user_id ) ) {
			$this->user_id = $user_id;
		}
	}

	/**
	 * @param WP_User $user
	 */
	public function set_user( $user ) {
		if ( $user instanceof WP_User ) {
			$this->set_user_id( $user->ID );
		}
	}

	/**
	 * @return int|null
	 */
	public function get_user_id() {
		return $this->user_id;
	}

	/**
	 * @param int $action
	 */
	public function set_action( $action ) {
		if ( ! is_int( $action ) ) {
			return;
		}

		$all_actions = array(
			self::ACTION_LOAD_ORDER,
			self::ACTION_PICK_ITEM,
			self::ACTION_RESET_ORDER,
			self::ACTION_PICK_ORDER,
		);

		if ( in_array( $action, $all_actions, true ) ) {
			$this->action = $action;
		}
	}

	/**
	 * @return int|null
	 */
	public function get_action() {
		return $this->action;
	}

	public function get_humanized_action() {
		return self::get_action_label( $this->action );
	}

	/**
	 * @param string $date
	 */
	public function set_action_date_gmt( $date ) {
		if ( ! is_string( $date ) ) {
			return;
		}

		if ( (bool) DateTime::createFromFormat( 'Y-m-d H:i:s', $date ) ) {
			$this->action_date_gmt = $date;
		}
	}

	/**
	 * @return string|null
	 */
	public function get_action_date_gmt() {
		return $this->action_date_gmt;
	}

	/* ONLY PICK ITEM ACTION PROPS START */
	/**
	 * @param int $item_id
	 */
	public function set_order_item_id( $item_id ) {
		if ( is_numeric( $item_id ) && ( $item_id = (int) $item_id ) ) {
			$this->data['order_item_id'] = $item_id;
		}
	}

	/**
	 * @param WC_Order_Item_Product $order_item
	 */
	public function set_order_item( $order_item ) {
		if ( $order_item instanceof WC_Order_Item_Product ) {
			$this->set_order_item_id( $order_item->get_id() );
		}
	}

	/**
	 * @return int|null
	 */
	public function get_order_item_id() {
		return isset( $this->data['order_item_id'] ) ? $this->data['order_item_id'] : null;
	}

	/**
	 * @deprecated Use set_quantity_adjustment()
	 * @param float $qty
	 */
	public function set_quantity( $qty ) {
		if ( is_numeric( $qty ) && ( $qty = (float) $qty ) ) {
			$this->data['quantity'] = $qty;
		}
	}

	/**
	 * @deprecated Use get_quantity_adjustment()
	 * @return float|null
	 */
	public function get_quantity() {
		return isset( $this->data['quantity'] ) ? $this->data['quantity'] : null;
	}


	/**
	 * Log a quantity adjustment
	 *
	 * @param int $quantity The picked quantity adjustment for this action log
	 */
	public function set_quantity_adjustment( $quantity ) {
		if ( is_numeric( $quantity ) && ( $quantity = (float) $quantity ) ) {
			$this->data['action_quantity'] = $quantity;
		}
	}

	/**
	 * @return int The picked quantity adjustment for this action log
	 */
	public function get_quantity_adjustment() {
		$qty = isset( $this->data['action_quantity'] ) ? $this->data['action_quantity'] : null;
		if ( $qty === null ) {
			return (int) $this->get_quantity(); // old data pre 2.3.x
		}
		return (int) $qty;
	}

	/**
	 * Logs the total quantity picked at the time of this log
	 *
	 * @param int $quantity The total picked quantity at the time of this log
	 */
	public function set_total_picked_quantity( $quantity ) {
		if ( is_numeric( $quantity ) && ( $quantity = (float) $quantity ) ) {
			$this->data['total_picked_quantity'] = $quantity;
		}
	}

	/**
	 * @return int The total picked quantity at the time of this log
	 */
	public function get_total_picked_quantity() {
		$qty = isset( $this->data['total_picked_quantity'] ) ? $this->data['total_picked_quantity'] : null;
		if ( $qty === null ) {
			return null; // not recorded pre 2.3.x
		}
		return (int) $qty;
	}

	/* ONLY PICK ITEM ACTION PROPS FINISH */

	public function get_humanized_data() {
		$data = array();
		if ( $this->action === self::ACTION_PICK_ITEM ) {
			if ( $this->get_order_item_id() === null ) {
				return $data;
			}

			$order_item = new WC_Order_Item_Product( $this->get_order_item_id() );

			$data = array(
				'order_item_id'         => array(
					'label' => __( 'Order item', 'woocommerce-pickingpal' ),
					'value' => sprintf( '%s (%s)', $order_item->get_product()->get_name( 'edit' ), $this->get_order_item_id() ),
				),
				'quantity_adjustment'   => array(
					'label' => __( 'Picked Quantity', 'woocommerce-pickingpal' ),
					'value' => $this->get_quantity_adjustment(),
				),
				'total_picked_quantity' => array(
					'label' => __( 'Total Picked', 'woocommerce-pickingpal' ),
					'value' => $this->get_total_picked_quantity(),
				),
			);

			// Backwards compatibility for data stored < v2.3.0
			if ( $this->get_total_picked_quantity() === null ) {
				unset( $data['quantity_adjustment'] );
				unset( $data['total_picked_quantity'] );
				$data['quantity'] = array(
					'label' => __( 'Quantity', 'woocommerce-pickingpal' ),
					'value' => $this->get_quantity(),
				);
			}
		}

		return $data;
	}

	/**
	 * @return bool
	 */
	public function save() {
		return $this->id === null ? $this->create() : $this->update();
	}

	/**
	 * @return bool
	 */
	public function create() {
		global $wpdb;
		$table_name = self::get_table_name();

		$data = json_encode( $this->data );
		$data = $data !== false ? $data : '';

		$query = $wpdb->prepare(
			"INSERT INTO {$table_name} 
						(`order_id`, `user_id`, `action_date_gmt`,`action`, `data`) VALUES (%d, %d, %s, %d, %s)",
			$this->order_id,
			$this->user_id,
			$this->action_date_gmt,
			$this->action,
			$data
		);

		$insert_result = $wpdb->query( $query );

		return $insert_result;
	}

	/**
	 * @return bool
	 */
	private function update() {
		global $wpdb;
		$table_name = self::get_table_name();

		$data = json_encode( $this->data );
		$data = $data !== false ? $data : '';

		$query = $wpdb->prepare(
			"UPDATE {$table_name} SET
						`order_id` = %d, `user_id` = %d, `action_date_gmt` = %s,`action` = %d, `data` = %s WHERE {$table_name}.id = %d",
			$this->order_id,
			$this->user_id,
			$this->action_date_gmt,
			$this->action,
			$data,
			$this->id
		);

		$update_result = $wpdb->query( $query );

		return $update_result;
	}

	/**
	 * @param array $args
	 *
	 * @return array<int, self>
	 */
	public static function get( $args ) {
		global $wpdb;
		$table_name = self::get_table_name();

		$order_id_list = isset( $args['order_id'] ) && is_array( $args['order_id'] ) ? $args['order_id'] : array();
		$user_id_list  = isset( $args['user_id'] ) && is_array( $args['user_id'] ) ? $args['user_id'] : array();
		$action_list   = isset( $args['action'] ) && is_array( $args['action'] ) ? $args['action'] : array();

		$order_item_id_list = isset( $args['order_item_id'] ) && is_array( $args['order_item_id'] ) ? $args['order_item_id'] : array();

		$allowed_order_by = array( 'order_id', 'user_id', 'action_date_gmt', 'action' );
		$order_by         = isset( $args['order_by'] ) && in_array( $args['order_by'], $allowed_order_by, true ) ? $args['action'] : 'order_id';

		$order_id_where = count( $order_id_list ) > 0 ? sprintf( 'order_id IN (%s)', implode( ',', $order_id_list ) ) : '1=1';
		$user_id_where  = count( $user_id_list ) > 0 ? sprintf( 'user_id IN (%s)', implode( ',', $user_id_list ) ) : '1=1';
		$action_where   = count( $action_list ) > 0 ? sprintf( 'action IN (%s)', implode( ',', $action_list ) ) : '1=1';

		$query = $wpdb->prepare(
			"
						SELECT * FROM {$table_name} 
						WHERE %s AND %s AND %s ORDER BY %s
						",
			$order_id_where,
			$user_id_where,
			$action_where,
			$order_by
		);

		$result_rows = $wpdb->get_results( $query, ARRAY_A );
		if ( ! $result_rows ) {
			return array();
		}

		$results = array();
		foreach ( $result_rows as $row ) {
			$results[ $row->id ] = self::fill_from_row( new self(), $row );
		}

		return $results;
	}

	/**
	 * @param int $id
	 *
	 * @return bool
	 */
	protected function load_using_id( $id ) {
		global $wpdb;
		$table_name = self::get_table_name();

		$query  = $wpdb->prepare( "SELECT * FROM {$table_name} WHERE id = %s", $id );
		$result = $wpdb->get_row( $query, ARRAY_A );
		if ( ! $result ) {
			return false;
		}

		self::fill_from_row( $this, $result );

		return true;
	}

	/**
	 * @param self  $instance
	 * @param array $from_data
	 *
	 * @return self
	 */
	public static function fill_from_row( $instance, $from_data ) {
		$instance->id = (int) $from_data['id'];

		$instance->set_order_id( (int) $from_data['order_id'] );
		$instance->set_user_id( (int) $from_data['user_id'] );
		$instance->set_action_date_gmt( (string) $from_data['action_date_gmt'] );
		$instance->set_action( (int) $from_data['action'] );

		$data = json_decode( $from_data['data'], true );
		if ( is_array( $data ) ) {
			$instance->data = $data;
		}

		return $instance;
	}
}
