<?php

/**
 * Main WC Picking Pal Class
 *
 * @class WC_PickingPal
 */
class WC_PickingPal {
	protected $version  = '2.6.2';
	protected $is_pro   = null;
	protected $metakeys = array(
		'printed_pick_ticket' => '_wc_pick_ticket_printed',
		'picker_data'         => '_wc_pick_arr',
		'pickingpal_version'  => '_wc_pickingpal_version',
	);

	/**
	 * @var object The single instance of the class
	 */
	protected static $_instance            = null;
	protected static $api_manager_instance = null;

	/**
	 * @var
	 */
	public $override_sku = 'PP-OVRD-ITM-QTY';

	/**
	 * @var WCPP_Settings
	 */
	public $settings_handler;

	/**
	 * @var WCPP_Helper
	 */
	public $helper;

	/**
	 * @var
	 */
	public $pick_reset_override_sku = '';

	private $hpos_support = true;


	/**
	 * Main WC_PickingPal Instance
	 *
	 * Ensures only one instance of WooCommerce is loaded or can be loaded.
	 *
	 * @static
	 * @see WCPP()
	 * @return WC_PickingPal Main instance
	 */
	public static function instance() {
		if ( is_null( self::$_instance ) ) {
			self::$_instance = new self();
		}
		return self::$_instance;
	}


	/**
	 * WC_PickingPal Constructor
	 */
	public function __construct() {

		$this->init_api_manager(
			WCPP_PLUGIN_FILE,
			$this->get_app_product_id(),
			$this->get_version(),
			'plugin',
			$this->get_website(),
			$this->get_app_name()
		);
	}

	public function init() {
		$this->init_installer();
		$this->init_pick_order_status_store();
		$this->get_settings();
		$this->get_helper();
		$this->includes();
		$this->init_hooks();
		if ( $this->is_pro() ) {
			$this->init_pro_extension();
		}
	}

	private function get_settings() {
		include_once 'admin/class-wcpp-settings.php';
		$this->settings_handler = WCPP_Settings::getInstance();
	}

	private function get_helper() {
		include_once 'admin/class-wcpp-helper.php';
		$this->helper = new WCPP_Helper();
	}

	private function init_pick_order_status_store() {
		include_once 'admin/class-wcpp-order-actions-log-store.php';
		include_once 'admin/class-wcpp-order-extend.php';
	}

	private function init_installer() {
		include_once 'class-wcpp-install.php';
	}

	/**
	 * Include required files used in admin and on the frontend.
	 */
	private function includes() {
		include_once 'class-wcpp-visibility-controller.php';
		include_once 'interfaces/interface-wc-order-validate.php';
		include_once 'class-wc-order-validate-impl.php';

		if ( is_admin() ) {
			include_once 'admin/class-wcpp-admin.php';
			// must load this class after WooCommerce
			add_action(
				'plugins_loaded',
				function () {
					include_once 'class-wc-order-compat.php';
				}
			);
		} else {
			include_once 'class-wcpp-frontend.php';
		}
		do_action( 'wcpp_includes' );
	}

	/**
	 * Hook actions
	 */
	private function init_hooks() {
		add_action(
			'before_woocommerce_init',
			function () {
				// declare HPOS support
				if ( class_exists( \Automattic\WooCommerce\Utilities\FeaturesUtil::class ) ) {
					\Automattic\WooCommerce\Utilities\FeaturesUtil::declare_compatibility( 'custom_order_tables', WCPP_PLUGIN_FILE, $this->hpos_support );
				}
			}
		);
		add_action(
			'plugins_loaded',
			function () {
				load_plugin_textdomain( 'woocommerce-pickingpal', false, dirname( plugin_basename( WCPP_PLUGIN_FILE ) ) . '/i18n/languages' );
			}
		);
		do_action( 'wcpp_init_hooks' );
	}

	/**
	 * Get the plugin url.
	 *
	 * @return string
	 */
	public function plugin_url() {
		return untrailingslashit( plugins_url( '/', WCPP_PLUGIN_FILE ) );
	}

	/**
	 * Get the plugin main class path.
	 *
	 * @return string
	 */
	public function plugin_path() {
		return untrailingslashit( plugin_dir_path( WCPP_PLUGIN_FILE ) );
	}

	/**
	 * Get .js files path
	 *
	 * @return string
	 */
	public function plugin_js_url() {
		return untrailingslashit( $this->plugin_url() . '/assets/js/' );
	}

	/**
	 * Get .css files path
	 *
	 * @return string
	 */
	public function plugin_css_url() {
		return untrailingslashit( $this->plugin_url() . '/assets/css/' );
	}

	/**
	 * Get views files path
	 *
	 * @return string
	 */
	public function plugin_views_path() {
		return untrailingslashit( $this->plugin_path() . '/view/' );
	}

	/**
	 * Get template files path
	 *
	 * @return string
	 */
	public function plugin_templates_path() {
		return untrailingslashit( $this->plugin_path() . '/woocommerce-pickingpal-template/' );
	}

	/**
	 * Get template files url
	 *
	 * @return string
	 */
	public function plugin_templates_url() {
		return untrailingslashit( $this->plugin_url() . '/woocommerce-pickingpal-template/' );
	}

	/**
	 * Get template file path and url
	 *
	 * @@param string $template_name Name for check existing
	 * @@param bool $default_path
	 * @return array|bool Array with url and path
	 */
	public function return_url_and_path_existing_template( $template_name, $default_path = true ) {
		$theme_template = $this->find_theme_template( $template_name );
		if ( $theme_template ) {
			return $theme_template;
		}
		if ( ! $default_path ) {
			return false;
		}
		$template['url']  = $this->plugin_templates_url();
		$template['path'] = $this->plugin_templates_path();
		return $template;
	}

	/**
	 * Used to check if the theme has a template file override for our plugin.
	 *
	 * @param string $template_name
	 *
	 * @return array|null Array with url and path
	 */
	public function find_theme_template( $template_name ) {
		$template_details = null;
		if ( file_exists( trailingslashit( get_stylesheet_directory() ) . 'woocommerce/woocommerce-pickingpal-template/' . $template_name . '.php' ) ) {
			$template_details         = array();
			$template_details['url']  = trailingslashit( get_stylesheet_directory_uri() ) . 'woocommerce/woocommerce-pickingpal-template/';
			$template_details['path'] = trailingslashit( get_stylesheet_directory() ) . 'woocommerce/woocommerce-pickingpal-template/';
		}
		return $template_details;
	}

	/**
	 * Get images url
	 *
	 * @return string
	 */
	public function plugin_imgs_url() {
		return untrailingslashit( $this->plugin_url() . '/assets/img/' );
	}

	/**
	 * Get sounds url
	 *
	 * @return string
	 */
	public function plugin_sounds_url() {
		return untrailingslashit( $this->plugin_url() . '/assets/sounds/' );
	}

	/**
	 * Get sounds url
	 *
	 * @return string
	 */
	public function plugin_assets_url() {
		return untrailingslashit( $this->plugin_url() . '/assets/' );
	}

	/**
	 * Get Ajax URL.
	 *
	 * @return string
	 */
	public function ajax_url() {
		return admin_url( 'admin-ajax.php', 'relative' );
	}

	public function is_pro() {
		if ( $this->is_pro === null ) {
			if ( file_exists( $this->get_pro_extension_path() ) ) {
				$this->is_pro = true;
			} else {
				$this->is_pro = false;
			}
		}
		return $this->is_pro;
	}

	public function is_activated() {
		return self::get_api_manager()->is_activated_license();
	}

	public function is_hpos_enabled() {
		if ( $this->hpos_support ) {
			if ( class_exists( '\Automattic\WooCommerce\Utilities\OrderUtil' ) ) {
				return \Automattic\WooCommerce\Utilities\OrderUtil::custom_orders_table_usage_is_enabled();
			}
		}
		return false;
	}

	public function get_app_name() {
		return $this->is_pro() ? 'PickingPal Pro' : 'PickingPal';
	}

	public function get_app_product_id() {
		return $this->is_pro() ? 1513 : 1360;
	}

	public function get_app_product_name() {
		return $this->get_app_name() . ' for WooCommerce';
	}

	public function get_website() {
		return 'https://pickingpal.com/';
	}

	public function get_version() {
		return $this->version;
	}

	public static function get_api_manager() {
		return self::$api_manager_instance;
	}

	/**
	 * Returns the meta key from a given meta ID
	 * If the meta ID is not found, returns null
	 *
	 * @param string $id
	 *
	 * @return string|null
	 */
	public function get_meta_key( $id ) {
		if ( isset( $this->metakeys[ $id ] ) ) {
			return $this->metakeys[ $id ];
		}
		return null;
	}

	/**
	 * Handle deprecated properties
	 *
	 * @param string $name
	 * @return mixed
	 */
	public function __get( $name ) {
		$deprecated_metakey_properties = array(
			'metakey_printed'  => 'printed_pick_ticket',
			'metakey_pick_arr' => 'picker_data',
			'metakey_version'  => 'pickingpal_version',
		);
		if ( isset( $deprecated_metakey_properties[ $name ] ) ) {
			$metakey_id = $deprecated_metakey_properties[ $name ];
			trigger_error( "Accessing $name is deprecated, use get_meta_key('$metakey_id') method instead.", E_USER_DEPRECATED );
			return $this->get_meta_key( $metakey_id );
		}
	}

	protected function init_api_manager( $file, $product_id, $software_version, $plugin_or_theme, $api_url, $software_title = '', $text_domain = '' ) {
		$manager_class = 'WC_AM_Client';
		if ( ! class_exists( $manager_class ) ) {
			include_once 'wc-am-client.php';
		}
		if ( is_null( self::$api_manager_instance ) ) {
			self::$api_manager_instance = new $manager_class( $file, $product_id, $software_version, $plugin_or_theme, $api_url, $software_title, $text_domain );
		}
		return self::$api_manager_instance;
	}

	protected function get_pro_extension_path() {
		return plugin_dir_path( WCPP_PLUGIN_FILE ) . '/pro_version/class-wcpp-loader-pro.php';
	}

	protected function init_pro_extension() {
		if ( file_exists( $this->get_pro_extension_path() ) ) {
			require_once $this->get_pro_extension_path();
			new WCPP_Loader_Pro();
		}
	}
}


/**
 * Returns the main instance of WC_PickingPal to prevent the need to use globals.
 *
 * @return WC_PickingPal
 */
function WCPP() {
	return WC_PickingPal::instance();
}
