<?php

defined( 'ABSPATH' ) || exit;

class WCPP_Visibility_Controller {
	const UNKNOWN_STATUS      = 0;
	const VISIBLE_STATUS      = 1;
	const HIDE_STATUS         = - 1;
	const NO_SENSE_STATUS     = - 2; // having no sense at all
	const INACCESSIBLE_STATUS = - 3; // in accessible in base version

	const ALL_STATUSES = array(
		self::UNKNOWN_STATUS,
		self::VISIBLE_STATUS,
		self::HIDE_STATUS,
		self::NO_SENSE_STATUS,
		self::INACCESSIBLE_STATUS,
	);

	const PRODUCT_PAGE_VIEW = 'product_page';
	const PICK_TICKET_VIEW  = 'pick_ticket';
	const PICK_LIST_VIEW    = 'pick_list';
	const PACKING_SLIP_VIEW = 'packing_slip';
	const PICK_WINDOW_VIEW  = 'pick_window';

	const CUSTOMER_ORDER_NOTES_FEATURE = 'customer_order_notes';
	const PRODUCT_IMAGE_FEATURE        = 'product_image';

	protected $statuses;
	protected $additional_sku;
	protected $warehouses;

	public function __construct() {
		$this->statuses        = array();
		$this->additional_skus = array();
		$this->warehouses      = array();

		$this->load();
		$this->rewrite_no_sense_statuses();
		$this->rewrite_pro_statuses();
	}

	/**
	 * @return WCPP_Visibility_Controller
	 */
	public static function init() {
		if ( class_exists( 'WCPP_Visibility_Controller_Pro' ) ) {
			return new WCPP_Visibility_Controller_Pro();
		}

		return new self();
	}

	protected function load() {
		$visibility = WCPP()->settings_handler->get_option( 'visibility' );

		foreach ( WCPP()->settings_handler->get_skus( 'only_enabled' ) as $key => $sku_data ) {
			$this->additional_skus[ $key ] = ! empty( $sku_data['label'] ) ? $sku_data['label'] : $key;
		}

		foreach ( WCPP()->settings_handler->get_warehouse_locations( 'only_enabled' ) as $key => $location ) {
			$this->warehouses[ $key ] = ! empty( $location['label'] ) ? $location['label'] : $key;
		}

		foreach ( $this->get_all_features() as $what_item ) {
			$this->statuses[ $what_item ] = array();
			foreach ( $this->get_all_views() as $where_item ) {
				if ( isset( $visibility[ $what_item ][ $where_item ] ) ) {
					$status = $visibility[ $what_item ][ $where_item ];

					if ( array_search( $status, self::ALL_STATUSES, true ) === false ) {
						if ( $status === 'true' ) {
							$status = self::VISIBLE_STATUS;
						} elseif ( $status === 'false' ) {
							$status = self::HIDE_STATUS;
						} else {
							$status = self::UNKNOWN_STATUS;
						}
					}
				} else {
					$status = self::UNKNOWN_STATUS;
				}
				$this->statuses[ $what_item ][ $where_item ] = $status;
			}
		}
	}

	protected function rewrite_no_sense_statuses() {
		foreach ( array_keys( $this->warehouses ) as $key ) {
			$this->statuses[ $key ][ $this::PRODUCT_PAGE_VIEW ] = self::NO_SENSE_STATUS;
		}

		$this->statuses[ $this::CUSTOMER_ORDER_NOTES_FEATURE ][ $this::PRODUCT_PAGE_VIEW ] = self::NO_SENSE_STATUS;
		$this->statuses[ $this::CUSTOMER_ORDER_NOTES_FEATURE ][ $this::PICK_LIST_VIEW ]    = self::NO_SENSE_STATUS;

		$this->statuses[ $this::PRODUCT_IMAGE_FEATURE ][ $this::PRODUCT_PAGE_VIEW ] = self::NO_SENSE_STATUS;
	}

	protected function rewrite_pro_statuses() {
		foreach ( array_keys( $this->additional_skus ) as $key ) {
			$this->statuses[ $key ][ $this::PICK_TICKET_VIEW ]  = self::INACCESSIBLE_STATUS;
			$this->statuses[ $key ][ $this::PICK_LIST_VIEW ]    = self::INACCESSIBLE_STATUS;
			$this->statuses[ $key ][ $this::PACKING_SLIP_VIEW ] = self::INACCESSIBLE_STATUS;
			$this->statuses[ $key ][ $this::PICK_WINDOW_VIEW ]  = self::INACCESSIBLE_STATUS;
		}

		$this->statuses[ $this::CUSTOMER_ORDER_NOTES_FEATURE ][ $this::PICK_TICKET_VIEW ]  = self::INACCESSIBLE_STATUS;
		$this->statuses[ $this::CUSTOMER_ORDER_NOTES_FEATURE ][ $this::PACKING_SLIP_VIEW ] = self::INACCESSIBLE_STATUS;
		$this->statuses[ $this::CUSTOMER_ORDER_NOTES_FEATURE ][ $this::PICK_WINDOW_VIEW ]  = self::INACCESSIBLE_STATUS;

		$this->statuses[ $this::PRODUCT_IMAGE_FEATURE ][ $this::PICK_TICKET_VIEW ]  = self::INACCESSIBLE_STATUS;
		$this->statuses[ $this::PRODUCT_IMAGE_FEATURE ][ $this::PICK_LIST_VIEW ]    = self::INACCESSIBLE_STATUS;
		$this->statuses[ $this::PRODUCT_IMAGE_FEATURE ][ $this::PACKING_SLIP_VIEW ] = self::INACCESSIBLE_STATUS;
	}

	/**
	 * @param string $what
	 * @param string $where
	 *
	 * @return int
	 */
	public function get_status( $what, $where ) {
		return isset( $this->statuses[ $what ][ $where ] ) ? $this->statuses[ $what ][ $where ] : self::UNKNOWN_STATUS;
	}

	/**
	 * @param string $what
	 * @param string $where
	 * @param string $status
	 *
	 * @return bool
	 */
	public function set_status( $what, $where, $status ) {
		if ( ! isset( $this->statuses[ $what ][ $where ] ) ) {
			return false;
		}

		if ( ! in_array(
			$status,
			array(
				self::UNKNOWN_STATUS,
				self::VISIBLE_STATUS,
				self::HIDE_STATUS,
				self::INACCESSIBLE_STATUS,
			)
		) ) {
			return false;
		}

		$this->statuses[ $what ][ $where ] = $status;

		return true;
	}

	/**
	 * @return array
	 */
	public function get_statuses() {
		return $this->statuses;
	}

	/**
	 * @param string $what
	 * @param string $where
	 *
	 * @return bool
	 */
	public function is_visible( $what, $where ) {
		return $this->get_status( $what, $where ) === self::VISIBLE_STATUS;
	}

	public function get_view_label( $where ) {
		$labels = array(
			self::PRODUCT_PAGE_VIEW => __( 'Product Page', 'woocommerce-pickingpal' ),
			self::PICK_TICKET_VIEW  => __( 'Pick Ticket', 'woocommerce-pickingpal' ),
			self::PICK_LIST_VIEW    => __( 'Pick List', 'woocommerce-pickingpal' ),
			self::PACKING_SLIP_VIEW => __( 'Packing Slip', 'woocommerce-pickingpal' ),
			self::PICK_WINDOW_VIEW  => __( 'Pick Window', 'woocommerce-pickingpal' ),
		);

		return isset( $labels[ $where ] ) ? $labels[ $where ] : $where;
	}

	public function get_all_views() {
		return array(
			self::PRODUCT_PAGE_VIEW,
			self::PICK_TICKET_VIEW,
			self::PICK_LIST_VIEW,
			self::PACKING_SLIP_VIEW,
			self::PICK_WINDOW_VIEW,
		);
	}

	public function get_feature_label( $feature ) {
		$labels = array_merge(
			$this->additional_skus,
			$this->warehouses,
			array(
				self::CUSTOMER_ORDER_NOTES_FEATURE => __( 'Customer provided note', 'woocommerce-pickingpal' ),
				self::PRODUCT_IMAGE_FEATURE        => __( 'Product Image', 'woocommerce-pickingpal' ),
			)
		);

		return isset( $labels[ $feature ] ) ? $labels[ $feature ] : $feature;
	}

	public function get_all_features() {
		return array_merge(
			array_keys( $this->additional_skus ),
			array_keys( $this->warehouses ),
			array(
				self::CUSTOMER_ORDER_NOTES_FEATURE,
				self::PRODUCT_IMAGE_FEATURE,
			)
		);
	}
}
